// DOMContentLoaded is fired once the document has been loaded and parsed,
// but without waiting for other external resources to load (css/images/etc)
// That makes the app more responsive and perceived as faster.
// https://developer.mozilla.org/Web/Reference/Events/DOMContentLoaded

DEBUG = false;

function createIcon(iconURL) {
  var p = new Promise(function(resolve, reject) {
    var canvas, ctx, image;

    canvas = document.createElement('CANVAS');
    ctx = canvas.getContext('2d');
    image = new Image();

    image.crossOrigin = 'Anonymous';
    image.onload = function() {
      var dataURL;
      canvas.height = image.height;
      canvas.width = image.width;
      ctx.drawImage(image, 0, 0);

      // Define the image format here more dynamically
      dataURL = canvas.toDataURL('image/png');

      // Resolve the promise
      resolve(dataURL);

      // Clear the memory
      canvas = null; 
    };

    image.onerror = function() {
      reject(new Error('createIcon: Can not convert image to base64'));
    };

    // The image will load after setting an src attribute
    image.src = iconURL;
  });
  return p;
}

var old_console = console;
if (DEBUG === true) {
  console = function() {
    var self = this;
    var print = function() {
      for (i in arguments) {
        $('.b-log').append(arguments[i] + ' ');
      }
      $('.b-log').append('\n');
      old_console.log.apply(old_console, arguments);
    };
    self.log = print;
    self.info = print;
    self.error = print;
    return this;
  }()
}

window.addEventListener('DOMContentLoaded', function() {
  // We'll ask the browser to use strict code to help us catch errors earlier.
  // https://developer.mozilla.org/Web/JavaScript/Reference/Functions_and_function_scope/Strict_mode
  'use strict';

  var translate = navigator.mozL10n.get;

  // We want to wait until the localisations library has loaded all the strings.
  // So we'll tell it to let us know once it's ready.
  navigator.mozL10n.once(main);

  // ---

  function main() {
//     var message = document.getElementById('message');
//     // We're using textContent because inserting content from external sources into your page using innerHTML can be dangerous.
//     // https://developer.mozilla.org/Web/API/Element.innerHTML#Security_considerations
//     message.textContent = translate('message');
    var timer = new Timer();
    
    navigator.mozSetMessageHandler && navigator.mozSetMessageHandler("alarm", function (mozAlarm) {
      console.info("alarm fired: " + JSON.stringify(mozAlarm.data));
      timer.alert(mozAlarm.data);
    });

    document.addEventListener("visibilitychange", function() {
      console.log('visibility', document.visibilityState );
      if (!navigator.mozAlarms) {
        return;
      }
      if (document.visibilityState === 'visible') { //only for FxOS
        timer.startCountdown();
      } else {
        timer.stopCountdown();
      }
    })
    
    console.log('start app', timer);
    //TODO timer.continue() show progress window if progress was in place before window was closed
    $('.i-button.m-work').on('click', function(){
      console.log('start timer', 25);
      if (DEBUG === true) {
        timer.start(1*60, 'work');
      } else {
        timer.start(25*60, 'work');
      }
    });
    $('.i-button.m-pause').on('click', function(){
      console.log('start timer', 5);
      if (DEBUG === true) {
        timer.start(31, 'break');
      } else {
        timer.start(5*60, 'break');
      }
    });
    $('.c100').on('click', function(){
      timer.stop();
    });
  }

});

Timer = function() {
  var self = this;
  var el = $('.b-timer-screen');
  var el_progress = el.find('.c100');
  var el_time_left = el.find('.i-time-left');
  var el_time_total = el.find('.i-time-total');
  var timer,
    total_time,
    seconds_left,
    session,
    last_notification = null,
    alarm,
    old_date;

  self.seconds_left = 0;
  
  self.start = function(seconds, session_type) {
    clearInterval(timer);
    last_notification != null? last_notification.close() : null;
    total_time = seconds;
    self.seconds_left = seconds;
    session = session_type;
    localStorage.setItem('session_type', session_type);

    el.find('.i-button.m-stop').removeClass('mdi-navigation-check');
    el.find('.i-button.m-stop').addClass('mdi-av-stop');
    self.update(self.seconds_left, total_time);
    el.show();
    
    old_date = Date.now();
    self.startCountdown(self.seconds_left, total_time);
    console.log('start', el, timer);

    var dt  = new Date(Date.now() + total_time*1000 + 1);
    var data = {
      'session': session_type,
      'fire_date': dt,
      'start_date': Date.now()
    };
    if (navigator.mozAlarms) {
      var request = navigator.mozAlarms.add(dt, "ignoreTimezone", data);
      request.onsuccess = function () {
        alarm = this.result;
        console.info("The alarm has been scheduled", dt, alarm);
      };
      request.onerror = function () {
        console.error("An error occurred: " + this.error.name);
      };
    };
  };

  self.stop = function() {
    console.log('stop');
    el.hide();
    self.done(0, total_time);
    console.log('last_notification', last_notification);
    last_notification != null? last_notification.close() : null;
  };

  self.alert = function(data){
    console.log('alert', data);
    self.done(0, total_time);
    el.show();
    last_notification != null? last_notification.close() : null;
    var body, sound;
    var session = session || localStorage.getItem('session_type') || null;
    if (session === 'work') {
      body = "Take a break."
      sound = 'sounds/ding.ogg'
    } else if (session === 'break') {
      body = "Time to work!"
      sound = 'sounds/dingding.ogg'
    }
    self.toForeground();
    createIcon('img/icons/icon32x32.png').then(function(dataURL) {
      last_notification = new Notification('Pomidor session is over',{
        'body': body,
        'renotify': true,
        'sound': sound,
        'tag': 'PomidorEnded',
        'icon': dataURL
      });
      last_notification.onclick = function() {
        self.toForeground();
      }
    });
  };

  self.toForeground = function() {
    if (typeof(navigator.mozApps) !== 'object') {
      console.log('no mozApps');
      return;
    }
    var request = navigator.mozApps.getSelf();
    request.onsuccess = function (e) {
      var app = e.target.result;
      console.log('onsuccess', app, e, e.target, e.target.result, app.manifest.name);
      console.log('request', request, request.result, request.result.manifest.name)
      if (app) {
        app.launch(request.result.manifest.name);
      }
    };
  };
  
  self.done = function(seconds, total_seconds) {
    console.log('done');
    clearInterval(timer);
    self.seconds_left = 0;
    self.cancel_timer();

    el.find('.i-button.m-stop').removeClass('mdi-av-stop');
    el.find('.i-button.m-stop').addClass('mdi-navigation-check');

    var str = Math.trunc(seconds/60) + ':' + ((seconds%60 > 9) ? seconds%60 : '0'+seconds%60);
    el.find('.i-time-left').text(str);
    percent = 100;
    var el_progress = el.find('.c100');
    var old_progress = el_progress.attr('class').match(/p\d+/)[0];
    el_progress.removeClass(old_progress);
    el_progress.addClass('p' + percent);
  };

  self.cancel_timer = function() {
    console.log('removing', alarm);
    navigator.mozAlarms && navigator.mozAlarms.remove(alarm);
  };

  self.startCountdown = function(seconds_left, total_seconds) {
    if (self.seconds_left === 0) return;
    var now = Date.now();
    var diff = Math.round((now - old_date)/1000);
    console.log('seconds_left', self.seconds_left, diff);
    self.seconds_left -= diff;
    self.update(self.seconds_left, total_time);
    old_date = now;
    timer = setInterval(function(){
      var now = Date.now();
      var diff = Math.round((now - old_date)/1000);
      console.log('seconds_left', self.seconds_left, diff);
      self.seconds_left -= diff;
      self.update(self.seconds_left, total_time);
      old_date = now;
    }, 1000);
  };
  
  self.stopCountdown = function() {
    clearInterval(timer);
  };
  
  self.update = function(seconds, total_seconds) {
    console.log('update', seconds, total_seconds);
    if (seconds <= 0) {
      self.alert();
    }
    var str = Math.trunc(seconds/60) + ':' + ((seconds%60 > 9) ? seconds%60 : '0'+seconds%60);
    el_time_left.text(str);
    var str = Math.trunc(total_seconds/60) + ':' + ((total_seconds%60 > 9) ? total_seconds%60 : '0'+total_seconds%60);
    el_time_total.text(str);
    var percent = 100 - Math.round(seconds/total_seconds*100);
    percent = (percent == 0) ? percent = 1 : percent;
    var old_progress = el_progress.attr('class').match(/p\d+/)[0];
    el_progress.removeClass(old_progress);
    el_progress.addClass('p' + percent);
    console.log('update', str, percent);
  };

  return self;
};