function GameState(){    
}

GameState.DIRECT_NONE		= -1;
GameState.DIRECT_LEFT		= 0;
GameState.DIRECT_TOP_LEFT 	= 1;
GameState.DIRECT_TOP_RIGHT 	= 2;
GameState.DIRECT_RIGHT 		= 3;
GameState.DIRECT_BOTTOM_RIGHT   = 4;
GameState.DIRECT_BOTTOM_LEFT 	= 5;
GameState.STATE_DATA_SIZE       = 37;

GameState.prototype = (function(){
    var cells = [];
    
    for(var i = 0; i < 19; i++)
        cells[i] = new Cell();      
    
    var mStateCells = [];
    for(var i = 0; i < cells.length; i++)
        mStateCells[i] = 0;
    
    var mEmptyCellCount = 0;
    var mMovedCellCount = 0;
    var mMaxLevel = 0;
    var mMoveCount = 0;
    var mFirstScore = 0;
    var mSecondScore = 0;
    var mLastBlockLevel = 0;
    
    var mGameEnd = false;
    var mMergedCounts = [];
    
    for(var i = 0; i < 14; i++)
        mMergedCounts[i] = 0;
    
    /**
	 * 19 bytes - each byte represent level of a cell
	 * 1 byte - indicate that the game end or not
	 * 4 bytes -  the move count
	 * 4 bytes - the first score
	 * 4 bytes - the second score
	 * 1 byte - the direction of last move
	 * 2 bytes - 1 byte for position, 1 byte for level of new cell.
	 * Max to 2 new cell
	 */
    var mState = new Int8Array(GameState.STATE_DATA_SIZE);
    
    var ROW_180 = [      [0, 1, 2], 
                      [ 3,  4,  5,  6],
                    [ 7,  8,  9, 10, 11],
                      [12, 13, 14, 15],
                        [16, 17, 18]    ];
                    
    var ROW_120 = [      [2, 6, 11], 
                      [ 1,  5,  10,  15],
                    [ 0,  4,  9, 14, 18],
                      [3, 8, 13, 17],
                        [7, 12, 16]    ];  
                    
    var ROW_60 = [      [0, 3, 7], 
                      [ 1,  4,  8,  12],
                    [ 2,  5,  9, 13, 16],
                      [6, 10, 14, 17],
                        [11, 15, 18]    ];
                    
    var MULTIPLIER = Math.sqrt(3);
         
    
    function removeEmptyCell (position){
        for(var i = 0; i < mEmptyCellCount; i++){
            if(mStateCells[i] === position){
                mStateCells[i] = mStateCells[mEmptyCellCount - 1];
		mEmptyCellCount--;
		return;
            }
	}
    }
    
    function addEmptyCell(position){
        for(var i = 0; i < mEmptyCellCount; i++){
            if(mStateCells[i] === position)
		return;
	}
		
	mStateCells[mEmptyCellCount] = position;
	mEmptyCellCount++;
    }
    
    function getRandomEmptyCell(removeEmptyCell){
        if(mEmptyCellCount === 0)
            return -1;
 		
	var index = getRandomInt(0, mEmptyCellCount);
	var position = mStateCells[index];
		
	if(removeEmptyCell){
            mStateCells[index] = mStateCells[mEmptyCellCount - 1];
            mEmptyCellCount--;
	}
		
	return position;
    }
    
    function getRandomInt(min, max) {
        return Math.floor(Math.random() * (max - min)) + min;
    }
    
    function addMovedCell(position){
        for(var i = 0; i < mMovedCellCount; i++)
            if(mStateCells[mStateCells.length - i - 1] === position)
		return;
	 		
 	mStateCells[mStateCells.length - mMovedCellCount - 1] = position;
 	mMovedCellCount++;
    }

    function getLevelState(levels){
        if(levels === null)
            levels = [];
 		
 	for(var i = 0; i < cells.length; i++)
            levels[i] = cells[i].level;
 		
        return levels;
    }
    
    function setLevelState(levels){
 	mEmptyCellCount = 0;
 	mMovedCellCount = 0;
 	mMaxLevel = 0;
 		
 	for(var i = 0; i < cells.length; i++){
            if(cells[i] === null)
		cells[i] = new Cell();
 			
            cells[i].type = Cell.TYPE_DEFAULT;
            cells[i].level = levels[i];
 			
            mMaxLevel = Math.max(mMaxLevel, cells[i].level);
 			
            if(cells[i].level === 0){
 		mStateCells[mEmptyCellCount] = i;
 		mEmptyCellCount++;
            }
 	}
    }

    function saveData(direction, newCellPos){
 	//write level state - 19 bytes
 	getLevelState(mState);
 		 		
 	//write game end - 1 byte
 	mState[19] = mGameEnd ? 1 : 0;
 		
 	//write move count, score - 4 bytes each
 	toBytes(mMoveCount, mState, 20);
 	toBytes(mFirstScore, mState, 24);
 	toBytes(mSecondScore, mState, 28);
 		
 	//write direction - 1 bytes
 	mState[32] = direction;
 		
 	//write position and level of new cell - 1 byte each
 	var index = 33;
 	for(var i = 0; i < newCellPos.length; i++){
            if(newCellPos[i] >= 0){
 		mState[index] = newCellPos[i];
 		mState[index + 1] = cells[newCellPos[i]].level;
 		index += 2;
            }
 	}
 		
 	for(var i = index; i < mState.length; i++)
            mState[i] = -1;
    }
    
    function isMovable(pos1, pos2){
 	if(cells[pos1].level > 0 && cells[pos2].level > 0)
            return cells[pos1].level === cells[pos2].level;
 		
 	if(cells[pos1].level === 0 || cells[pos2].level === 0)
            return cells[pos1].level + cells[pos2].level > 0;
 			
 	return cells[pos1].level + cells[pos2].level === 0;
    }

    function generateNewCell(){
        var position = getRandomEmptyCell(true);	
	if(position < 0)
            return -1;
		               
	//init value
	var cell = cells[position];	
	cell.level = getRandomInt(0, mMaxLevel > 3 ? 2 : 1) + 1;
	cell.type = Cell.TYPE_IN;
		
	mMaxLevel = Math.max(mMaxLevel, cell.level);
						
	return position;
    }
    
    function generateNewBlock(){		
        if(mMoveCount % Config.BLOCK_STEP !== 0)
            return -1;
				
	var maxLevel = Math.min(Config.LEVEL_BLOCK_MAX, mMaxLevel);
	if(maxLevel < Config.LEVEL_BLOCK_START)
            return -1;
				
	var position = getRandomEmptyCell(true);
	if(position < 0)
            return -1;
						
	//init value		
	var cell = cells[position];
	cell.type = Cell.TYPE_IN;
	cell.level = -(getRandomInt(0, maxLevel - Config.LEVEL_BLOCK_MIN + 1) + Config.LEVEL_BLOCK_MIN);
	//ensure that next block level and last block level aren't same
	if(cell.level === mLastBlockLevel){			
            if(cell.level === -maxLevel)
		cell.level = -Config.LEVEL_BLOCK_MIN;
            else
		cell.level -= 1;
	}
		
	mLastBlockLevel = cell.level;
		
	return position;
    }
        
    function findNonEmptyCol(cols, start, asc){
        if(asc)
            for(var i = start; i < cols.length; i++){
		if(cells[cols[i]].level !== 0)
                    return i;
            }
	else
            for(var i = start; i >= 0; i--){
		if(cells[cols[i]].level !== 0)
                    return i;
            }				
		
	return -1;
    }
    
    function findEmptyCol(cols, start, end, asc){
        var memo = -1;
		
	if(asc)	
            for(var i = end - 1; i >= start; i--){
		if(cells[cols[i]].level === 0)
                    memo = i;
		else
                    return memo;
            }
	else
            for(var i = end + 1; i <= start; i++){
		if(cells[cols[i]].level === 0)
                    memo = i;
		else
                    return memo;
            }		
		
        return memo;
    }
    
    function moveCellsByRow(seft, rows, asc){
        var result = 0;
		
	seft.clearMovedCell();
				
	for(var i = 0; i < rows.length; i++){
            var cols = rows[i];
            var col = asc ? -1 : cols.length;			
            var empty_col;
            var next_col;
			
            col = findNonEmptyCol(cols, asc ? col + 1 : col - 1, asc);
            
            while(col >= 0){
		var curCell = cells[cols[col]];
				
		empty_col = findEmptyCol(cols, asc ? 0 : cols.length - 1, col, asc);
		var emptyCell = empty_col < 0 ? null : cells[cols[empty_col]];
						
		next_col = findNonEmptyCol(cols, asc ? col + 1 : col - 1, asc);
		var nextCell = next_col < 0 ? null: cells[cols[next_col]];
					
		if(curCell.level < 0){ 		// if curCell is a block
                    if(nextCell === null) 	// finish this row, move to next row
			break;					
                    else if(curCell.level + nextCell.level === 0){ //if nextCell can remove curCell's block
			curCell.type = Cell.TYPE_MOVE_BLOCK;
			curCell.pos1 = cols[col];
			curCell.pos2 = cols[next_col];
			curCell.level = nextCell.level;
						
			nextCell.type = Cell.TYPE_DEFAULT;
			nextCell.level = 0;
						
			addEmptyCell(cols[next_col]);
			addMovedCell(cols[col]);
						
			col = next_col;
						
			result += Math.pow(Config.BASE_VALUE, curCell.level) * 4;
                    }
                    else	//process next col
                        col = asc ? next_col - 1 : next_col + 1;
		}
		else if(nextCell === null || curCell.level !== nextCell.level){	//if doesn't have any, or curCell and nextCell doesn't have same level				
                    if(emptyCell !== null){									//move curCell to emptyCell if possible						
			emptyCell.type = Cell.TYPE_MOVE;
			emptyCell.pos1 = cols[col];
			emptyCell.pos2 = -1;
			emptyCell.level = curCell.level;
						
			curCell.type = Cell.TYPE_DEFAULT;
			curCell.level = 0;
												
			removeEmptyCell(cols[empty_col]);
			addEmptyCell(cols[col]);
			addMovedCell(cols[empty_col]);
						
			mMaxLevel = Math.max(mMaxLevel, emptyCell.level);
                    }
                    else
			curCell.type = Cell.TYPE_DEFAULT;
					
                    if(nextCell === null) 	// finish this row, move to next row
                        break;
                    else			//process next col
                        col = asc ? next_col - 1 : next_col + 1;	
		}
		else{                                   //if curCell and nextCell have same level
                    if(emptyCell !== null){		//move curCell and nextCell to emptyCell if possible
			emptyCell.type = Cell.TYPE_MOVE;
			emptyCell.pos1 = cols[col];
			emptyCell.pos2 = cols[next_col];
			emptyCell.level = curCell.level + 1;
						
			curCell.type = Cell.TYPE_DEFAULT;
			curCell.level = 0;
						
			nextCell.type = Cell.TYPE_DEFAULT;
			nextCell.level = 0;
						
			removeEmptyCell(cols[empty_col]);
			addEmptyCell(cols[col]);
			addEmptyCell(cols[next_col]);
			addMovedCell(cols[empty_col]);
						
			mMaxLevel = Math.max(mMaxLevel, emptyCell.level);
			result += Math.pow(Config.BASE_VALUE, emptyCell.level);
                    }
                    else{						//if doesn't have emptyCell, then move nextCell to curCell
			curCell.type = Cell.TYPE_MOVE;
			curCell.pos1 = cols[col];
			curCell.pos2 = cols[next_col];
			curCell.level = nextCell.level + 1;
						
			nextCell.type = Cell.TYPE_DEFAULT;
			nextCell.level = 0;
					
			addEmptyCell(cols[next_col]);
			addMovedCell(cols[col]);
						
			mMaxLevel = Math.max(mMaxLevel, curCell.level);
			result += Math.pow(Config.BASE_VALUE, curCell.level);
                    }
					
                    col = next_col;
		}
                
                col = findNonEmptyCol(cols, asc ? col + 1 : col - 1, asc);
            }			
        }	
		
	var count = seft.getMovedCellCount();		
	if(count > 0){
            mMoveCount++;
            if(mMoveCount % 2 === 0)
		mSecondScore += result;
            else
		mFirstScore += result;
	}
		
	return count > 0;
    }

    function toBytes(value, des, offset){
        for(var i = offset + 3; i >= offset; i--){
            des[i] = value & 0x000000FF;
            value = value >> 8;
        }
	
        return des;
    }
        
    function toInt(data, offset){
        var length = Math.min(4, data.length - offset);
        var value = 0;
        for(var i = 0; i < length; i++)
            if(data[offset + i] < 0)
                value = (value << 8) + 256 + data[offset + i];
            else
                value = (value << 8) + data[offset + i];
        
        return value;
    }

    return {        
        getCells: function(){
            return cells;
        },
        
        reset: function(){
            for(var i = 0; i < cells.length; i++){
                cells[i].level = 0;
                cells[i].type = Cell.TYPE_DEFAULT;
            }

            mMaxLevel = 0;
            for(var i = 0; i < cells.length; i++)
                mStateCells[i] = i;

            mEmptyCellCount = cells.length;
            mMoveCount = 0;
            mFirstScore = 0;
            mSecondScore = 0;
            mLastBlockLevel = 0;
            mGameEnd = false;

            saveData(GameState.DIRECT_NONE, [generateNewCell(), generateNewCell()]);
        },
        
        setupLocation: function(edge, cellPadding, anchorX, anchorY){        
            for(var row = 0; row < ROW_180.length; row++){			
                for(var col = 0; col < ROW_180[row].length; col++){
                    var index = ROW_180[row][col];

                    cells[index].y = row * (edge * 3 / 2 + cellPadding) + anchorY;

                    if(row === 2)
                        cells[index].x = col * (edge * MULTIPLIER + cellPadding) + anchorX;
                    else if(row === 1 || row === 3)
                        cells[index].x = (col + 0.5) * (edge * MULTIPLIER + cellPadding) + anchorX;
                    else
                        cells[index].x = (col + 1) * (edge * MULTIPLIER + cellPadding) + anchorX;	
                }					
            }		
        },
        
        getMovedCellCount: function(){
            return mMovedCellCount;
        },
    
        getMovedCell: function(index){
            return mStateCells[mStateCells.length - index - 1];
        },
        
        getMergedCellCount: function(){
            return mMergedCounts;
        },
        
        clearMovedCell: function (){
            mMovedCellCount = 0;
        },
        
        getData: function(){            
            return mState;
        },
                
        restoreData: function(data){  		
            setLevelState(data);
            mGameEnd = data.length > 19 && data[19] !== 0;

            if(data.length > 20){
                mMoveCount = toInt(data, 20);
                mFirstScore = toInt(data, 24);
                mSecondScore = toInt(data, 28);
            }
            else{
                mMoveCount = 0;
                mFirstScore = 0;
                mSecondScore = 0;
            }

            var length = Math.min(data.length, mState.length);
            for(var i = 0; i < length; i++)
                mState[i] = data[i];

            for(var i = length; i < mState.length; i++)
                mState[i] = -1;
        },
    
        restoreTurn: function(data){
            var direction = data[32];
            switch (direction) {
                case GameState.DIRECT_LEFT:
                    moveCells(ROW_180, true);
                    break;
                case GameState.DIRECT_RIGHT:
                    moveCells(ROW_180, false);
                    break;
                case GameState.DIRECT_TOP_LEFT:
                    moveCells(ROW_120, true);
                    break;
                case GameState.DIRECT_BOTTOM_RIGHT:
                    moveCells(ROW_120, false);
                    break;
                case GameState.DIRECT_TOP_RIGHT:
                    moveCells(ROW_60, true);
                    break;
                case GameState.DIRECT_BOTTOM_LEFT:
                    moveCells(ROW_60, false);
                    break;
            }
		
            for(var i = 33; i < data.length; i += 2){
                var pos = data[i];
                var level = data[i + 1];

                if(pos >= 0){
                    var cell = cells[pos];		
                    cell.level = level;
                    cell.type = Cell.TYPE_IN;

                    mMaxLevel = Math.max(mMaxLevel, cell.level);
                }
            }

            return direction;
        },
    
        getMoveCount: function(){
            return mMoveCount;
        },
    
        getScore: function(){
            return mFirstScore + mSecondScore;
        },
    
        getFirstScore: function(){
            return mFirstScore;
        },
 	
        getSecondScore: function(){
            return mSecondScore;
        },
 	
        getMaxLevel: function(){
            return mMaxLevel;
        },
    
        isWin: function(){
            var win = !mGameEnd && mMaxLevel >= Config.LEVEL_WIN;
            if(win)
                mGameEnd = true;
 		
            return win;
        },
    
        isOver: function(){ 	 		 		
            for(var r = 0; r < ROW_180.length; r++){
                var cols = ROW_180[r];
                var shift_up = 0;
                var shift_down = 0;
                switch (r) {
                    case 1:
                        shift_up = -1;
                        shift_down = 0;
                        break;
                    case 2:
                        shift_up = -1;
                        shift_down = -1;
                        break;
                    case 3:
                        shift_up = 0;
                        shift_down = -1;
                        break;					
                }

                for(var c = 0; c < cols.length; c++){
                    if(c > 0 && isMovable(cols[c], cols[c - 1]))
                        return false;

                    if(c < cols.length - 1 && isMovable(cols[c], cols[c + 1]))
                        return false;

                    if(r > 0){
                        var prev_cols = ROW_180[r - 1];

                        if(c + shift_up > 0 && isMovable(cols[c], prev_cols[c + shift_up]))
                            return false;
                        else if(c + shift_up < prev_cols.length - 1 && isMovable(cols[c], prev_cols[c + shift_up + 1]))
                            return false;
                    }

                    if(r < ROW_180.length - 1){
                        var next_cols = ROW_180[r + 1];

                        if(c + shift_down > 0 && isMovable(cols[c], next_cols[c + shift_down]))
                            return false;
                        else if(c + shift_down < next_cols.length - 1 && isMovable(cols[c], next_cols[c + shift_down + 1]))
                            return false;
                    }
                }
            }

            mGameEnd = true;
            return true;
        },
    
        haveInCell: function(){
            for(var i = 0; i < cells.length; i++)
                if(cells[i].type === Cell.TYPE_IN)
                    return true;

            return false;
        },
      
        moveCells: function(direction, createNewCell){
            var result = false;
            switch (direction) {
                case GameState.DIRECT_LEFT:
                    result = moveCellsByRow(this, ROW_180, true);
                    break;
                case GameState.DIRECT_RIGHT:
                    result = moveCellsByRow(this, ROW_180, false);
                    break;
                case GameState.DIRECT_TOP_LEFT:
                    result = moveCellsByRow(this, ROW_120, true);
                    break;
                case GameState.DIRECT_BOTTOM_RIGHT:
                    result = moveCellsByRow(this, ROW_120, false);
                    break;
                case GameState.DIRECT_TOP_RIGHT:
                    result = moveCellsByRow(this, ROW_60, true);
                    break;
                case GameState.DIRECT_BOTTOM_LEFT:
                    result = moveCellsByRow(this, ROW_60, false);
                    break;
            }
		
            if(result){	
                for(var i = 0; i < mMergedCounts.length; i++)
                    mMergedCounts[i] = 0;

                for(var i = 0; i < this.getMovedCellCount(); i++){
                    var cell = cells[this.getMovedCell(i)];
                    if(cell.pos2 >= 0)
                        mMergedCounts[cell.level - 1]++;
                }

                if(createNewCell){
                    var pos = generateNewCell();
                    var pos_block = (pos >= 0) ? generateNewBlock() : -1;

                    saveData(direction, [pos, pos_block]);	
                }
                else{
                    saveData(direction, null);
                }
            }

            return result;
        }
        
    };
    
}());