function GameView(){}

GameView.GRAVITY_LEFT               = 0x01;
GameView.GRAVITY_RIGHT              = 0x02;
GameView.GRAVITY_CENTER_HORIZONTAL  = 0x04;
GameView.HORIZONTAL_GRAVITY_MASK    = 0x0F;

GameView.GRAVITY_TOP                = 0x10;
GameView.GRAVITY_BOTTOM             = 0x20;
GameView.GRAVITY_CENTER_VERTICAL    = 0x40;
GameView.VERTICAL_GRAVITY_MASK      = 0xF0;

GameView.SET_STATE_TASK_RESET = 0;
GameView.SET_STATE_TASK_RESTORE_DATA = 1;
GameView.SET_STATE_TASK_RESTORE_TURN = 2;

GameView.prototype = (function(){
    var mContext;
    
    var mWidth = 0;
    var mHeight = 0;
    
    var mCellPadding = 8;
    var mPaddingLeft = 0;
    var mPaddingTop = 0;
    var mPaddingRight = 0;
    var mPaddingBottom = 0;
    
    var mAnchorX = 0.0;
    var mAnchorY = 0.0;
    var mGravity = GameView.GRAVITY_CENTER_HORIZONTAL | GameView.GRAVITY_CENTER_VERTICAL;
    
    var mState = new GameState();
    
    var mCellDrawable = new HexaDrawable();
    mCellDrawable.setBaseEdge(HexaDrawable.DEFAULT_EDGE);
    
    var MULTIPLIER = Math.sqrt(3);
    
    var mRenderInfo = new RenderInfo();
    
    var mStateData = [];
    
    var mStartPoint = new Point();
    var mEndPoint = new Point();
    var mScrolling = false;
	
    var mTouchSlop = 32;
		
    var mTouchable = true;	
    var mGenerateCell = true;
    var mBlockDirections = [];
    var mSoundOn = true;
    
    var mGameStateListener;
    var mMoveCellTask;
    var mSetStateTask;
    
    var mCacheRenderInfo = new RenderInfo();
    
    var mDispatchAnimEnd = true;    
        
    function MoveCellTask(){
		
	var mDirection = 0;
	var mCreateNewCell = true;
		
	this.setArgs = function(direction, createNewCell){
            mDirection = direction;
            mCreateNewCell = createNewCell;
	};
		
	this.run = function(){
            var prev_score = mState.getScore();
            var move = mState.moveCells(mDirection, mCreateNewCell);
            if(move)
		dispatchOnGameStateChangedEvent(mState.getData(), mState.getMergedCellCount(), prev_score, mState.getScore());            
            else
		setRenderInfo(false, 0, GameState.DIRECT_NONE, null);
        };
		
    }
    
    function SetStateTask(){
		
        var mData;
        var mMode = 0;

        this.setArgs = function(data, mode){
            mData = data;
            mMode = mode;
        };

        this.run = function() {
            switch (mMode) {
                case GameView.SET_STATE_TASK_RESET:
                    mState.reset();				
                    dispatchOnGameStateChangedEvent(mState.getData(), null, 0, 0);
                    break;
                case GameView.SET_STATE_TASK_RESTORE_DATA:
                    mState.restoreData(mData);				
                    dispatchOnGameStateChangedEvent(mState.getData(), null, mState.getScore(), mState.getScore());
                    break;
                case GameView.SET_STATE_TASK_RESTORE_TURN:
                    var prev_score = mState.getScore();	
                    mRenderInfo.mAnimDirection = mState.restoreTurn(mData);	
                    dispatchOnGameStateChangedEvent(mState.getData(), mState.getMergedCellCount(), prev_score, mState.getScore());
                    break;
            }
        };
    }
    
    function RenderInfo(){
        var mAnimating = false;
        var mAnimFactor = 0.0;
        var mAnimDirection = 0;
        var mStartAnimTime = 0;
        var mRunnable;
        
        this.set = function(animating, animFactor, animDirection, startAnimTime, runnable){
            mAnimating = animating;
            mAnimFactor = animFactor;
            mAnimDirection = animDirection;
            mStartAnimTime = startAnimTime;
            mRunnable = runnable;
        };

        this.copy = function(info){
            info.set(mAnimating, mAnimFactor, mAnimDirection, mStartAnimTime, mRunnable);

            return info;
        };
        
        this.isAnimating = function(){
            return mAnimating;
        };
        
        this.getAnimFactor = function(){
            return mAnimFactor;
        };
        
        this.setAnimFactor = function(animFactor){
            mAnimFactor = animFactor;
        };
        
        this.getAnimDirection = function(){
            return mAnimDirection;
        };
        
        this.getStartAnimTime = function(){
            return mStartAnimTime;
        };
        
        this.getRunnable = function(){
            return mRunnable;
        };
        
        this.setRunnable = function(runnable){
            mRunnable = runnable;
        };

        this.equals = function(info) {
            return info.isAnimating() === mAnimating && info.getAnimFactor() === mAnimFactor && info.getAnimDirection() === mAnimDirection && info.getStartAnimTime() === mStartAnimTime;
        };
    }
    
    function Point(){
        var mX;
        var mY;
        
        this.set = function(x, y){
            mX = x;
            mY = y;
        };
        
        this.getX = function(){
            return mX;
        };
        
        this.getY = function(){
            return mY;
        };
    }
            
    function getDistance(start, end){
	var dx = end.getX() - start.getX();
        var dy = end.getY() - start.getY();
		
	return Math.sqrt(dx * dx + dy * dy);
    }
	
    function getDirection(start, end){
	var dx = end.getX() - start.getX();
	var dy = end.getY() - start.getY();
		
	var angle = Math.atan2(dy, dx);
		
	if(angle < 0)
            angle = 2 * Math.PI + angle;
		
	var result;
				
	if(angle < Math.PI / 6)
            result = GameState.DIRECT_RIGHT;		
	else if(angle < Math.PI / 2)
            result = GameState.DIRECT_BOTTOM_RIGHT;
	else if(angle < Math.PI * 5 / 6)
            result = GameState.DIRECT_BOTTOM_LEFT;
	else if(angle < Math.PI * 7 / 6)
            result = GameState.DIRECT_LEFT;
        else if(angle < Math.PI * 3 / 2)
            result = GameState.DIRECT_TOP_LEFT;
	else if(angle < Math.PI * 11 / 6)
            result = GameState.DIRECT_TOP_RIGHT;
	else
            result = GameState.DIRECT_RIGHT;	
		
	if(mBlockDirections != null)
            for(var i = 0; i < mBlockDirections.length; i++)
		if(result === mBlockDirections[i]){
                    result = GameState.DIRECT_NONE;
                    break;
		}
		
	return result;
    }
    
    function onAnimEnd(){
        setRenderInfo(false, 0, GameState.DIRECT_NONE, null);
		
        var cells = mState.getCells();
	for(var i = 0; i < cells.length; i++)
            cells[i].type = Cell.TYPE_DEFAULT;
				
	if(mGameStateListener != null)
            mGameStateListener.onGameAnimEnd();
		
	//check if game end
	if(mState.isWin()){
            if(mGameStateListener != null)
		mGameStateListener.onGameEnded(true, mState.getScore());
	}
	else if(mState.isOver()){
            if(mGameStateListener != null)
		mGameStateListener.onGameEnded(false, mState.getScore());
	}
    }
        
    function dispatchOnGameStateChangedEvent(data, mergedCounts, prev_score, score){		
        for(var i = 0; i < data.length; i++)
            mStateData[i] = data[i];
				
	if(mGameStateListener != null)
            mGameStateListener.onGameStateChanged(mStateData, mergedCounts, prev_score, score);
    }
    
    function setRenderInfo(anim, factor, direction, runnable){		
	if(anim)
            mRenderInfo.set(anim, factor, direction, Date.now(), runnable);
        else
            mRenderInfo.set(anim, factor, direction, 0, runnable);
    }
    
    function renderNormal(context){	
        var cells = mState.getCells();
        for(var i = 0; i < cells.length; i++){
            var cell = cells[i];
            
            if(cell.level >= 0)
		mCellDrawable.setProperties(
                        Config.getCellColor(cell.level),
                        cell.level, 
                        Config.getCellTextColor(cell.level), 
                        0, 1, 
                        cell.x, cell.y)
			.draw(context);
            else
		mCellDrawable.setProperties(
			Config.CELL_BLOCK_COLOR_BG, 
			-cell.level, 
                        Config.CELL_BLOCK_COLOR_TEXT, 
			0, 1, 
			cell.x, cell.y)
			.draw(context);
	}
    }
    
    function renderInAnimation(context, factor){        
        var cells = mState.getCells();
        for(var i = 0; i < cells.length; i++){
            var cell = cells[i];
            switch (cell.type) {
		case Cell.TYPE_DEFAULT:
		case Cell.TYPE_MOVE:
		case Cell.TYPE_MOVE_BLOCK:	
                    if(cell.level >= 0)
			mCellDrawable.setProperties(
                                Config.getCellColor(cell.level), 
				cell.level, 
                                Config.getCellTextColor(cell.level), 
				0, 1, 
				cell.x, cell.y)
				.draw(context);
                    else
			mCellDrawable.setProperties(
				Config.CELL_BLOCK_COLOR_BG, 
				-cell.level, 
                                Config.CELL_BLOCK_COLOR_TEXT, 
				0, 1, 
				cell.x, cell.y)
				.draw(context);
                    break;
		case Cell.TYPE_IN:
                    mCellDrawable.setProperties(
                            Config.getCellColor(0), 
                            -1, 
                            Config.getCellTextColor(0), 
                            0, 1, 
                            cell.x, cell.y)
                            .draw(context);
					
                    if(cell.level > 0)
			mCellDrawable.setProperties(
				Config.getCellColor(cell.level), 
				cell.level, 
                                Config.getCellTextColor(cell.level), 
				0, Config.ANIM_IN_START_SCALE + (1 - Config.ANIM_IN_START_SCALE) * factor, 
				cell.x, cell.y)
				.draw(context);
                    else
			mCellDrawable.setProperties(
				Config.CELL_BLOCK_COLOR_BG, 
				-cell.level, 
                                Config.CELL_BLOCK_COLOR_TEXT, 
				0, Config.ANIM_IN_START_SCALE + (1 - Config.ANIM_IN_START_SCALE) * factor, 
				cell.x, cell.y)
				.draw(context);
                    break;
            }
        }		
    }
    
    function renderMoveAnimation(context, factor, direction){		
	var cells = mState.getCells();
        for(var i = 0; i < cells.length; i++){
            var cell = cells[i];
            switch (cell.type) {
		case Cell.TYPE_DEFAULT:
                    if(cell.level >= 0)
			mCellDrawable.setProperties(
                                Config.getCellColor(cell.level), 
				cell.level, 
                                Config.getCellTextColor(cell.level), 
				0, 1, 
				cell.x, cell.y)
				.draw(context);
                    else
			mCellDrawable.setProperties(
				Config.CELL_BLOCK_COLOR_BG, 
				-cell.level, 
                                Config.CELL_BLOCK_COLOR_TEXT, 
				0, 1, 
				cell.x, cell.y)
				.draw(context);
                    break;
		case Cell.TYPE_IN:
                    mCellDrawable.setProperties(
                            Config.getCellColor(0), 
                            -1, 
                            Config.getCellTextColor(0), 
                            0, 1, 
                            cell.x, cell.y)
                            .draw(context);
                    break;
                case Cell.TYPE_MOVE:
                    if(cell.pos2 < 0 || cells[cell.pos1] !== cell){
                        mCellDrawable.setProperties(
                                Config.getCellColor(0), 
                                -1, 
                                Config.getCellTextColor(0), 
                                0, 1, 
				cell.x, cell.y)
				.draw(context);
                    }
                    break;
            }
        }
		
	for(var i = 0, count = mState.getMovedCellCount(); i < count; i++){
            var cell = cells[mState.getMovedCell(i)];
						
            if(cell.type === Cell.TYPE_MOVE_BLOCK){		//one cell move to block cell
		var srcCell = cells[cell.pos2];
				
		var x = (cell.x - srcCell.x) * factor + srcCell.x;
		var y = (cell.y - srcCell.y) * factor + srcCell.y;
				
		mCellDrawable.setProperties(
			Config.CELL_BLOCK_COLOR_BG, 
			cell.level, 
                        Config.CELL_BLOCK_COLOR_TEXT, 
			0, 1, 
			cell.x, cell.y)
			.draw(context);
				
		mCellDrawable.setProperties(
			Config.getCellColor(cell.level), 
			cell.level, 
                        Config.getCellTextColor(cell.level), 
                        0, 1, 
			x, y)
			.draw(context);
            }
            else if(cell.pos2 < 0){			//one cell move to empty cell	
		var srcCell = cells[cell.pos1];			
		mCellDrawable.setProperties(
			Config.getCellColor(cell.level), 
			cell.level, 
                        Config.getCellTextColor(cell.level), 
			0, 1, 
			(cell.x - srcCell.x) * factor + srcCell.x, (cell.y - srcCell.y) * factor + srcCell.y)
			.draw(context);
            }
            else if(cells[cell.pos1] === cell){ // one cell move to non-empty cell	
		var prev_level = cell.level - 1;
		var srcCell = cells[cell.pos2];
				
		var x = (cell.x - srcCell.x) * factor + srcCell.x;
		var y = (cell.y - srcCell.y) * factor + srcCell.y;
				
		if(isCross(cell.x, cell.y, x, y, mCellDrawable.getBaseWidth(), mCellDrawable.getBaseHeight(), direction)){
                    if(factor < 1)
                        mCellDrawable.setProperties(
				Config.getCellColor(cell.level), 
				prev_level, 
                                Config.getCellTextColor(cell.level), 
				0, 1, 
                                x, y)
				.draw(context);
					
                    mCellDrawable.setProperties(
                            Config.getCellColor(cell.level), 
                            cell.level, 
                            Config.getCellTextColor(cell.level), 
                            0, Config.ANIM_MOVE_MAX_SCALE * (1 + factor) / 2, 
                            cell.x, cell.y)
                            .draw(context);
                }
                else{					
                    mCellDrawable.setProperties(
                            Config.getCellColor(prev_level), 
                            prev_level, 
                            Config.getCellTextColor(prev_level), 
                            0, 1, 
                            x, y)
                            .draw(context);
					
                    mCellDrawable.setProperties(
                            Config.getCellColor(prev_level), 
                            prev_level, 
                            Config.getCellTextColor(prev_level), 
                            0, 1, 
                            cell.x, cell.y)
                            .draw(context);
                }
            }
            else{	//move 2 cell to empty cell
		var prev_level = cell.level - 1;
				
		var srcCell = cells[cell.pos1];				
		var x1 = (cell.x - srcCell.x) * factor + srcCell.x;
		var y1 = (cell.y - srcCell.y) * factor + srcCell.y;
				
		srcCell = cells[cell.pos2];
		var x2 = (cell.x - srcCell.x) * factor + srcCell.x;
		var y2 = (cell.y - srcCell.y) * factor + srcCell.y;
				
		if(isCross(cell.x, cell.y, x1, y1, mCellDrawable.getBaseWidth(), mCellDrawable.getBaseHeight(), direction)){	
                    if(factor < 1)
			mCellDrawable.setProperties(
                                Config.getCellColor(cell.level), 
                                prev_level, 
                                Config.getCellTextColor(cell.level), 
				0, 1, 
				x2, y2)
				.draw(context);
					
                    mCellDrawable.setProperties(
                            Config.getCellColor(cell.level), 
                            cell.level, 
                            Config.getCellTextColor(cell.level), 
                            0, Config.ANIM_MOVE_MAX_SCALE * (1 + factor) / 2, 
                            x1, y1)
                            .draw(context);
		}
		else{					
                    mCellDrawable.setProperties(
                            Config.getCellColor(prev_level), 
                            prev_level, 
                            Config.getCellTextColor(prev_level), 
                            0, 1, 
                            x2, y2)
                            .draw(context);
					
                    mCellDrawable.setProperties(
                            Config.getCellColor(prev_level), 
                            prev_level, 
                            Config.getCellTextColor(prev_level), 
                            0, Config.ANIM_MOVE_MAX_SCALE * (1 + factor) / 2, 
                            x1, y1)
                            .draw(context);
		}
            }
	}
    }
    
    function isCross(x1, y1, x2, y2, width, height, direction){
        switch (direction) {
            case GameState.DIRECT_LEFT:						
		return x2 >= x1 && x2 <= x1 + width;
            case GameState.DIRECT_TOP_LEFT:	
		return x2 >= x1 && x2 <= x1 + width && y2 >= y1 && y2 <= y1 + height;
            case GameState.DIRECT_BOTTOM_LEFT:
		return x2 >= x1 && x2 <= x1 + width && y2 <= y1 && y2 + height>= y1;
            case GameState.DIRECT_RIGHT:
		return x2 <= x1 && x2 + width >= x1;
            case GameState.DIRECT_TOP_RIGHT:
		return x2 <= x1 && x2 + width >= x1 && y2 >= y1 && y2 <= y1 + height;
            case GameState.DIRECT_BOTTOM_RIGHT:
		return x2 <= x1 && x2 + width >= x1 && y2 <= y1 && y2 + height>= y1;
	}
		
	return false;
    }
        
    return{
        setProperties: function(context, width, height, cellPadding, paddingLeft, paddingTop, paddingRight, paddingBottom, gravity){
            mContext = context;

            mWidth = width;
            mHeight = height;

            mCellPadding = cellPadding;
            mPaddingLeft = paddingLeft;
            mPaddingTop = paddingTop;
            mPaddingRight = paddingRight;
            mPaddingBottom = paddingBottom;

            mGravity = gravity;

            mCellDrawable.setContext(mContext);

            var width = mWidth - mPaddingLeft - mPaddingRight;
            var height = mHeight - mPaddingTop - mPaddingBottom;
            var edge = Math.min((width - mCellPadding * 4) / MULTIPLIER / 5, (height - mCellPadding * 4) / 8);

            mCellDrawable.setBaseEdge(edge);
            for(var i = 1; i <= 14; i++)
                mCellDrawable.setTextLevel(i);

            mCellDrawable.setTextLevel(0);

            switch (mGravity & GameView.HORIZONTAL_GRAVITY_MASK) {
                case GameView.GRAVITY_LEFT:
                    mAnchorX = mPaddingLeft;
                    break;
                case GameView.GRAVITY_RIGHT:
                    mAnchorX = mPaddingLeft + width - edge * MULTIPLIER * 5 - mCellPadding * 4;
                    break;
                case GameView.GRAVITY_CENTER_HORIZONTAL:
                    mAnchorX = mPaddingLeft + (width - edge * MULTIPLIER * 5 - mCellPadding * 4) / 2;
                    break;
            }

            switch (mGravity & GameView.VERTICAL_GRAVITY_MASK) {
                case GameView.GRAVITY_TOP:
                    mAnchorY = mPaddingTop;
                    break;
                case GameView.GRAVITY_BOTTOM:
                    mAnchorY = mPaddingTop + height - edge * 8 - mCellPadding * 4;
                    break;
                case GameView.GRAVITY_CENTER_VERTICAL:
                    mAnchorY = mPaddingTop + (height - edge * 8 - mCellPadding * 4) / 2;
                    break;
            }

            mState.setupLocation(edge, mCellPadding, mAnchorX, mAnchorY);
        },

        setSoundOn: function(on){
            mSoundOn = on;
        },

        getScore: function(){
            return mState.getScore();
        },

        getMoveCount: function(){
            return mState.getMoveCount();
        },

        getCellLocation: function(index){
            var rect = new Rect();
            var cell = mState.getCells()[index];
            rect.set(cell.x, cell.y, cell.x + mCellDrawable.getBaseWidth(), cell.y + mCellDrawable.getBaseHeight());

            return rect;
        },

        restoreData: function(data){
            if(mSetStateTask == null)
                mSetStateTask = new SetStateTask();

            mSetStateTask.setArgs(data, GameView.SET_STATE_TASK_RESTORE_DATA);		
            setRenderInfo(false, 0, GameState.DIRECT_NONE, mSetStateTask);
        },

        restoreTurn: function(data){		
            if(mSetStateTask == null)
                mSetStateTask = new SetStateTask();

            mSetStateTask.setArgs(data, GameView.SET_STATE_TASK_RESTORE_TURN);		
            setRenderInfo(true, 0, GameState.DIRECT_NONE, mSetStateTask);
        },

        reset: function(){
            if(mSetStateTask == null)
                mSetStateTask = new SetStateTask();

            mSetStateTask.setArgs(null, GameView.SET_STATE_TASK_RESET);				
            setRenderInfo(true, 0, GameState.DIRECT_NONE, mSetStateTask);		
        },

        moveCells: function(direction, createNewCell){
            if(mMoveCellTask == null)
                mMoveCellTask = new MoveCellTask();

            mMoveCellTask.setArgs(direction, createNewCell);		
            setRenderInfo(true, 0, direction, mMoveCellTask);		
        },

        setTouchable: function(touchable){
            mTouchable = touchable;
        },

        setGenerateCell: function(enable){
            mGenerateCell = enable;
        },

        setBlockDirection: function(directions){
            mBlockDirections = directions;
        },

        isWin: function(){
            return mState.isWin();
        },

        onTouchEvent: function(action, event) {
            if(mRenderInfo.isAnimating() || !mTouchable)
                return false;

            switch (action) {
                case MOUSE_ACTION_DOWN:
                    mScrolling = true;
                    mStartPoint.set(event.x, event.y);
                    break;
                case MOUSE_ACTION_MOVE:
                    if(mScrolling){
                        mEndPoint.set(event.x, event.y);
                        if(getDistance(mStartPoint, mEndPoint) > mTouchSlop){
                            mScrolling = false;
                            var direction = getDirection(mStartPoint, mEndPoint);
                            if(direction !== GameState.DIRECT_NONE)
                                this.moveCells(direction, mGenerateCell);
                            else if(mGameStateListener != null)
                                mGameStateListener.onGameMoveBlocked();
                        }
                    }				
                    break;
                case MOUSE_ACTION_UP:
                    if(mScrolling){
                        mScrolling = false;
                        mEndPoint.set(event.x, event.y);
                        if(getDistance(mStartPoint, mEndPoint) > mTouchSlop){                            
                            var direction = getDirection(mStartPoint, mEndPoint);
                            if(direction !== GameState.DIRECT_NONE)
                                this.moveCells(direction, mGenerateCell);
                            else if(mGameStateListener != null)
                                mGameStateListener.onGameMoveBlocked();
                        }
                    }
                    break;
            }

            return true;
        },
    
        onKeyDown: function(event){
            if(mRenderInfo.isAnimating() || !mTouchable || mScrolling)
                return false;
            
            switch(event.keyCode){
                case 113: //q
                case 81: //Q
                    this.moveCells(GameState.DIRECT_TOP_LEFT, mGenerateCell);
                    break;
                case 119: //w
                case 87: //W    
                    this.moveCells(GameState.DIRECT_TOP_RIGHT, mGenerateCell);
                    break;
                case 97: //a
                case 65: //A    
                    this.moveCells(GameState.DIRECT_LEFT, mGenerateCell);
                    break;
                case 115: //s
                case 83: //S    
                    this.moveCells(GameState.DIRECT_RIGHT, mGenerateCell);
                    break;
                case 122: //z
                case 90: //Z    
                    this.moveCells(GameState.DIRECT_BOTTOM_LEFT, mGenerateCell);
                    break;
                case 120: //x
                case 88: //X    
                    this.moveCells(GameState.DIRECT_BOTTOM_RIGHT, mGenerateCell);
                    break;
            }
            
            return true;
        },
    
        setOnGameStateChangedListener: function(listener){
            mGameStateListener = listener;
        },
        
        preUpdate: function(){
            mDispatchAnimEnd = true;
        },

        update: function(next){
            if(mRenderInfo != null){
                if(mRenderInfo.getRunnable() != null){
                    mRenderInfo.getRunnable().run();
                    mRenderInfo.setRunnable(null);
                }

                //advance the animation
                if(mRenderInfo.isAnimating()){						
                    var value = (next - mRenderInfo.getStartAnimTime()) / Config.ANIM_DURATION;
                    //animation end
                    if(value > 1){
                        if(mRenderInfo.getAnimDirection() !== GameState.DIRECT_NONE){
                            mState.clearMovedCell();	
                            if(mState.haveInCell())
                                setRenderInfo(true, 0, GameState.DIRECT_NONE, null);
                            else if(mDispatchAnimEnd){
                                onAnimEnd();
                                mDispatchAnimEnd = false;
                            }
                        }
                        else if(mDispatchAnimEnd){
                            onAnimEnd();
                            mDispatchAnimEnd = false;
                        }
                    }
                    else{
                        if(mRenderInfo.getAnimDirection() === GameState.DIRECT_NONE)
                            mRenderInfo.setAnimFactor(Config.getOvershootInterpolation(value));
                        else
                            mRenderInfo.setAnimFactor(Config.getDecelerateInterpolation(value));	
                    }
                }					

                if(!mCacheRenderInfo.equals(mRenderInfo)){                
                    mRenderInfo.copy(mCacheRenderInfo);
                    return true;
                }			
            }

            return false;
        },

        render: function(){
            mContext.save();
            
            if(mCacheRenderInfo.isAnimating()){            
                if(mCacheRenderInfo.getAnimDirection() === GameState.DIRECT_NONE)
                    renderInAnimation(mContext, mCacheRenderInfo.getAnimFactor());
                else
                    renderMoveAnimation(mContext, mCacheRenderInfo.getAnimFactor(), mCacheRenderInfo.getAnimDirection());
            }
            else
                renderNormal(mContext);
            
            mContext.restore();
        }
        
    };
    
})();

function OnGameStateChangedListener(){}

OnGameStateChangedListener.prototype = (function(){
    
    this.onGameStateChanged = null;
		
    this.onGameEnded = null;
		
    this.onGameAnimEnd = null;
		
    this.onGameMoveBlocked = null;
    
    return {
        setListener: function(gameStateChanged, gameEnded, gameAnimEnd, gameMoveBlocked){
            this.onGameStateChanged = gameStateChanged;
            this.onGameEnded = gameEnded;
            this.onGameAnimEnd = gameAnimEnd;
            this.onGameMoveBlocked = gameMoveBlocked;
        }
    };
    
})();