// Cross-browser support for requestAnimationFrame
requestAnimationFrame = window.requestAnimationFrame || 
        window.webkitRequestAnimationFrame || 
        window.msRequestAnimationFrame || 
        window.mozRequestAnimationFrame;

var MOUSE_ACTION_DOWN = 1;
var MOUSE_ACTION_MOVE = 2;
var MOUSE_ACTION_UP = 3;

var canvas = document.getElementById('mainCanvas');
var context = canvas.getContext('2d');

var FRAME_DURATION = 1000 / 60;
var next;
var sleepTime;
var needRender = true;
var count;

var mUndoCount = 0;
var mStateDatas = [];
for(var i = 0; i <= Config.MAX_UNDO; i++)            
    mStateDatas[i] = new Int8Array(GameState.STATE_DATA_SIZE);

var mGameView = new GameView();
var mScoreView = new ScoreView();
var mAvatarView = new AvatarView();
var mRestartButton = new Button();
var mUndoButton = new Button();
var mLoginButton = new Button();
var mLeaderboardButton = new Button();
var mConfirmView = new ConfirmView();
var mViews = [mGameView, mScoreView, mAvatarView, mRestartButton, mUndoButton, mLoginButton, mLeaderboardButton, mConfirmView];
var mFocusView;

var mCellPadding = 8;
var mPadding = 16;
var mProfileHeight = 136;
var mScoreHeight = 48;
var mScoreY = 88;
var mAvatarSize = 72;
var mAvatarBorder = 4;
var mAvatarMargin = 8;

var mButtonSize = 48;
var mUndoTextSize = 10;

var mConfirmMsgTextSize = 16;
var mConfirmOptTextSize = 12;
var mConfirmPadding = 8;

var ID_RESTART = 0;
var ID_GAME_END = 1;
var ID_LOG_OUT = 2;

var mNotifTime;

var onMouseDown = function(evt){
    for(var i = mViews.length - 1; i >= 0; i--){
        if(mViews[i].onTouchEvent(MOUSE_ACTION_DOWN, getMousePos(evt))){
            mFocusView = mViews[i];
            return;
        }
    }
    
    mFocusView = null;
};

var onMouseMove = function(evt){
    if(mFocusView != null)
        mFocusView.onTouchEvent(MOUSE_ACTION_MOVE, getMousePos(evt));
};

var onMouseUp = function(evt){
    if(mFocusView != null){
        mFocusView.onTouchEvent(MOUSE_ACTION_UP, getMousePos(evt));
        mFocusView = null;
    }
};

var onKeyDown = function(evt){    
    mGameView.onKeyDown(evt);
};

var onGameStateChanged = function(data, mergedCounts, prev_score, score){
    addStateData(data);	    		
    updateAchievementProgress(mergedCounts);
    
    mScoreView.setText(score + '');       
};

var onGameEnded = function(win, score){
    if(Player.loaded){
        
        LeaderboardManager.submitScore(score, Config.LEAD_SINGLEPLAY, null, function(newDaily, newWeekly, newAllTime){
            if(newAllTime)
                showNotif("Congratulation, you have achieved a new high score!");     
            else if(newDaily)
                showNotif("Congratulation, you have achieved a new daily high score!");     
        });
    }
    
    mConfirmView.show(ID_GAME_END, [(win ? 'Congratulation' : 'Game Over'), score + ''], win ? Config.BG_WIN : Config.BG_OVER, 'Restart', 'Continue', onConfirmSelected);
};

var onGameAnimEnd = function(){
    
};

var onGameMoveBlocked = function(){
        
};

var onRestartClicked = function(){
    mConfirmView.show(ID_RESTART, ['Do you want to', 'restart game?'], Config.BG_SCORE, 'Yes', 'No', onConfirmSelected);
};

var onUndoClicked = function(){
    if(mUndoCount <= 0 || mGameView.getMoveCount() === 0)
        return;
    	
    var data = new Int8Array(GameState.STATE_DATA_SIZE);
    arrayCopy(mStateDatas[1], 0, data, 0, mStateDatas[1].length);
    
    for(var i = 2; i < mStateDatas.length; i++)
        arrayCopy(mStateDatas[i], 0, mStateDatas[i - 2], 0, mStateDatas[i].length);    
    	        
    mGameView.restoreData(data);
    	
    mUndoCount--;    
    
    mUndoButton.setText(mUndoCount + '');
    if(mUndoCount == 0)
        mUndoButton.setVisible(false);
};

var onAvatarClicked = function(){
    if(!Player.loaded)
        login.showLoginDialog();
    else
        mConfirmView.show(ID_LOG_OUT, ['Do you want to', 'log out?'], Config.BG_SCORE, 'Yes', 'No', onConfirmSelected);    
};

var onLeaderboardClicked = function(){
    if(!Player.loaded)
        login.showLoginDialog();
    else
        LeaderboardTable.showLeaderboard(Config.LEAD_SINGLEPLAY);
};

var onAchievementUnlocked = function(achivementId, unlock){
    
};

var onConfirmSelected = function(id, select){
    switch(id){
        case ID_RESTART:
            if(select === ConfirmView.SELECT_OPT1){
                restart();
                mConfirmView.hide();
            }
            else if(select === ConfirmView.SELECT_OPT2)
                mConfirmView.hide();
            break;
        case ID_GAME_END:
            if(select === ConfirmView.SELECT_OPT1){
                restart();
                mConfirmView.hide();
            }
            else if(select === ConfirmView.SELECT_OPT2)
                mConfirmView.hide();
            break;
        case ID_LOG_OUT:  
            if(select === ConfirmView.SELECT_OPT1){
                login.logout();
                mConfirmView.hide();
            }
            else if(select === ConfirmView.SELECT_OPT2)
                mConfirmView.hide();  
            break;
    }
};

function printStateDatas(){
    text = 'undo = ' + mUndoCount + ' states: ';
    for(var i = 0; i <= mUndoCount; i++){
        text += '{';
        for(var j = 0; j < 19; j++)
            text += mStateDatas[i][j] + ',';
        text += '}, ';
    }
    
    console.log(text);
}

function addStateData(data){
    if(data == null)
        return;
           	
    for(var i = Math.min(mStateDatas.length - 1, mUndoCount); i > 0; i--)	
        arrayCopy(mStateDatas[i - 1], 0, mStateDatas[i], 0, mStateDatas[i].length);    
    	
    arrayCopy(data, 0, mStateDatas[0], 0, data.length);
}

function updateAchievementProgress(counts){
    if(counts == null || !Player.loaded)
        return;
    	
    var id_first = 0;
    var id_500 = 0;
    var id_1000 = 0;
    	
    for(var i = 7; i < counts.length; i++){
    	switch (i) {
            case 7: //256
		id_first = Config.ACH_FIRST_256_TILE;
		id_500 = Config.ACH_500TH_256_TILE;
		id_1000 = Config.ACH_1000TH_256_TILE;
		break;
            case 8: //512
		id_first = Config.ACH_FIRST_512_TILE;
		id_500 = Config.ACH_500TH_512_TILE;
		id_1000 = Config.ACH_1000TH_512_TILE;
		break;
            case 9: //1024
		id_first = Config.ACH_FIRST_1024_TILE;
		id_500 = Config.ACH_500TH_1024_TILE;
		id_1000 = Config.ACH_1000TH_1024_TILE;
		break;
            case 10: //2048
		id_first = Config.ACH_FIRST_2048_TILE;
		id_500 = Config.ACH_500TH_2048_TILE;
		id_1000 = Config.ACH_1000TH_2048_TILE;
		break;
            case 11: //4096
		id_first = Config.ACH_FIRST_4096_TILE;
		id_500 = Config.ACH_500TH_4096_TILE;
		id_1000 = Config.ACH_1000TH_4096_TILE;
		break;
            case 12: //8192
		id_first = Config.ACH_FIRST_8192_TILE;
		id_500 = null;
		id_1000 = null;
		break;
            case 13: //16384
		id_first = Config.ACH_FIRST_16384_TILE;
		id_500 = null;
		id_1000 = null;
		break;
        }
    }
    		
    if(counts[i] > 0){            
        AchievementManager.unlockAchievement(id_first, onAchievementUnlocked);
        if(id_500 != null)
            AchievementManager.submitProgress(id_500, counts[i], onAchievementUnlocked());
        if(id_1000 != null)
            AchievementManager.submitProgress(id_1000, counts[i], onAchievementUnlocked());
    }
}

function showNotif(text){ 
    console.log(text);
    $('#p_notif').text(text);
    classie.add(document.querySelector('#tn-notif'), 'tn-box-active');
    mNotifTime = Date.now();
    var time = mNotifTime;
    setTimeout(function() {
        if(time === mNotifTime)
            classie.remove(document.querySelector('#tn-notif'), 'tn-box-active');
    }, 3000);
};

function layout(){
    var windowHeight = $(window).innerHeight();
    var deviceFrameY = (windowHeight - $('#device_frame').outerHeight()) / 2;
    var canvasY = deviceFrameY + 96;
    var containerY = Math.max(0, (windowHeight - $('#container').outerHeight()) / 2);
    
    $('#device_frame').css({ top: deviceFrameY + 'px'});
    $('#mainCanvas').css({ top: canvasY + 'px' });
    $('#container').css({ top: containerY + 'px'});
   
    mGameView.setProperties(context, 
                                canvas.width, canvas.height,
                                mCellPadding,
                                mPadding, mProfileHeight + mPadding, mPadding, mPadding,
                                GameView.GRAVITY_CENTER_HORIZONTAL | GameView.GRAVITY_CENTER_VERTICAL);
                                
    mScoreView.setProperties(context,
                                canvas.width, mScoreHeight,
                                0, mScoreY,
                                Config.TEXT_SCORE, Config.BG_SCORE);
  
    mAvatarView.setProperties(context, 
                                mAvatarSize,
                                mAvatarMargin, mAvatarMargin,
                                Player.profileUrl,
                                Config.BG_SCORE, mAvatarBorder);
                                
    mRestartButton.setProperties(context,
                                mButtonSize, mButtonSize,
                                canvas.width - mButtonSize, mScoreY + mScoreHeight,
                                'res/bt_restart.png',
                                Config.TEXT_SCORE, 0);       
                                
    mUndoButton.setProperties(context,
                                mButtonSize, mButtonSize,
                                canvas.width - mButtonSize * 2, mScoreY + mScoreHeight,
                                'res/bt_undo.png',
                                Config.TEXT_SCORE, mUndoTextSize);  
                                
    mLoginButton.setProperties(context,
                                mButtonSize, mButtonSize,
                                mAvatarMargin + mAvatarSize, mScoreY - mButtonSize,
                                'res/bt_login.png',
                                Config.TEXT_SCORE, 0);     
                                
    mLeaderboardButton.setProperties(context,
                                mButtonSize, mButtonSize,
                                canvas.width - mButtonSize, mScoreY - mButtonSize,
                                'res/bt_leaderboard.png',
                                Config.TEXT_SCORE, 0);     
                                
    mLoginButton.setVisible(!Player.loaded);
    
    mConfirmView.setProperties(context,
                                canvas.width, canvas.height, 
                                mConfirmMsgTextSize, mConfirmOptTextSize,
                                Config.BG_SCORE, Config.TEXT_SCORE,
                                mConfirmPadding);
}

function main() {    
    next = Date.now();
    
    if(needRender)
        render();
    
    count = 1;
    preUpdate();
    
    do{
        next += Math.max(5, Math.floor(FRAME_DURATION / count));
	count++;
        
        if(next > Date.now())
            needRender = update(next);
        
        sleepTime = next - Date.now();
    }
    while(sleepTime <= 0);
    
    setTimeout(function () {
        window.requestAnimationFrame(main);
    }, sleepTime);
};

function restart() {     
    for(var i = 0; i < mStateDatas.length; i++){    		
    	for(var j = 0; j < mStateDatas[i].length; j++)
            mStateDatas[i][j] = 0;
    }
    
    mUndoCount = Config.MAX_UNDO;
    mUndoButton.setText(mUndoCount + '');
    mUndoButton.setVisible(true);
    
    mGameView.reset();
    mGameView.setGenerateCell(true);
    mGameView.setBlockDirection(null);
    mGameView.setTouchable(true);
}

function preUpdate(){
    mGameView.preUpdate();
}

function update(next) {
    var result = false;
    for(var i = 0; i < mViews.length; i++)
        if(mViews[i].update(next) && !result)
            result = true;        
    
    return result;
}

function render() {
    context.beginPath();
    context.rect(0, 0, canvas.width, canvas.height);
    context.closePath();
    context.fillStyle = Config.BG_CANVAS;
    context.fill();
    
    for(var i = 0; i < mViews.length; i++)
        mViews[i].render();    
}

function getMousePos(evt) {
    var rect = canvas.getBoundingClientRect();
    return {
        x: evt.clientX - rect.left,
        y: evt.clientY - rect.top
    };
}

function arrayCopy(src, src_start, des, des_start, length){
    for(var i = 0; i < length; i++)
        des[des_start + i] = src[src_start + i];
}

canvas.addEventListener('mousedown', onMouseDown, false);
canvas.addEventListener('mousemove', onMouseMove, false);
canvas.addEventListener('mouseup', onMouseUp, false);
window.addEventListener('keypress', onKeyDown, true);

var mListener = new OnGameStateChangedListener();
mListener.setListener(onGameStateChanged, onGameEnded, onGameAnimEnd, onGameMoveBlocked);
mGameView.setOnGameStateChangedListener(mListener);

mUndoButton.setOnClickListener(onUndoClicked);
mRestartButton.setOnClickListener(onRestartClicked);
mAvatarView.setOnClickListener(onAvatarClicked);
mLoginButton.setOnClickListener(onAvatarClicked);
mLeaderboardButton.setOnClickListener(onLeaderboardClicked);

$(window).resize(function(){
    layout();
});

layout();
restart();
main();

