/**
 * 矩阵的数据显示层
 *
 * @class
 * @extends xx.Class
 * @extends jQuery
 */
xx.MatrixLayer = xx.Class.extend({

    /**
     * @type {xx.MatrixData}
     */
    matrixData: null,

    /**
     * @type {Boolean}
     */
    _touchable: true,

    /**
     *
     * @param $originalItem
     */
    getItemByOriginal: function ($originalItem) {
        return $originalItem[0].$item;
    },

    /**
     * 检测连成一条线的Item并清除
     * @param $item
     */
    checkAndClear: function ($item) {
        var lines = this.matrixData.getLines($item, 4);
        var result = [];
        $(lines).each(function () {
            result.push($(this).length);
            $(this).each(function () {
                // 将源Item设置为EMPTY样式
                this.clear();
            });
        });
        return result;
    },

    /**
     * @type {Function}
     *
     * @private
     */
    _moveCallback: null,


    /**
     * 设置Item移动事件回调函数
     *
     * @param callback
     */
    setMoveCallback: function (callback) {
        this._moveCallback = callback;
    },

    /**
     * 返回Item移动事件回调函数
     *
     * @returns {Function}
     */
    getMoveCallback: function () {
        return this._moveCallback;
    },

    /**
     * 返回当前矩阵是否处理鼠标/触摸事件
     *
     * @returns {Boolean}
     */
    getTouchable: function () {
        return this._touchable;
    },

    /**
     * 设置当前矩阵是否处理鼠标/触摸事件
     *
     * @param {Boolean} touchable
     */
    setTouchable: function (touchable) {
        this._touchable = touchable;
    },

    /**
     * @type {Function}
     */
    refill: null,


    /**
     * 为不可移动至的Item绘制 BLOCK 样式
     *
     * @param $item
     */
    drawBlockStyle: function ($item) {
        // 将所有的 EMPTY 样式的 ITEM 设置为 BLOCK 样式
        this.find(".item.EMPTY").addClass("BLOCK");

        // 清除可到达的 ITEM 的 BLOCK 样式
        this._clearNearbyBlockStyle($item);
    },

    /**
     * 清除给定元素相邻的元素的 BLOCK 样式，并将被清除的 ITEM 作为数组返回
     *
     * @param $item
     * @private
     */
    _clearNearbyBlockStyle: function ($item) {
        // 获取Item的位置
        var r = $item._rowIndex, c = $item._colIndex;
        var items = [], tmp;
        // UP
        if (r > 0) {
            tmp = this._checkAndClearBlockStyle(this.matrixData.getItem(r - 1, c));
            if (tmp) {
                items.push(tmp);
            }
        }
        // DOWN
        if (r < this.matrixData.getRowNum() - 1) {
            tmp = this._checkAndClearBlockStyle(this.matrixData.getItem(r + 1, c));
            if (tmp) {
                items.push(tmp);
            }
        }
        // LEFT
        if (c > 0) {
            tmp = this._checkAndClearBlockStyle(this.matrixData.getItem(r, c - 1));
            if (tmp) {
                items.push(tmp);
            }
        }
        // DOWN
        if (c < this.matrixData.getColNum() - 1) {
            tmp = this._checkAndClearBlockStyle(this.matrixData.getItem(r, c + 1));
            if (tmp) {
                items.push(tmp);
            }
        }
        //
        for (var i = 0; i < items.length; i++) {
            this._clearNearbyBlockStyle(items[i]);
        }
    },

    /**
     * 如果传入的Item可以作为移动目标地，则移除其BLOCK样式，并返回该Item，否则返回null
     *
     * @param $item
     * @returns {*}
     * @private
     */
    _checkAndClearBlockStyle: function ($item) {
        if ($item.isClear() && $item.hasClass("BLOCK")) {
            $item.removeClass("BLOCK");
            return $item;
        }
        return null;
    },

    /**
     * 清楚所有Item的 BLOCK 样式
     */
    clearBlockStyle: function () {
        this.find(".item.BLOCK").removeClass("BLOCK");
    },

    /**
     * 将源Item移动至目标Item
     * <br>
     * 将源Item的样式复制到目标Item，并将源Item设置为EMPTY样式
     *
     * @param {xx.MatrixItem} $srcItem
     * @param {xx.MatrixItem} $target
     */
    moveTo: function ($srcItem, $target) {
        // 如果当前 Item 为 BLOCK 样式则不可移动至该 Item
        if ($target.isBlock() || $srcItem.isClear()) {
            return;
        }
        // 源Item样式
        var itemClass = $srcItem.itemClass;

        // 设置目标Item的样式为源样式
        $target.setItemClass(itemClass);

        // 将源Item设置为EMPTY样式
        $srcItem.clear();

        // 移除源Item、目标Item的选中状态
        $srcItem.setItemSelected(false);
        $target.setItemSelected(false);

        setTimeout(function () {
            // 清除连线的Item，并调用Callback
            $target.matrix.getMoveCallback()($target.matrix.checkAndClear($target));
        }, 200);
    },

    /**
     * 类型的矩阵
     */
    getClassesMatrix: function () {
        var classes = [];
        //
        for (var i = 0; i < this.matrixData.getRowNum(); i++) {
            classes[i] = classes[i] ? classes[i] : [];
            for (var j = 0; j < this.matrixData.getColNum(); j++) {
                classes[i][j] = this.matrixData.getItem(i, j).getItemClass();
            }
        }
        return classes;
    }
})
;

xx.MatrixLayer.createMatrixLayer = function (row, col) {
    var matrixLayer = new xx.MatrixLayer();
    matrixLayer.matrixData = xx.MatrixData.createMatrixData(row, col);
    return matrixLayer;
};


/**
 * 矩阵的元素Item
 *
 * @class
 * @extends xx.Class
 * @extends jQuery
 */
xx.MatrixItem = xx.Class.extend({

    /**
     * @type {xx.MatrixLayer}
     */
    matrix: null,

    /**
     *
     * @type {String}
     */
    itemClass: "EMPTY",

    /**
     * @type {Number}
     */
    _rowIndex: -1,

    /**
     * @type {Number}
     */
    _colIndex: -1,

    /**
     * Item开始拖拽时触发，设置样式信息及源Item's ID
     *
     * @param {jQuery.Event} e
     */
    onDragstart: function (e) {

        // 设置源Item's ID
        e.originalEvent.dataTransfer.setData("SrcItemId", e.target.id);
        //
        this.matrix.drawBlockStyle(this);
    },

    /**
     * 在某一拖拽事件拖拽至Item上时触发，用于判断是否可以Drop
     *
     * @param {jQuery.Event} e
     */
    onDragover: function (e) {
        if (e.originalEvent.dataTransfer.getData("SrcItemId") != e.target.id  // 判断当前Item是否是源Item
            && this.isClear() // 判断当前 Item 是否为 EMPTY 样式
            && !this.isBlock() // 判断当前 Item 是否为 BLOCK 样式
            ) {
            // 允许拖入
            e.preventDefault();
        }
    },

    /**
     * 在Item拖拽后放至另一Item时触发，完成拖拽事件
     * <br>
     *  将源Item移动至目标Item
     *
     * @param {jQuery.Event} e
     */
    onDrop: function (e) {

        // 取消事件默认动作
        e.preventDefault();

        // 将源Item移动至目标Item
        this.matrix.moveTo(this.matrix.getItemByOriginal($("#" + e.originalEvent.dataTransfer.getData("SrcItemId"))), this);
    },

    /**
     * 在Item被点击时触发，对于可选择的Item，增加样式selected
     */
    onClick: function () {

        if (this.isSelectable()) {
            /* 如果目标可选定 */

            // 设置目标选中状态
            this.setItemSelected(true);
        } else if (this.isClear && $("div.matrix .matrixTable tr td .item.selected").length > 0) {
            /* 如果目标不可选定 && 当前目标为EMPTY && 存在已存在选定的元素 */

            // 移动
            this.matrix.moveTo(this.matrix.getItemByOriginal($("div.matrix .matrixTable tr td .item.selected")), this);
        }
    },

    /**
     * 判断给定的Item类型是否是可被选定的
     *
     * @returns {Boolean}
     */
    isSelectable: function () {
        return xx.MatrixLayer.selectableItemClasses.inArray(this.itemClass);
    },

    /**
     * 设置指定元素为选中状态
     *
     * @param {Boolean} selected
     */
    setItemSelected: function (selected) {
        //
        if (selected) {
            if (this.hasClass("selected")) {
                // 如果已被选择
                return;
            }
            // 清除 BLOCK 样式
            this.matrix.clearBlockStyle();
            // 移除上一个选中元素的selected样式
            $("div.matrix .matrixTable tr td .item.selected").removeClass("selected");
            // 添加selected样式
            this.addClass("selected");
            // 更改不能作为移动目的Item的样式
            this.matrix.drawBlockStyle(this);
        } else {
            // 清除 BLOCK 样式
            this.matrix.clearBlockStyle();
            // 移除selected样式
            this.removeClass("selected");
        }
    },

    /**
     * 设置ItemClass
     *
     * @param {String} itemClass
     */
    setItemClass: function (itemClass) {
        this.removeClass(this.itemClass).addClass(itemClass);
        this.itemClass = itemClass;
    },

    /**
     * 返回ItemClass
     *
     * @return {String}
     */
    getItemClass: function () {
        return this.itemClass;
    },

    /**
     * 将当前元素设置为EMPTY
     */
    clear: function () {
        this.setItemClass("EMPTY");
        this.setItemSelected(false)
    },

    /**
     *
     * @returns {boolean}
     */
    isClear: function () {
        return this.itemClass == "EMPTY";
    },

    /**
     *
     * @returns {boolean}
     */
    isBlock: function () {
        return this.hasClass("BLOCK");
    },

    /**
     * 判断指定的元素是否与本元素在同一条直线
     *
     * @param {xx.MatrixItem} item
     * @returns {boolean}
     */
    isInLine: function (item) {
        return  this.itemClass != "EMPTY" && this.itemClass == item.itemClass;
    }
})
;

/**
 * 记录哪些类型是可以被选定的
 *
 * @type {Array}
 */
xx.MatrixLayer.selectableItemClasses = ["PURPLE", "CYAN", "GREEN", "RED", "BLUE", "YELLOW"];

/**
 * 记录哪些类型是阻塞的
 *
 * @type {Array}
 */
xx.MatrixLayer.blockItemClasses = ["PURPLE", "CYAN", "GREEN", "RED", "BLUE", "YELLOW", "BLOCK"];

/**
 * 随机生成指定长度的矩阵元素类型
 *
 * @param {Number} num 数量
 * @return {Array}
 */
xx.MatrixLayer.generateRandomClass = function (num) {
    var rtn = [];
    for (var i = 0; i < num; i++) {
        rtn.push(xx.MatrixLayer.selectableItemClasses[Math.floor(Math.random() * xx.MatrixLayer.selectableItemClasses.length)]);
    }
    return rtn;
};


(function () {

    // 初始化TD元素的ID，该ID会递增
    var matrixItemId = 53000;

    $(".matrix").each(function () {

        var $this = $(this);

        // 获取矩阵的行数和列数
        var row = parseInt($this.attr("row"));
        var col = parseInt($this.attr("col"));

        // 插入Table元素
        var $matrix = $.extend(xx.MatrixLayer.createMatrixLayer(row, col), $("<table class='matrixTable'></table>"));
        $this.append($matrix);
        $matrix.parent()[0].matrix = $matrix;

        // 创建矩阵
        for (var i = 0; i < row; i++) {
            // 创建行
            var $tr = $("<tr></tr>");
            // 在每行中添加列
            for (var j = 0; j < col; j++) {
                // 创建Item
                var $td = $("<td></td>");
                var $item = $.extend(new xx.MatrixItem(), $("<div class='item EMPTY' draggable='true'></div>"));
                $item[0].$item = $item;

                // 将Item设置到矩阵数据中
                if (!$matrix.matrixData.m_matrix[i]) {
                    $matrix.matrixData.m_matrix[i] = [];
                }
                $matrix.matrixData.m_matrix[i][j] = $item;

                // 设置Item的父矩阵
                $item.matrix = $matrix;

                // 设置属性
                $item.attr("id", "matrixItem" + matrixItemId++);
                $item.attr("_rowIndex", "" + i);
                $item.attr("_colIndex", "" + j);
                $item._rowIndex = i;
                $item._colIndex = j;

                // 判断是否为静态的
                if (!$this.attr("static")) {
                    // 绑定拖拽相关事件
                    (function (_$item) {
                        _$item.bind("dragstart", function (e) {
                            if ($matrix.getTouchable()) {
                                _$item.onDragstart(e);
                            }
                        });
                        _$item.bind("dragover", function (e) {
                            if ($matrix.getTouchable()) {
                                _$item.onDragover(e);
                            }
                        });
                        _$item.bind("drop", function (e) {
                            if ($matrix.getTouchable()) {
                                _$item.onDrop(e);
                            }
                        });
                        _$item.bind("click", function () {
                            if ($matrix.getTouchable()) {
                                _$item.onClick();
                            }
                        });
                    })($item);
                }
                // 初始化为 EMPTY
                $item.setItemClass("EMPTY");
                //
                // 添加至Table中
                $td.append($item);
                $tr.append($td);
            }
            $matrix.append($tr);
        }
    });
})();

