/*jshint browser: true, devel: true */
/*global remoteStorage: true, RemoteStorage: true, Gesture: true, tiles: true, utils: true, Event: true, scrap: true, saveScraped: true, $:true, $$: true, Showdown: true */
/*exported: alir */
/**
    Alir
    Copyright (C) 2013  Clochix

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.
*/

/**
 * @TODO use notifications
 * https://developer.mozilla.org/en-US/docs/WebAPI/Using_Web_Notifications
 */

var config,
    _;
config = {
  gesture: false,
  vibrate: false,
  rs: {
    login: '',
    syncIntervalOn: 60,
    syncIntervalOff: 300
  },
  dropBox: {
    apiKey: ''
  },
  google: {
    clientId: '',
    apiKey: ''
  },
  lang: 'en-US',
  menu: true,
  menuBottom: false,
  alarmInterval: 60,
  logLevel: 'info',
  proxy: 'http://',
  bookmarks: {},
  alarms: {},
  style: {
    fontSize: 1,
    'font-size': 'sans-serif'
  },
  first: true,
  theme: '',
  logs: ''
};
function displayItem(item) {
  "use strict";
  if (typeof item !== 'object') {
    utils.log("Trying to display undefined item", "error");
  }
  switch (item.type) {
  case 'article':
    window.articles.addToList(item);
    break;
  case 'feed':
    window.feeds.display(item);
    break;
  default:
    // try to display and save item as an article
    if (item.title && (item.text || item.html)) {
      item.type = 'article';
      window.articles.addToList(item);
      remoteStorage.alir.private.remove(item.id);
      remoteStorage.alir.saveArticle(item);
      utils.log(utils.format("Wrong item type %s for %s", item.type, item.title), "warning");
    } else {
      utils.log(utils.format("Unable to display item %s of type %s", item.id, item.type), "error");
    }
  }
}
function Alir() {
  "use strict";
  var self = this,
      dynamicSheet = document.getElementById('userCss').sheet,
      slider = document.getElementById('styleFontSize'),
      family = document.getElementById('styleFontFamily');
  RemoteStorage.eventHandling(this, "configLoaded", "statusUpdated");
  // style {{
  (function () {
    function updateStyle(conf) {
      while (dynamicSheet.cssRules[0]) {
        dynamicSheet.deleteRule(0);
      }
      slider.value = conf.fontSize;
      family.value = conf.fontFamily;
      dynamicSheet.insertRule("#articleShowTile .content .html, #styleFontSizeSample { font-size: " + conf.fontSize + "rem; font-family: " + conf.fontFamily  + " }", 0);
    }
    function onSizeChanged(event) {
      window.config.style.fontSize = slider.value;
      updateStyle(window.config.style);
    }
    function onFamilyChanged(event) {
      window.config.style.fontFamily = family.value;
      updateStyle(window.config.style);
    }
    self.on('configLoaded', function (conf) {
      updateStyle(conf.style);
    });
    slider.addEventListener('change', onSizeChanged);
    slider.addEventListener('input',  onSizeChanged);
    family.addEventListener('change', onFamilyChanged);
    self.styleReset = function () {
      window.config.style.fontSize = 1;
      window.config.style.fontFamily = 'sans-serif';
      updateStyle(window.config.style);
    };
  }());
  // }}
  // status {{
  (function () {
    //jshint maxstatements: 25
    var status;
    status = {
      installed: false,
      online: navigator.onLine,
      user: true,
      visible: !document.hidden,
      rs: remoteStorage.connected,
      fxos: typeof window.MozActivity !== 'undefined' // @FIXME
    };
    if (status.online) {
      document.body.classList.add('net');
    } else {
      document.body.classList.remove('net');
    }
    if (status.fxos) {
      document.body.classList.add('fxos');
    } else {
      document.body.classList.remove('fxos');
    }
    // when navigator goes online / offline
    function windowOnline(e) {
      status.online = navigator.onLine;
      if (status.online) {
        document.body.classList.add('net');
        remoteStorage.startSync();
      } else {
        document.body.classList.remove('net');
        remoteStorage.stopSync();
      }
      self._emit('statusUpdated', status);
    }
    // when visibility of application change
    function documentVisibility() {
      status.visible = !document.hidden;
      self._emit('statusUpdated', status);
    }
    // when user toggle online / offline state
    function userOnline(online) {
      if (online) {
        remoteStorage.alir.goOnline();
        document.body.classList.add('online');
        status.user = true;
      } else {
        remoteStorage.alir.goOffline();
        document.body.classList.remove('online');
        document.body.classList.remove('sync');
        status.user = false;
      }
      self._emit('statusUpdated', status);
    }
    window.addEventListener("offline", windowOnline, false);
    window.addEventListener("online", windowOnline, false);
    document.addEventListener("visibilitychange", documentVisibility, false);
    if (!remoteStorage.connected) {
      document.body.classList.remove('online');
    }
    remoteStorage.on("disconnected", function (e) {
      document.body.classList.remove('online');
      status.rs = false;
      self._emit('statusUpdated', status);
    });
    remoteStorage.remote.on("connected", function (e) {
      document.getElementById('prefToken').value = remoteStorage.remote.token;
      document.body.classList.add('online');
      status.rs = true;
      self._emit('statusUpdated', status);
    });
    // check if application is installed {
    if (window.navigator.mozApps) {
      (function () {
        var request = window.navigator.mozApps.getSelf();
        request.onsuccess = function () {
          if (request.result) {
            status.installed = true;
            document.body.classList.remove('hosted');
            document.body.classList.add('installed');
          }
        };
      }());
    } else {
      status.installed = false;
    }
    // }
    // Toggle user online state {
    self.onoff = function () {
      if (document.body.classList.contains('online')) {
        userOnline(false);
      } else {
        if (remoteStorage.connected) {
          userOnline(true);
        } else {
          if (document.getElementById('rsLogin').value !== '') {
            remoteStorage.widget.view.events.connect(new Event(""));
          } else {
            window.alir.ui.message(_('notConnected'), 'error');
          }
        }
      }
    };
    // }
    self.getStatus = function () {
      return status;
    };
  }());
  // }}
  this.install = function () {
    (function () {
      var host, request;
      host = window.location.protocol + '//' + window.location.host;
      if (window.alir.getStatus().fxos) {
        request = window.navigator.mozApps.install(host + '/hosted.webapp');
      } else {
        request = window.navigator.mozApps.installPackage(host + "/package.manifest");
      }
      request.onerror = function () {
        utils.log("Install Error : " + this.error.name, "error");
        utils.log(this.error, 'error');
        window.alir.ui.message(_('installKo'), "info");
      };
      request.onsuccess = function () {
        window.alir.ui.message(_('installOk'), "info");
      };
    }());
  };
  this.update = function () {
    (function () {
      var request = window.navigator.mozApps.getSelf();
      request.onsuccess = function () {
        if (request.result) {
          request.result.checkForUpdate();
        } else {
          alert("Called from outside of an app");
        }
      };
      request.onerror = function () {
        utils.log("Error: " + request.error.name, "error");
      };
    }());
  };
  this.getAll = function () {
    var startTime = {};

    ['article', 'feed'].forEach(function (type) {
      startTime[type] = window.performance.now();
      remoteStorage.alir.private.getAll(type + '/').then(function (objects) {
        utils.log(utils.format("All %s got in %s", type, Math.round((window.performance.now() - startTime[type]))), "debug");
        if (objects) {
          Object.keys(objects).forEach(function (key) {
            try {
              objects[key].id   = key;
              objects[key].type = type;
              if (type === 'article' && (objects[key].loaded === false || objects[key].title === '???')) {
                self.reload(key);
              }
              displayItem(objects[key]);
            } catch (e) {
              utils.log(utils.format("Error on %s for key %s : %s / %s", type, key, objects[key], e.toString(), "error"));
            }
          });
        }
        tiles.go("articleList");
        utils.log(utils.format("All %s displayed in %s", type, Math.round((window.performance.now() - startTime[type]))), "debug");
      });
    });
  };
  this.rs = {
    "setState": function (state) {
      var actions = {
        "connect": $("#prefRS [data-method=connect]").classList,
        "disconnect": $("#settingsSync [data-method=disconnect]").classList,
        "sync": $("#settingsSync [data-method=sync]").classList,
        "reset": $("#settingsSync [data-method=reset]").classList
      };
      switch (state) {
      case "initial":
        actions.connect.remove("hidden");
        actions.disconnect.add("hidden");
        actions.sync.add("hidden");
        //actions.reset.add("hidden");
        break;
      case "connected":
        actions.connect.add("hidden");
        actions.disconnect.remove("hidden");
        actions.sync.remove("hidden");
        //actions.reset.remove("hidden");
        break;
      default:
        utils.log("unknown state " + state, "warning");
      }
    },
    "setView": function () {
      if (typeof remoteStorage.widget.view === 'undefined') {
        remoteStorage.widget.display();
      }
    },
    "connect": function () {
      remoteStorage.widget.view.form.userAddress.value = $('#rsLogin').value;
      remoteStorage.widget.view.events.connect(new Event(""));
    },
    "connectDropbox": function () {
      remoteStorage.widget.view.connectDropbox();
    },
    "connectDrive": function () {
      remoteStorage.widget.view.connectGdrive();
    },
    "sync": function () {
      remoteStorage.sync.addTask('/alir/');
      remoteStorage.sync.sync();
    },
    "reset": function () {
      remoteStorage.widget.view.events.reset(new Event(""));
    },
    "disconnect": function () {
      remoteStorage.widget.view.events.disconnect(new Event(""));
    },
    "cacheReset": function () {
      remoteStorage.caching.reset();
    },
    "status": ""
  };
  this.reset = function () {
    config.testMode = true;
    localStorage.clear();
    this.rs.setView();
    remoteStorage.widget.view.events.reset(new Event(""));
  };
  this.ui = {
    clearLogs: function () {
      document.getElementById('debugLog').innerHTML =  "";
      document.body.classList.remove('error');
    },
    inspect: function () {
      remoteStorage.inspect();
    },
    message: function (msg, level) {
      self.ui.messagebar.classList.remove('info');
      self.ui.messagebar.classList.remove('error');
      if (typeof msg !== 'undefined') {
        level = level || 'info';
        self.ui.messagebar.classList.add(level);
        self.ui.messagebar.textContent = msg;
        if (level === 'info') {
          window.setTimeout(function () {
            self.ui.message();
          }, 5000);
        }
      } else {
        self.ui.messagebar.textContent = '';
      }
    },
    widgetShow: function () {
      document.getElementById('rsWidget').classList.toggle("hidden");
    },
    setTheme: function (theme) {
      var cl = document.body.classList;
      [].slice.call(document.body.classList).forEach(function (c) {
        if (c.substr(0, 5) === 'theme') {
          cl.remove(c);
        }
      });
      if (theme !== '') {
        cl.add(theme);
      }
    },
    toggleMenu: function (show) {
      if (show === true) {
        document.getElementById('menu').classList.add("show");
        document.body.classList.add("menu");
      } else if (show === false) {
        document.getElementById('menu').classList.remove("show");
        document.body.classList.remove("menu");
      } else {
        document.getElementById('menu').classList.toggle("show");
        document.body.classList.toggle("menu");
      }
    },
    choice: function (choices, title) {
      var tile = document.querySelector('[data-tile=prompt]'),
          tileContent = tile.querySelector('.content'),
          button;
      tile.querySelector('.title').textContent = title || '';
      choices.push({'object': 'tiles', 'method': 'back', 'l10nId': 'back'});
      tileContent.innerHTML = '';
      choices.forEach(function (choice) {
        button = document.createElement('button');
        ['object', 'method', 'params', 'l10nId'].forEach(function (d) {
          if (typeof choice[d] !== 'undefined') {
            button.dataset[d] = choice[d];
          }
        });
        if (typeof choice.l10nId !== 'undefined') {
          button.innerHTML = _(choice.l10nId);
        } else if (typeof choice.content !== 'undefined') {
          button.innerHTML = choice.content;
        }
        tileContent.appendChild(button);
      });
      tiles.go('prompt');
    },
    tip: function (e, text) {
      var rect = e.getBoundingClientRect();
      this.tooltip.textContent = text;
      this.tooltip.style.display = 'block';
      this.tooltip.style.top = (rect.top   + rect.height + 5) + 'px';
      this.tooltip.style.left = (rect.left + (rect.width / 2) - 90) + 'px';
    },
    point: function (e, scroll) {
      var rect = e.getBoundingClientRect();
      if (scroll !== false) {
        window.scrollTo(0, window.scrollY - 90 + e.getBoundingClientRect().top);
      }
      this.pointer.style.display = 'block';
      this.pointer.style.top  = (-40 + rect.top  + rect.height / 2) + 'px';
      this.pointer.style.left = (-40 + rect.left + rect.width / 2) + 'px';
    },
    top: function () {
      var scrollbar = $("#menu .scrollbar"),
          endlistener,
          duration = 0.5;
      endlistener = function () {
        document.body.style.transition = "none";
        scrollbar.style.transition     = "none";
        document.body.removeEventListener("transitionend", endlistener);
      };
      self.ui.toggleMenu(false);
      document.body.style.marginTop = - window.scrollY + "px";
      window.scrollTo(0, 0);
      document.body.style.transition = "margin-top " + duration + "s";
      scrollbar.style.transition = "top " + duration + "s";
      document.body.style.marginTop = 0;
      document.body.addEventListener("transitionend", endlistener, false);
    }
  };
  this.ui.messagebar = document.getElementById('messagebar');
  this.ui.messagebar.addEventListener('click', function () {
    self.ui.message();
  });
  this.ui.tooltip = document.getElementById('tooltip');
  this.ui.pointer = document.getElementById('pointer');
  this.ui.tooltip.addEventListener('click', function () {
    this.style.display = 'none';
    this.tooltip.textContent = '';
  }, false);
  (function () {
    var help  = document.getElementById('help'),
        text  = help.querySelector('.content'),
        prev  = help.querySelector('.prev'),
        close = help.querySelector('.close'),
        next  = help.querySelector('.next'),
        elmt,
        curr  = 0,
        doScroll;
    close.addEventListener('click', function () {
      help.style.display = 'none';
      self.ui.pointer.style.display = 'none';
      tiles._emit('helpClosed', tiles.getCurrent());
    }, false);
    next.addEventListener('click', function () {
      curr++;
      if (curr === elmt.length) {
        curr = 0;
      }
      display();
    }, false);
    prev.addEventListener('click', function () {
      curr--;
      if (curr === -1) {
        curr = elmt.length - 1;
      }
      display();
    }, false);
    function display() {
      if (elmt[curr]) {
        text.textContent = _(elmt[curr].dataset.help);
        if (elmt[curr].dataset.helpNoScroll) {
          self.ui.point(elmt[curr], false);
        } else {
          self.ui.point(elmt[curr], doScroll);
        }
        help.style.display = 'block';
      }
    }
    self.ui.help = function (sel, scroll) {
      doScroll = scroll;
      if (sel) {
        elmt = $$(sel + " [data-help]");
      } else {
        tiles._emit('help', tiles.getCurrent());
        elmt = $$('[data-help]', tiles.getCurrentTile());
        self.ui.toggleMenu(false);
      }
      // remove hidden elements
      elmt = elmt.filter(function (e) {
        var rect = e.getBoundingClientRect();
        return rect.width > 0 && rect.height > 0;
      });
      window.scrollTo(0, 0);
      // Wait for menu to disapear
      window.setTimeout(function () {
        curr = 0;
        display();
      }, 500);
    };
  }());
}
window.alir = new Alir();
window.alir.on('statusUpdated', function (status) {
  "use strict";
  //console.log(status);
});
// network depends on alir
// @FIXME move creation of all global objects elsewhere
window.network = new window.Network();

window.link = {
  // Open link in external browser
  open: function (site) {
    "use strict";
    var href = document.getElementById('linkRef').textContent,
        openURL;
    if (site) {
      href = site + window.encodeURIComponent(href);
    }
    if (typeof window.MozActivity !== 'undefined') {
      try {
        openURL = new window.MozActivity({
          name: "view",
          data: {
            type: "url",
            url: href
          }
        });
      } catch (e) {
        window.open(href);
      }
    } else {
      window.open(href);
    }
    tiles.back();
  },
  scrap: function (href) {
    "use strict";
    if (typeof href === 'undefined') {
      href = document.getElementById('linkRef').textContent;
      tiles.back();
    }
    try {
      utils.log("Scraping " + href, "info");
      window.alir.ui.message(_("scraping") + ' ' + href, "info");
      scrap(href, function (err, res) {
        if (err) {
          utils.log(err.toString(), 'error');
          window.alir.ui.message(_('scrapingError', {url: href}), 'error');
          res.loaded = false;
        }
        saveScraped(res);
      });
    } catch (e) {
      utils.log(e.toString(), "error");
      window.alir.ui.message(_('scrapingError', {url: href}), 'error');
    }
  },
  share: function () {
    "use strict";
    var href = document.getElementById('linkRef').textContent,
        request;
    if (typeof window.MozActivity !== 'undefined') {
      try {
        request = new window.MozActivity({
          name: "share",
          data: {
            type: "url",
            url: href
          }
        });
        request.onerror = function () {
          if (request.error.name !== 'USER_ABORT') {
            utils.log("Error sharing : " + request.error.name, "error");
            window.alir.ui.message(_('sharingError'), 'error');
          }
        };
      } catch (e) {
        utils.log("Error sharing : " + e, "error");
        window.alir.ui.message(_('sharingError'), 'error');
      }
    } else {
      // @TODO How to share ???
      utils.log("Sorry, share is not available", "error");
    }
    tiles.back();
  },
  facebook: function () {
    "use strict";
    var txt = window.encodeURIComponent(document.getElementById('linkText').value);
    this.open('https://www.facebook.com/sharer/sharer.php?t=' + txt + '&u=');
  },
  google: function () {
    "use strict";
    this.open('https://plus.google.com/share?url=');
  },
  twitter: function () {
    "use strict";
    var txt = window.encodeURIComponent(document.getElementById('linkText').value);
    this.open('https://twitter.com/intent/tweet?text=' + txt + '&url=');
  }
};

RemoteStorage.defineModule('alir', function module(privateClient, publicClient) {
  "use strict";

  // Define a common data type using JSON Schema
  privateClient.declareType('article', {
    "description": "Article",
    "type": "object",
    "properties": {
      "id": {
        "type": "string",
        "format": "id"
      },
      "title": {
        "type": "string"
      },
      "html": {
        "type": "string"
      },
      "text": {
        "type": "string"
      },
      "tags": {
        "type": "array"
      },
      "notes": {
        "type": "object"
      },
      "flags": {
        "type": "object"
      },
      "alternate": {
        "type": "array"
      },
      "loaded": {
        "type": "boolean"
      }
    }
  });

  privateClient.declareType('feed', {
    "description": "Feed",
    "type": "object",
    "properties": {
      "id": {
        "type": "string",
        "format": "id"
      },
      "title": {
        "type": "string"
      },
      "url": {
        "type": "string"
      },
      "lastUpdated": {
        "type": "string"
      },
      "type": {
        "type": "string"
      },
      "short": {
        "type": "boolean"
      },
      "articles": {
        "type": "object"
      }
    }
  });

  function getObject(type, key, cb) {
    var path = type + '/' + key;
    remoteStorage.alir.private.getObject(path).then(function (obj) {
      if (obj) {
        obj.id = key;
        cb(obj);
      } else {
        utils.log("Unable to load " + path, "error");
        path = '/' + path;
        remoteStorage.alir.private.getObject(path).then(function (obj) {
          if (obj) {
            obj.id = key;
          } else {
            utils.log("Unable to load " + path, "error");
            window.alir.ui.message(_('itemNotFound', {type: type}), "error");
          }
          cb(obj);
        });
      }
    });
  }
  return {
    exports: {
      getArticle: function (key, cb) {
        getObject('article', key, cb);
      },
      saveArticle: function (article) {
        article.type = 'article';
        return privateClient.storeObject('article', 'article/' + article.id, article);
      },
      getFeed: function (key, cb) {
        getObject('feed', key, cb);
      },
      saveFeed: function (feed) {
        feed.type = 'feed';
        return privateClient.storeObject('feed', 'feed/' + feed.id, feed);
      },
      goOffline: function () {
        remoteStorage.setSyncInterval(config.rs.syncIntervalOff * 1000);
      },
      goOnline: function () {
        remoteStorage.setSyncInterval(config.rs.syncIntervalOn * 1000);
        remoteStorage.startSync();
      },
      private: privateClient,
      public: publicClient
    }
  };
});
function initUI() {
  // jshint maxstatements: 60
  "use strict";
  function clicked(elmt) {
    if (typeof elmt !== 'undefined' && elmt instanceof Element) {
      elmt.classList.add('clicked');
      window.setTimeout(function () {
        elmt.classList.remove('clicked');
      }, 400);
    }
    if (config.vibrate && typeof window.navigator.vibrate === 'function') {
      window.navigator.vibrate(50);
    }
  }
  // configuration {{
  (function () {
    // jshint maxcomplexity: 12
    var conf = localStorage.getItem('config');
    function firstUse() {
      // Create first use content
      function doFirst() {
        var article,
            xhr = new XMLHttpRequest({ mozSystem: true });
        document.documentElement.lang = document.webL10n.getLanguage();
        document.documentElement.dir = document.webL10n.getDirection();
        if (config.first === true) {
          // if mozSystem is not available, init a CORS Proxy
          if (typeof xhr.mozSystem !== 'boolean' || xhr.mozSystem !== true) {
            config.proxy = window.location.protocol + '//www.corsproxy.com/';
            document.getElementById('proxyUrl').value = config.proxy;
          }
          article = {
            id:    utils.uuid(),
            title: _('firstArticleTitle'),
            html:  new Showdown.converter().makeHtml(_('firstArticleMd')),
            date:  Date.now(),
            flags: {
              editable: false
            }
          };
          remoteStorage.alir.saveArticle(article);

          // Guided tour
          window.alir.ui.toggleMenu(true);
          window.alir.ui.help('#menu', false);
        }
      }
      if (document.webL10n.getReadyState() === 'complete') {
        doFirst();
      } else {
        document.addEventListener('localized', doFirst);
      }
    }
    if (conf) {
      conf = JSON.parse(conf);
      utils.merge(config, conf);

      config.first = conf.first;
      firstUse();

      if (typeof conf.lang !== 'undefined') {
        $('#settingsLang select').value = conf.lang;
        if (document.webL10n.getLanguage() !== conf.lang) {
          document.webL10n.setLanguage(conf.lang);
        }
      }

      if (conf.rs.login) {
        $('#rsLogin').value = conf.rs.login;
      }
      if (conf.dropBox.apiKey) {
        $('#dropboxApiKey').value = conf.dropBox.apiKey;
        remoteStorage.setApiKeys('dropbox', {api_key: conf.dropBox.apiKey});
      }
      if (conf.google.apiKey && conf.google.clientId) {
        $('#driveClientId').value = conf.goole.clientId;
        $('#driveApiKey').value   = conf.google.apiKey;
        remoteStorage.setApiKeys('googledrive', {client_id: conf.google.clientId, api_key: conf.google.apiKey});
      }
      if (conf.menu) {
        document.body.classList.remove('menu-right');
        document.body.classList.add('menu-left');
      } else {
        document.body.classList.remove('menu-left');
        document.body.classList.add('menu-right');
      }
      if (conf.menuBottom) {
        document.body.classList.add('bottom');
        document.getElementById('prefMenuBottom').checked = true;
      } else {
        document.getElementById('prefMenuBottom').checked = false;
      }
      (function () {
        var alarm = $('#alarmInterval'),
            alarmValue = $('#alarmIntervalValue');
        function onIntervalChanged(event) {
          var value = event.target.value;
          conf.alarmInterval     = value;
          alarmValue.textContent = value;
          if (event.type === 'change' && typeof window.alarms !== "undefined") {
            window.alarms.reset(function () {
              window.alarms.plan();
            });
          }
        }
        alarm.value            = conf.alarmInterval;
        alarmValue.textContent = conf.alarmInterval;
        alarm.addEventListener('change', onIntervalChanged);
        alarm.addEventListener('input', onIntervalChanged);
      }());
      // sync interval {{
      ['On', 'Off'].forEach(function (mode) {
        var range      = $('#rsSyncInterval' + mode),
            rangeValue = $('#rsSyncInterval' + mode + 'Value'),
            prefName   = 'syncInterval' + mode;
        function onIntervalChanged(event) {
          var value = event.target.value;
          conf.rs[prefName] = value;
          rangeValue.textContent = value;
          if (event.type === 'change') {
            if (mode === 'On') {
              remoteStorage.setSyncInterval(value * 1000);
            }
            //if (mode === 'Off') {
            //  remoteStorage.setBackgroundSyncInterval(value * 1000);
            //}
          }
        }
        range.value            = conf.rs[prefName];
        rangeValue.textContent = conf.rs[prefName];
        range.addEventListener('change', onIntervalChanged);
        range.addEventListener('input', onIntervalChanged);
      });
      remoteStorage.setSyncInterval(conf.rs.syncIntervalOn * 1000);
      //remoteStorage.setBackgroundSyncInterval(conf.rs.syncIntervalOff * 1000);
      // }}

      $('#settingsLoglevel select').value = conf.logLevel;
      utils.logLevel = conf.logLevel;

      $('#settingsTheme select').value = conf.theme;

      $('#proxyUrl').value = conf.proxy;

      window.alir.ui.setTheme(conf.theme);

      if (typeof conf.bookmarks === 'undefined') {
        conf.bookmarks = {};
      }

      if (conf.logs !== '') {
        document.getElementById('debugLog').innerHTML += conf.logs;
      }

      config = conf;
    } else {
      firstUse();
    }
    window.alir._emit('configLoaded', config);
  }());
  // }}

  $$('form').forEach(function (form) {
    form.addEventListener('submit', function (e) {
      e.preventDefault();
      return false;
    });
  });

  // Actions {{
  (function () {

    // Generic event Listener
    document.body.addEventListener('click', function (ev) {
      //jshint curly: false
      var params = [],
          target  = ev.target,
          dataset = target.dataset,
          listeners,
          obj, path;

      listeners = [
        {
          // Reset button
          sel: "button[type=reset]",
          action: function (elmt) {
            var res = utils.parent(elmt, function (el) { return el.tagName === 'FIELDSET'; });
            if (res !== null) {
              $$("input", res).forEach(function (el) {
                el.value = '';
              });
            }
          }
        },
        // Links inside content
        {
          sel: "#articleShowTile a[href][target]",
          action: function (elmt) {
            document.getElementById("linkRef").textContent = elmt.href;
            document.getElementById("linkRefFull").textContent = elmt.href;
            document.getElementById("linkText").value = '';
            tiles.go('link');
          }
        },
        // Containers
        {
          sel: ".container .header",
          action: function (elmt) {
            var target = utils.parent(elmt, function (el) { return el.classList.contains('container'); });
            if (target) {
              target.classList.toggle('folded');
            }
          }
        }
      ];
      listeners.forEach(function (listener) {
        if (utils.match(ev.target, listener.sel)) {
          ev.preventDefault();
          clicked(ev.target);
          listener.action(ev.target);
        }
      });
      if (typeof dataset.method === 'undefined' && typeof ev.target.parentNode.dataset.method !== 'undefined') {
        target  = ev.target.parentNode;
        dataset = ev.target.parentNode.dataset;
      }
      if (dataset.method && typeof dataset.object !== 'undefined') {
        if (typeof dataset.params !== "undefined") {
          params = dataset.params.split(',');
        }
        path = dataset.object.split('.');
        obj = window;
        do {
          obj = obj[path.shift()];
        } while (typeof obj !== 'undefined' && path.length > 0);
        if (typeof obj === 'undefined') {
          utils.log("Unknown data object " + dataset.object, "error");
        } else {
          if (typeof obj[dataset.method] !== 'function') {
            utils.log(utils.format("Object %s has no method %s", dataset.object, dataset.method), "error");
          } else {
            clicked(target);
            ev.preventDefault();
            obj[dataset.method].apply(obj, params);
            return false;
          }
        }
      }
    });
    document.body.addEventListener('focus', function (ev) {
      if (ev.target.tagName === 'TEXTAREA') {
        // Move carret to the end of textarea
        ev.target.selectionStart = ev.target.value.length;
      }
    }, true);
    // JK keyboard navigation inside lists
    document.body.addEventListener('keypress', function (ev) {
      var parent;
      if (ev.target.tagName === 'A' && ev.keyCode === 0 && (ev.charCode === 106 || ev.charCode === 107)) {
        parent = utils.parent(ev.target, function (e) {return e.tagName === "LI"; });
        if (parent) {
          if (ev.charCode === 106) {
            parent = parent.previousElementSibling;
          } else {
            parent = parent.nextElementSibling;
          }
          if (parent) {
            parent = parent.querySelector("a");
          }
        }
        if (parent) {
          parent.focus();
        }
      }
    }, false);

  })();
  // }}

  remoteStorage.on("wire-busy", function (e) {
    document.body.classList.add('sync');
  });
  remoteStorage.on("wire-done", function (e) {
    document.body.classList.remove('sync');
  });
  remoteStorage.remote.on("wire-busy", function (e) {
    document.body.classList.add('sync');
  });
  remoteStorage.remote.on("wire-done", function (e) {
    document.body.classList.remove('sync');
  });
  remoteStorage.on("error", function (e) {
    if (e instanceof RemoteStorage.SyncError) {
      if (window.alir.getStatus().online === true) {
        utils.log("remoteStorage sync error: " + e, "error");
      }
    } else {
      utils.log("remoteStorage error: " + e, "error");
    }
  });

  remoteStorage.on("features-loaded", function (e) {
    var features = [];
    remoteStorage.features.forEach(function (feature) {
      if (feature.supported === true) {
        features.push(feature.name);
      }
    });
    if (features.indexOf('Dropbox') !== -1) {
      $('#prefDropbox').classList.remove('hidden');
    } else {
      $('#prefDropbox').classList.add('hidden');
    }
    if (features.indexOf('GoogleDrive') !== -1) {
      $('#prefDrive').classList.remove('hidden');
    } else {
      $('#prefDrive').classList.add('hidden');
    }

  });
  // Gestures {{
  (function () {
    var gestureEvents;
    gestureEvents = {
      gesture: function (e) {
        var items;
        function getItems() {
          var current  = $("#list > .current"),
              previous,
              next,
              items = $$("#list > li"),
              currentIndex;
          if (current === null) {
            return false;
          }
          if (items.length === 1) {
            previous = next = current;
          } else {
            items.some(function (e, i) {
              if (e.classList.contains("current")) {
                currentIndex = i;
              } else {
                return false;
              }
            });
            if (currentIndex === 0) {
              previous = items[items.length - 1];
              next     = items[currentIndex + 1];
            } else if (currentIndex === items.length - 1) {
              previous = items[currentIndex - 1];
              next     = items[0];
            } else {
              previous = items[currentIndex - 1];
              next     = items[currentIndex + 1];
            }
          }
          return [previous, current, next];
        }
        switch (e.detail.dir) {
        case 'E':
          items = getItems();
          if (items) {
            config.bookmarks[items[1].dataset.key] = window.scrollY / document.querySelector("[data-tile].shown").clientHeight;
            items[1].classList.add('hideRight');
            items[2].classList.add('showLeft');
            window.setTimeout(function () {
              items[1].classList.remove('current');
              items[1].classList.remove('hideRight');
              items[2].classList.remove('showLeft');
              window.articles.show(items[2].dataset.key);
            }, 500);
          }
          window.articles.ui.menu(false);
          break;
        case 'W':
          items = getItems();
          if (items) {
            config.bookmarks[items[1].dataset.key] = window.scrollY / document.querySelector("[data-tile].shown").clientHeight;
            items[1].classList.add('hideLeft');
            items[0].classList.add('showRight');
            window.setTimeout(function () {
              items[1].classList.remove('current');
              items[1].classList.remove('hideLeft');
              items[0].classList.remove('showRight');
              window.articles.show(items[0].dataset.key);
            }, 500);
          }
          window.articles.ui.menu(false);
          break;
        }
      }
    };
    if (config.gesture) {
      Gesture.attach(document.getElementById('articleShow'), gestureEvents);
    }
  }());
  // Delete article with left swipe in list
  Gesture.attach(document.getElementById('main'), {
    gesture: function (e) {
      var target;
      if (e.detail.dir === 'W') {
        target = utils.parent(e.detail.startTarget, function (e) { return typeof e.dataset.key !== 'undefined'; });
        if (target) {
          window.articles.delete(target.dataset.key, target.dataset.title);
        }
      }
    }
  });
  // }}
  // Tags {{
  (function () {
    var input = $('#tagTile [name=tagInput]');
    input.addEventListener('change', function () {
      tiles.back(input.value);
    });
    $('#tagList').addEventListener("click", function (event) {
      if (event.target.dataset && event.target.dataset.tag) {
        tiles.back(event.target.dataset.tag);
      }
    });
    $('#tagTile [name="save"]').addEventListener('click', function () {
      tiles.back(input.value);
    });
  }());
  // }}
  // Filters {{
  (function () {
    var filter = document.getElementById('listFilter');
    filter.addEventListener("input", window.articles.ui.updateFilter);
    filter.addEventListener("change", window.articles.ui.updateFilter);
    document.querySelector("#listFilter + button").addEventListener("click", function () {
      filter.value = '';
      window.articles.ui.updateFilter();
    });
  }());
  // }}
  // {{ Settings

  $('#rsLogin').addEventListener('change', function () {
    config.rs.login = this.value;
    remoteStorage.widget.view.form.userAddress.value = this.value;
  });
  remoteStorage.on('ready', function () {
    utils.log(utils.format("Ready fired at %s", Math.round(window.performance.now())), "debug");
    window.alir.rs.setState(remoteStorage.remote.connected ? 'connected' : 'initial');
    window.alir.rs.status = 'ready';
  });
  remoteStorage.on('disconnected', function () {
    window.alir.rs.setState('initial');
  });
  window.alir.rs.setState('initial');
  document.getElementById('dropboxApiKey').addEventListener('change', function () {
    remoteStorage.setApiKeys('dropbox', {api_key: this.value});
    config.dropBox.apiKey = this.value;
  });
  document.getElementById('driveClientId').addEventListener('change', function () {
    remoteStorage.setApiKeys('googledrive', {client_id: this.value, api_key: $('#driveApiKey').value});
    config.google.clientId = this.value;
  });
  document.getElementById('driveApiKey').addEventListener('change', function () {
    remoteStorage.setApiKeys('googledrive', {client_id: $('#driveClientId').value, api_key: this.value});
    config.google.apiKey = this.value;
  });
  document.body.addEventListener('change', function (ev) {
    // jshint maxcomplexity: 11
    var val = ev.target.value;
    if (ev.target.dataset.target) {
      switch (ev.target.dataset.target) {
      case 'gesture':
        config.gesture = ev.target.checked;
        //if (config.gesture) {
        //  Gesture.attach(document.getElementById('articleShow'), gestureEvents);
        //} else {
        //  Gesture.detach(document.getElementById('articleShow'), gestureEvents);
        //}
        break;
      case 'vibrate':
        config.vibrate = ev.target.checked;
        break;
      case 'lang':
        config.lang = val;
        document.webL10n.setLanguage(val);
        break;
      case 'logLevel':
        config.logLevel = val;
        utils.logLevel  = val;
        break;
      case 'theme':
        config.theme = val;
        window.alir.ui.setTheme(val);
        break;
      case 'proxy':
        config.proxy = val;
        break;
      case 'prefMenuBottom':
        val = ev.target.checked;
        config.menuBottom = val;
        document.body.classList.toggle('bottom');
        break;
      case 'prefMenuLeft':
        val = ev.target.checked;
        config.menu = val;
        if (val) {
          document.body.classList.remove('menu-right');
          document.body.classList.add('menu-left');
        } else {
          document.body.classList.remove('menu-left');
          document.body.classList.add('menu-right');
        }
        break;
      }
    }
  });
  // }}

  // Link share text {{
  (function () {
    $$('.textareaLength').forEach(function (l) {
      var t = l.previousElementSibling;
      function onInput(event) {
        l.textContent = t.value.length;
      }
      t.addEventListener('change', onInput, false);
      t.addEventListener('input', onInput, false);
      t.addEventListener('focus', onInput, false);
    });
  }());
  // }}

  // Manage scroll
  (function () {
    var height = document.querySelector("[data-tile].shown").clientHeight,
        scroll = $("#menu .scrollbar");
    //scroll.style.height = (window.innerHeight / document.querySelector("[data-tile].shown").clientHeight * 100) + '%';
    function checkSize() {
      var tile, h, header, hh, tmp;
      tile = document.querySelector("[data-tile].shown");
      header = document.querySelector('.shown > header,.shown > * > header');
      if (tile) {
        h = tile.clientHeight;
        if (header) {
          if (config.menuBottom) {
            hh = header.getBoundingClientRect().top - parseInt(window.getComputedStyle(header).marginTop, 10);
          } else {
            hh = header.getBoundingClientRect().bottom - parseInt(window.getComputedStyle(header).marginBottom, 10);
          }
          tmp = (1 - (window.scrollY / (tile.clientHeight - document.body.clientHeight)));
          if (tmp < -0.5) {
            tmp = 1;
          }
          if (config.menuBottom) {
            scroll.style.top = (window.scrollY / (document.querySelector("[data-tile].shown").clientHeight + hh) * 100) + '%';
          } else {
            scroll.style.top = "calc("  + (hh * tmp) + "px + " + (window.scrollY / document.querySelector("[data-tile].shown").clientHeight * 100) + '%)';
          }
        } else {
          scroll.style.top = (window.scrollY / document.querySelector("[data-tile].shown").clientHeight * 100) + '%';
        }
        if (h !== height || scroll.style.height === '') {
          height = h;
          if (header) {
            scroll.style.height = ((window.innerHeight - header.clientHeight) / document.querySelector("[data-tile].shown").clientHeight * 100) + '%';
          } else {
            scroll.style.height = (window.innerHeight / document.querySelector("[data-tile].shown").clientHeight * 100) + '%';
          }
        }
      }
    }
    window.tiles.on('shown', checkSize);
    setInterval(checkSize, 250);
    document.addEventListener('scroll', checkSize);
  })();

  // Preferences
  (function () {
    ['gesture', 'vibrate'].forEach(function (pref) {
      var elmt = document.getElementById('pref' + pref[0].toUpperCase() + pref.substr(1));
      if (elmt) {
        elmt.checked = config[pref];
      }
    });
  }());

  function onHash() {
    var hash   = window.location.hash.substr(1),
        params = hash.split('|').slice(1),
        url;
    hash = hash.split('|')[0];
    if (hash === 'articleShow' && params.length > 0 && window.articles.getCurrentId() !== params[0]) {
      window.articles.show(params[0]);
      return true;
    }
    if (hash === 'scrap' && params.length > 0) {
      try {
        url = decodeURIComponent(params[0]);
        utils.log("Scraping " + url);
        scrap(url, function (err, res) {
          saveScraped(res);
          if (err) {
            utils.log(err.toString(), 'error');
          }
        });
      } catch (e) {
        utils.log(e.toString(), "error");
      }
      return true;
    }
    if (tiles.exists(hash) && tiles.getCurrent() !== hash && ['articleShow', 'link', 'noteEdit', 'noteView', 'prompt', 'imgShow'].indexOf(hash) === -1) {
      tiles.show(hash);
      return true;
    }
    return false;
  }
  window.addEventListener("hashchange", onHash, false);

  if (!onHash()) {
    // display default tile
    tiles.show('articleList');
  }

  // prepare bookmarklet
  (function () {
    //jshint scripturl: true
    window.name = "Alir";
    document.getElementById('bookmarklet').href = "javascript:window.open('" + utils.format("%s//%s/#scrap|", window.location.protocol, window.location.host) + "' + encodeURIComponent(window.location.toString()), 'Alir')";
  }());

  if (typeof document.getElementById("authFrame").setVisible === "function") {
    // the application is installed, override auth methods
    (function () {
      var frame = document.getElementById("authFrame"), displayed = true;
      frame.setVisible(false);
      function done() {
        frame.removeEventListener('mozbrowserlocationchange', onLocationChange);
        tiles.back();
        frame.setVisible(false);
        displayed = false;
      }
      function onLocationChange(e) {
        var hash, params = {}, domain;
        if (e.detail !== frame.getAttribute("src")) {
          frame.setAttribute("src", e.detail);
        }
        hash = e.detail.split('#');
        if (hash.length === 2) {
          hash = hash[1];
          hash.split('&').forEach(function (p) {
            var tmp = p.split('=');
            params[decodeURIComponent(tmp[0])] = decodeURIComponent(tmp[1]);
          });
          if (params.access_token) {
            RemoteStorage.Authorize._rs_init(remoteStorage);
            remoteStorage._emit("features-loaded");
            done();
            window.alir.rs.setState('connected');
            window.alir.rs.sync();
          }
        }
        if (displayed === true) {
          domain = new RegExp("(http.?://([^/]+).*/)([^?]*).?(.*$)").exec(e.detail);
          if (domain !== null && domain[2] === 'localhost') {
            // something strange is happening, let the user decide
            done();
          }
        }
      }
      remoteStorage.on('authing', function () {
        tiles.go("auth");
        frame.setVisible(true);
      });
      RemoteStorage.Authorize.getLocation = function () {
        var location = frame.getAttribute("src");
        if (location === null) {
          location = "http://localhost/";
        }
        return {
          href: location,
          toString: function () {
            return location;
          }
        };
      };
      RemoteStorage.Authorize.setLocation = function (location) {
        frame.setAttribute("src", location);
        frame.addEventListener('mozbrowserlocationchange', onLocationChange);
      };
    }());
  }
}
// }}

window.addEventListener('load', function () {
  //jshint maxstatements: 30
  "use strict";
  _ = document.webL10n.get;
  var hasPending = false,
      isActivity = false;
  // Check if application is installed
  if (typeof navigator.mozHasPendingMessage === 'function' && typeof navigator.mozSetMessageHandler === 'function') {
    if (navigator.mozHasPendingMessage('alarm')) {
      hasPending = true;
      navigator.mozSetMessageHandler("alarm", window.feeds.handleAlarmMessage);
    }
    if (navigator.mozHasPendingMessage('activity')) {
      hasPending = true;
      isActivity = true;
      navigator.mozSetMessageHandler("activity", window.activityHandler);
    }
    if (navigator.mozHasPendingMessage('notification')) {
      hasPending = true;
      navigator.mozSetMessageHandler("notification", function (notification) {
        utils.log(notification, "debug");
      });
    }
  }

  remoteStorage.access.claim('alir', 'rw');
  remoteStorage.caching.enable('/alir/');
  window.alir.on('configLoaded', function (conf) {
    //remoteStorage.enableLog();
    remoteStorage.displayWidget("rsWidget");
    if (conf.rs.login) {
      remoteStorage.widget.view.form.userAddress.value = conf.rs.login;
    }
  });
  // config is loaded during initUI, so this must be registered BEFORE
  window.alir.on('configLoaded', function () {
    Object.keys(window.config.alarms).forEach(function (d) {
      window.alarms.set(new Date(d));
    });
    window.alarms.plan();
  });
  initUI();
  if (isActivity === false) {
    remoteStorage.alir.private.on('change', function onChange(ev) {
      var elmt, item, id;
      id = ev.relativePath.split('/').pop();
      if (typeof ev.oldValue === 'undefined' && typeof ev.newValue !== 'undefined') {
        //console.log("Create " + ev.relativePath);
        if (typeof ev.newValue.id === 'undefined') {
          ev.newValue.id = id;
        }
        if (typeof ev.newValue.type === 'undefined') {
          ev.newValue.type = ev.relativePath.split('/').shift();
        }
        displayItem(ev.newValue);
      } else if (typeof ev.oldValue !== 'undefined' && typeof ev.newValue === 'undefined') {
        //console.log("Delete " + ev.relativePath);
        elmt = document.getElementById(id);
        if (elmt) {
          elmt.parentNode.removeChild(elmt);
        }
        delete config.bookmarks[ev.relativePath];
      } else if (typeof ev.oldValue !== 'undefined' && typeof ev.newValue !== 'undefined') {
        //console.log("Update " + ev.relativePath);
        if (typeof ev.newValue.id === 'undefined') {
          ev.newValue.id = id;
        }
        if (typeof ev.newValue.type === 'undefined') {
          ev.newValue.type = ev.relativePath.split('/').shift();
        }
        item = displayItem(ev.newValue);
      }
    });
  }
  window.tiles.on('shown', function (name) {
    if (name === 'settings') {
      document.body.classList.remove('error');
    }
  });
  //@TODO Remove this
  // This is just a migration step for previous contents
  /*
  (function () {
    remoteStorage.alir.private.getAll('').then(function (all) {
      if (typeof all === 'object') {
        Object.keys(all).forEach(function (key) {
          if (key.substr(-1) !== '/') {
            utils.log("Migrating article " + all[key].title, "info");
            all[key].id = key;
            remoteStorage.alir.saveArticle(all[key]);
            remoteStorage.alir.private.remove(key);
          }
        });
      } else {
        console.log('Nothing to migrate');
      }
    });
  }());
  */
  //window.alir.getAll();

}, false);
window.addEventListener('unload', function () {
  "use strict";
  config.logs = document.getElementById('debugLog').innerHTML;
  config.first = false;
  if (config.testMode !== true) {
    localStorage.setItem('config', JSON.stringify(config));
  }
}, false);
window.onerror = function myErrorHandler(errorMsg, url, lineNumber) {
  "use strict";
  try {
    utils.log("[error] %s at %s %s", errorMsg, url, lineNumber, "error");
  } catch (e) {
    console.log(e);
  }
};
