/**
 * @filter           Brightness / Contrast
 * @description      Provides additive brightness and multiplicative contrast control.
 * @param brightness -1 to 1 (-1 is solid black, 0 is no change, and 1 is solid white)
 * @param contrast   -1 to 1 (-1 is solid gray, 0 is no change, and 1 is maximum contrast)
 *
 * @filter           Hue / Saturation
 * @description      Provides rotational hue and multiplicative saturation control. RGB color space
 *                   can be imagined as a cube where the axes are the red, green, and blue color
 *                   values. Hue changing works by rotating the color vector around the grayscale
 *                   line, which is the straight line from black (0, 0, 0) to white (1, 1, 1).
 *                   Saturation is implemented by scaling all color channel values either toward
 *                   or away from the average color channel value.
 * @param hue        -1 to 1 (-1 is 180 degree rotation in the negative direction, 0 is no change,
 *                   and 1 is 180 degree rotation in the positive direction)
 * @param saturation -1 to 1 (-1 is solid gray, 0 is no change, and 1 is maximum contrast)
 */

//precision highp float;
precision mediump float;
//precision lowp float;

uniform sampler2D image;

varying vec2 src_position;

uniform float hue;
uniform float saturation;

uniform float brightness;
uniform float contrast;
 
uniform float temperature;

const vec3 warmFilter = vec3(0.93, 0.54, 0.0);

const mat3 RGBtoYIQ = mat3(0.299, 0.587, 0.114, 0.596, -0.274, -0.322, 0.212, -0.523, 0.311);
const mat3 YIQtoRGB = mat3(  1.0, 0.956, 0.621,   1.0, -0.272, -0.647,   1.0, -1.105, 1.702);

void main() {
    vec4 color = texture2D(image, src_position);

    /* --------------------------------------------------------------
     * temperature
     */

    vec3 processed = vec3(
        (color.r < 0.5 ? (2.0 * color.r * warmFilter.r) : (1.0 - 2.0 * (1.0 - color.r) * (1.0 - warmFilter.r))), //adjusting temperature
        (color.g < 0.5 ? (2.0 * color.g * warmFilter.g) : (1.0 - 2.0 * (1.0 - color.g) * (1.0 - warmFilter.g))), 
        (color.b < 0.5 ? (2.0 * color.b * warmFilter.b) : (1.0 - 2.0 * (1.0 - color.b) * (1.0 - warmFilter.b)))
    );

    color = vec4(mix(color.rgb, processed, temperature), color.a);
    // --------------------------------------------------------------
    
    /* --------------------------------------------------------------
     * brightness / contrast 
     */
    
    color.rgb += brightness;
    if (contrast > 0.0) {
        color.rgb = (color.rgb - 0.5) / (1.0 - contrast) + 0.5;
    } else {
        color.rgb = (color.rgb - 0.5) * (1.0 + contrast) + 0.5;
    }
    // --------------------------------------------------------------
    
    /* --------------------------------------------------------------
     * hue / saturation 
     */
    
    // hue adjustment, wolfram alpha: RotationTransform[angle, {1, 1, 1}][{x, y, z}] 
    float angle = hue * 3.14159265;
    float s = sin(angle), c = cos(angle);
    vec3 weights = (vec3(2.0 * c, -sqrt(3.0) * s - c, sqrt(3.0) * s - c) + 1.0) / 3.0;
    float len = length(color.rgb);
    color.rgb = vec3(
        dot(color.rgb, weights.xyz),
        dot(color.rgb, weights.zxy),
        dot(color.rgb, weights.yzx)
    );
    
    // saturation adjustment
    float average = (color.r + color.g + color.b) / 3.0;
    if (saturation > 0.0) {
        color.rgb += (average - color.rgb) * (1.0 - 1.0 / (1.001 - saturation));
    } else {
        color.rgb += (average - color.rgb) * (-saturation);
    }
    // --------------------------------------------------------------
    
    gl_FragColor = color;
}
