/**
 * Loads additional information about a bible verse.
 *
 * JSON format:
 * {
 *    "Ge 1:1" : [
 *      a,
 *      b,
 *    ],
  *   ...
 * }
 *
 * Call loadForBibleText()
 *
 * @param urlTemplate {String}  e.g. dataURL("bible/footnotes/%BOOKCODE%.footnotes.json")
 * @param property {String}  which property to store it in, e.g. "footnotes"
 */
function LoadBibleAnnotation(storage, urlTemplate, property) {
  assert(storage instanceof Storage);
  assert(property && typeof(property) == "string");
  assert(urlTemplate && typeof(urlTemplate) == "string");
  this._urlTemplate = urlTemplate;
  this._property = property;
  this._storage = storage;
  this._loaded = {};
}
LoadBibleAnnotation.prototype = {
  _storage : null, // {Storage}
  _loaded : null, // {Map book.code -> true/false}   have loaded this already

  /**
   * @param bt {BibleText}
   * @param successCallback {Function({Array of annotation})}
   */
  loadForBibleText : function(bt, successCallback, errorCallback) {
    var self = this;
    this.loadForBibleBook(bt.book, function(storage) {
      storage.getAll(null, function(subjs) {
        var result = [];
        subjs.forEach(function(subj) {
          if (bt.contains(subj)) {
            result = result.concat(subj[self._property]);
          }
        });
        successCallback(result);
      }, errorCallback);
    }, errorCallback);
  },

  /**
   * Convenience function to load the file from a URL,
   * parse it and return the DB.
   * @param book {BibleText.books[n]}
   * @param successCallback {Function(db {Storage})}
   * @param errorCallback
   */
  loadForBibleBook : function(book, successCallback, errorCallback) {
    if (this._loaded[book.code]) {
      successCallback(this._storage);
      return;
    }
    var self = this;
    var url = this._urlTemplate.replace("%BOOKCODE%", book.code);
    console.log("Loading " + url);
    this._loadFromURL(url, function(storage) {
      self._loaded[book.code] = true;
      successCallback(storage);
    }, errorCallback);
  },

  /**
   * Convenience function to load the file from a URL,
   * parse it and return the DB.
   * @param url {String}
   * @param successCallback {Function(db {Storage})}
   * @param errorCallback
   */
  _loadFromURL : function(url, successCallback, errorCallback) {
    var self = this;
    //console.log("Loading URL " + url);
    loadURL(url, "json", function(text) {
      successCallback(self._load(text));
    }, errorCallback);
  },

  /**
   * Entry function to parse
   * @param json {JSON or String}   JSON file contents
   * @returns {Storage}   DB with the objects
   */
  _load : function(json) {
    if (typeof(json) == "string") {
      json = JSON.parse(json);
    }
    assert(typeof(json) == "object", "need JSON");

    var self = this;
    for (var verseRef in json) {
      var bt = self.getBibleText(verseRef)
      bt[self._property] = json[verseRef];
    };

    return this._storage;
  },

  getBibleText: function(ref) {
    var existing = this._storage.getID("bible-" + ref);
    if (existing) {
      return existing;
    }
    //console.log("  loading " + ref);
    var bt = new BibleText(ref);
    bt.id = "bible-" + ref;
    this._storage.add(bt);
    return bt;
  },
}
