/*
 * C25K / B210K by Syncostyle
 * c25k@syncostyle.com
 */

/*
 * Constants
 */
 
var PROGRAM = 'C25K',
    TIMER_MARKER_WIDTH = 10,
	TIMER_BAR_HEIGHT = 48,
	CONTROL_BUTTON_HEIGHT = 64,
	START_COLOR = '#27ae60', START_COLOR_HOVER = '#2ecc71',
	RESUME_COLOR = '#27ae60', RESUME_COLOR_HOVER = '#2ecc71',
	PAUSE_COLOR = '#2c3e50', PAUSE_COLOR_HOVER = '#34495e',
	STOP_COLOR = '#c0392b', STOP_COLOR_HOVER = '#e74c3c',
	AUDIO_WALK = 1,
	AUDIO_RUN  = 2,
	AUDIO_WARMUP = 3,
	AUDIO_COOLDOWN = 4,
	AUDIO_FINISH = 5,
    AUDIO_HALFWAY = 6,
	STATE_STOPPED = 1,
	STATE_STARTED = 2,
	STATE_PAUSED = 3,
	STATE_SETTINGS = 10;

var messages = {
	"warmup" : "Warm Up",
	"run" : "Run",
	"walk" : "Walk",
	"cooldown" : "Cool Down",
	"complete" : "Finished"
};

/*
 * Workout Status Codes
 */
var WORKOUT_NOT_STARTED = 0,
	WORKOUT_STARTED		= 1,
	WORKOUT_COMPLETE	= 2;
	
var workoutStatusText = [ 'Not Started', 'Attempted', 'Completed' ];

/*
 * Variables
 */
 
var timeRun, lastTimeChecked, totalTime, currentStage, timeToNextStage, timeSinceLastStage, timerId, selectedWorkout, workoutLength, state, enableScroll, wakelockAcquired, timeSinceLastSave, soundEnabled, vibrationEnabled, halfwayNotificationEnabled, halfwayNotificationAlreadyGiven;

/*
 * Timers
 */
var scrollTimer = null; // to detect end of scrolling

// wait for device to be ready
document.addEventListener('deviceready', onDeviceReady, false);

var deviceos;

function onDeviceReady() {
	state = STATE_STOPPED;
	wakelockAcquired = false;

	deviceos = typeof device !== "undefined" ? device.platform.toLowerCase() : 'none';
	deviceVersion = typeof device !== "undefined" ? device.version : 'none';
    
    $('body').addClass(PROGRAM);
    $('#header-bar-title').html(PROGRAM + ' Trainer');
    
	loadSettings();

	$('#topTimerBar').css('top', '-' + TIMER_BAR_HEIGHT + 'px');
	$('#topTimerBarPointer').css('top', '-' + TIMER_BAR_HEIGHT + 'px');
	
	$('#countdownTimer').hide();
	$('#exerciseInstruction').hide();
	
	$('.controlButton').hide();
	$('#start').show();
	
	fillExerciseSelection();
	
	/* If the highest attempted workout is complete, select the next one (as long
	 * as it isn't the last workout). Otherwise select the highest attempted. */
	var latestWorkout = getLatestWorkout() || 0;
	if (getWorkoutStatus(latestWorkout) == WORKOUT_COMPLETE && latestWorkout < workouts.length)
		selectExercise(latestWorkout + 1);
	else
		selectExercise(latestWorkout);
	
	updateLayout();
    
    $('body').removeClass('invisible');
	
	$('.exerciseSelectionChoice').click(function() {
		selectExercise($(this).data('workoutid'), true);
	});
	
	// Event listener for the settings button in the header
	$('#header-bar-settings').click(function() {
		showSettingsMenu();
	});
	
	// Event listeners for the settings
	$('.settingButton').click(function() {
		var type = $(this).data('setting');
		
		if (type == 'sound') {
			setSound(!soundEnabled);
		} else if (type == 'vibration') {
			setVibration(!vibrationEnabled);
		} else if (type == 'halfway') {
			setHalfwayNotification(!halfwayNotificationEnabled);
		}
	});
	
	// Event listener for the soft back button
	$('#header-bar-back').click(function() {
		if(state === STATE_SETTINGS) {
			onBackButton();
		}
	});
	
	// Event listener to resize if the screen size changes
	$(window).resize(updateLayout);
	
	$(document).on('backbutton', onBackButton);
	
	// Momentum scrolling doesn't work on 2.3, so use a shim
	if(deviceos === 'android' && parseInt(deviceVersion) < 3) {
		touchScroll('exerciseSelection');
	} else {
		setTimeout(function(){scrollListener('exerciseSelection')}, 500);
	}
}

function onBackButton() {
	if(state === STATE_SETTINGS) {
		hideSettingsMenu();
	} else if(state !== STATE_STOPPED) {
		navigator.notification.confirm('You are in the middle of a workout. Are you sure you want to exit?', function(r) {
			if(r==1) { 
				cancelRun();
				navigator.app.exitApp();
			}
		}, 'Exit?', ['Yes, stop run and exit', 'No, keep running']);	
	} else {
		navigator.app.exitApp();
	}
}

/*
 * Listens for scrolling the exercise selection div
 */
function scrollListener(id) {
	var s = function(e) {
		selectNearestExercise(false);
		
		if(deviceos !== 'firefoxos') {
			if(scrollTimer !== null) {
				clearTimeout(scrollTimer);
			}
			
			scrollTimer = setTimeout(function() {
				selectNearestExercise(true);
			}, 500);
		}
	}
	
	$('#' + id)[0].addEventListener('scroll', s, false);
}

/*
 * Show and hide the settings menu
 */
 
function showSettingsMenu() {
	if(state === STATE_STOPPED) {
		state = STATE_SETTINGS;
		$('#header-bar-title').html('Settings');
		$('#header-bar-settings').css({'visibility':'hidden'});
		$('#header-bar-back').css({'visibility':'visible'});
		$('#settingsMenu').show();
	}
}

function hideSettingsMenu() {
	if(state === STATE_SETTINGS) {
		state = STATE_STOPPED;
		$('#header-bar-title').html(PROGRAM + ' Trainer');
		$('#header-bar-settings').css({'visibility':'visible'});
		$('#header-bar-back').css({'visibility':'hidden'});
		$('#settingsMenu').hide();
	}
}

/*
 * Saves the workout status and timestamp of that status
 */
function saveWorkoutStatus(id, status) {
	window.localStorage.setItem(id + 's', status);
	window.localStorage.setItem(id + 't', Date.now());
	
	// save this workout as the most recently attempted
	window.localStorage.setItem('latest', id);
}

/*
 * Saves the progress of the current workout for later resuming
 */
function saveCurrentWorkoutProgress() {
	window.localStorage.setItem(selectedWorkout + 'p', timeRun);
}

/*
 * Returns the status code of the give workout
 */
 
function getWorkoutStatus(id) {
	return (window.localStorage.getItem(id + 's') * 1) || WORKOUT_NOT_STARTED;
}

/*
 * Returns a string with the status of the given workout e.g. "attempted yesterday"
 */
function getWorkoutStatusText(id) {
	var status = getWorkoutStatus(id), 
		timestamp = window.localStorage.getItem(id + 't');
	
	if (status == WORKOUT_NOT_STARTED) {
		return workoutStatusText[status];
	} else {
		/* The Date object complains about Invalid Date if the timestamp is passed
		 * as a string. Multiplying by 1 forces it to an int. */
		var d = new Date(timestamp * 1);
		var r = getRelativeDate(d);
		
		if(r === 'today' && status == WORKOUT_STARTED) {
			return 'Press Start to Resume';
		} else {	
			return workoutStatusText[status] + ' ' + r;
		}
	}
}

/*
 * Returns the highest attempted workout
 */
function getLatestWorkout() {
	// if latest doesn't exist, check for highest (from older versions)
	return (window.localStorage.getItem('latest') || window.localStorage.getItem('highest') || 0) * 1;
}

/*
 * Compares a given date to now, and returns a string describing the time difference
 * e.g. 'yesterday', 'last week'
 */ 
function getRelativeDate(d1) {

	// use today's date at 23:59:59 so that all calculations can be done simply
	// by subtracting full days
	var d2 = new Date();
	d2.setHours(23);
	d2.setMinutes(59);
	d2.setSeconds(59);
	
	var	difference = Math.floor((d2 - d1)/1000); // find difference in seconds
	
	// TODO: Fix this to not just say today is any time in the past 24 hours
	// Maybe get the current date to be midnight tonight
	if(difference < 86400) {
		return 'today';
	} else if (difference < 86400 * 2) {
		return 'yesterday';
	} else if (difference < 86400 * 7) {
		return Math.floor((difference / 86400)) + ' days ago';
	} else if (difference < 86400 * 14) {
		return 'last week';
	} else if (difference < 86400 * 30) {
		// making it simple and assuming 30 days in a month
		return Math.floor((difference / (86400*7))) + ' weeks ago';
	} else {
		return 'over a month ago';
	}
		
}

function selectExercise(workoutNumber, scroll){
	setWorkout(workoutNumber);
	
	// Mark as selected, and deselect the previous choice
	$('.selectedExerciseSelectionChoice').removeClass('selectedExerciseSelectionChoice');
	$('#workout' + workoutNumber).addClass('selectedExerciseSelectionChoice');
	
	// scroll to the selected exercise
	if(scroll !== false) {
		$.scrollTo($('#workout' + workoutNumber).width() * workoutNumber, 200, overlayTimerBar);
	}
}

function selectNearestExercise(scroll) {
	if(state === STATE_STOPPED) {
		var distanceScrolled = $('#exerciseSelection')[0].scrollLeft,
			exerciseWidth = $('.exerciseSelectionChoice').width();
		
		var exerciseToSnap = Math.floor(((distanceScrolled) / exerciseWidth) + 0.5);
		
		selectExercise(exerciseToSnap, scroll);
	}
}

function fillExerciseSelection() {
	
	var exerciseSelectionDiv = $('#exerciseSelection');

	exerciseSelectionDiv.html(''); // clear the div's contents
	
	for(var i=0;i<workouts.length;i++){
		var workoutId = 'workout' + i;
		var workoutClasses = 'exerciseSelectionChoice';
		
		if (getWorkoutStatus(i) == WORKOUT_COMPLETE) {
			workoutClasses += ' selectionComplete';
		}
		
		exerciseSelectionDiv.append('<div id="' + workoutId + '" class="' + workoutClasses + '" data-workoutid="' + i + '">' 
			+ '<span id="' + workoutId + 'Name" class="workoutName">' + getFullWorkoutName(i) + '</span>'
			+ '<div id="' + workoutId + 'TimerBar" class="timerBar"></div>'
			+ '<div id="' + workoutId + 'Summary" class="exerciseSummary">' + getWorkoutStatusText(i) + '</div>'
			+ '</div>');
		fillBar('#' + workoutId + 'TimerBar', i);
	}
}

/*
 * Takes the number of the workout and returns which week and day it is
 * e.g. workout 0 is week 1 day 1, workout 10 is week 4 day 2
 */
function getFullWorkoutName(workoutNumber) {
	var week = Math.floor(workoutNumber / 3) + 1;
	var day  = (workoutNumber % 3) + 1;
	
	return "Week " + week + " Day " + day;
}

function updateLayout() {
	// set negative margins so text is properly centred
	$('#main').css('margin-top', ($('#main').height() / -2) + 'px');
	
	// Set the height of the exercise selection
	$('#exerciseSelection').css('height', $(window).height() - CONTROL_BUTTON_HEIGHT);
	
	$('.exerciseSelectionChoice').css('margin-top', ( ($(window).height()) - $('.exerciseSelectionChoice').height())/2 + 'px');
	
	if(state === STATE_STOPPED) {
		$.scrollTo($('#workout' + selectedWorkout).width() * selectedWorkout, 1, overlayTimerBar);
	}
}

/*
 * Places the functional timer bar invisibly over the one shown for the selected
 * workout
 */
function overlayTimerBar() {
	var barLocation = $('#workout' + selectedWorkout +'TimerBar').offset(),
		barWidth = $('#workout' + selectedWorkout + 'TimerBar').width(),
		barHeight = $('#workout' + selectedWorkout + 'TimerBar').height();
	
	$('#topTimerBar').css('top', barLocation.top).css('left', barLocation.left).width(barWidth).height(barHeight);
	$('#topTimerBar').css('display','none');
}

function fillBar(targetBar, workoutNumber) {

	$(targetBar).html('');

	var l = workouts[workoutNumber].length,
		exerciseType;
		
	totalTime = 0;
	
	for (var i=0; i<l; i++) {
		totalTime += workouts[workoutNumber][i];
	}
	
	for (var i=0; i<l; i++) {
		
		if(i===0) {
			exerciseType = 'warmup';
		} else if(i===l-1) {
			exerciseType = 'cooldown';
		} else if( (i % 2) === 1) {
			exerciseType = 'run';
		} else {
			exerciseType = 'walk';
		}
		
		$(targetBar).append('<li id="stage' + i + '" class="' + exerciseType + '" style="width: ' + (workouts[workoutNumber][i] / totalTime * 100) + '%"></li>');
		
	}
	
}

function setWorkout(workout) {
	selectedWorkout = workout;
	$('.currentExerciseName').html(getFullWorkoutName(workout));
	$('#start')[0].setAttribute('onclick', 'onStartButton(' + workout + ')');
}

/*
 * Called when the start button is pressed. If the workout has been previously
 * completed, the user will be asked if they want to do it again. Otherwise the
 * workout will begin immediately.
 */
function onStartButton(workoutId) {
	if(getWorkoutStatus(workoutId) == WORKOUT_COMPLETE) {
		navigator.notification.confirm('You have already completed this workout. Are you sure you want to do it again?', checkResponse, 'Run again?', ['Yes, run again', 'No']);
	} else {
		startRun(workoutId);
	}
	
	function checkResponse(r) {
		if(r === 1)
			startRun(workoutId);
	}
}

function onPauseButton() {
	pauseRun();
}

function onResumeButton() {
	resumeRun();
}

/*
 * Called when the user presses the stop button. If it is pressed during the
 * Cool Down stage (i.e. the last stage), we should still consider the run
 * complete. Otherwise the run has not been completed.
 */
function onStopButton() {
	if(currentStage == workouts[selectedWorkout].length)
		completeRun();
	else
		cancelRun();
}

function onDidItButton() {
	endRun();
}

/*
 * Starts the currently selected workout. This will remove the selection area and 
 * replace it with a timer.
 */
function startRun(workoutId) {	
	selectedWorkout = workoutId;
	
	// Calculate the length of tihs workout in ms
	workoutLength = 0;
	for(var i=0, len = workouts[selectedWorkout].length; i<len; i++) {
		workoutLength += workouts[selectedWorkout][i]
	}

	state = STATE_STARTED;

	fillBar('#topTimerBar', selectedWorkout);
	$('#topTimerBar').css('display', 'block');
	
	var previousProgress = (window.localStorage.getItem(selectedWorkout + 'p') * 1) || 0,
		previousAttempt = (window.localStorage.getItem(selectedWorkout + 't') * 1) || 0;
	
	if((new Date().getTime() - previousAttempt) < 86400000) {
		timeRun = previousProgress;
	} else {
		timeRun = 0;
	}
	
	if((timeRun * 2) > workoutLength) {
		halfwayNotificationAlreadyGiven = true;
	} else {
		halfwayNotificationAlreadyGiven = false;
	}
	
	timeSinceLastSave = 0;
	
	if(timeRun > 0) {
		setStageByTime(timeRun);
	} else {
		currentStage = -1;
		nextStage();
	}
	
	lastTimeChecked = new Date().getTime();
	timerId = setInterval(nextTick, 200);
	
	$('#start').hide();
	$('#resume').hide();	
	$('#pause').show();
	$('#stop').hide();
	
	$('#exerciseInstruction').show();
	$('#workoutStatus').hide();
	$('#countdownTimer').show();
	
	$('#exerciseSelection').hide();
	$('#exerciseInfo').show();
	
	// Animate the bar moving from the centre to the top
	$('#topTimerBar').animate({top:'0px', left:'0px', width:'100%', height:TIMER_BAR_HEIGHT + 'px'}, 500);
	
	$('#topTimerBarPointer').animate({top:'0px', left:'0px'}, 500);
	
	updateBar();
	updateLayout();
	
	saveWorkoutStatus(selectedWorkout, WORKOUT_STARTED);
	saveCurrentWorkoutProgress();
	
	acquireWakelock();
}

/*
 * Pauses the run, which stops the timer but does not reset anything.
 */
function pauseRun() {
	state = STATE_PAUSED;

	clearInterval(timerId);
	
	$('#start').hide();
	$('#pause').hide();
	$('#resume').show();
	$('#stop').show();
	
	releaseWakelock();
}

/*
 * Resumes a run after being paused, essentially starting the timer again
 */
function resumeRun() {
	state = STATE_STARTED;

	timerId = setInterval(nextTick, 200);
	lastTimeChecked = new Date().getTime();
	updateBar();
	
	$('#start').hide();
	$('#pause').show();
	$('#resume').hide();
	$('#stop').hide();
	
	acquireWakelock();
}

/*
 * Ends a run. Hides the timer and returns the user to the workout selection screen
 */
function endRun() {
	state = STATE_STOPPED;

	clearInterval(timerId);
	
	$('#topTimerBar').animate({'top':'-' + TIMER_BAR_HEIGHT + 'px'}, 600);
	$('#topTimerBarPointer').animate({'top':'-' + TIMER_BAR_HEIGHT + 'px'}, 600);
	
	$('#countdownTimer').hide();
	
	$('.controlButton').hide();
	$('#start').show();
	
	$('#exerciseInstruction').hide();
	$('#exerciseInfo').hide();
	
	fillExerciseSelection();
	selectExercise(selectedWorkout, false);
	$('#exerciseSelection').show();
	
	window.plugins.statusBarNotification.clear();
	
	updateLayout();
	
	releaseWakelock();
}

/*
 * Marks the workout as completed and then ends the run
 */
function completeRun() {
	$('#exerciseInstruction').html('Finished');
	saveWorkoutStatus(selectedWorkout, WORKOUT_COMPLETE);
	window.localStorage.removeItem(selectedWorkout + 'p');

	window.plugins.statusBarNotification.clear();
	
	clearInterval(timerId);
	
	$('#countdownTimer').hide();
	
	$('.controlButton').hide();
	$('#didIt').show();
	
	releaseWakelock();
}

/*
 * Called when the user ends the run early
 */
function cancelRun() {
	endRun();
}

/*
 * Returns the type of exercise for the given stage
 * Parameters: n: the position in the exercise array
 */
function getStageType(n) {
	/*
	 * All exercise arrays follow a pattern. The first and last entries are warmup and
	 * Cool Down. Between these, it alternates between run and walk. e.g. 1, 3, 5, etc are
	 * run, and 2, 4, 6 etc. are walk
	 */
	
	if(n === 0) {
		return 'warmup';
	} else if (n === workouts[selectedWorkout].length - 1) {
		return 'cooldown';
	} else if ( (n % 2) === 1 ) {
		return 'run';
	} else {
		return 'walk';
	};
}

/*
 * Moves to the next stage
 */
function nextStage() {
	
	// Set the previous stage to complete
	$('#stage' + currentStage).removeClass();
	$('#stage' + currentStage).addClass('complete');
	
	if(timeRun >= totalTime) {
		playAudio(AUDIO_FINISH);
		window.plugins.statusBarNotification.notify('Finished', getFullWorkoutName(selectedWorkout));
		completeRun();
	} else {
		setStage(++currentStage);
	}

	updateLayout();
}

/*
 * Set the stage by providing the id within the workout. e.g. for the fourth stage, id
 * will be 3. An optional parameter, timePassed, can also be given, to start from a 
 * point within the stage, rather than the start of it.
 */
function setStage(id, timePassed) {
	if(timePassed > 0) {
		timeToNextStage = workouts[selectedWorkout][currentStage] - timePassed;
        timeSinceLastStage = timePassed;
	} else {
		timeToNextStage = workouts[selectedWorkout][currentStage];
        timeSinceLastStage = 0;
    }
		
	var e = getStageType(currentStage);
	
	$('#exerciseInstruction').html(messages[e]);
	window.plugins.statusBarNotification.notify(messages[e], getFullWorkoutName(selectedWorkout));
	
	if(e == 'walk')
		playAudio(AUDIO_WALK);
	else if(e == 'run')
		playAudio(AUDIO_RUN);
	else if(e == 'warmup')
		playAudio(AUDIO_WARMUP);
	else if(e == 'cooldown')
		playAudio(AUDIO_COOLDOWN);

	if(vibrationEnabled)
		navigator.notification.vibrate(1000);
}

/* 
 * Takes a time in milliseconds and sets the stage based on that. Useful for resuming
 */
function setStageByTime(timeInMs) {
	var t = timeInMs,
		stage = 0;
	
	while(t>0) {
		if(t > workouts[selectedWorkout][stage]) {
			$('#stage' + stage).removeClass();
			$('#stage' + stage).addClass('complete');
			t -= workouts[selectedWorkout][stage];
			stage++;
		} else {
			break;
		}
	}
	
	currentStage = stage;
	setStage(currentStage, t);
}

function nextTick() {
	var currentTime = new Date().getTime(),
	timePassed = currentTime - lastTimeChecked;
	
	timeRun += timePassed;
	timeSinceLastSave += timePassed;
    timeSinceLastStage += timePassed;
	timeToNextStage -= timePassed;
	
	lastTimeChecked = currentTime;
	
	updateBar();
	updateTime();
	
	// Save the progress every 15 seconds so we're not constantly writing
	if(timeSinceLastSave > 15000) {
		saveCurrentWorkoutProgress();
		timeSinceLastSave = 0;
	}
	
	// Give a notification if you've reached half way
	if(halfwayNotificationEnabled && !halfwayNotificationAlreadyGiven && (timeRun * 2) > workoutLength) {
        notifyHalfway();
	}
}

function notifyHalfway() {
    // Only play the notification when we are 15 seconds clear of a stage either side to avoid confusion
    if(timeSinceLastStage > 15000 && timeToNextStage > 15000) {
        halfwayNotificationAlreadyGiven = true;
        playAudio(AUDIO_HALFWAY);
        console.log('You are half way there');
    } else {
        console.log('Halfway but too close');
    }
}

function updateBar() {
	
	if(timeToNextStage <= 0) {
		nextStage();
	}
	
	if(timeRun > 0) {
		var m = ((timeRun / totalTime) * $('#topTimerBar').width()) - TIMER_MARKER_WIDTH;
	
		if (m < 0)
			m = 0;
		
		if (m < $('#topTimerBar').width() - TIMER_MARKER_WIDTH)
			$('#topTimerBarPointer').css('left', m + 'px');
	}
}

/*
 * Updates both the full timer and current stage countdown
 */
function updateTime() {
	var m = 0, s = 0;
	
	// First the standard timer
	s = Math.floor(timeRun / 1000);
	m = Math.floor(s / 60);
	
	if (s >= 60) {	
		s = s % 60;
	}
	
	$('#standardTimer').html(m + ':' + (s < 10 ? '0' : '') + s);
	
	// Now the countdown
	
	s = Math.ceil(timeToNextStage / 1000);
	m = Math.floor(s / 60);
	
	if (s >= 60) {
		s = s % 60;
	}
	
	$('#countdownTimer').html(m + ':' + (s < 10 ? '0' : '') + s);
}

/*
 * Plays an audio notification based on the parameter type
 */
function playAudio(type) {
	if(soundEnabled) {
		if(type == AUDIO_WALK) {
			src = 'walk';
		} else if (type == AUDIO_RUN) {
			src = 'run';
		} else if (type == AUDIO_WARMUP) {
			src = 'warmup';
		} else if (type == AUDIO_COOLDOWN) {
			src = 'cooldown';
		} else if (type == AUDIO_FINISH) {
			// There are multiple audio files for finishing, depending on what time of week it is
            var dayOfWeek = selectedWorkout % 3;
            
            if (dayOfWeek == 0) {
                src = 'done_greatstart';
            } else if (dayOfWeek == 1) {
                src = 'done_doinggreat';
            } else {
                // if they've completed the final workout, play a special message
                if(selectedWorkout == (workouts.length - 1) && PROGRAM === 'C25K') {
                    src = 'complete';
                } else {
                    src = 'done_weekdown';
                }
            }
		} else if (type == AUDIO_HALFWAY) {
            src = 'halfway';
        }
		
		if(deviceos === 'android') {
			try {
				window.plugins.properAudio.play(src, function(){}, function(){});
			} catch (e) {
				console.log('Failed to play sound: ' + e);
			}
		} else if (deviceos === 'firefoxos') {
			var a = new Audio();
	        a.mozAudioChannelType = 'notification'; 
			a.src = '/audio/' + src + '.ogg';
			a.play();
		}
	}
}

/*
 * Acquire and release wakelock
 */
function acquireWakelock() {
	if(deviceos === 'android') {
		window.plugins.powerManagement.acquire(function(){wakelockAcquired=true;}, function(){wakelockAcquired=false;});
	} else {
		wakelockAcquired = navigator.requestWakeLock('screen');
	}
}

function releaseWakelock() {
	// trying to release without having the wakelock leads to an exception
	if(deviceos === 'android' && wakelockAcquired) {
		window.plugins.powerManagement.release(function(){wakelockAcquired=false;}, function(){wakelockAcquired=true;});
	} else if (wakelockAcquired) {
		wakelockAcquired.unlock();
	}
}

/*
 * Set the sound setting
 */
 
function setSound(enabled) {
	if(enabled == true) {
		soundEnabled = true;
		window.localStorage.setItem('sound', '1');
		$('#soundSetting').html('ON').addClass('on').removeClass('off');
	} else {
		soundEnabled = false;
		window.localStorage.setItem('sound', '0');
		$('#soundSetting').html('OFF').addClass('off').removeClass('on');
	}
}

/*
 * Set the vibration setting
 */
 
function setVibration(enabled) {
	if(enabled == true) {
		vibrationEnabled = true;
		window.localStorage.setItem('vibration', '1');
		$('#vibrationSetting').html('ON').addClass('on').removeClass('off');
	} else {
		vibrationEnabled = false;
		window.localStorage.setItem('vibration', '0');
		$('#vibrationSetting').html('OFF').addClass('off').removeClass('on');
	}
}

/*
 * Set the halfway notification setting
 */
 
function setHalfwayNotification(enabled) {
	if(enabled == true) {
		halfwayNotificationEnabled = true;
		window.localStorage.setItem('halfwayNotification', '1');
		$('#halfwayNotificationSetting').html('ON').addClass('on').removeClass('off');
	} else {
		halfwayNotificationEnabled = false;
		window.localStorage.setItem('halfwayNotification', '0');
		$('#halfwayNotificationSetting').html('OFF').addClass('off').removeClass('on');
	}
}

/*
 * Load all settings from localStorage
 */
 
function loadSettings() {
	var s = window.localStorage.getItem('sound');
	if(s == null)
		s = true;
	setSound(s);
	
	var v = window.localStorage.getItem('vibration');
	if(v == null)
		v = true;
	setVibration(v);
	
	var h = window.localStorage.getItem('halfwayNotification');
	if(h == null)
		h = true;
	setHalfwayNotification(h);
}
 
var workouts;

if(PROGRAM === 'B210K') {
    workouts = [
        [ 300000, 600000, 60000, 600000, 60000, 600000, 60000, 600000, 300000],
        [ 300000, 600000, 60000, 600000, 60000, 600000, 60000, 600000, 300000],
        [ 300000, 600000, 60000, 600000, 60000, 600000, 60000, 600000, 300000],
        [ 300000, 900000, 60000, 900000, 60000, 900000, 300000],
        [ 300000, 900000, 60000, 900000, 60000, 900000, 300000],
        [ 300000, 900000, 60000, 900000, 60000, 900000, 300000],
        [ 300000, 1020000, 60000, 1020000, 60000, 1020000, 300000],
        [ 300000, 1020000, 60000, 1020000, 60000, 1020000, 300000],
        [ 300000, 1020000, 60000, 1020000, 60000, 1020000, 300000],
        [ 300000, 1080000, 60000, 1080000, 60000, 1080000, 300000],
        [ 300000, 1080000, 60000, 1080000, 60000, 1080000, 300000],
        [ 300000, 1080000, 60000, 1080000, 60000, 1080000, 300000],
        [ 300000, 1320000, 60000, 1320000, 300000],
        [ 300000, 1500000, 60000, 1500000, 300000],
        [ 300000, 1800000, 60000, 1800000, 300000],
        [ 300000, 3600000, 300000],
        [ 300000, 3600000, 300000],
        [ 300000, 3600000, 300000]
    ];
} else {
    workouts = [ 
        [ 300000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 300000 ],
        [ 300000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 300000 ],
        [ 300000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 60000, 90000, 300000 ],
        [ 300000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 300000 ],
        [ 300000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 300000 ],
        [ 300000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 90000, 120000, 300000 ],
        [ 300000, 90000, 90000, 180000, 180000, 90000, 90000, 180000, 180000, 300000 ],
        [ 300000, 90000, 90000, 180000, 180000, 90000, 90000, 180000, 180000, 300000 ],
        [ 300000, 90000, 90000, 180000, 180000, 90000, 90000, 180000, 180000, 300000 ],
        [ 300000, 180000, 90000, 300000, 150000, 180000, 90000, 300000, 300000 ],
        [ 300000, 180000, 90000, 300000, 150000, 180000, 90000, 300000, 300000 ],
        [ 300000, 180000, 90000, 300000, 150000, 180000, 90000, 300000, 300000 ],
        [ 300000, 300000, 180000, 300000, 180000, 300000, 300000 ],
        [ 300000, 480000, 300000, 480000, 300000 ],
        [ 300000, 1200000, 300000 ],
        [ 300000, 300000, 180000, 480000, 180000, 300000, 300000 ],
        [ 300000, 600000, 180000, 600000, 300000 ],
        [ 300000, 1320000, 300000 ],
        [ 300000, 1500000, 300000 ],
        [ 300000, 1500000, 300000 ],
        [ 300000, 1500000, 300000 ],
        [ 300000, 1680000, 300000 ],
        [ 300000, 1680000, 300000 ],
        [ 300000, 1680000, 300000 ],
        [ 300000, 1800000, 300000 ],
        [ 300000, 1800000, 300000 ],
        [ 300000, 1800000, 300000 ]
    ];
}
//workouts[0] = [ 3000, 6000, 9000, 6000, 9000, 6000, 9000, 6000, 9000, 6000, 9000, 6000, 9000, 6000, 9000, 6000, 9000, 3000 ];

// Debug functions
function _adjustTime(amount) {
	timeRun += amount;
	timeToNextStage -= amount;
}