define(["require", "exports", 'domain/service/utils'], function(require, exports, __utils__) {
    var utils = __utils__;
    

    var GRAVITY = 300;

    var Hiyoko = (function () {
        function Hiyoko(potential) {
            this.potential = potential;
            this.status = new Status();
            this.groundIndex = 0;
            this.status.life = potential.life;
        }
        Hiyoko.prototype.update = function (grounds) {
            this.status.life += this.potential.recoveryPower;
            if (this.status.life > this.potential.life) {
                this.status.life = this.potential.life;
            }

            if (!this.on(grounds[this.groundIndex])) {
                var i = this.groundIndex + 1;
                if (grounds.length > i && this.on(grounds[i])) {
                    this.groundIndex = i;
                }
            }
            this.move();
            if (grounds.length > this.groundIndex + 1)
                this.hitProcess(grounds[this.groundIndex], grounds[this.groundIndex + 1]);
        };

        Hiyoko.prototype.move = function () {
            switch (this.status.state) {
                case State.WALK:
                case State.CHARGE:
                case State.AIR:
                    this.status.x += this.potential.speed;
                    break;
            }
            switch (this.status.state) {
                case State.AIR:
                case State.FALL:
                    this.status.velocity -= GRAVITY;
                    this.status.y += this.status.velocity;
                    break;
            }
        };

        Hiyoko.prototype.hitProcess = function (ground, nextGround) {
            switch (this.status.state) {
                case State.WALK:
                    var distanceToCliff = ground.end - this.status.x;
                    if (distanceToCliff <= this.potential.chargeStartDistance) {
                        this.status.state = State.CHARGE;
                    }
                    break;
                case State.CHARGE:
                    if (this.status.time >= this.potential.chargeTime) {
                        this.status.time = 0;
                        this.jump(nextGround);
                        break;
                    }
                    this.status.time++;
                    break;
                case State.AIR:
                    if (!this.on(ground))
                        break;
                    if (ground.height < this.status.y)
                        break;
                    var prevY = this.status.y - this.status.velocity;
                    if (ground.height >= prevY) {
                        this.status.x = ground.begin - 8;
                        this.status.state = State.FALL;
                        break;
                    }
                    this.status.y = ground.height;
                    this.status.state = State.WALK;
                    break;
                case State.FALL:
                    if (this.status.y < -320 * 1000) {
                        this.status.state = State.DEAD;
                    }
                    break;
            }
        };

        Hiyoko.prototype.jump = function (ground) {
            this.status.velocity = range(GRAVITY + 1, this.potential.jumpPower + this.potential.jumpHeightWeight * (ground.height - this.status.y) + this.potential.jumpWidthWeight * (ground.begin - this.status.x), this.potential.chargeWeight * this.potential.chargeTime);
            this.status.state = State.AIR;
        };

        Hiyoko.prototype.on = function (ground) {
            return ground.end > this.status.x + 8 && this.status.x - 8 >= ground.begin;
        };
        return Hiyoko;
    })();
    exports.Hiyoko = Hiyoko;

    function range(min, value, max) {
        return Math.max(min, Math.min(value, max));
    }

    var Status = (function () {
        function Status() {
            this.x = 0;
            this.y = 0;
            this.velocity = 0;
            this.state = State.WALK;
            this.time = 0;
        }
        return Status;
    })();
    exports.Status = Status;

    (function (State) {
        State[State["WALK"] = 0] = "WALK";
        State[State["CHARGE"] = 1] = "CHARGE";
        State[State["AIR"] = 2] = "AIR";
        State[State["FALL"] = 3] = "FALL";
        State[State["DEAD"] = 4] = "DEAD";
    })(exports.State || (exports.State = {}));
    var State = exports.State;

    function isAir(state) {
        return state === State.AIR || state === State.FALL;
    }

    var Potential = (function () {
        function Potential(speed, jumpPower, chargeStartDistance, chargeTime, chargeWeight, jumpHeightWeight, jumpWidthWeight, life, recoveryPower) {
            this.speed = speed;
            this.jumpPower = jumpPower;
            this.chargeStartDistance = chargeStartDistance;
            this.chargeTime = chargeTime;
            this.chargeWeight = chargeWeight;
            this.jumpHeightWeight = jumpHeightWeight;
            this.jumpWidthWeight = jumpWidthWeight;
            this.life = life;
            this.recoveryPower = recoveryPower;
        }
        return Potential;
    })();
    exports.Potential = Potential;

    var HiyokoFactory = (function () {
        function HiyokoFactory() {
        }
        HiyokoFactory.createNew = function () {
            return Enumerable.repeat(null, 10).select(function (x) {
                var hiyoko = new Hiyoko(createPotential());
                hiyoko.status.x = utils.randomRange(-50 * 1000, 50 * 1000);
                return hiyoko;
            }).toArray();
        };

        HiyokoFactory.createNext = function (hiyokos) {
            var need = hiyokos.length;
            var shuffled = Enumerable.from(select(hiyokos)).shuffle();
            return Enumerable.repeat(shuffled).selectMany(function (x) {
                return x;
            }).buffer(2).select(function (x) {
                return crossover(x[0].potential, x[1].potential);
            }).selectMany(function (x) {
                return x;
            }).take(need).select(function (x) {
                return new Hiyoko(x);
            }).toArray();
        };
        return HiyokoFactory;
    })();
    exports.HiyokoFactory = HiyokoFactory;

    function select(hiyokos) {
        return Enumerable.from(hiyokos).orderByDescending(function (x) {
            return x.status.x;
        }).take(hiyokos.length / 2 | 0).toArray();
    }

    function createPotential() {
        return new Potential(utils.randomRange(2900, 3000), utils.randomRange(0, 1000), utils.randomRange(3 * 1000, 30 * 1000), utils.randomRange(1, 2), utils.randomRange(1, 10000), utils.randomRange(0, 0.04 * 2), utils.randomRange(0, 0.07 * 2), utils.randomRange(1, 100), utils.randomRange(0, 100));
    }

    function crossover(p1, p2) {
        if (Math.random() < 0.05) {
            p1 = createPotential();
        }
        var selector = Enumerable.generate(function () {
            return Math.random() >= 0.5;
        }, 9).toArray();
        return [
            new Potential((selector[0] ? p1.speed : p2.speed) + (Math.random() * 100 - 50 | 0), selector[1] ? p1.jumpPower : p2.jumpPower, selector[2] ? p1.chargeStartDistance : p2.chargeStartDistance, selector[3] ? p1.chargeTime : p2.chargeTime, selector[4] ? p1.chargeWeight : p2.chargeWeight, selector[5] ? p1.jumpHeightWeight : p2.jumpHeightWeight, selector[6] ? p1.jumpWidthWeight : p2.jumpWidthWeight, selector[7] ? p1.life : p2.life, selector[8] ? p1.recoveryPower : p2.recoveryPower),
            new Potential((p1.speed + p2.speed) / 2, (p1.jumpPower + p2.jumpPower) / 2, (p1.chargeStartDistance + p2.chargeStartDistance) / 2, (p1.chargeTime + p2.chargeTime) / 2, (p1.chargeWeight + p2.chargeWeight) / 2, (p1.jumpHeightWeight + p2.jumpHeightWeight) / 2, (p1.jumpWidthWeight + p2.jumpWidthWeight) / 2, (p1.life + p2.life) / 2, (p1.recoveryPower + p2.recoveryPower) / 2)
        ];
    }
});
//@ sourceMappingURL=file:///E:/Developments/node/hiyokorun/public/js/domain/entity/hiyoko.js.map
