(function(exports) {

  var noteStrings = ['C', 'C#', 'D', 'D#', 'E', 'F',
                     'F#', 'G', 'G#', 'A', 'A#', 'B'];

  var FFT_SIZE = 1024; // lower gives better performance, worse accuracy

  function Tuner() {
    this.context = new AudioContext();
    this.start();
  }

  Tuner.prototype.stop = function() {
    if (!this.stream) { return; }
    this.stream = null;
    this.source = null;
    this.analyzer = null;
    this.buf = null;
  }

  Tuner.prototype.start = function() {
    if (this.stream) { return; }
    navigator.mozGetUserMedia({ audio: true }, (stream) => {
      this.stream = stream;
      this.source = this.context.createMediaStreamSource(stream);
      this.analyzer = this.context.createAnalyser();
      this.analyzer.fftSize = FFT_SIZE;
      this.source.connect(this.analyzer);
      this.buf = new Uint8Array(this.analyzer.frequencyBinCount*2);
      this.read();
    }, console.log.bind(console));
  }


  Tuner.prototype.read = function() {
    if (!this.stream) {
      return;
    }

    this.analyzer.getByteTimeDomainData(this.buf);

    // Autocorrelation algorithm via Chris Wilson:
    // <https://github.com/cwilso/PitchDetect>
    // MIT License. Copyright (c) 2014 Chris Wilson.
    var MIN_SAMPLES = FFT_SIZE / 256 | 0;
    var MAX_SAMPLES = FFT_SIZE / 2 | 0;
    var SIZE = MAX_SAMPLES;
    var best_offset = -1;
    var best_correlation = 0;
    var rms = 0;

    if (this.buf.length < (SIZE + MAX_SAMPLES - MIN_SAMPLES))
      return;  // Not enough data

    for (var i=0;i<SIZE;i++) {
      var val = (this.buf[i] - 128)/128;
      rms += val*val;
    }
    rms = Math.sqrt(rms/SIZE);

    for (var offset = MIN_SAMPLES; offset <= MAX_SAMPLES; offset++) {
      var correlation = 0;

      for (var i=0; i<SIZE; i++) {
        correlation += Math.abs(((this.buf[i] - 128)/128)-((this.buf[i+offset] - 128)/128));
      }
      correlation = 1 - (correlation/SIZE);
      if (correlation > best_correlation) {
        best_correlation = correlation;
        best_offset = offset;
      }
    }

    if ((rms > 0.01) && (best_correlation > 0.01)) {
      this.confidence = best_correlation * rms * 10000;
      this.hertz = this.context.sampleRate / best_offset;
      var note = Math.round( 12 * (Math.log( this.hertz / 440 )/Math.log(2) ) ) + 69;
      this.noteName = noteStrings[note%12] || "";
      this.cents = Math.round(1200 * Math.log( this.hertz / (440 * Math.pow(2,(note-69)/12)))/Math.log(2));
    } else {
      this.confidence = 0;
      this.noteName = '?';
    }
    this.onpitchchange && this.onpitchchange(this);
    window.requestAnimationFrame(this.read.bind(this));
  }

  exports.Tuner = Tuner;

})(window);
