/**
 * bullets.js
 *
 * contains all classes and logic for bullet objects
 *
 * Copyright(c) 2013 Pavle Goloskokovic
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
 * http://creativecommons.org/licenses/by-nc-sa/3.0/
 */

/**
 * Class representing a bullet object
 * @type {*}
 */
var Bullet = Class.extend({
    init: function(bdamage,pdamage,speed,x,y,frame,explosion){
        this.bluntDamage = bdamage;
        this.piercingDamage = pdamage;
        this.speed = speed;
        this.x = x;
        this.y = y;
        this.frame = frame;
        this.explosion = explosion;
    },
    bluntDamage: 0,
    piercingDamage: 0,
    speed: 0,
    x: 0, y: 0,
    velX: 0, velY: 0,
    frame: '',
    explosion: null,
    update : function(){},
    drawBullet: function(){
        drawSprite(this.frame,this.x,this.y);
    }
});

/**
 * Bullet used for shooting from JunkTower
 * @type {*}
 */
var JunkBullet = Bullet.extend({
    init: function(x,y,enemy){
        this._super(15,5,10,x,y,"bullets/junk.png",JunkExplosion);
        this.targetEnemy = enemy;
        this.nearDistance = 15;
    },
    targetEnemy: null,
    nearDistance: 0,
    update : function(){

        this.move();

        // Checks if bullet collided with target enemy
        // If collision occurred decrease enemy's energy
        if (inCircle(this.targetEnemy, this.x, this.y, this.nearDistance)) {
            if(Math.random()<=0.85){
                this.targetEnemy.health -= this.bluntDamage / this.targetEnemy.bluntArmor + this.piercingDamage / this.targetEnemy.piercingArmor;
            }
            this.targetEnemy = null;
            game.bullets.remove(this);
            game.animations.push(new this.explosion(this.x, this.y));
        }
    },

    // Moves bullet towards enemy
    move: function(){
        var angle = Math.atan2(this.targetEnemy.y - this.y, this.targetEnemy.x - this.x);
        this.velX = Math.cos(angle) * this.speed;
        this.velY = Math.sin(angle) * this.speed;

        this.x += this.velX;
        this.y += this.velY;
    }
});

/**
 * Bullet used for shooting from PlasmaTower
 * @type {*}
 */
var PlasmaBullet = Bullet.extend({
    init: function(x,y,enemy){

        this._super(15,20,10,x,y,"",PlasmaExplosion);
        this.targetX = enemy.x;
        this.targetY = enemy.y;

        this.curX = this.x;
        this.curY = this.y;

        var dist = vectorMagnitude(this.x,this.targetX,this.y,this.targetY);
        var t = dist / this.speed;
        this.hgtVel = - t * this.g / 2;

        var angle = Math.atan2(this.targetY - this.y, this.targetX - this.x);
        this.velX = Math.cos(angle) * this.speed;
        this.velY = Math.sin(angle) * this.speed;

    },
    frameIndex: 0,
    hgt: 0, hgtVel: 0,
    targetX: 0, targetY: 0,
    curX: 0, curY: 0,
    g: 5,
    effectArea: 75,
    update : function(){

        this.frameIndex = (this.frameIndex+1)%3;

        // Checks if bullet reached target coordinate
        // If true decrease energy of all enemies in a certain radius
        if (this.hgt + this.hgtVel > 0) {

            for(var i=0; i<game.enemies.length; i++){
                if(inCircle(game.enemies[i],this.x,this.y,this.effectArea)){
                    var distance = vectorMagnitude(this.x,game.enemies[i].x,this.y,game.enemies[i].y);
                    game.enemies[i].health -= (this.bluntDamage / game.enemies[i].bluntArmor + this.piercingDamage / game.enemies[i].piercingArmor)*(1-distance/this.effectArea);
                }
            }
            game.bullets.remove(this);
            game.animations.push(new this.explosion(this.x, this.y));
        }

        this.move();
    },

    // Moves bullet towards target coordinate
    move: function(){

        this.curX += this.velX;
        this.curY += this.velY;


        this.hgt += this.hgtVel;
        this.hgtVel += this.g;

        this.x = this.curX;
        this.y = this.curY + this.hgt;
    },
    drawBullet: function(){
        drawSprite("bullets/plasma000"+(this.frameIndex+1)+".png",this.x,this.y);
    }

});

/**
 * Bullet used for shooting from RocketTower
 * @type {*}
 */
var RocketBullet = Bullet.extend({
    init: function(x,y,enemy){
        this._super(5,15,10,x,y,"bullets/rocket.png",RocketExplosion);
        this.targetEnemy = enemy;
        this.nearDistance = 15;
        this.velY = this.speed;
    },
    targetEnemy: null,
    nearDistance: 0,
    rotation: -90,
    deltaStep: 0,
    update : function(){

        // Checks if current target enemy died or left the map
        // If true try to find nearest enemy for new target or explode
        if(game.enemies.indexOf(this.targetEnemy)==-1){
            if(game.enemies.length>0){

                var nextTarget = game.enemies[0];
                var nextTargetDistance = vectorMagnitude(nextTarget.x,this.x,nextTarget.y,this.y);

                for(var i=1; i<game.enemies.length; i++){

                    var currentEnemy = game.enemies[i];
                    var currentEnemyDistance = vectorMagnitude(currentEnemy.x,this.x,currentEnemy.y,this.y);

                    if( currentEnemyDistance < nextTargetDistance
                     ||( currentEnemyDistance == nextTargetDistance
                      && currentEnemy.progress > nextTarget.progress)){

                        nextTarget = currentEnemy;
                        nextTargetDistance = currentEnemyDistance;
                    }
                }

                this.targetEnemy = nextTarget;

            }else{
                game.bullets.remove(this);
                game.animations.push(new this.explosion(this.x, this.y));
                return;
            }
        }

        this.move();

        // Checks if bullet collided with target enemy
        // If collision occurred decrease enemy's energy
        if (inCircle(this.targetEnemy, this.x, this.y, this.nearDistance)) {
            this.targetEnemy.health -= this.bluntDamage / this.targetEnemy.bluntArmor + this.piercingDamage / this.targetEnemy.piercingArmor;
            this.targetEnemy = null;
            game.bullets.remove(this);
            game.animations.push(new this.explosion(this.x, this.y));
        }
    },

    // Moves bullet towards current target enemy
    move: function(){

        var wantedAngle = Math.atan2(this.targetEnemy.y - this.y, this.targetEnemy.x - this.x);
        var currentAngle = this.rotation / 180 * Math.PI;

        if(Math.abs(wantedAngle-currentAngle)>Math.PI){
            wantedAngle -= sign(wantedAngle)*Math.PI*2;
        }
        currentAngle = (1-this.deltaStep)*currentAngle + this.deltaStep*wantedAngle;

        if(this.deltaStep<0.3){
            this.deltaStep+=0.02;
        }

        this.rotation = currentAngle * 180 / Math.PI;
        this.velX = Math.cos(currentAngle) * this.speed;
        this.velY = Math.sin(currentAngle) * this.speed;

        this.x += this.velX;
        this.y += this.velY;
    },
    drawBullet: function(){
        drawSprite(this.frame,this.x,this.y,this.rotation);
    }
});

/**
 * Class representing bullet explosion animation
 * @type {*}
 */
var Explosion = Class.extend({
    init: function(x,y,framePrefix,numOfFrames){
        this.x = x;
        this.y = y;
        this.framePrefix = framePrefix;
        this.numOfFrames = numOfFrames;
    },
    x: 0,
    y: 0,
    framePrefix: '',
    currentFrame: 1,
    numOfFrames: 0,

    draw : function(){
        drawSprite(this.framePrefix + String("000" + this.currentFrame).slice(-4) + ".png",this.x,this.y);
    },
    update : function(){
        this.currentFrame++;
        if(this.currentFrame>this.numOfFrames){
            game.animations.remove(this);
        }
    }
});

/**
 * Bullet explosion animation for JunkBullet
 * @type {*}
 */
var JunkExplosion = Explosion.extend({
    init: function(x,y){
        this._super(x,y,"explosions/junk/JunkExplosion",5);
        gSM.playSound("./audio/junk_explode.ogg");
    }
});

/**
 * Bullet explosion animation for PlasmaBullet
 * @type {*}
 */
var PlasmaExplosion = Explosion.extend({
    init: function(x,y){
        this._super(x,y,"explosions/plasma/plasma-explosion",9);
        gSM.playSound("./audio/plasma_explode.ogg");
    }
});

/**
 * Bullet explosion animation for RocketBullet
 * @type {*}
 */
var RocketExplosion = Explosion.extend({
    init: function(x,y){
        this._super(x,y,"explosions/rocket/rocket-explosion",11);
        gSM.playSound("./audio/rocket_explode.ogg");
    }
});


