/**
 * enemies.js
 *
 * contains all classes and logic for enemy objects
 *
 * Copyright(c) 2013 Pavle Goloskokovic
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
 * http://creativecommons.org/licenses/by-nc-sa/3.0/
 */

var NEAR_DISTANCE = 10;
var PATH_WIDTH = 45;

/**
 * Class representing an enemy object
 * @type {*}
 */
var Enemy = Class.extend({

    init: function(velocity,health,barmor,parmor){
        this.path = game.maps[game.level].path;
        this.velocity = velocity;
        this.health = this.maxHealth = this.initialHealth = health;
        this.bluntArmor = barmor;
        this.piercingArmor = parmor;
        this.pathShift = Math.random();
        var curPoint = displacePointOfPath(this.path, 0, this.pathShift, PATH_WIDTH);
        this.x = curPoint.x;
        this.y = this.curY = curPoint.y;

        this.findNextTarget();
    },

    velX: 0, velY: 0, velocity: 0,
    currPathIndex: 0, path: null, pathShift: 0,
    currTarget: null,
    currentPathSegmentDistance: 0,
    progress: 0,
    x: 0, y: 0, curY: 0, height: 0,
    hgt: 0, hgtCounter: 0,
    health: 100, maxHealth: 100, initialHealth: 100,
    bluntArmor: 0, piercingArmor: 0,
    jumpInterval: 1.1, jumpAmp: 5,
    frame: '', framePrefix: '',
    deadScale: 0.642,
    deadRot: 90,
    worthDead: 0,
    worthLive: 1,

    drawEnemy : function(){
        drawSprite(this.frame,this.x,this.y);
        this.drawHealthBar();
    },

    drawHealthBar: function(){

        var hw = 34;
        var hh = 4;

        game.ctx.fillStyle = "#CC0033";
        game.ctx.fillRect(this.x - hw/2, this.y - this.height/2 - hh - 5, hw, hh);

        game.ctx.fillStyle = "#33CC00";
        game.ctx.fillRect(this.x - hw/2, this.y - this.height/2 - hh - 5, Math.max(hw*this.health/this.maxHealth,0), hh);

        game.ctx.strokeStyle = "#000";
        game.ctx.strokeRect(this.x - hw/2, this.y - this.height/2 - hh - 5, hw, hh);

    },

    /**
     * function that calculates next path point for enemy
     */
    findNextTarget: function(){
        this.currPathIndex++;
        if (this.currPathIndex >= this.path.length) {
            this.leaveLevel();
            return;
        }
        this.currTarget = displacePointOfPath(this.path, this.currPathIndex, this.pathShift, PATH_WIDTH);
        var lastPoint = this.path[this.currPathIndex-1];
        var nextPoint = this.path[this.currPathIndex];
        this.currentPathSegmentDistance = vectorMagnitude(lastPoint.x,nextPoint.x,lastPoint.y,nextPoint.y);
    },

    leaveLevel: function(){

        game.enemies.remove(this);

        game.status.lives-=this.worthLive;
        if(game.status.lives<=0){
            game.endGame();
            return;
        }
        if(game.enemies.length == 0 && game.waveManager.ended){
            if(game.status.lives > 0){
                game.endGame();
            }
        }
    },

    update : function(){

        // check if enemy died

        if(this.health<=0){
            game.enemies.remove(this);
            game.animations.push(new Dead(this.x,this.y,this.deadRot,this.deadScale));

            gSM.playSound("./audio/dead.ogg");

            game.status.money+=this.worthDead;

            if(game.enemies.length == 0 && game.waveManager.ended){
                if(game.status.lives > 0){
                    game.endGame();
                }
            }
            return;
        }

        // move enemy

        this.x += this.velX;
        this.curY += this.velY;
        this.y = this.curY + this.hgt;
        this.hgt = Math.sin(this.hgtCounter) * this.jumpAmp;
        this.hgtCounter = (this.hgtCounter + this.jumpInterval) % 360;

        var dist = vectorMagnitude(this.x,this.currTarget.x,this.y,this.currTarget.y);
        if (dist < NEAR_DISTANCE) {
            this.findNextTarget();
        }

        // calculate enemy's path progress
        this.progress = this.currPathIndex - dist/this.currentPathSegmentDistance;

        // calculates enemy's rotation
        var angle = Math.atan2(this.currTarget.y - this.y, this.currTarget.x - this.x);
        this.velX = Math.cos(angle) * this.velocity;
        this.velY = Math.sin(angle) * this.velocity;

        // choosing correct sprite for drawing depending on enemy's rotation
        var xStr,yStr;
        if(this.velX<0){
            xStr = "Left";
        }else{
            xStr = "Right";
        }
        if(this.velY<0){
            yStr = "Top";
        }else{
            yStr = "Down";
        }
        if(Math.abs(this.velX)<0.25*this.velocity){
            this.frame = yStr
        }else if(Math.abs(this.velX)>0.75*this.velocity){
            this.frame = xStr;
        }else{
            this.frame = yStr + xStr;
        }

        //this.frame = this.framePrefix + mapFrames(this.frame) + ".png";
         this.frame ="enemies/patita/patita.png"
    }
});

/**
 * A function that shifts the enemies from the original path
 * @param path
 * @param pathIndex
 * @param shift
 * @param pathWidth
 * @returns {{}}
 */
function displacePointOfPath(path, pathIndex, shift, pathWidth)
{
    var startPoint = (pathIndex == 0) ? null : game.maps[game.level].path[pathIndex-1];
    var endPoint = (pathIndex == path.length - 1) ? null : game.maps[game.level].path[pathIndex+1];

    var points = constructLateralPoints(startPoint, game.maps[game.level].path[pathIndex],endPoint,pathWidth);

    var retPoint = {};
    retPoint.x = points.point1.x + (points.point2.x - points.point1.x) * shift;
    retPoint.y = points.point1.y + (points.point2.y - points.point1.y) * shift;

    return retPoint;
}

/**
 * A function for constructing lateral points
 * @param startPoint
 * @param midPoint
 * @param endPoint
 * @param distance
 * @returns {{}}
 */
function constructLateralPoints(startPoint, midPoint, endPoint, distance)
{
    var targetAngle;

    if (startPoint !== null && endPoint !== null) {
        targetAngle = (lineAngle(startPoint.x, startPoint.y, midPoint.x, midPoint.y) + lineAngle(midPoint.x, midPoint.y, endPoint.x, endPoint.y)) / 2 + Math.PI / 2;
    }
    else if (startPoint == null && endPoint != null)
    {
        targetAngle = lineAngle(midPoint.x, midPoint.y, endPoint.x, endPoint.y) + Math.PI / 2;

    }
    else if (startPoint != null && endPoint == null)
    {
        targetAngle = lineAngle(startPoint.x, startPoint.y, midPoint.x, midPoint.y) + Math.PI / 2;
    }

    var point1 = {};
    var point2 = {};

    point1.x = midPoint.x + Math.cos(targetAngle)*distance;
    point1.y = midPoint.y + Math.sin(targetAngle)*distance;

    point2.x = midPoint.x - Math.cos(targetAngle)*distance;
    point2.y = midPoint.y - Math.sin(targetAngle)*distance;

    var retPair = {};
    retPair.point1 = point1;
    retPair.point2 = point2;

    return retPair;
}

/**
 * A function that calculates an angle of line
 * @param x1
 * @param y1
 * @param x2
 * @param y2
 * @returns {number}
 */
function lineAngle(x1, y1, x2, y2){
    return Math.atan2( (y2-y1) , (x2-x1) );
}

/**
 * A function that maps direction represented as a string to animation frame number
 * @param str
 * @returns {number}
 */
function mapFrames(str){
    switch(str){
        case "Top":
            return 1;
            break;
        case "Down":
            return 5;
            break;
        case "Left":
            return 7;
            break;
        case "Right":
            return 3;
            break;
        case "TopLeft":
            return 8;
            break;
        case "TopRight":
            return 2;
            break;
        case "DownLeft":
            return 6;
            break;
        case "DownRight":
            return 4;
            break;
        default:
            return 5;
            break;
    }
}

/**
 * Bear enemy class
 * @type {*}
 */
var Rabbit = Enemy.extend({
    init: function(){
        this._super(2,20,2,2);
        this.framePrefix = "enemies/patita/patita";
        this.height = 32;
        this.worthDead = 10;
    }
});

/**
 * Cat enemy class
 * @type {*}
 */
//var Cat = Enemy.extend({
//    init: function(){
//        this._super(3.5,46,1,4);
//        this.jumpInterval = 0.05;
//        this.jumpAmp = 12;
//        this.framePrefix = "enemies/cat/cat000";
//        this.height = 42;
//        this.worthDead = 15;
//    }
//});

/**
 * Dog enemy class
 * @type {*}
 */
//var Dog = Enemy.extend({
//    init: function(){
//        this._super(3,50,4,1);
//        this.jumpInterval = 0.2;
//        this.jumpAmp = 3;
//        this.framePrefix = "enemies/dog/dog000";
//        this.height = 40;
//        this.worthDead = 20;
//    }
//});

/**
 * Rabbit enemy class
 * @type {*}
 */
//var Rabbit = Enemy.extend({
//    init: function(){
//        this._super(2,20,2,2);
//        this.framePrefix = "enemies/rabbit/rabbit000";
//        this.height = 32;
//        this.worthDead = 10;
//    }
//});

/**
 * Enemy disintegration animation
 * @type {*}
 */
var Dead = Class.extend({
    init: function(x,y,rot,scale){
        this.x = x;
        this.y = y;
        this.rot = rot;
        this.scale = scale;
    },
    x: 0,
    y: 0,
    framePrefix: 'enemies/dead/dead000',
    currentFrame: 0,
    rot: 0,
    scale: 0,
    draw : function(){
    drawSprite(this.framePrefix + Math.floor(this.currentFrame/2+1) + ".png",this.x,this.y,0,this.scale);
   //drawSprite("enemies/patita/patita.png",this.x,this.y,0,this.scale);
    },
    update : function(){
        this.currentFrame++;
        if(this.currentFrame>7){
           game.animations.remove(this);
        }
    }
});

