/**
 * interface.js
 *
 * contains classes and logic for game interface
 *
 * Copyright(c) 2013 Pavle Goloskokovic
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
 * http://creativecommons.org/licenses/by-nc-sa/3.0/
 */

/**
 * Class for representing page background map
 * @type {*}
 */
var Background = TiledMapClass.extend({

    init: function(){
        this.canvas = document.getElementById('bg_canvas');
        this.ctx = this.canvas.getContext('2d');
        var v = viewport();
        this.h = this.canvas.height = v.height;
        this.w = this.canvas.width = v.width;

        this.load("tld/backgroundMap.json");
    },

    h: 0,
    w: 0,
    canvas: null,

    /**
     * Function that parses map definition file created using Tided
     * Since map is generated randomly we place tiles which are used for filling layer
     * at the beginning of each layer.
     * Here we collect IDs of tiles and use them to fill layers data array with those values randomly.
     * We also get layer fill property indicating the percentage of the map that needs to be filled.
     * @param mapJSON
     */
    parseMapJSON: function (mapJSON) {

        this.currMapData = JSON.parse(mapJSON);

        // parsing layers data
        for(var layerIdx = 0; layerIdx < this.currMapData.layers.length; layerIdx++) {

            var layer = this.currMapData.layers[layerIdx];
            var data = layer.data;
            var dataNum = data.length;

            // fill percentage value
            var fill = parseInt(layer.properties.fill)/100;

            var tileIds = [];
            var tileIdsNum = 0;

            // collecting tile IDs values for map filling
            // tile IDs are placed at the beginning of layer's data array

            for(var i=0; i<dataNum; i++){
                if(data[i]!=0){
                    tileIds.push(data[i]);
                }else{
                    break;
                }
            }

            tileIdsNum = tileIds.length;

            // generating layer data definition
            for(i=0;i<dataNum;i++){
                if(Math.random()<fill){
                    data[i] = tileIds[Math.floor(Math.random()*tileIdsNum)];
                }else{
                    data[i] = 0;
                }
            }
        }

        var map = this.currMapData;

        this.numXTiles = map.width;
        this.numYTiles = map.height;

        this.tileSize.x = map.tilewidth;
        this.tileSize.y = map.tileheight;

        this.pixelSize.x = this.numXTiles * this.tileSize.x;
        this.pixelSize.y = this.numYTiles * this.tileSize.y;

        this.tilesets = [];

        for(i = 0; i < map.tilesets.length; i++) {

            var ts = {

                "firstgid": this.currMapData.tilesets[i].firstgid,

                "image": map.tilesets[i].image,
                "imageheight": this.currMapData.tilesets[i].imageheight,
                "imagewidth": this.currMapData.tilesets[i].imagewidth,
                "name": this.currMapData.tilesets[i].name,

                "numXTiles": Math.floor(this.currMapData.tilesets[i].imagewidth / this.tileSize.x),
                "numYTiles": Math.floor(this.currMapData.tilesets[i].imageheight / this.tileSize.y)
            };

            this.tilesets.push(ts);
        }

        this.ready = true;

        this.draw();
    },
    draw: function(){

        if(!this.ready){
            return;
        }

        var tx = Math.floor((this.w - this.pixelSize.x)/2);
        var ty = Math.floor((this.h - this.pixelSize.y)/2);

        this.ctx.translate(tx,ty);

        this._super();

        this.ctx.setTransform(1, 0, 0, 1, 0, 0);
    },
    update : function(){
        var v = viewport();
        this.h = this.canvas.height = v.height;
        this.w = this.canvas.width = v.width;
    }
});

/**
 * Function for positioning DOM elements on a page.
 * Gets called on page load event and on page resize event.
 */
function positionElements(){
    var v = viewport();
    var top =  0 + "px"; //(v.height-CANVAS_HEIGHT) + "px";
    var left = 0 + "px"; //(v.width-CANVAS_WIDTH) + "px";
    var gameInterface = document.getElementById('game_interface');
    var gameCanvas = document.getElementById('game_canvas');
    gameCanvas.style.left = gameInterface.style.left = left;
    gameCanvas.style.top =  gameInterface.style.top = top;
	gameCanvas.style.width = 788 + "px";
	//gameCanvas.style.width = gameInterface.style.width ="100%"; //(CANVAS_WIDTH-200) + "px";
	gameCanvas.style.height = 1280 + "px";
	//gameCanvas.style.margin = gameInterface.style.margin =0 + "px";
	//gameCanvas.style.margin = gameInterface.style.padding =0 + "px";
    gameCanvas.classList.add('glow');
    if(game && game.background){
        game.background.update();
        game.background.draw();
    }
}

function showInstructions(){
    document.getElementById('instructions').style.display = 'block';
    document.getElementById('menu_state').style.display = 'none';
}

function hideInstructions(){
    document.getElementById('instructions').style.display = '';
    document.getElementById('menu_state').style.display = 'block';
}

function showAbout(){
    document.getElementById('about').style.display = 'block';
    document.getElementById('menu_state').style.display = 'none';
}

function hideAbout(){
    document.getElementById('about').style.display = '';
    document.getElementById('menu_state').style.display = 'block';
}