/**
 * map.js
 *
 * contains classes and logic for map rendering
 *
 * Copyright(c) 2013 Pavle Goloskokovic
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
 * http://creativecommons.org/licenses/by-nc-sa/3.0/
 */

/**
 * Class used in course examples form map rendering
 * @type {*}
 */
var TiledMapClass = Class.extend({

    init: function(ctx){
       this.ctx = ctx;
    },
    ctx: null,

    currMapData: null, // JSON of the map.json file.
    tilesets: null,
    numXTiles: 0,
    numYTiles: 0,
    tileSize: {
        "x": 0,
        "y": 0
    },
    pixelSize: {
        "x": 0,
        "y": 0
    },
    ready: false,

    path: null,
    mapItems: null,
    platformsInFront: null,
    platformsBehind: null,

    load: function (map) {
        var thiz = this;
        xhrGet(map, function (index) {
            thiz.parseMapJSON(this.response);
        });
    },

    /**
     * Function that parses map definition file created using Tided.
     * Information about map items, platforms positions and path points
     * are included in map definition as object layers. This function also
     * parses this information and generates corresponding objects.
     * @param mapJSON
     */
    parseMapJSON: function (mapJSON) {

        this.currMapData = JSON.parse(mapJSON);

        var map = this.currMapData;

        this.numXTiles = map.width;
        this.numYTiles = map.height;

        this.tileSize.x = map.tilewidth;
        this.tileSize.y = map.tileheight;

        this.pixelSize.x = this.numXTiles * this.tileSize.x;
        this.pixelSize.y = this.numYTiles * this.tileSize.y;

        this.tilesets = [];

        for(var i = 0; i < map.tilesets.length; i++) {

            var ts = {

                "firstgid": this.currMapData.tilesets[i].firstgid,

                "image": map.tilesets[i].image,
                "imageheight": this.currMapData.tilesets[i].imageheight,
                "imagewidth": this.currMapData.tilesets[i].imagewidth,
                "name": this.currMapData.tilesets[i].name,

                "numXTiles": Math.floor(this.currMapData.tilesets[i].imagewidth / this.tileSize.x),
                "numYTiles": Math.floor(this.currMapData.tilesets[i].imageheight / this.tileSize.y)
            };

            this.tilesets.push(ts);
        }

        for(var layerIdx = 0; layerIdx < this.currMapData.layers.length; layerIdx++) {

            var layer = this.currMapData.layers[layerIdx];

            if(layer.type != "objectgroup") continue;

            // if layer is an object layer we parse its definition
            // to create corresponding objects

            switch(layer.name){
                // we define path array for this map
                case "path":
                    this.path = [];
                    // sort path points by their ordinal number
                    layer.objects.sort(function(foo,bar){return (parseInt(foo.properties.ord)<parseInt(bar.properties.ord)?-1:1);});
                    for(i=0; i<layer.objects.length; i++){
                        // add point to path array
                        this.path.push({x: layer.objects[i].x, y: layer.objects[i].y});
                    }
                    break;
                // we define map items array for this map
                case "mapItems":
                    this.mapItems = [];
                    for(i=0; i<layer.objects.length; i++){
                        // we take object type which represents name of the class
                        // and get that class object form "window" map
                        this.mapItems.push(new window[layer.objects[i].type](layer.objects[i].x,layer.objects[i].y));
                    }
                    break;
                // we define platforms array for this map for platforms
                // that need to be drawn in front of the enemies
                case "platformsInFront":
                    this.platformsInFront = [];
                    for(i=0; i<layer.objects.length; i++){
                        var platform = new Platform(layer.objects[i].x,layer.objects[i].y,true);
                        this.platformsInFront.push(platform);
                        game.platforms.push(platform);
                    }
                    break;
                // we define platforms array for this map for platforms
                // that need to be drawn behind the enemies
                case "platformsBehind":
                    this.platformsBehind = [];
                    for(i=0; i<layer.objects.length; i++){
                        var platform = new Platform(layer.objects[i].x,layer.objects[i].y,false);
                        this.platformsBehind.push(platform);
                        game.platforms.push(platform);
                    }
                    break;
            }
        }

        // check if map definition is missing any of the required layers
        // and display an error if true
        if( this.platformsInFront == null || this.platformsInFront.length == 0
         || this.platformsBehind == null || this.platformsBehind.length == 0
         || this.mapItems == null || this.mapItems.length == 0
         || this.path == null || this.path.length == 0){
            console.error("Map JSON file doesn't contain all required layers or their objects (platforms, mapItems or path)!");
            return;
        }

        this.ready = true;
    },

    getTilePacket: function (tileIndex) {

        var pkt = {
            "img": null,
            "px": 0,
            "py": 0
        };

        var tile;
        for(tile = this.tilesets.length - 1; tile >= 0; tile--) {
            if(this.tilesets[tile].firstgid <= tileIndex) break;
        }
        pkt.img = this.tilesets[tile].image;

        var localIdx = tileIndex - this.tilesets[tile].firstgid;

        var lTileX = Math.floor(localIdx % this.tilesets[tile].numXTiles);
        var lTileY = Math.floor(localIdx / this.tilesets[tile].numXTiles);

        pkt.px = (lTileX * this.tileSize.x);
        pkt.py = (lTileY * this.tileSize.y);

        return pkt;
    },

    draw: function () {

        if(!this.ready) return;

        for(var layerIdx = 0; layerIdx < this.currMapData.layers.length; layerIdx++) {

            if(this.currMapData.layers[layerIdx].type != "tilelayer") continue;

            var dat = this.currMapData.layers[layerIdx].data;

            for(var tileIDX = 0; tileIDX < dat.length; tileIDX++) {

                var tID = dat[tileIDX];
                if(tID === 0) continue;

                var tPKT = this.getTilePacket(tID);

                var worldX = tileIDX % this.numXTiles * this.tileSize.x;
                var worldY = Math.floor(tileIDX / this.numXTiles) * this.tileSize.y;

                drawTile(tPKT.img, tPKT.px, tPKT.py, this.tileSize.x, this.tileSize.y, worldX, worldY, this.tileSize.x, this.tileSize.y, this.ctx);
                //ctx.drawImage(tPKT.img, tPKT.px, tPKT.py, this.tileSize.x, this.tileSize.y, worldX/16, worldY/16, this.tileSize.x/16, this.tileSize.y/16);
            }
        }
    }
});

/**
 * Superclass for all map items
 * @type {*}
 */
var MapItem = Class.extend({
    init: function(x,y,frame){
        this.x = x;
        this.y = y;
        this.frame = frame;
    },
    x: 0,
    y: 0,
    frame: '',
    drawMapItem : function(){
        drawSprite(this.frame,this.x,this.y);
    },
    update : function(){}
});

/**
 * Status bar map item
 * @type {*}
 */
var StatusBar = MapItem.extend({/*535,582*/
    init: function(x,y){
        this._super(x,y,"map/status-bar.png");
    },
    drawMapItem : function(){

        var wm = game.waveManager;

        drawSprite(this.frame,this.x,this.y);

        if(!wm.waves[wm.currWave].ended){
            drawSprite("map/status-bar-enemy.png",this.x-83,this.y);
        }

        game.ctx.fillStyle = "#fff";
        game.ctx.font = "bold 16px Arial";
        game.ctx.fillText((game.status.lives<0?"0":game.status.lives),this.x-12,this.y+6);
        game.ctx.fillText(game.status.money,this.x+33,this.y+6);
        game.ctx.font = "12px Arial";

        //pause btn
        game.ctx.fillText((wm.currWave+1) + "/" + wm.waves.length,this.x-68,this.y+6);
        if(game.isPaused){
            drawSprite("map/status-bar-play.png",this.x+80,this.y);
        }else{
            drawSprite("map/status-bar-pause.png",this.x+80,this.y);
        }

        //game.ctx.strokeRect(this.x + 65, this.y - 15, 30, 30);

    }
});

/**
 * Class representing animated group of rocks
 * @type {}
 */
var Rock1 = MapItem.extend({
    init: function(x,y){
        this._super(x,y,"map/rocks/rocks1.png");
    },
    rotation : 0,
    rotationDirection: 1,
    translationDirection: 0,
    drawMapItem : function(){
        drawSprite(this.frame,this.x,this.y, this.rotation);
    },
    update : function(){
        this.rotation = this.rotation + 0.1 * this.rotationDirection;
        if(this.rotation >= 5 || this.rotation <= -4){
            this.rotationDirection*=-1;
        }
        var xStep, yStep;
        switch (this.translationDirection){
            case 0:
                xStep = -7.2;
                yStep = -11.25;
                if(this.x<60-7.2 || this.y<100-11.25){
                    this.x=60-7.2;
                    this.y=100-11.25;
                    this.translationDirection = (this.translationDirection+1)%4;
                }
                break;
            case 1:
                xStep = 7.2 + 10.15;
                yStep = 11.25 - 22.7;
                if(this.x>60+10.15 || this.y<100-22.7){
                    this.x=60+10.15;
                    this.y=100-22.7;
                    this.translationDirection = (this.translationDirection+1)%4;
                }
                break;
            case 2:
                xStep = -10.15 + 29.1;
                yStep = 22.7 - 1.6;
                if(this.x>60+29.1 || this.y>100-1.6){
                    this.x=60+29.1;
                    this.y=100-1.6;
                    this.translationDirection = (this.translationDirection+1)%4;
                }
                break;
            case 3:
                xStep = -29.1;
                yStep = 1.6;
                if(this.x<60 || this.y>101){
                    this.x=60;
                    this.y=100;
                    this.translationDirection = (this.translationDirection+1)%4;
                }
                break;
        }
        this.x += xStep / 60;
        this.y += yStep / 60;
    }

});

/**
 * Class representing animated group of rocks
 * @type {}
 */
var Rock2 = MapItem.extend({
    init: function(x,y){
        this._super(0,0,"map/rocks/rocks2.png");
        this.rootX = x;
        this.rootY = y;
    },
    rootX: 301,
    rootY: 78,
    translationDirection: 1,
    rotationDirection: 1,
    washingMachine: "map/wm.png",
    wmRotation: 0,
    drawMapItem : function(){

        var wmCos = Math.cos(this.wmRotation);

        if(wmCos>0){
            drawSprite(this.frame,this.x+this.rootX,this.y+this.rootY);
        }
        drawSprite(this.washingMachine,
            this.x+this.rootX+Math.sin(this.wmRotation)*70,
            this.y+this.rootY+wmCos*35+10,
            (this.wmRotation*30)%360);
        if(wmCos<0){
            drawSprite(this.frame,this.rootX+this.x,this.rootY+this.y);
        }
    },
    update : function(){
        this.x += 0.037 * this.translationDirection;
        this.y +=  - 0.1 * this.translationDirection;
        if(this.x >= 7.45 || this.y <= -19 ){
            this.translationDirection=-1;
        }else if(this.x<=0 || this.y>=0){
            this.translationDirection=1;
        }
        this.wmRotation = (this.wmRotation+0.02)%360;
    }

});

/**
 * Class representing animated group of rocks
 * @type {}
 */
var Rock3 = MapItem.extend({
    init: function(x,y){
        this._super(x,y,"map/rocks/rocks3.png");
    },
    rotation : 0,
    rotationDirection: 1,
    drawMapItem : function(){
        drawSprite(this.frame,this.x,this.y, this.rotation);
    },
    update : function(){
        this.rotation = this.rotation + 0.3 * this.rotationDirection;
        if(this.rotation >= 31.5 || this.rotation <= -24.2){
            this.rotationDirection*=-1;
        }
    }
});

/**
 * Class representing animated group of rocks
 * @type {}
 */
var Rock4 = MapItem.extend({
    init: function(x,y){
        this._super(x,y,"map/rocks/rocks4.png");
    },
    rotation : 0,
    drawMapItem : function(){
        drawSprite(this.frame,this.x,this.y, this.rotation);
    },
    update : function(){
        this.rotation = this.rotation + 0.524;
    }
});

/**
 * Class representing animated group of rocks
 * @type {}
 */
var Rock5 = MapItem.extend({
    init: function(x,y){
        this._super(0,0,"map/rocks/rocks5.png");
        this.rootX = x;
        this.rootY = y;
    },
    rootX: 94,
    rootY: 514,
    translationDirection: 1,
    drawMapItem : function(){
        drawSprite(this.frame,this.rootX+this.x,this.rootY+this.y);
    },
    update : function(){
        this.x += 0.014 * this.translationDirection;
        this.y +=  - 0.19 * this.translationDirection;
        if(this.x >= 3.95 || this.y <= -16 ){
            this.translationDirection=-1;
        }else if(this.x<=-4.0 || this.y>=5.55){
            this.translationDirection=1;
        }
    }
});

/**
 * Class representing animated group of rocks
 * @type {}
 */
var Rock6 = MapItem.extend({
    init: function(x,y){
        this._super(x,y,"map/rocks/rocks6.png");
    },
    rotation : 0,
    drawMapItem : function(){
        drawSprite(this.frame,this.x,this.y, this.rotation);
    },
    update : function(){
        this.rotation = this.rotation + 0.586;
    }
});

var Antenna = MapItem.extend({
    init: function(x,y){
        this._super(x,y,"map/antena.png");
    },
    rotation : 0,
    drawMapItem : function(){
        drawSprite(this.frame,this.x,this.y, this.rotation);
    },
    update : function(){
        this.rotation = this.rotation + 1.094;
    }
});

/**
 * Class representing alien observer :)
 * @type {}
 */
var Alien = MapItem.extend({
    init: function(x,y){
        this._super(x,y);
    },
    rotation : 0,
    currentFrame: 0,
    drawMapItem : function(){
        var frame = "map/alien/alien000" + (Math.floor(this.currentFrame/3) + 1) + ".png";
        drawSprite(frame ,this.x,this.y, this.rotation);
    },
    update : function(){
        this.currentFrame = (this.currentFrame+1)%27;
    }
});

/**
 * Class representing globe
 * @type {}
 */
var Globe = MapItem.extend({
    init: function(x,y){
        this._super(x,y,"map/globe.png");
    }
});