/*Copyright 2011 Google Inc. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 #limitations under the License.*/

/**
 * soundManager.js
 *
 * HTML5 audio element fallback for browsers that don't support Web Audio API
 *
 * Copyright(c) 2013 Pavle Goloskokovic
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
 * http://creativecommons.org/licenses/by-nc-sa/3.0/
 */

//----------------------------
SoundManager = Class.extend({

    clips: {},
    loopInstances: [],
    webAudio:true,
    enabled: true,
    mute:false,
    _context:null,
    _mainNode:null,
    //----------------------------
    init: function()
    {
        //http://www.html5rocks.com/en/tutorials/webaudio/intro/
        //WEBAUDIO API
        try {
            this._context = new webkitAudioContext();
            this._mainNode = this._context.createGainNode(0);
            this._mainNode.connect(this._context.destination);
        }
        catch(e) {
            // if Web Audio API is not supported in user's browser
            // use HTML5 audio elements for playing sound
            console.error('Web Audio API is not supported in this browser!');
            this.webAudio = false;

            // if user's browser does not support HTML5 audio element
            // disable sound manager
            if(!document.createElement('audio').canPlayType){
                console.error('HTML5 audio element is not supported in this browser!');
                this.enabled = false;
            }
        }


    },
    //----------------------------
    loadAsync: function(path, callbackFcn)
    {
        if(!this.enabled){return;}

        if(this.clips[path])
        {
            if(callbackFcn){callbackFcn(this.clips[path].s);}
            return;
        }

        var clip = {s:null,b:null,l:false};
        clip.s = new Sound();
        clip.s.path = path;
        this.clips[path] = clip;

        if(this.webAudio){

            var request = new XMLHttpRequest();
            request.open('GET', path, true);
            request.responseType = 'arraybuffer';
            request.onload = function() {
                gSM._context.decodeAudioData(request.response,
                    function(buffer)
                    {
                        clip.b = buffer;
                        clip.l = true;
                        if(callbackFcn){callbackFcn(clip.s);}
                    },
                    function(data)
                    {
                        Logger.log("failed");
                    });
            };
            request.send();

        }else{

            // create audio element for given file

            var audio = new Audio(path);
            audio.preload = 'auto';
            clip.s.webAudio = false;

            // acts as on load event listener
            audio.addEventListener( 'canplaythrough', function(ev){

                // removing event listener since in some browsers
                // this event fires more than once
                this.removeEventListener('canplaythrough',arguments.callee,false);

                clip.l = true;
                clip.b = audio;
                if(callbackFcn){
                    callbackFcn(clip.s);
                }

            }, false );
        }
    },

    //----------------------------
    isLoaded:function(path)
    {
        if(!this.enabled){return false;}

        var sd = this.clips[path];
        if(sd == null)
            return false;
        return sd.l;
    },
    //----------------------------
    toggleMute: function()
    {
        if(!this.enabled){return;}

        this.mute = !this.mute;

        if(this.webAudio){

            if(this.mute){
                this._mainNode.gain.value = 0;
            }else{
                this._mainNode.gain.value = 1;
            }

        }else{

            for(var i=0; i<this.loopInstances.length; i++){
                if(this.mute){
                    this.loopInstances[i].pause();
                }else{
                    this.loopInstances[i].play();
                }
            }
        }

        var muteBtn = this.canvas = document.getElementById('mute-toggle');
        if(muteBtn){
            muteBtn.classList.toggle('mute-off');
            muteBtn.classList.toggle('mute-on');
        }
    },
    //----------------------------
    stopAll: function()
    {
        if(!this.enabled){return;}

        if(this.webAudio){

            this._mainNode.disconnect();
            this._mainNode = this._context.createGainNode(0);
            this._mainNode.connect(this._context.destination);
            if(this.mute){
                this._mainNode.gain.value = 0;
            }else{
                this._mainNode.gain.value = 1;
            }

        }else{

            for(var i=0; i<this.loopInstances.length; i++){
                this.loopInstances[i].pause();
            }
            this.loopInstances = [];
        }


    },
    //----------------------------
    playSound: function(path,settings)
    {
        if(!this.enabled){return false;}

        var looping = false;
        var volume = 0.3;
        if(settings)
        {
            if(settings.looping)
                looping = settings.looping;
            if(settings.volume)
                volume = settings.volume;
        }

        var sd = this.clips[path];
        if(sd == null)
            return false;
        if(sd.l == false)
            return false;

        if(this.webAudio){

            var currentClip = this._context.createBufferSource(); // creates a sound source
            currentClip.buffer = sd.b;                           // tell the source which sound to play
            currentClip.gain.value = volume;
            currentClip.connect(this._mainNode);
            currentClip.loop = looping;
            currentClip.noteOn(0);                               // play the source now

            return true;

        }else{

            // we clone original audio element to avoid
            // multiple sound source downloads
            var a = sd.b.cloneNode(true);
            a.volume = volume;

            // we keep track of looping audio instances
            // in a global array so we can pause and stop
            // them individually when needed
            if(looping){
                this.loopInstances.push(a);
                a.addEventListener('ended', function() {
                    this.currentTime = 0;
                    this.play();
                }, false);
            }
            if(!this.mute){
                a.play();
            }

            return true;
        }
    }
});


//----------------------------
Sound = Class.extend({
    //----------------------------
    init: function(  ) {
    },
    path: null,
    webAudio: true,
    //----------------------------
    play: function(loop,volume) {
        gSM.playSound(this.path,{looping:loop, volume:volume});
    }
});
//----------------------------

var gSM = new SoundManager();

var initSoundManager = function () {

    if(!gSM.enabled){return;}

    //gSM.loadAsync("./audio/menu.ogg", function(s){
    //    if(game.status.state != GameState.MENU_STATE
    //    && game.status.state != undefined){
    //        return;
    //    }
    //    gSM.playSound("./audio/menu.ogg",{looping:true,volume:0.07});
    //});

    //gSM.loadAsync("./audio/play.ogg", function(s){
    //    if(game.status.state!=GameState.PLAY_STATE){
    //        return;
    //    }
    //    gSM.playSound("./audio/play.ogg",{looping:true,volume:0.07});
    //});

    //gSM.loadAsync("./audio/end-win.ogg", function(s){
    //    if(game.status.state!=GameState.END_STATE){
    //        return;
    //    }
    //    gSM.playSound("./audio/end-win.ogg",{looping:true,volume:0.07});
    //});

    //gSM.loadAsync("./audio/end-lost.ogg", function(s){
    //    if(game.status.state!=GameState.END_STATE){
    //        return;
    //    }
    //    gSM.playSound("./audio/end-lost.ogg",{looping:true,volume:0.07});
    //});

    gSM.loadAsync("./audio/coins.ogg");
    gSM.loadAsync("./audio/junk_explode.ogg");
    gSM.loadAsync("./audio/plasma_explode.ogg");
    gSM.loadAsync("./audio/rocket_shoot.ogg");
    gSM.loadAsync("./audio/rocket_explode.ogg");
    gSM.loadAsync("./audio/dead.ogg");
};
