/**
 * towers.js
 *
 * contains all classes and logic for platforms and towers objects
 *
 * Copyright(c) 2013 Pavle Goloskokovic
 *
 * This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
 * http://creativecommons.org/licenses/by-nc-sa/3.0/
 */

/**
 * Class representing a platform object
 * @type {*}
 */
var Platform = Class.extend({
    init: function(x,y,isInFront){
        this.x = x;
        this.y = y;
        this.isInFront = isInFront;
        this.frame = "towers/platform.png";
        this.lightPeriod = 15 + Math.random() * 10;
    },
    x: 0,
    y: 0,
    frame: '',
    lightPeriod: 20,
    lightFrame:0,
    tower: null,
    displayDialog: false,
    buyDialogFrame: "towers/platform_buy_menu.png",
    sellDialogFrame: "towers/platform_sell_menu.png",
    isInFront: false,

    drawPlatform : function(){
        drawSprite(this.frame,this.x,this.y);
        drawSprite("towers/platform_light000"+(Math.floor(this.lightFrame/this.lightPeriod) + 1)+".png",this.x-14,this.y-35);
        if(this.tower !== null){
            this.tower.drawTower();
        }

        if(this.displayDialog){

            //sellDialog
            if(this.tower !== null){

                drawSprite(this.sellDialogFrame,this.x,this.y-35);

                /*
                game.ctx.strokeRect(this.x - 40, this.y - 55, 40, 40);//sell
                game.ctx.strokeRect(this.x, this.y - 55, 40, 40);//X
                */

            //buyDialog
            }else{

                drawSprite(this.buyDialogFrame,this.x+13,this.y-35);

                /*
                game.ctx.strokeRect(this.x - 58, this.y - 61, 37, 57);//junk area
                game.ctx.strokeRect(this.x - 15, this.y - 61, 37, 57);//plasma area
                game.ctx.strokeRect(this.x + 27, this.y - 61, 37, 57);//rocket area
                game.ctx.strokeRect(this.x + 72, this.y - 52, 20, 20);//X
                */
            }
        }
    },
    update : function(){
        this.lightFrame = (this.lightFrame+1) % (2*this.lightPeriod);
        if(this.tower !== null){
            this.tower.update();
        }
    },
    setTower: function(tower){
        this.tower = tower;
    },
    clearTower: function(){
        this.tower = null;
        this.displayDialog = false;
    }
});

/**
 * Class representing a tower object
 * @type {*}
 */
var Tower = Class.extend({
    init: function(shootInterval,shootRange,price,platform,frame){
        this.shootInterval = shootInterval;
        this.shootRange = shootRange;
        this.buyPrice = price;
        this.sellPrice = Math.floor(price * 0.6);
        this.platform = platform;
        this.frame = frame;
    },
    shootInterval: 0,
    shootRange: 0,
    buyPrice: 0,
    sellPrice: 0,
    bulletType: null,
    bulletXOffset: 0,
    bulletYOffset: 0,
    platform: null,
    platformYOffset: 0,
    frame: '',
    shootSound: null,
    lastTickTime: null,
    pausedDeltaTimeMs: 0,

    drawTower : function(){
        if(this.platform !== null){
            drawSprite(this.frame,this.platform.x,this.platform.y+this.platformYOffset);
        }
    },
    update : function(){
        if(this.platform !== null){
            this.tick();
        }
    },
    /**
     * Function that measures time and checks if
     * enough time passed so that the tower can shoot again
     */
    tick: function(){
        if(this.platform !== null){
            if(this.lastTickTime == null || (new Date()).getTime()-this.lastTickTime.getTime()>=this.shootInterval){
                if(this.lastTickTime == null){
                    this.lastTickTime = new Date();
                }else{
                    this.lastTickTime = new Date(this.lastTickTime.getTime() + this.shootInterval);
                }
                this.shoot();
            }
        }
    },
    shoot: function(){

        if(this.platform == null){return;}

        var distance;

        // looping through enemies array trying to
        // find enemy in range to shoot at
        for(var i=0; i<game.enemies.length; i++){

            distance = vectorMagnitude(this.platform.x,game.enemies[i].x,this.platform.y,game.enemies[i].y);

            if(distance < this.shootRange){

                this.playShootSound();

                game.bullets.push(new this.bulletType(this.platform.x,this.platform.y+this.bulletYOffset,game.enemies[i]));
                break;
            }
        }
    },
    playShootSound: function(){}
});

/**
 * Class representing a tower that shoots JunkBullet objects
 * @type {*}
 */
var JunkTower = Tower.extend({
    init: function(platform){
        this._super(500,140,100,platform,"towers/junk.png");
        this.bulletType = JunkBullet;
        this.platformYOffset = 2;
        this.bulletYOffset = -25;
    }
});

/**
 * Class representing a tower that shoots PlasmaBullet objects
 * @type {*}
 */
var PlasmaTower = Tower.extend({
    init: function(platform){
        this._super(1200,160,150,platform,"towers/plasma.png");
        this.bulletType = PlasmaBullet;
        this.platformYOffset = -6;
        this.bulletYOffset = -37;
    }
});

/**
 * Class representing a tower that shoots RocketBullet objects
 * @type {*}
 */
var RocketTower = Tower.extend({
    init: function(platform){
        this._super(700,200,200,platform,"towers/rocket.png");
        this.bulletType = RocketBullet;
        this.platformYOffset = 11;
        this.bulletYOffset = -10;
    },
    playShootSound: function(){
        gSM.playSound("./audio/rocket_shoot.ogg",{volume:0.1});
    }
});

/**
 * Function for pushing platform in front of enemies
 * for drawing purposes
 * @param platform
 */
function pushPlatformInFront(platform){
    if(!platform.isInFront
    && game.maps[game.level].platformsBehind.remove(platform)){
        game.maps[game.level].platformsInFront.push(platform);
    }
}
/**
 * Function for pushing platform behind enemies
 * for drawing purposes
 * @param platform
 */
function pushPlatformBehind(platform){
    if(!platform.isInFront
    && game.maps[game.level].platformsInFront.remove(platform)){
        game.maps[game.level].platformsBehind.push(platform);
    }
}