/**
 * @author Gregory Tardivel
 */
var jsDraw = { 
  AUTHOR: 'Gregory Tardivel',
  DESCRIPTION: 'Library for drawing on a HTML5 canvas',
  REVISION: '1.0.1'
};
(function() {
  'use strict';
}());
jsDraw.CRAYON = 0;
jsDraw.GOMME = 1;
jsDraw.REMPLIR = 2;
jsDraw.position = function(elt) {
  var retour = {x: 0, y: 0};
  retour.x += elt.offsetLeft;
  retour.y += elt.offsetTop;
  while (elt.parentElement) {
    elt = elt.parentElement;
    retour.x += elt.offsetLeft;
    retour.y += elt.offsetTop;
  }
  return retour;
};
jsDraw.locate = function(elt, evt) {
  var x = 0;
  var y = 0;
  var touches = evt.changedTouches;
  if (touches) {
    // FirefoxOS
    x = touches[0].pageX - jsDraw.position(elt).x;
    y = touches[0].pageY - jsDraw.position(elt).y;
  } else {
    // Web
    x = evt.pageX - jsDraw.position(elt).x;
    y = evt.pageY - jsDraw.position(elt).y;
  }
  return {x: x, y: y};
};
jsDraw.pipette = function(canvas, evt) {
  var ctx = canvas.getContext('2d');
  var pick = jsDraw.locate(canvas, evt);
  var x = pick.x;
  var y = pick.y;
  var pixel = ctx.getImageData(x, y, 1, 1).data;
  return new jsDraw.Color({
    r: pixel[0], g: pixel[1], b: pixel[2], a: pixel[3]
  });
};
jsDraw.Color = function(rgb) {
  'use strict';
  var r = rgb.r,
  g = rgb.g,
  b = rgb.b,
  a = rgb.a;
  if (a === undefined) {
    a = 255;
  }
  if (r > 255 || g > 255 || b > 255 || a > 255) {
    throw 'Invalid color';
  }
  return {
    r: r,
    g: g,
    b: b,
    a: a,
    toRGBA: function() {
      return 'rgba(' + r + ',' + g + ',' + b + ',' + (a / 255) + ')';
    },
    equals: function(color) {
      return r === color.r && g === color.g && b === color.b && a === color.a;
    }
  };
};
jsDraw.Dessin = function(parent) {
  'use strict';
  var canvas = document.createElement('canvas');
  canvas.style.border = '1px solid #555';
  canvas.style.margin = 'auto';
  canvas.style.display = 'block';
  if (parent !== undefined) {
    parent.appendChild(canvas);
  } else {
    document.body.appendChild(canvas);
  }
  var saves = [], idxUndo = undefined,
  backgroundColor = new jsDraw.Color({
    r: 255, g: 255, b: 255
  }),
  color = new jsDraw.Color({
    r: 0, g: 0, b: 0
  }),
  events = [],
  activUndo = true,
  drawWidth = 4,
  action = jsDraw.CRAYON,
  context = canvas.getContext('2d'),
  drawing = false,
  lock = false,
  previousX = -1,
  previousY = -1,
  currentX = -1,
  currentY = -1,
  currentLayerData, newLayerData,
  configure = function() {
    context.lineWidth = drawWidth;
    if (action === jsDraw.CRAYON) {
      context.strokeStyle = color.toRGBA();
      context.fillStyle = color.toRGBA();
    } else if (action === jsDraw.GOMME) {
      context.strokeStyle = backgroundColor.toRGBA();
      context.fillStyle = backgroundColor.toRGBA();
    }
    context.lineJoin = 'round';
    context.lineCap = 'round';
  },
  locate = function(event) {
    var pick = jsDraw.locate(canvas, event);
    currentX = pick.x;
    currentY = pick.y;
    if (previousX === -1 && previousY === -1) {
      previousX = currentX;
      previousY = currentY;
    }
  },
  line = function(x1, y1, x2, y2) {
    context.beginPath();
    context.moveTo(x1, y1);
    context.lineTo(x2, y2);
    context.closePath();
    context.stroke();
  },
  arc = function(x, y, r, s, e) {
    context.beginPath();
    context.arc(x, y, r, s, e);
    context.closePath();
    context.fill();
  },
  point = function(event) {
    var mid = drawWidth / 2;
    arc(currentX, currentY, drawWidth / 2, 0, 2 * Math.PI);
  },
  draw = function(event) {
    if (drawing) {
      context = canvas.getContext('2d');
      locate(event);
      if (events['ondraw']) {
        events['ondraw'].call({
          x: currentX, y: currentY
        });
      }
      configure();
      line(previousX, previousY, currentX, currentY);
      previousX = currentX;
      previousY = currentY;
    }
  },
  reset = function() {
    drawing = false;
    previousX = -1;
    previousY = -1;
    if (activUndo === true) {
      if (idxUndo !== undefined) {
        saves.splice(idxUndo + 1, saves.length - idxUndo - 1);
        idxUndo = undefined;
      }
      if (saves.length > 5) {
        saves.splice(0, 1);
      }
      saves.push(context.getImageData(0, 0, canvas.width, canvas.height));
    }
  },
  colorPixel = function(pixelPos) {
    newLayerData.data[pixelPos] = color.r;
    newLayerData.data[pixelPos + 1] = color.g;
    newLayerData.data[pixelPos + 2] = color.b;
    newLayerData.data[pixelPos + 3] = color.a !== undefined ? color.a : 255;
  },
  matchNewColor = function(pixelPos, color) {
    var r = newLayerData.data[pixelPos],
    g = newLayerData.data[pixelPos + 1],
    b = newLayerData.data[pixelPos + 2],
    a = newLayerData.data[pixelPos + 3];
    return r === color.r && g === color.g && b === color.b && a === color.a;
  },
  matchStartColor = function(pixelPos, color) {
    var r = currentLayerData.data[pixelPos],
    g = currentLayerData.data[pixelPos + 1],
    b = currentLayerData.data[pixelPos + 2],
    a = currentLayerData.data[pixelPos + 3];
    return r === color.r && g === color.g && b === color.b && a === color.a;
  },
  floodFill = function(pixel) {
    var newPos, x, y, pixelPos, reachLeft, reachRight;
    var pixelStack = [[pixel.x, pixel.y]];
    while (pixelStack.length) {
      newPos = pixelStack.pop();
      x = newPos[0];
      y = newPos[1];
      // Get current pixel position
      pixelPos = (y * canvas.width + x) * 4;
      if (matchNewColor(pixelPos, color)) {
        continue;
      }
      // Go up as long as the color matches and are inside the canvas
      while (y >= 0 && matchStartColor(pixelPos, pixel.color)) {
        y -= 1;
        pixelPos -= canvas.width * 4;
      }
      pixelPos += canvas.width * 4;
      y += 1;
      reachLeft = false;
      reachRight = false;
      // Go down as long as the color matches and in inside the canvas
      while (y <= canvas.height && matchStartColor(pixelPos, pixel.color)) {
        y += 1;
        colorPixel(pixelPos);
        if (x > 0) {
          if (matchStartColor(pixelPos - 4, pixel.color)) {
            if (!reachLeft) {
              // Add pixel to stack
              pixelStack.push([x - 1, y]);
              reachLeft = true;
            }
          } else if (reachLeft) {
            reachLeft = false;
          }
        }
        if (x < canvas.width) {
          if (matchStartColor(pixelPos + 4, pixel.color)) {
            if (!reachRight) {
              // Add pixel to stack
              pixelStack.push([x + 1, y]);
              reachRight = true;
            }
          } else if (reachRight) {
            reachRight = false;
          }
        }
        pixelPos += canvas.width * 4;
      }
    }
    context.putImageData(newLayerData, 0, 0);
  },
  remplir = function(event) {
    currentLayerData = context.getImageData(0, 0, canvas.width, canvas.height);
    newLayerData = context.getImageData(0, 0, canvas.width, canvas.height);
    var pick = jsDraw.locate(canvas, event);
    var cible = new jsDraw.Pixel({
      canvas: canvas,
      x: pick.x,
      y: pick.y
    });
    floodFill(cible);
  },
  handleStart = function(evt) {
    evt.preventDefault();
    if (!lock) {
      if (action != jsDraw.REMPLIR) {
        drawing = true;
        locate(evt);
        configure();
        point(evt);
      } else {
        remplir(evt);
      }
    }
  },
  handleMove = function(evt) {
    evt.preventDefault();
    if (!lock) {
      draw(evt);
    }
  };
  // FirefoxOS
  canvas.addEventListener('touchstart', handleStart, false);
  canvas.addEventListener('touchend', reset, false);
  canvas.addEventListener('touchmove', handleMove, false);
  // PC
  canvas.addEventListener('mousedown', handleStart, false);
  canvas.addEventListener('mousemove', handleMove, false);
  canvas.addEventListener('mouseup', reset, false);
  return {
    canvas: canvas,
    color: function(p) {
      if (p !== undefined) {
        color = p;
      }
      return color;
    },
    addEventListener: function(event, callback) {
      events[event] = callback;
    },
    lock: function(p) {
      if (p !== undefined) {
        lock = p;
      }
      return lock;
    },
    fill: function(color) {
      backgroundColor = color;
      var context = canvas.getContext('2d');
      context.fillStyle = color.toRGBA();
      context.fillRect(0, 0, canvas.width, canvas.height);
      reset();
    },
    load: function(image) {
      var context = canvas.getContext('2d');
      context.drawImage(image, 0, 0, image.width, image.height, 0, 0, canvas.width, canvas.height);
    },
    mode: function(param) {
      action = param;
    },
    resize: function(width, height) {
      canvas.width = width;
      canvas.height = height;
    },
    size: function(w) {
      if (w !== undefined) {
        drawWidth = w;
      }
      return drawWidth;
    },
    clearUndo: function() {
      idxUndo = undefined;
      saves = [];
    },
    undo: function() {
      if (activUndo === true) {
        if (idxUndo === undefined) {
          idxUndo = saves.length - 1;
        }
        idxUndo--;
        if (idxUndo < 0) {
          idxUndo = 0;
        }
        if (idxUndo >= 0) {
          context.clearRect(0, 0, canvas.width, canvas.height);
          context.putImageData(saves[idxUndo], 0, 0);
        }
      }
    },
    redo: function() {
      if (activUndo) {
        idxUndo++
        if (idxUndo > saves.length - 1) {
          idxUndo = saves.length - 1;
        }
        if (idxUndo >= 0 && idxUndo < saves.length) {
          context.clearRect(0, 0, canvas.width, canvas.height);
          context.putImageData(saves[idxUndo], 0, 0);
        }
      }
    },
    setActivUndo: function(b) {
      activUndo = b;
    }
  };
};
jsDraw.Pixel = function(json) {
  'use strict';
  var x = json.x,
  y = json.y,
  ctx = json.canvas.getContext('2d'),
  data = ctx.getImageData(x, y, 1, 1).data,
  color = new jsDraw.Color({r: data[0], g: data[1], b: data[2], a: data[3]});
  return {
    x: x,
    y: y,
    color: color
  };
};
jsDraw.ColorPicker = function(callback) {
  'use strict';
  var fenetre = document.createElement('div');
  fenetre.id = 'dJSDrawColor';
  var imageColor = new Image();
  imageColor.src = './css/images/color-picker.png';
  var color = new jsDraw.Color({
    r: 0, g: 0, b: 0
  });
  var dColor = document.createElement('div');
  dColor.id = 'dJSDrawCurrent';
  dColor.style.backgroundColor = color.toRGBA();
  var bClose = document.createElement('button');
  bClose.id = 'bJSDrawClose';
  bClose.onclick = function() {
    hide(callback);
  }
  var canvas = document.createElement('canvas');
  canvas.style.marginTop = '20px';
  var actif = false,
  handleSelect = function(evt) {
    evt.preventDefault();
    if (evt.type == 'mousedown' || evt.type == 'touchstart') {
      actif = true;
    }
    if (evt.type == 'mouseup' || evt.type == 'touchend') {
      actif = false;
    }
    if (actif) {
      color = jsDraw.pipette(canvas, evt);
      dColor.style.backgroundColor = color.toRGBA();
    }
  },
  hide = function(callback) {
    fenetre.style.display = 'none';
    if (callback) {
      callback.call({
        color: color
      });
    }
  },
  resize = function() {
    var width = document.body.clientWidth * 0.9;
    var height = document.body.clientHeight * 0.9;
    if (height < width) {
      width = height;
    } else {
      height = width;
    }
    fenetre.style.left = (document.body.clientWidth - width) / 2 + 'px';
    fenetre.style.top = (document.body.clientHeight - height - 20) / 2 + 'px';
    fenetre.style.width = width + 'px';
    fenetre.style.height = height + 'px';
    canvas.width = width;
    canvas.height = height;
    var ctx = canvas.getContext('2d');
    ctx.drawImage(imageColor, 0, 0, imageColor.width, imageColor.height, 0, 0, width, height);
  };
  resize();
  fenetre.appendChild(dColor);
  fenetre.appendChild(bClose);
  fenetre.appendChild(canvas);
  document.getElementsByTagName('body')[0].appendChild(fenetre);
  // FirefoxOS
  canvas.addEventListener('touchstart', handleSelect, false);
  canvas.addEventListener('touchend', handleSelect, false);
  canvas.addEventListener('touchmove', handleSelect, false);
  // PC
  canvas.addEventListener('mousedown', handleSelect, false);
  canvas.addEventListener('mousemove', handleSelect, false);
  canvas.addEventListener('mouseup', handleSelect, false);
  return {
    hide: hide,
    resize: resize,
    show: function() {
      fenetre.style.display = 'block';
    }
  };
};
