"use strict";

//var shutterSpeedValues = [1 / 16000, 1 / 8000, 1 / 4000, 1 / 2000, 1 / 1000, 1 / 500, 1 / 250, 1 / 125, 1 / 60, 1 / 30, 1 / 15, 1 / 8, 1 / 4, 1 / 2, 1, 2, 4, 8, 15, 30, 60];
//var apertureValues = [1, 1.4, 2, 2.8, 4, 5.6, 8, 11, 16, 22, 32, 45, 64];
//var isoValues = [50, 100, 200, 400, 800, 1600, 3200, 6400, 12800, 25600, 51200, 102400, 204800, 409600];

var shutterSpeedValues = [1 / 16000, 1 / 12500, 1 / 10000, 1 / 8000, 1 / 6400, 1 / 5000, 1 / 4000, 1 / 3200, 1 / 2500, 1 / 2000, 1 / 1600, 1 / 1250, 1 / 1000, 1 / 800, 1 / 640, 1 / 500, 1 / 400, 1 / 320, 1 / 250, 1 / 200, 1 / 160, 1 / 125, 1 / 100, 1 / 80, 1 / 60, 1 / 50, 1 / 40, 1 / 30, 1 / 25, 1 / 20, 1 / 15, 1 / 13, 1 / 10, 1 / 8, 1 / 6, 1 / 5, 1 / 4, 1 / 3.2, 1 / 2.5, 1 / 2, 1 / 1.6, 1 / 1.3, 1, 1.3, 1.6, 2, 2.5, 3.2, 4, 5, 6, 8, 10, 13, 15, 20, 25, 30, 40, 50, 60];
var apertureValues = [1, 1.1, 1.2, 1.4, 1.6, 1.8, 2, 2.2, 2.5, 2.8, 3.2, 3.5, 4, 4.5, 5, 5.6, 6.3, 7.1, 8, 9, 10, 11, 13, 14, 16, 18, 20, 22, 25, 29, 32, 36, 40, 45, 51, 57, 64];
var isoValues = [50, 64, 80, 100, 125, 160, 200, 250, 320, 400, 500, 640, 800, 1000, 1250, 1600, 2000, 2500, 3200, 4000, 5000, 6400, 8000, 10000, 12800, 16000, 20000, 25600, 32000, 40000, 51200, 64000, 80000, 102400, 128000, 160000, 204800, 256000, 320000, 409600];

// Exposure "osztály"

var Exposure = function (shutterSpeed, aperture, iso) {
   this.shutterSpeed = shutterSpeed;
   this.aperture = aperture;
   this.iso = iso;
};

Exposure.prototype.CalculateExposureDifferenceByType = function (exp, type) {
   if (typeof(exp) != "object" || typeof(type) != "string") {
      return Number.NaN;
   }

   switch (type) {
      case 's':
         return Math.log2(exp.shutterSpeed / this.shutterSpeed);

      case 'a':
         return LogSqrt2(this.aperture / exp.aperture); // A nevező és a számláló fordított, mert negált érték kell

      case 'i':
         return Math.log2(exp.iso / this.iso);

      default:
         return Number.NaN;
   }
};

Exposure.prototype.CalculateExposureDifference = function (exp) {
   if (typeof(exp) != "object") {
      return Number.NaN;
   }

   return this.CalculateExposureDifferenceByType(exp, 's') + this.CalculateExposureDifferenceByType(exp, 'a') + this.CalculateExposureDifferenceByType(exp, 'i');
};

Exposure.prototype.CorrectExposureDifferenceByType = function (corrVal, type) {
   var index = -1;

   if (typeof(corrVal) != "number" || isNaN(corrVal) || typeof(type) != "string") {
      return index;
   }

   switch (type) {
      case 's':
         index = GetClosestNeighbourIndex(shutterSpeedValues, this.shutterSpeed * Math.pow(0.5, corrVal));
         this.shutterSpeed = shutterSpeedValues[index];
         break;

      case 'a':
         index = GetClosestNeighbourIndex(apertureValues, this.aperture * Math.pow(Math.sqrt(2), corrVal));
         this.aperture = apertureValues[index];
         break;

      case 'i':
         index = GetClosestNeighbourIndex(isoValues, this.iso * Math.pow(0.5, corrVal));
         this.iso = isoValues[index];
         break;
   }

   return index;
};

Exposure.prototype.ToString = function () {
   var s = "Shutter speed: " + FormatShutterSpeed(this.shutterSpeed) + ";   ";
   s += "Aperture: " + FormatAperture(this.aperture) + ";   ";
   s += "ISO: " + FormatISO(this.iso);
   return s;
};

// "Statikus" metódusok

function GetRealExposureDifferenceAsThirds(difference) {
   difference = Math.round(difference * 3);
   var fraction = difference % 3;
   var integer = (difference - fraction) / 3;
   return [integer, fraction];
}

function GetClosestNeighbourIndex(valueArray, searchValue) {
   var minIndex = 0;
   var minDiff = Math.abs(valueArray[0] - searchValue);

   for (var i = 1; i < valueArray.length; i++) {
      var tmpDiff = Math.abs(valueArray[i] - searchValue);

      if (tmpDiff < minDiff) {
         minIndex = i;
         minDiff = tmpDiff;
      }
   }

   return minIndex;
}

function LogSqrt2(val) {
   // logA(B) = logC(B) / logC(A)
   return Math.log2(val) / Math.log2(Math.sqrt(2));
}

function FormatShutterSpeed(val) {
   return (val < 1 ? "1/" + (Math.round(10 / val) / 10) : val) + " s";
}

function FormatAperture(val) {
   return "f/" + val;
}

function FormatISO(val) {
   return val;
}
