// DOMContentLoaded is fired once the document has been loaded and parsed,
// but without waiting for other external resources to load (css/images/etc)
// That makes the app more responsive and perceived as faster.
// https://developer.mozilla.org/Web/Reference/Events/DOMContentLoaded
window.addEventListener('DOMContentLoaded', function() {

  // We'll ask the browser to use strict code to help us catch errors earlier.
  // https://developer.mozilla.org/Web/JavaScript/Reference/Functions_and_function_scope/Strict_mode
  'use strict';

  //var apiURL = 'https://developer.mozilla.org/search.json?q=';
  var appSecureHost = 'https://www.flexiauthor.com';
  var apiURL = appSecureHost + '/en/contributors-api/contributors/index?featured=1&keyword=';
  var errorMsg = document.getElementById('error');
  var searchInput = document.getElementById('term');
  var results = document.getElementById('results');
  var request = null;
  var translate = navigator.mozL10n.get;
  var form = document.querySelector('form');

  // Forms will take the values in the input fields they contain
  // and send them to a server for further processing,
  // but since we want to stay in this page AND make a request to another server,
  // we will listen to the 'submit' event, and prevent the form from doing what
  // it would usually do, using preventDefault.
  // Read more about it here:
  // https://developer.mozilla.org/Web/API/event.preventDefault
  //
  // Then we search without leaving this page, just as we wanted.
  form.addEventListener('submit', function(e) {
      e.preventDefault();
      search();
  }, false);

  // We want to wait until the localisations library has loaded all the strings.
  // So we'll tell it to let us know once it's ready.
  navigator.mozL10n.once(search);

  // ---

  function search() {

    // Are we searching already? Then stop that search
    if(request && request.abort) {
      request.abort();
    }


    results.textContent = translate('searching');

    // We will be using the 'hidden' attribute throughout the app rather than a
    // 'hidden' CSS class because it enhances accessibility.
    // See: http://www.whatwg.org/specs/web-apps/current-work/multipage/editing.html#the-hidden-attribute
    results.hidden = false;
    errorMsg.hidden = true;


    var term = searchInput.value;
    if(term.length === 0) {
      term = searchInput.placeholder;
    }

    var url = apiURL + term;

    // If you don't set the mozSystem option, you'll get CORS errors (Cross Origin Resource Sharing)
    // You can read more about CORS here: https://developer.mozilla.org/docs/HTTP/Access_control_CORS
    request = new XMLHttpRequest({ mozSystem: true });

    request.open('get', url, true);
    request.responseType = 'json';
    
    request.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
    request.setRequestHeader('X-Range', 'items=0-10');

    // We're setting some handlers here for dealing with both error and
    // data received. We could just declare the functions here, but they are in
    // separate functions so that search() is shorter, and more readable.
    request.addEventListener('error', onRequestError);
    request.addEventListener('load', onRequestLoad);

    request.send();

  }


  function onRequestError() {

    var errorMessage = request.error;
      if(!errorMessage) {
        errorMessage = translate('searching_error');
      }
      showError(errorMessage);

  }


  function onRequestLoad() {

    var response = request.response;
    
    if(response === null) {
      showError(translate('searching_error'));
      return;
    }

    results.textContent = '';

    var documents = response.items;

    if(documents.length === 0) {

      var p = document.createElement('p');
      p.textContent = translate('search_no_results');
      results.appendChild(p);

    } else {

      documents.forEach(function(doc) {
        appendAuthorProfile(doc);
      });

    }

    // And once we have all the content in place, we can show it.
    results.hidden = false;

  }

  function appendAuthorProfile(author){

        var panel = document.createElement('div');
        panel.setAttribute("class", "panel panel-default");

        var profile = document.createElement('div');
        profile.setAttribute("class", "panel-body media");
        
         var photo = document.createElement('img');
         photo.src = author.photo;
         photo.setAttribute("class", "thumbnail pull-left");
         photo.style = "max-height:150px;";

        var photoLink = document.createElement('a');
        photoLink.href = appSecureHost + author._links.self.href;
        photoLink.appendChild(photo);
        photoLink.addEventListener('click', function(evt) {
          evt.preventDefault();
          window.open(this.href, 'overlay');
        });
        profile.appendChild(photoLink);
    

        var docLink = document.createElement('a');
        docLink.innerHTML ='<h4>' + author.formattedName + '</h4>';
        docLink.href = appSecureHost + author._links.self.href;

        docLink.addEventListener('click', function(evt) {
          evt.preventDefault();
          window.open(evt.target.href, 'overlay');
        });

        profile.appendChild(docLink);

       
        var bookCount = document.createElement('p');
        bookCount.setAttribute("class", "");
       var bcount = 0; 
       if(author.bookCount){
          bcount = author.bookCount;
        }
        bookCount.innerHTML = 'Book Count: <span class="badge">' + bcount + '</span>';
        profile.appendChild(bookCount);
      

        panel.appendChild(profile);
    
        results.appendChild(panel);
    
  }
  
  function showError(text) {
    errorMsg.textContent = text;
    errorMsg.hidden = false;
    results.hidden = true;
  }

});
