(function (ng) {
    'use strict';
    var module = ng.module('lrInfiniteScroll', []);

    module.directive('lrInfiniteScroll', ['$timeout', function (timeout) {
        return{
            link: function (scope, element, attr) {
                var
                    lengthThreshold = attr.scrollThreshold || 50,
                    timeThreshold = attr.timeThreshold || 400,
                    handler = scope.$eval(attr.lrInfiniteScroll),
                    promise = null,
                    lastRemaining = 9999;

                lengthThreshold = parseInt(lengthThreshold, 10);
                timeThreshold = parseInt(timeThreshold, 10);

                if (!handler || !ng.isFunction(handler)) {
                    handler = ng.noop;
                }

                element.bind('scroll', function () {
                    var
                        remaining = element[0].scrollHeight - (element[0].clientHeight + element[0].scrollTop);

                    //if we have reached the threshold and we scroll down
                    if (remaining < lengthThreshold && (remaining - lastRemaining) < 0) {

                        //if there is already a timer running which has no expired yet we have to cancel it and restart the timer
                        if (promise !== null) {
                            timeout.cancel(promise);
                        }
                        promise = timeout(function () {
                            handler();
                            promise = null;
                        }, timeThreshold);
                    }
                    lastRemaining = remaining;
                });
            }

        };
    }]);
})(angular);

/**
 * Copyright 2012 Craig Campbell
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Mousetrap is a simple keyboard shortcut library for Javascript with
 * no external dependencies
 *
 * @version 1.2.2
 * @url craig.is/killing/mice
 */
(function() {

    /**
     * mapping of special keycodes to their corresponding keys
     *
     * everything in this dictionary cannot use keypress events
     * so it has to be here to map to the correct keycodes for
     * keyup/keydown events
     *
     * @type {Object}
     */
    var _MAP = {
            8: 'backspace',
            9: 'tab',
            13: 'enter',
            16: 'shift',
            17: 'ctrl',
            18: 'alt',
            20: 'capslock',
            27: 'esc',
            32: 'space',
            33: 'pageup',
            34: 'pagedown',
            35: 'end',
            36: 'home',
            37: 'left',
            38: 'up',
            39: 'right',
            40: 'down',
            45: 'ins',
            46: 'del',
            91: 'meta',
            93: 'meta',
            224: 'meta'
        },

        /**
         * mapping for special characters so they can support
         *
         * this dictionary is only used incase you want to bind a
         * keyup or keydown event to one of these keys
         *
         * @type {Object}
         */
        _KEYCODE_MAP = {
            106: '*',
            107: '+',
            109: '-',
            110: '.',
            111 : '/',
            186: ';',
            187: '=',
            188: ',',
            189: '-',
            190: '.',
            191: '/',
            192: '`',
            219: '[',
            220: '\\',
            221: ']',
            222: '\''
        },

        /**
         * this is a mapping of keys that require shift on a US keypad
         * back to the non shift equivelents
         *
         * this is so you can use keyup events with these keys
         *
         * note that this will only work reliably on US keyboards
         *
         * @type {Object}
         */
        _SHIFT_MAP = {
            '~': '`',
            '!': '1',
            '@': '2',
            '#': '3',
            '$': '4',
            '%': '5',
            '^': '6',
            '&': '7',
            '*': '8',
            '(': '9',
            ')': '0',
            '_': '-',
            '+': '=',
            ':': ';',
            '\"': '\'',
            '<': ',',
            '>': '.',
            '?': '/',
            '|': '\\'
        },

        /**
         * this is a list of special strings you can use to map
         * to modifier keys when you specify your keyboard shortcuts
         *
         * @type {Object}
         */
        _SPECIAL_ALIASES = {
            'option': 'alt',
            'command': 'meta',
            'return': 'enter',
            'escape': 'esc'
        },

        /**
         * variable to store the flipped version of _MAP from above
         * needed to check if we should use keypress or not when no action
         * is specified
         *
         * @type {Object|undefined}
         */
        _REVERSE_MAP,

        /**
         * a list of all the callbacks setup via Mousetrap.bind()
         *
         * @type {Object}
         */
        _callbacks = {},

        /**
         * direct map of string combinations to callbacks used for trigger()
         *
         * @type {Object}
         */
        _direct_map = {},

        /**
         * keeps track of what level each sequence is at since multiple
         * sequences can start out with the same sequence
         *
         * @type {Object}
         */
        _sequence_levels = {},

        /**
         * variable to store the setTimeout call
         *
         * @type {null|number}
         */
        _reset_timer,

        /**
         * temporary state where we will ignore the next keyup
         *
         * @type {boolean|string}
         */
        _ignore_next_keyup = false,

        /**
         * are we currently inside of a sequence?
         * type of action ("keyup" or "keydown" or "keypress") or false
         *
         * @type {boolean|string}
         */
        _sequence_type = false;

    /**
     * loop through the f keys, f1 to f19 and add them to the map
     * programatically
     */
    for (var i = 1; i < 20; ++i) {
        _MAP[111 + i] = 'f' + i;
    }

    /**
     * loop through to map numbers on the numeric keypad
     */
    for (i = 0; i <= 9; ++i) {
        _MAP[i + 96] = i;
    }

    /**
     * cross browser add event method
     *
     * @param {Element|HTMLDocument} object
     * @param {string} type
     * @param {Function} callback
     * @returns void
     */
    function _addEvent(object, type, callback) {
        if (object.addEventListener) {
            object.addEventListener(type, callback, false);
            return;
        }

        object.attachEvent('on' + type, callback);
    }

    /**
     * takes the event and returns the key character
     *
     * @param {Event} e
     * @return {string}
     */
    function _characterFromEvent(e) {

        // for keypress events we should return the character as is
        if (e.type == 'keypress') {
            return String.fromCharCode(e.which);
        }

        // for non keypress events the special maps are needed
        if (_MAP[e.which]) {
            return _MAP[e.which];
        }

        if (_KEYCODE_MAP[e.which]) {
            return _KEYCODE_MAP[e.which];
        }

        // if it is not in the special map
        return String.fromCharCode(e.which).toLowerCase();
    }

    /**
     * checks if two arrays are equal
     *
     * @param {Array} modifiers1
     * @param {Array} modifiers2
     * @returns {boolean}
     */
    function _modifiersMatch(modifiers1, modifiers2) {
        return modifiers1.sort().join(',') === modifiers2.sort().join(',');
    }

    /**
     * resets all sequence counters except for the ones passed in
     *
     * @param {Object} do_not_reset
     * @returns void
     */
    function _resetSequences(do_not_reset, max_level) {
        do_not_reset = do_not_reset || {};

        var active_sequences = false,
            key;

        for (key in _sequence_levels) {
            if (do_not_reset[key] && _sequence_levels[key] > max_level) {
                active_sequences = true;
                continue;
            }
            _sequence_levels[key] = 0;
        }

        if (!active_sequences) {
            _sequence_type = false;
        }
    }

    /**
     * finds all callbacks that match based on the keycode, modifiers,
     * and action
     *
     * @param {string} character
     * @param {Array} modifiers
     * @param {Event|Object} e
     * @param {boolean=} remove - should we remove any matches
     * @param {string=} combination
     * @returns {Array}
     */
    function _getMatches(character, modifiers, e, remove, combination) {
        var i,
            callback,
            matches = [],
            action = e.type;

        // if there are no events related to this keycode
        if (!_callbacks[character]) {
            return [];
        }

        // if a modifier key is coming up on its own we should allow it
        if (action == 'keyup' && _isModifier(character)) {
            modifiers = [character];
        }

        // loop through all callbacks for the key that was pressed
        // and see if any of them match
        for (i = 0; i < _callbacks[character].length; ++i) {
            callback = _callbacks[character][i];

            // if this is a sequence but it is not at the right level
            // then move onto the next match
            if (callback.seq && _sequence_levels[callback.seq] != callback.level) {
                continue;
            }

            // if the action we are looking for doesn't match the action we got
            // then we should keep going
            if (action != callback.action) {
                continue;
            }

            // if this is a keypress event and the meta key and control key
            // are not pressed that means that we need to only look at the
            // character, otherwise check the modifiers as well
            //
            // chrome will not fire a keypress if meta or control is down
            // safari will fire a keypress if meta or meta+shift is down
            // firefox will fire a keypress if meta or control is down
            if ((action == 'keypress' && !e.metaKey && !e.ctrlKey) || _modifiersMatch(modifiers, callback.modifiers)) {

                // remove is used so if you change your mind and call bind a
                // second time with a new function the first one is overwritten
                if (remove && callback.combo == combination) {
                    _callbacks[character].splice(i, 1);
                }

                matches.push(callback);
            }
        }

        return matches;
    }

    /**
     * takes a key event and figures out what the modifiers are
     *
     * @param {Event} e
     * @returns {Array}
     */
    function _eventModifiers(e) {
        var modifiers = [];

        if (e.shiftKey) {
            modifiers.push('shift');
        }

        if (e.altKey) {
            modifiers.push('alt');
        }

        if (e.ctrlKey) {
            modifiers.push('ctrl');
        }

        if (e.metaKey) {
            modifiers.push('meta');
        }

        return modifiers;
    }

    /**
     * actually calls the callback function
     *
     * if your callback function returns false this will use the jquery
     * convention - prevent default and stop propogation on the event
     *
     * @param {Function} callback
     * @param {Event} e
     * @returns void
     */
    function _fireCallback(callback, e, combo) {

        // if this event should not happen stop here
        if (Mousetrap.stopCallback(e, e.target || e.srcElement, combo)) {
            return;
        }

        if (callback(e, combo) === false) {
            if (e.preventDefault) {
                e.preventDefault();
            }

            if (e.stopPropagation) {
                e.stopPropagation();
            }

            e.returnValue = false;
            e.cancelBubble = true;
        }
    }

    /**
     * handles a character key event
     *
     * @param {string} character
     * @param {Event} e
     * @returns void
     */
    function _handleCharacter(character, e) {
        var callbacks = _getMatches(character, _eventModifiers(e), e),
            i,
            do_not_reset = {},
            max_level = 0,
            processed_sequence_callback = false;

        // loop through matching callbacks for this key event
        for (i = 0; i < callbacks.length; ++i) {

            // fire for all sequence callbacks
            // this is because if for example you have multiple sequences
            // bound such as "g i" and "g t" they both need to fire the
            // callback for matching g cause otherwise you can only ever
            // match the first one
            if (callbacks[i].seq) {
                processed_sequence_callback = true;

                // as we loop through keep track of the max
                // any sequence at a lower level will be discarded
                max_level = Math.max(max_level, callbacks[i].level);

                // keep a list of which sequences were matches for later
                do_not_reset[callbacks[i].seq] = 1;
                _fireCallback(callbacks[i].callback, e, callbacks[i].combo);
                continue;
            }

            // if there were no sequence matches but we are still here
            // that means this is a regular match so we should fire that
            if (!processed_sequence_callback && !_sequence_type) {
                _fireCallback(callbacks[i].callback, e, callbacks[i].combo);
            }
        }

        // if you are inside of a sequence and the key you are pressing
        // is not a modifier key then we should reset all sequences
        // that were not matched by this key event
        if (e.type == _sequence_type && !_isModifier(character)) {
            _resetSequences(do_not_reset, max_level);
        }
    }

    /**
     * handles a keydown event
     *
     * @param {Event} e
     * @returns void
     */
    function _handleKey(e) {

        // normalize e.which for key events
        // @see http://stackoverflow.com/questions/4285627/javascript-keycode-vs-charcode-utter-confusion
        if (typeof e.which !== 'number') {
            e.which = e.keyCode;
        }

        var character = _characterFromEvent(e);

        // no character found then stop
        if (!character) {
            return;
        }

        if (e.type == 'keyup' && _ignore_next_keyup == character) {
            _ignore_next_keyup = false;
            return;
        }

        _handleCharacter(character, e);
    }

    /**
     * determines if the keycode specified is a modifier key or not
     *
     * @param {string} key
     * @returns {boolean}
     */
    function _isModifier(key) {
        return key == 'shift' || key == 'ctrl' || key == 'alt' || key == 'meta';
    }

    /**
     * called to set a 1 second timeout on the specified sequence
     *
     * this is so after each key press in the sequence you have 1 second
     * to press the next key before you have to start over
     *
     * @returns void
     */
    function _resetSequenceTimer() {
        clearTimeout(_reset_timer);
        _reset_timer = setTimeout(_resetSequences, 1000);
    }

    /**
     * reverses the map lookup so that we can look for specific keys
     * to see what can and can't use keypress
     *
     * @return {Object}
     */
    function _getReverseMap() {
        if (!_REVERSE_MAP) {
            _REVERSE_MAP = {};
            for (var key in _MAP) {

                // pull out the numeric keypad from here cause keypress should
                // be able to detect the keys from the character
                if (key > 95 && key < 112) {
                    continue;
                }

                if (_MAP.hasOwnProperty(key)) {
                    _REVERSE_MAP[_MAP[key]] = key;
                }
            }
        }
        return _REVERSE_MAP;
    }

    /**
     * picks the best action based on the key combination
     *
     * @param {string} key - character for key
     * @param {Array} modifiers
     * @param {string=} action passed in
     */
    function _pickBestAction(key, modifiers, action) {

        // if no action was picked in we should try to pick the one
        // that we think would work best for this key
        if (!action) {
            action = _getReverseMap()[key] ? 'keydown' : 'keypress';
        }

        // modifier keys don't work as expected with keypress,
        // switch to keydown
        if (action == 'keypress' && modifiers.length) {
            action = 'keydown';
        }

        return action;
    }

    /**
     * binds a key sequence to an event
     *
     * @param {string} combo - combo specified in bind call
     * @param {Array} keys
     * @param {Function} callback
     * @param {string=} action
     * @returns void
     */
    function _bindSequence(combo, keys, callback, action) {

        // start off by adding a sequence level record for this combination
        // and setting the level to 0
        _sequence_levels[combo] = 0;

        // if there is no action pick the best one for the first key
        // in the sequence
        if (!action) {
            action = _pickBestAction(keys[0], []);
        }

        /**
         * callback to increase the sequence level for this sequence and reset
         * all other sequences that were active
         *
         * @param {Event} e
         * @returns void
         */
        var _increaseSequence = function(e) {
                _sequence_type = action;
                ++_sequence_levels[combo];
                _resetSequenceTimer();
            },

            /**
             * wraps the specified callback inside of another function in order
             * to reset all sequence counters as soon as this sequence is done
             *
             * @param {Event} e
             * @returns void
             */
            _callbackAndReset = function(e) {
                _fireCallback(callback, e, combo);

                // we should ignore the next key up if the action is key down
                // or keypress.  this is so if you finish a sequence and
                // release the key the final key will not trigger a keyup
                if (action !== 'keyup') {
                    _ignore_next_keyup = _characterFromEvent(e);
                }

                // weird race condition if a sequence ends with the key
                // another sequence begins with
                setTimeout(_resetSequences, 10);
            },
            i;

        // loop through keys one at a time and bind the appropriate callback
        // function.  for any key leading up to the final one it should
        // increase the sequence. after the final, it should reset all sequences
        for (i = 0; i < keys.length; ++i) {
            _bindSingle(keys[i], i < keys.length - 1 ? _increaseSequence : _callbackAndReset, action, combo, i);
        }
    }

    /**
     * binds a single keyboard combination
     *
     * @param {string} combination
     * @param {Function} callback
     * @param {string=} action
     * @param {string=} sequence_name - name of sequence if part of sequence
     * @param {number=} level - what part of the sequence the command is
     * @returns void
     */
    function _bindSingle(combination, callback, action, sequence_name, level) {

        // make sure multiple spaces in a row become a single space
        combination = combination.replace(/\s+/g, ' ');

        var sequence = combination.split(' '),
            i,
            key,
            keys,
            modifiers = [];

        // if this pattern is a sequence of keys then run through this method
        // to reprocess each pattern one key at a time
        if (sequence.length > 1) {
            _bindSequence(combination, sequence, callback, action);
            return;
        }

        // take the keys from this pattern and figure out what the actual
        // pattern is all about
        keys = combination === '+' ? ['+'] : combination.split('+');

        for (i = 0; i < keys.length; ++i) {
            key = keys[i];

            // normalize key names
            if (_SPECIAL_ALIASES[key]) {
                key = _SPECIAL_ALIASES[key];
            }

            // if this is not a keypress event then we should
            // be smart about using shift keys
            // this will only work for US keyboards however
            if (action && action != 'keypress' && _SHIFT_MAP[key]) {
                key = _SHIFT_MAP[key];
                modifiers.push('shift');
            }

            // if this key is a modifier then add it to the list of modifiers
            if (_isModifier(key)) {
                modifiers.push(key);
            }
        }

        // depending on what the key combination is
        // we will try to pick the best event for it
        action = _pickBestAction(key, modifiers, action);

        // make sure to initialize array if this is the first time
        // a callback is added for this key
        if (!_callbacks[key]) {
            _callbacks[key] = [];
        }

        // remove an existing match if there is one
        _getMatches(key, modifiers, {type: action}, !sequence_name, combination);

        // add this call back to the array
        // if it is a sequence put it at the beginning
        // if not put it at the end
        //
        // this is important because the way these are processed expects
        // the sequence ones to come first
        _callbacks[key][sequence_name ? 'unshift' : 'push']({
            callback: callback,
            modifiers: modifiers,
            action: action,
            seq: sequence_name,
            level: level,
            combo: combination
        });
    }

    /**
     * binds multiple combinations to the same callback
     *
     * @param {Array} combinations
     * @param {Function} callback
     * @param {string|undefined} action
     * @returns void
     */
    function _bindMultiple(combinations, callback, action) {
        for (var i = 0; i < combinations.length; ++i) {
            _bindSingle(combinations[i], callback, action);
        }
    }

    // start!
    _addEvent(document, 'keypress', _handleKey);
    _addEvent(document, 'keydown', _handleKey);
    _addEvent(document, 'keyup', _handleKey);

    var Mousetrap = {

        /**
         * binds an event to mousetrap
         *
         * can be a single key, a combination of keys separated with +,
         * an array of keys, or a sequence of keys separated by spaces
         *
         * be sure to list the modifier keys first to make sure that the
         * correct key ends up getting bound (the last key in the pattern)
         *
         * @param {string|Array} keys
         * @param {Function} callback
         * @param {string=} action - 'keypress', 'keydown', or 'keyup'
         * @returns void
         */
        bind: function(keys, callback, action) {
            _bindMultiple(keys instanceof Array ? keys : [keys], callback, action);
            _direct_map[keys + ':' + action] = callback;
            return this;
        },

        /**
         * unbinds an event to mousetrap
         *
         * the unbinding sets the callback function of the specified key combo
         * to an empty function and deletes the corresponding key in the
         * _direct_map dict.
         *
         * the keycombo+action has to be exactly the same as
         * it was defined in the bind method
         *
         * TODO: actually remove this from the _callbacks dictionary instead
         * of binding an empty function
         *
         * @param {string|Array} keys
         * @param {string} action
         * @returns void
         */
        unbind: function(keys, action) {
            if (_direct_map[keys + ':' + action]) {
                delete _direct_map[keys + ':' + action];
                this.bind(keys, function() {}, action);
            }
            return this;
        },

        /**
         * triggers an event that has already been bound
         *
         * @param {string} keys
         * @param {string=} action
         * @returns void
         */
        trigger: function(keys, action) {
            _direct_map[keys + ':' + action]();
            return this;
        },

        /**
         * resets the library back to its initial state.  this is useful
         * if you want to clear out the current keyboard shortcuts and bind
         * new ones - for example if you switch to another page
         *
         * @returns void
         */
        reset: function() {
            _callbacks = {};
            _direct_map = {};
            return this;
        },

       /**
        * should we stop this event before firing off callbacks
        *
        * @param {Event} e
        * @param {Element} element
        * @return {boolean}
        */
        stopCallback: function(e, element, combo) {

            // if the element has the class "mousetrap" then no need to stop
            if ((' ' + element.className + ' ').indexOf(' mousetrap ') > -1) {
                return false;
            }

            // stop for input, select, and textarea
            return element.tagName == 'INPUT' || element.tagName == 'SELECT' || element.tagName == 'TEXTAREA' || (element.contentEditable && element.contentEditable == 'true');
        }
    };

    // expose mousetrap to the global object
    window.Mousetrap = Mousetrap;

    // expose mousetrap as an AMD module
    if (typeof define === 'function' && define.amd) {
        define(Mousetrap);
    }
}) ();

(function() { var h,aa=aa||{},k=this,ba=function(){},ca=function(a){var b=typeof a;if("object"==b)if(a){if(a instanceof Array)return"array";if(a instanceof Object)return b;var c=Object.prototype.toString.call(a);if("[object Window]"==c)return"object";if("[object Array]"==c||"number"==typeof a.length&&"undefined"!=typeof a.splice&&"undefined"!=typeof a.propertyIsEnumerable&&!a.propertyIsEnumerable("splice"))return"array";if("[object Function]"==c||"undefined"!=typeof a.call&&"undefined"!=typeof a.propertyIsEnumerable&&
!a.propertyIsEnumerable("call"))return"function"}else return"null";else if("function"==b&&"undefined"==typeof a.call)return"object";return b},m=function(a){return"array"==ca(a)},da=function(a){var b=ca(a);return"array"==b||"object"==b&&"number"==typeof a.length},n=function(a){return"string"==typeof a},p=function(a){return"function"==ca(a)},ea=function(a){var b=typeof a;return"object"==b&&null!=a||"function"==b},fa=function(a,b,c){return a.call.apply(a.bind,arguments)},ga=function(a,b,c){if(!a)throw Error();
if(2<arguments.length){var d=Array.prototype.slice.call(arguments,2);return function(){var c=Array.prototype.slice.call(arguments);Array.prototype.unshift.apply(c,d);return a.apply(b,c)}}return function(){return a.apply(b,arguments)}},q=function(a,b,c){q=Function.prototype.bind&&-1!=Function.prototype.bind.toString().indexOf("native code")?fa:ga;return q.apply(null,arguments)},ha=function(a,b){var c=Array.prototype.slice.call(arguments,1);return function(){var b=c.slice();b.push.apply(b,arguments);
return a.apply(this,b)}},r=Date.now||function(){return+new Date},s=function(a,b){var c=a.split("."),d=k;c[0]in d||!d.execScript||d.execScript("var "+c[0]);for(var e;c.length&&(e=c.shift());)c.length||void 0===b?d=d[e]?d[e]:d[e]={}:d[e]=b},t=function(a,b){function c(){}c.prototype=b.prototype;a.J=b.prototype;a.prototype=new c;a.ic=function(a,c,f){return b.prototype[c].apply(a,Array.prototype.slice.call(arguments,2))}};
Function.prototype.bind=Function.prototype.bind||function(a,b){if(1<arguments.length){var c=Array.prototype.slice.call(arguments,1);c.unshift(this,a);return q.apply(null,c)}return q(this,a)};var u=function(a){if(Error.captureStackTrace)Error.captureStackTrace(this,u);else{var b=Error().stack;b&&(this.stack=b)}a&&(this.message=String(a))};t(u,Error);u.prototype.name="CustomError";var ia=function(a,b){return a<b?-1:a>b?1:0};var v=Array.prototype,ja=v.indexOf?function(a,b,c){return v.indexOf.call(a,b,c)}:function(a,b,c){c=null==c?0:0>c?Math.max(0,a.length+c):c;if(n(a))return n(b)&&1==b.length?a.indexOf(b,c):-1;for(;c<a.length;c++)if(c in a&&a[c]===b)return c;return-1},ka=v.forEach?function(a,b,c){v.forEach.call(a,b,c)}:function(a,b,c){for(var d=a.length,e=n(a)?a.split(""):a,f=0;f<d;f++)f in e&&b.call(c,e[f],f,a)},la=v.some?function(a,b,c){return v.some.call(a,b,c)}:function(a,b,c){for(var d=a.length,e=n(a)?a.split(""):
a,f=0;f<d;f++)if(f in e&&b.call(c,e[f],f,a))return!0;return!1},na=function(a){var b;t:{b=ma;for(var c=a.length,d=n(a)?a.split(""):a,e=0;e<c;e++)if(e in d&&b.call(void 0,d[e],e,a)){b=e;break t}b=-1}return 0>b?null:n(a)?a.charAt(b):a[b]},oa=function(a,b){var c=ja(a,b),d;(d=0<=c)&&v.splice.call(a,c,1);return d},pa=function(a){return v.concat.apply(v,arguments)},qa=function(a){var b=a.length;if(0<b){for(var c=Array(b),d=0;d<b;d++)c[d]=a[d];return c}return[]};var ra="StopIteration"in k?k.StopIteration:Error("StopIteration"),sa=function(){};sa.prototype.next=function(){throw ra;};sa.prototype.Tb=function(){return this};var ta=function(a,b,c){for(var d in a)b.call(c,a[d],d,a)},ua=function(a){var b=[],c=0,d;for(d in a)b[c++]=a[d];return b},va=function(a){var b=[],c=0,d;for(d in a)b[c++]=d;return b},wa=function(a,b){var c;t:{for(c in a)if(b.call(void 0,a[c],c,a))break t;c=void 0}return c&&a[c]},xa="constructor hasOwnProperty isPrototypeOf propertyIsEnumerable toLocaleString toString valueOf".split(" "),ya=function(a,b){for(var c,d,e=1;e<arguments.length;e++){d=arguments[e];for(c in d)a[c]=d[c];for(var f=0;f<xa.length;f++)c=
xa[f],Object.prototype.hasOwnProperty.call(d,c)&&(a[c]=d[c])}};var w=function(a,b){this.o={};this.b=[];this.ea=this.g=0;var c=arguments.length;if(1<c){if(c%2)throw Error("Uneven number of arguments");for(var d=0;d<c;d+=2)this.set(arguments[d],arguments[d+1])}else a&&this.Aa(a)};w.prototype.q=function(){za(this);for(var a=[],b=0;b<this.b.length;b++)a.push(this.o[this.b[b]]);return a};w.prototype.C=function(){za(this);return this.b.concat()};w.prototype.Q=function(a){return x(this.o,a)};
w.prototype.remove=function(a){return x(this.o,a)?(delete this.o[a],this.g--,this.ea++,this.b.length>2*this.g&&za(this),!0):!1};var za=function(a){if(a.g!=a.b.length){for(var b=0,c=0;b<a.b.length;){var d=a.b[b];x(a.o,d)&&(a.b[c++]=d);b++}a.b.length=c}if(a.g!=a.b.length){for(var e={},c=b=0;b<a.b.length;)d=a.b[b],x(e,d)||(a.b[c++]=d,e[d]=1),b++;a.b.length=c}};h=w.prototype;h.get=function(a,b){return x(this.o,a)?this.o[a]:b};
h.set=function(a,b){x(this.o,a)||(this.g++,this.b.push(a),this.ea++);this.o[a]=b};h.Aa=function(a){var b;a instanceof w?(b=a.C(),a=a.q()):(b=va(a),a=ua(a));for(var c=0;c<b.length;c++)this.set(b[c],a[c])};h.B=function(){return new w(this)};h.jb=function(){za(this);for(var a={},b=0;b<this.b.length;b++){var c=this.b[b];a[c]=this.o[c]}return a};
h.Tb=function(a){za(this);var b=0,c=this.b,d=this.o,e=this.ea,f=this,g=new sa;g.next=function(){for(;;){if(e!=f.ea)throw Error("The map has changed since the iterator was created");if(b>=c.length)throw ra;var g=c[b++];return a?g:d[g]}};return g};var x=function(a,b){return Object.prototype.hasOwnProperty.call(a,b)};var Aa={id:"hitType",name:"t",valueType:"text",maxLength:void 0,defaultValue:void 0},Ba={id:"sessionControl",name:"sc",valueType:"text",maxLength:void 0,defaultValue:void 0},Ca={id:"description",name:"cd",valueType:"text",maxLength:2048,defaultValue:void 0},Da={Jc:Aa,jc:{id:"anonymizeIp",name:"aip",valueType:"boolean",maxLength:void 0,defaultValue:void 0},Vc:{id:"queueTime",name:"qt",valueType:"integer",maxLength:void 0,defaultValue:void 0},pc:{id:"cacheBuster",name:"z",valueType:"text",maxLength:void 0,
defaultValue:void 0},$c:Ba,Sc:{id:"nonInteraction",name:"ni",valueType:"boolean",maxLength:void 0,defaultValue:void 0},zc:Ca,jd:{id:"title",name:"dt",valueType:"text",maxLength:1500,defaultValue:void 0},Ac:{id:"dimension",name:"cd[1-9][0-9]*",valueType:"text",maxLength:150,defaultValue:void 0},Rc:{id:"metric",name:"cm[1-9][0-9]*",valueType:"integer",maxLength:void 0,defaultValue:void 0},lc:{id:"appId",name:"aid",valueType:"text",maxLength:150,defaultValue:void 0},mc:{id:"appInstallerId",name:"aiid",
valueType:"text",maxLength:150,defaultValue:void 0},Dc:{id:"eventCategory",name:"ec",valueType:"text",maxLength:150,defaultValue:void 0},Cc:{id:"eventAction",name:"ea",valueType:"text",maxLength:500,defaultValue:void 0},Ec:{id:"eventLabel",name:"el",valueType:"text",maxLength:500,defaultValue:void 0},Fc:{id:"eventValue",name:"ev",valueType:"integer",maxLength:void 0,defaultValue:void 0},bd:{id:"socialNetwork",name:"sn",valueType:"text",maxLength:50,defaultValue:void 0},ad:{id:"socialAction",name:"sa",
valueType:"text",maxLength:50,defaultValue:void 0},cd:{id:"socialTarget",name:"st",valueType:"text",maxLength:2048,defaultValue:void 0},md:{id:"transactionId",name:"ti",valueType:"text",maxLength:500,defaultValue:void 0},ld:{id:"transactionAffiliation",name:"ta",valueType:"text",maxLength:500,defaultValue:void 0},nd:{id:"transactionRevenue",name:"tr",valueType:"currency",maxLength:void 0,defaultValue:void 0},od:{id:"transactionShipping",name:"ts",valueType:"currency",maxLength:void 0,defaultValue:void 0},
pd:{id:"transactionTax",name:"tt",valueType:"currency",maxLength:void 0,defaultValue:void 0},xc:{id:"currencyCode",name:"cu",valueType:"text",maxLength:10,defaultValue:void 0},Nc:{id:"itemPrice",name:"ip",valueType:"currency",maxLength:void 0,defaultValue:void 0},Oc:{id:"itemQuantity",name:"iq",valueType:"integer",maxLength:void 0,defaultValue:void 0},Lc:{id:"itemCode",name:"ic",valueType:"text",maxLength:500,defaultValue:void 0},Mc:{id:"itemName",name:"in",valueType:"text",maxLength:500,defaultValue:void 0},
Kc:{id:"itemCategory",name:"iv",valueType:"text",maxLength:500,defaultValue:void 0},vc:{id:"campaignSource",name:"cs",valueType:"text",maxLength:100,defaultValue:void 0},tc:{id:"campaignMedium",name:"cm",valueType:"text",maxLength:50,defaultValue:void 0},uc:{id:"campaignName",name:"cn",valueType:"text",maxLength:100,defaultValue:void 0},sc:{id:"campaignKeyword",name:"ck",valueType:"text",maxLength:500,defaultValue:void 0},qc:{id:"campaignContent",name:"cc",valueType:"text",maxLength:500,defaultValue:void 0},
rc:{id:"campaignId",name:"ci",valueType:"text",maxLength:100,defaultValue:void 0},Ic:{id:"gclid",name:"gclid",valueType:"text",maxLength:void 0,defaultValue:void 0},yc:{id:"dclid",name:"dclid",valueType:"text",maxLength:void 0,defaultValue:void 0},Uc:{id:"pageLoadTime",name:"plt",valueType:"integer",maxLength:void 0,defaultValue:void 0},Bc:{id:"dnsTime",name:"dns",valueType:"integer",maxLength:void 0,defaultValue:void 0},dd:{id:"tcpConnectTime",name:"tcp",valueType:"integer",maxLength:void 0,defaultValue:void 0},
Zc:{id:"serverResponseTime",name:"srt",valueType:"integer",maxLength:void 0,defaultValue:void 0},Tc:{id:"pageDownloadTime",name:"pdt",valueType:"integer",maxLength:void 0,defaultValue:void 0},Wc:{id:"redirectResponseTime",name:"rrt",valueType:"integer",maxLength:void 0,defaultValue:void 0},ed:{id:"timingCategory",name:"utc",valueType:"text",maxLength:150,defaultValue:void 0},hd:{id:"timingVar",name:"utv",valueType:"text",maxLength:500,defaultValue:void 0},gd:{id:"timingValue",name:"utt",valueType:"integer",
maxLength:void 0,defaultValue:void 0},fd:{id:"timingLabel",name:"utl",valueType:"text",maxLength:500,defaultValue:void 0},Gc:{id:"exDescription",name:"exd",valueType:"text",maxLength:150,defaultValue:void 0},Hc:{id:"exFatal",name:"exf",valueType:"boolean",maxLength:void 0,defaultValue:"1"}};var Ea=function(a,b){this.width=a;this.height=b};Ea.prototype.B=function(){return new Ea(this.width,this.height)};Ea.prototype.floor=function(){this.width=Math.floor(this.width);this.height=Math.floor(this.height);return this};var y,Fa,Ga,Ha,Ia=function(){return k.navigator?k.navigator.userAgent:null};Ha=Ga=Fa=y=!1;var Ja;if(Ja=Ia()){var Ka=k.navigator;y=0==Ja.lastIndexOf("Opera",0);Fa=!y&&(-1!=Ja.indexOf("MSIE")||-1!=Ja.indexOf("Trident"));Ga=!y&&-1!=Ja.indexOf("WebKit");Ha=!y&&!Ga&&!Fa&&"Gecko"==Ka.product}var La=y,z=Fa,A=Ha,B=Ga,Ma=function(){var a=k.document;return a?a.documentMode:void 0},Na;
t:{var Oa="",Pa;if(La&&k.opera)var Qa=k.opera.version,Oa="function"==typeof Qa?Qa():Qa;else if(A?Pa=/rv\:([^\);]+)(\)|;)/:z?Pa=/\b(?:MSIE|rv)[: ]([^\);]+)(\)|;)/:B&&(Pa=/WebKit\/(\S+)/),Pa)var Ra=Pa.exec(Ia()),Oa=Ra?Ra[1]:"";if(z){var Sa=Ma();if(Sa>parseFloat(Oa)){Na=String(Sa);break t}}Na=Oa}
var Ta=Na,Ua={},C=function(a){var b;if(!(b=Ua[a])){b=0;for(var c=String(Ta).replace(/^[\s\xa0]+|[\s\xa0]+$/g,"").split("."),d=String(a).replace(/^[\s\xa0]+|[\s\xa0]+$/g,"").split("."),e=Math.max(c.length,d.length),f=0;0==b&&f<e;f++){var g=c[f]||"",l=d[f]||"",I=RegExp("(\\d*)(\\D*)","g"),F=RegExp("(\\d*)(\\D*)","g");do{var S=I.exec(g)||["","",""],T=F.exec(l)||["","",""];if(0==S[0].length&&0==T[0].length)break;b=ia(0==S[1].length?0:parseInt(S[1],10),0==T[1].length?0:parseInt(T[1],10))||ia(0==S[2].length,
0==T[2].length)||ia(S[2],T[2])}while(0==b)}b=Ua[a]=0<=b}return b},Va=k.document,Wa=Va&&z?Ma()||("CSS1Compat"==Va.compatMode?parseInt(Ta,10):5):void 0;!A&&!z||z&&z&&9<=Wa||A&&C("1.9.1");z&&C("9");var Xa=function(){};Xa.prototype.Ga=!1;Xa.prototype.na=function(){this.Ga||(this.Ga=!0,this.j())};Xa.prototype.j=function(){if(this.ob)for(;this.ob.length;)this.ob.shift()()};var D=function(a,b){this.type=a;this.currentTarget=this.target=b;this.defaultPrevented=this.N=!1;this.Ra=!0};D.prototype.j=function(){};D.prototype.na=function(){};D.prototype.preventDefault=function(){this.defaultPrevented=!0;this.Ra=!1};var Ya=function(a){Ya[" "](a);return a};Ya[" "]=ba;var Za=!z||z&&9<=Wa,$a=z&&!C("9"),ab=!B||C("528"),bb=A&&C("1.9b")||z&&C("8")||La&&C("9.5")||B&&C("528"),cb=A&&!C("8")||z&&!C("9");var E=function(a,b){D.call(this,a?a.type:"");this.relatedTarget=this.currentTarget=this.target=null;this.charCode=this.keyCode=this.button=this.screenY=this.screenX=this.clientY=this.clientX=this.offsetY=this.offsetX=0;this.metaKey=this.shiftKey=this.altKey=this.ctrlKey=!1;this.ib=this.state=null;if(a){var c=this.type=a.type;this.target=a.target||a.srcElement;this.currentTarget=b;var d=a.relatedTarget;if(d){if(A){var e;t:{try{Ya(d.nodeName);e=!0;break t}catch(f){}e=!1}e||(d=null)}}else"mouseover"==
c?d=a.fromElement:"mouseout"==c&&(d=a.toElement);this.relatedTarget=d;this.offsetX=B||void 0!==a.offsetX?a.offsetX:a.layerX;this.offsetY=B||void 0!==a.offsetY?a.offsetY:a.layerY;this.clientX=void 0!==a.clientX?a.clientX:a.pageX;this.clientY=void 0!==a.clientY?a.clientY:a.pageY;this.screenX=a.screenX||0;this.screenY=a.screenY||0;this.button=a.button;this.keyCode=a.keyCode||0;this.charCode=a.charCode||("keypress"==c?a.keyCode:0);this.ctrlKey=a.ctrlKey;this.altKey=a.altKey;this.shiftKey=a.shiftKey;this.metaKey=
a.metaKey;this.state=a.state;this.ib=a;a.defaultPrevented&&this.preventDefault()}};t(E,D);E.prototype.preventDefault=function(){E.J.preventDefault.call(this);var a=this.ib;if(a.preventDefault)a.preventDefault();else if(a.returnValue=!1,$a)try{if(a.ctrlKey||112<=a.keyCode&&123>=a.keyCode)a.keyCode=-1}catch(b){}};E.prototype.j=function(){};var db="closure_listenable_"+(1E6*Math.random()|0),eb=function(a){try{return!(!a||!a[db])}catch(b){return!1}},fb=0;var gb=function(a,b,c,d,e){this.H=a;this.proxy=null;this.src=b;this.type=c;this.fa=!!d;this.ha=e;this.key=++fb;this.removed=this.ga=!1},hb=function(a){a.removed=!0;a.H=null;a.proxy=null;a.src=null;a.ha=null};var G=function(a){this.src=a;this.h={};this.S=0};G.prototype.add=function(a,b,c,d,e){var f=a.toString();a=this.h[f];a||(a=this.h[f]=[],this.S++);var g=ib(a,b,d,e);-1<g?(b=a[g],c||(b.ga=!1)):(b=new gb(b,this.src,f,!!d,e),b.ga=c,a.push(b));return b};G.prototype.remove=function(a,b,c,d){a=a.toString();if(!(a in this.h))return!1;var e=this.h[a];b=ib(e,b,c,d);return-1<b?(hb(e[b]),v.splice.call(e,b,1),0==e.length&&(delete this.h[a],this.S--),!0):!1};
var jb=function(a,b){var c=b.type;if(!(c in a.h))return!1;var d=oa(a.h[c],b);d&&(hb(b),0==a.h[c].length&&(delete a.h[c],a.S--));return d};G.prototype.removeAll=function(a){a=a&&a.toString();var b=0,c;for(c in this.h)if(!a||c==a){for(var d=this.h[c],e=0;e<d.length;e++)++b,hb(d[e]);delete this.h[c];this.S--}return b};G.prototype.R=function(a,b,c,d){a=this.h[a.toString()];var e=-1;a&&(e=ib(a,b,c,d));return-1<e?a[e]:null};
var ib=function(a,b,c,d){for(var e=0;e<a.length;++e){var f=a[e];if(!f.removed&&f.H==b&&f.fa==!!c&&f.ha==d)return e}return-1};var kb="closure_lm_"+(1E6*Math.random()|0),lb={},mb=0,nb=function(a,b,c,d,e){if(m(b)){for(var f=0;f<b.length;f++)nb(a,b[f],c,d,e);return null}c=ob(c);return eb(a)?a.listen(b,c,d,e):pb(a,b,c,!1,d,e)},pb=function(a,b,c,d,e,f){if(!b)throw Error("Invalid event type");var g=!!e,l=qb(a);l||(a[kb]=l=new G(a));c=l.add(b,c,d,e,f);if(c.proxy)return c;d=rb();c.proxy=d;d.src=a;d.H=c;a.addEventListener?a.addEventListener(b.toString(),d,g):a.attachEvent(sb(b.toString()),d);mb++;return c},rb=function(){var a=tb,
b=Za?function(c){return a.call(b.src,b.H,c)}:function(c){c=a.call(b.src,b.H,c);if(!c)return c};return b},ub=function(a,b,c,d,e){if(m(b)){for(var f=0;f<b.length;f++)ub(a,b[f],c,d,e);return null}c=ob(c);return eb(a)?a.Ja(b,c,d,e):pb(a,b,c,!0,d,e)},vb=function(a,b,c,d,e){if(m(b))for(var f=0;f<b.length;f++)vb(a,b[f],c,d,e);else c=ob(c),eb(a)?a.Fa(b,c,d,e):a&&(a=qb(a))&&(b=a.R(b,c,!!d,e))&&wb(b)},wb=function(a){if("number"==typeof a||!a||a.removed)return!1;var b=a.src;if(eb(b))return jb(b.t,a);var c=a.type,
d=a.proxy;b.removeEventListener?b.removeEventListener(c,d,a.fa):b.detachEvent&&b.detachEvent(sb(c),d);mb--;(c=qb(b))?(jb(c,a),0==c.S&&(c.src=null,b[kb]=null)):hb(a);return!0},sb=function(a){return a in lb?lb[a]:lb[a]="on"+a},yb=function(a,b,c,d){var e=1;if(a=qb(a))if(b=a.h[b.toString()])for(b=qa(b),a=0;a<b.length;a++){var f=b[a];f&&f.fa==c&&!f.removed&&(e&=!1!==xb(f,d))}return Boolean(e)},xb=function(a,b){var c=a.H,d=a.ha||a.src;a.ga&&wb(a);return c.call(d,b)},tb=function(a,b){if(a.removed)return!0;
if(!Za){var c;if(!(c=b))t:{c=["window","event"];for(var d=k,e;e=c.shift();)if(null!=d[e])d=d[e];else{c=null;break t}c=d}e=c;c=new E(e,this);d=!0;if(!(0>e.keyCode||void 0!=e.returnValue)){t:{var f=!1;if(0==e.keyCode)try{e.keyCode=-1;break t}catch(g){f=!0}if(f||void 0==e.returnValue)e.returnValue=!0}e=[];for(f=c.currentTarget;f;f=f.parentNode)e.push(f);for(var f=a.type,l=e.length-1;!c.N&&0<=l;l--)c.currentTarget=e[l],d&=yb(e[l],f,!0,c);for(l=0;!c.N&&l<e.length;l++)c.currentTarget=e[l],d&=yb(e[l],f,
!1,c)}return d}return xb(a,new E(b,this))},qb=function(a){a=a[kb];return a instanceof G?a:null},zb="__closure_events_fn_"+(1E9*Math.random()>>>0),ob=function(a){return p(a)?a:a[zb]||(a[zb]=function(b){return a.handleEvent(b)})};var H=function(){this.t=new G(this);this.Cb=this};t(H,Xa);H.prototype[db]=!0;h=H.prototype;h.Da=null;h.addEventListener=function(a,b,c,d){nb(this,a,b,c,d)};h.removeEventListener=function(a,b,c,d){vb(this,a,b,c,d)};
h.dispatchEvent=function(a){var b,c=this.Da;if(c){b=[];for(var d=1;c;c=c.Da)b.push(c),++d}c=this.Cb;d=a.type||a;if(n(a))a=new D(a,c);else if(a instanceof D)a.target=a.target||c;else{var e=a;a=new D(d,c);ya(a,e)}var e=!0,f;if(b)for(var g=b.length-1;!a.N&&0<=g;g--)f=a.currentTarget=b[g],e=Ab(f,d,!0,a)&&e;a.N||(f=a.currentTarget=c,e=Ab(f,d,!0,a)&&e,a.N||(e=Ab(f,d,!1,a)&&e));if(b)for(g=0;!a.N&&g<b.length;g++)f=a.currentTarget=b[g],e=Ab(f,d,!1,a)&&e;return e};
h.j=function(){H.J.j.call(this);this.t&&this.t.removeAll(void 0);this.Da=null};h.listen=function(a,b,c,d){return this.t.add(String(a),b,!1,c,d)};h.Ja=function(a,b,c,d){return this.t.add(String(a),b,!0,c,d)};h.Fa=function(a,b,c,d){return this.t.remove(String(a),b,c,d)};var Ab=function(a,b,c,d){b=a.t.h[String(b)];if(!b)return!0;b=qa(b);for(var e=!0,f=0;f<b.length;++f){var g=b[f];if(g&&!g.removed&&g.fa==c){var l=g.H,I=g.ha||g.src;g.ga&&jb(a.t,g);e=!1!==l.call(I,d)&&e}}return e&&!1!=d.Ra};
H.prototype.R=function(a,b,c,d){return this.t.R(String(a),b,c,d)};var Bb=function(a,b){H.call(this);this.oa=a||1;this.L=b||k;this.Ba=q(this.Db,this);this.Ca=r()};t(Bb,H);h=Bb.prototype;h.enabled=!1;h.e=null;h.Db=function(){if(this.enabled){var a=r()-this.Ca;0<a&&a<0.8*this.oa?this.e=this.L.setTimeout(this.Ba,this.oa-a):(this.e&&(this.L.clearTimeout(this.e),this.e=null),this.dispatchEvent("tick"),this.enabled&&(this.e=this.L.setTimeout(this.Ba,this.oa),this.Ca=r()))}};h.start=function(){this.enabled=!0;this.e||(this.e=this.L.setTimeout(this.Ba,this.oa),this.Ca=r())};
h.stop=function(){this.enabled=!1;this.e&&(this.L.clearTimeout(this.e),this.e=null)};h.j=function(){Bb.J.j.call(this);this.stop();delete this.L};var Cb=function(a,b,c){if(p(a))c&&(a=q(a,c));else if(a&&"function"==typeof a.handleEvent)a=q(a.handleEvent,a);else throw Error("Invalid listener argument");return 2147483647<b?-1:k.setTimeout(a,b||0)};var J=function(a){this.Ea=a;this.b={}};t(J,Xa);var Db=[];J.prototype.listen=function(a,b,c,d){m(b)||(Db[0]=b,b=Db);for(var e=0;e<b.length;e++){var f=nb(a,b[e],c||this.handleEvent,d||!1,this.Ea||this);if(!f)break;this.b[f.key]=f}return this};J.prototype.Ja=function(a,b,c,d){return Eb(this,a,b,c,d)};var Eb=function(a,b,c,d,e,f){if(m(c))for(var g=0;g<c.length;g++)Eb(a,b,c[g],d,e,f);else{b=ub(b,c,d||a.handleEvent,e,f||a.Ea||a);if(!b)return a;a.b[b.key]=b}return a};
J.prototype.Fa=function(a,b,c,d,e){if(m(b))for(var f=0;f<b.length;f++)this.Fa(a,b[f],c,d,e);else c=c||this.handleEvent,e=e||this.Ea||this,c=ob(c),d=!!d,b=eb(a)?a.R(b,c,d,e):a?(a=qb(a))?a.R(b,c,d,e):null:null,b&&(wb(b),delete this.b[b.key]);return this};J.prototype.removeAll=function(){ta(this.b,wb);this.b={}};J.prototype.j=function(){J.J.j.call(this);this.removeAll()};J.prototype.handleEvent=function(){throw Error("EventHandler.handleEvent not implemented");};var K=function(){H.call(this);this.ia=new J(this);ab&&(bb?this.ia.listen(cb?document.body:window,["online","offline"],this.Wa):(this.Ya=ab?navigator.onLine:!0,this.e=new Bb(250),this.ia.listen(this.e,"tick",this.zb),this.e.start()))};t(K,H);K.prototype.zb=function(){var a=ab?navigator.onLine:!0;a!=this.Ya&&(this.Ya=a,this.Wa())};K.prototype.Wa=function(){this.dispatchEvent((ab?navigator.onLine:1)?"online":"offline")};
K.prototype.j=function(){K.J.j.call(this);this.ia.na();this.ia=null;this.e&&(this.e.na(),this.e=null)};var Hb=function(a){p(k.setImmediate)?k.setImmediate(a):(Fb||(Fb=Gb()),Fb(a))},Fb,Gb=function(){var a=k.MessageChannel;"undefined"===typeof a&&"undefined"!==typeof window&&window.postMessage&&window.addEventListener&&(a=function(){var a=document.createElement("iframe");a.style.display="none";a.src="";document.documentElement.appendChild(a);var b=a.contentWindow,a=b.document;a.open();a.write("");a.close();var c="callImmediate"+Math.random(),d=b.location.protocol+"//"+b.location.host,a=q(function(a){if(a.origin==
d||a.data==c)this.port1.onmessage()},this);b.addEventListener("message",a,!1);this.port1={};this.port2={postMessage:function(){b.postMessage(c,d)}}});if("undefined"!==typeof a){var b=new a,c={},d=c;b.port1.onmessage=function(){c=c.next;var a=c.lb;c.lb=null;a()};return function(a){d.next={lb:a};d=d.next;b.port2.postMessage(0)}}return"undefined"!==typeof document&&"onreadystatechange"in document.createElement("script")?function(a){var b=document.createElement("script");b.onreadystatechange=function(){b.onreadystatechange=
null;b.parentNode.removeChild(b);b=null;a();a=null};document.documentElement.appendChild(b)}:function(a){k.setTimeout(a,0)}};var Ib=function(a){Hb(function(){throw a;})},Nb=function(a,b){Jb||(Hb(Kb),Jb=!0);Lb.push(new Mb(a,b))},Jb=!1,Lb=[],Kb=function(){for(;Lb.length;){var a=Lb;Lb=[];for(var b=0;b<a.length;b++){var c=a[b];try{c.Vb.call(c.scope)}catch(d){Ib(d)}}}Jb=!1},Mb=function(a,b){this.Vb=a;this.scope=b};var Ob=function(a){a.prototype.then=a.prototype.then;a.prototype.$goog_Thenable=!0},Pb=function(a){if(!a)return!1;try{return!!a.$goog_Thenable}catch(b){return!1}};var M=function(a,b){this.k=0;this.r=void 0;this.l=this.m=null;this.ja=this.ya=!1;try{var c=this;a.call(b,function(a){L(c,2,a)},function(a){L(c,3,a)})}catch(d){L(this,3,d)}};M.prototype.then=function(a,b,c){return Qb(this,p(a)?a:null,p(b)?b:null,c)};Ob(M);M.prototype.cancel=function(a){0==this.k&&Nb(function(){var b=new Rb(a);Sb(this,b)},this)};
var Sb=function(a,b){if(0==a.k)if(a.m){var c=a.m;if(c.l){for(var d=0,e=-1,f=0,g;g=c.l[f];f++)if(g=g.la)if(d++,g==a&&(e=f),0<=e&&1<d)break;0<=e&&(0==c.k&&1==d?Sb(c,b):(d=c.l.splice(e,1)[0],Tb(c),d.za(b)))}}else L(a,3,b)},Vb=function(a,b){a.l&&a.l.length||2!=a.k&&3!=a.k||Ub(a);a.l||(a.l=[]);a.l.push(b)},Qb=function(a,b,c,d){var e={la:null,Qa:null,za:null};e.la=new M(function(a,g){e.Qa=b?function(c){try{var e=b.call(d,c);a(e)}catch(F){g(F)}}:a;e.za=c?function(b){try{var e=c.call(d,b);void 0===e&&b instanceof
Rb?g(b):a(e)}catch(F){g(F)}}:g});e.la.m=a;Vb(a,e);return e.la};M.prototype.Za=function(a){this.k=0;L(this,2,a)};M.prototype.$a=function(a){this.k=0;L(this,3,a)};
var L=function(a,b,c){if(0==a.k){if(a==c)b=3,c=new TypeError("Promise cannot resolve to itself");else{if(Pb(c)){a.k=1;c.then(a.Za,a.$a,a);return}if(ea(c))try{var d=c.then;if(p(d)){Wb(a,c,d);return}}catch(e){b=3,c=e}}a.r=c;a.k=b;Ub(a);3!=b||c instanceof Rb||Yb(a,c)}},Wb=function(a,b,c){a.k=1;var d=!1,e=function(b){d||(d=!0,a.Za(b))},f=function(b){d||(d=!0,a.$a(b))};try{c.call(b,e,f)}catch(g){f(g)}},Ub=function(a){a.ya||(a.ya=!0,Nb(a.Rb,a))};
M.prototype.Rb=function(){for(;this.l&&this.l.length;){var a=this.l;this.l=[];for(var b=0;b<a.length;b++){var c=a[b],d=this.r;2==this.k?c.Qa(d):(Tb(this),c.za(d))}}this.ya=!1};var Tb=function(a){for(;a&&a.ja;a=a.m)a.ja=!1},Yb=function(a,b){a.ja=!0;Nb(function(){a.ja&&Zb.call(null,b)})},Zb=Ib,Rb=function(a){u.call(this,a)};t(Rb,u);Rb.prototype.name="cancel";/*
 Portions of this code are from MochiKit, received by
 The Closure Authors under the MIT license. All other code is Copyright
 2005-2009 The Closure Authors. All Rights Reserved.
*/
var N=function(a,b){this.ba=[];this.Oa=a;this.Na=b||null;this.O=this.K=!1;this.r=void 0;this.wa=this.qb=this.va=!1;this.ca=0;this.m=null;this.ua=0};N.prototype.cancel=function(a){if(this.K)this.r instanceof N&&this.r.cancel();else{if(this.m){var b=this.m;delete this.m;a?b.cancel(a):(b.ua--,0>=b.ua&&b.cancel())}this.Oa?this.Oa.call(this.Na,this):this.wa=!0;this.K||this.A(new $b)}};N.prototype.Pa=function(a,b){this.va=!1;ac(this,a,b)};
var ac=function(a,b,c){a.K=!0;a.r=c;a.O=!b;bc(a)},dc=function(a){if(a.K){if(!a.wa)throw new cc;a.wa=!1}};N.prototype.G=function(a){dc(this);ac(this,!0,a)};N.prototype.A=function(a){dc(this);ac(this,!1,a)};N.prototype.Ub=function(a,b){return O(this,a,null,b)};var ec=function(a,b,c){O(a,b,b,c)},O=function(a,b,c,d){a.ba.push([b,c,d]);a.K&&bc(a);return a};N.prototype.then=function(a,b,c){var d,e,f=new M(function(a,b){d=a;e=b});O(this,d,function(a){a instanceof $b?f.cancel():e(a)});return f.then(a,b,c)};
Ob(N);
var fc=function(a){var b=new N;O(a,b.G,b.A,b);return b},gc=function(a){return la(a.ba,function(a){return p(a[1])})},bc=function(a){if(a.ca&&a.K&&gc(a)){var b=a.ca,c=hc[b];c&&(k.clearTimeout(c.da),delete hc[b]);a.ca=0}a.m&&(a.m.ua--,delete a.m);for(var b=a.r,d=c=!1;a.ba.length&&!a.va;){var e=a.ba.shift(),f=e[0],g=e[1],e=e[2];if(f=a.O?g:f)try{var l=f.call(e||a.Na,b);void 0!==l&&(a.O=a.O&&(l==b||l instanceof Error),a.r=b=l);Pb(b)&&(d=!0,a.va=!0)}catch(I){b=I,a.O=!0,gc(a)||(c=!0)}}a.r=b;d&&(l=q(a.Pa,a,
!0),d=q(a.Pa,a,!1),b instanceof N?(O(b,l,d),b.qb=!0):b.then(l,d));c&&(b=new ic(b),hc[b.da]=b,a.ca=b.da)},jc=function(a){var b=new N;b.G(a);return b},lc=function(){var a=kc,b=new N;b.A(a);return b},cc=function(){u.call(this)};t(cc,u);cc.prototype.message="Deferred has already fired";cc.prototype.name="AlreadyCalledError";var $b=function(){u.call(this)};t($b,u);$b.prototype.message="Deferred was canceled";$b.prototype.name="CanceledError";
var ic=function(a){this.da=k.setTimeout(q(this.Nb,this),0);this.$=a};ic.prototype.Nb=function(){delete hc[this.da];throw this.$;};var hc={};var mc=function(a,b){var c=Array.prototype.slice.call(arguments),d=c.shift();if("undefined"==typeof d)throw Error("[goog.string.format] Template required");return d.replace(/%([0\-\ \+]*)(\d+)?(\.(\d+))?([%sfdiu])/g,function(a,b,d,l,I,F,S,T){if("%"==F)return"%";var Xb=c.shift();if("undefined"==typeof Xb)throw Error("[goog.string.format] Not enough arguments");arguments[0]=Xb;return P[F].apply(null,arguments)})},P={s:function(a,b,c){return isNaN(c)||""==c||a.length>=c?a:a=-1<b.indexOf("-",0)?a+Array(c-
a.length+1).join(" "):Array(c-a.length+1).join(" ")+a},f:function(a,b,c,d,e){d=a.toString();isNaN(e)||""==e||(d=a.toFixed(e));var f;f=0>a?"-":0<=b.indexOf("+")?"+":0<=b.indexOf(" ")?" ":"";0<=a&&(d=f+d);if(isNaN(c)||d.length>=c)return d;d=isNaN(e)?Math.abs(a).toString():Math.abs(a).toFixed(e);a=c-d.length-f.length;return d=0<=b.indexOf("-",0)?f+d+Array(a+1).join(" "):f+Array(a+1).join(0<=b.indexOf("0",0)?"0":" ")+d},d:function(a,b,c,d,e,f,g,l){return P.f(parseInt(a,10),b,c,d,0,f,g,l)}};P.i=P.d;
P.u=P.d;var nc=function(a){if("function"==typeof a.q)return a.q();if(n(a))return a.split("");if(da(a)){for(var b=[],c=a.length,d=0;d<c;d++)b.push(a[d]);return b}return ua(a)},oc=function(a,b,c){if("function"==typeof a.forEach)a.forEach(b,c);else if(da(a)||n(a))ka(a,b,c);else{var d;if("function"==typeof a.C)d=a.C();else if("function"!=typeof a.q)if(da(a)||n(a)){d=[];for(var e=a.length,f=0;f<e;f++)d.push(f)}else d=va(a);else d=void 0;for(var e=nc(a),f=e.length,g=0;g<f;g++)b.call(c,e[g],d&&d[g],a)}};var Q=function(a){this.M=new w;this.Aa(arguments)};Q.prototype.set=function(a,b){this.M.set(a.name,{key:a,value:b})};Q.prototype.remove=function(a){this.M.remove(a.name)};Q.prototype.get=function(a){a=this.M.get(a.name,null);return null===a?null:a.value};Q.prototype.Aa=function(a){for(var b=0;b<a.length;b+=2)this.set(a[b],a[b+1])};var pc=function(a,b){ka(a.M.q(),function(a){b(a.key,a.value)})};Q.prototype.jb=function(){var a={};pc(this,function(b,c){a[b.id]=c});return a};
Q.prototype.B=function(){var a=new Q;a.M=this.M.B();return a};Q.prototype.toString=function(){var a={};pc(this,function(b,c){a[b.id]=c});return JSON.stringify(a)};var qc={id:"apiVersion",name:"v",valueType:"text",maxLength:void 0,defaultValue:void 0},rc={id:"appName",name:"an",valueType:"text",maxLength:100,defaultValue:void 0},sc={id:"appVersion",name:"av",valueType:"text",maxLength:100,defaultValue:void 0},tc={id:"clientId",name:"cid",valueType:"text",maxLength:void 0,defaultValue:void 0},uc={id:"language",name:"ul",valueType:"text",maxLength:20,defaultValue:void 0},vc={id:"libVersion",name:"_v",valueType:"text",maxLength:void 0,defaultValue:void 0},wc={id:"screenColors",
name:"sd",valueType:"text",maxLength:20,defaultValue:void 0},xc={id:"screenResolution",name:"sr",valueType:"text",maxLength:20,defaultValue:void 0},yc={id:"trackingId",name:"tid",valueType:"text",maxLength:void 0,defaultValue:void 0},zc={id:"viewportSize",name:"vp",valueType:"text",maxLength:20,defaultValue:void 0},Ac={kc:qc,nc:rc,oc:sc,wc:tc,Pc:uc,Qc:vc,Xc:wc,Yc:xc,kd:yc,qd:zc},Cc=function(a){if(!n(a))return a;var b=Bc(a,Da);if(ea(b))return b;b=Bc(a,Ac);if(ea(b))return b;b=/^dimension(\d+)$/.exec(a);
if(null!==b)return{id:a,name:"cd"+b[1],valueType:"text",maxLength:150,defaultValue:void 0};b=/^metric(\d+)$/.exec(a);if(null!==b)return{id:a,name:"cm"+b[1],valueType:"integer",maxLength:void 0,defaultValue:void 0};throw Error(a+" is not a valid parameter name.");},Bc=function(a,b){var c=wa(b,function(b){return b.id==a&&"metric"!=a&&"dimension"!=a});return ea(c)?c:null};var Dc=function(a,b){this.n=a;this.v=b};Dc.prototype.send=function(a,b){b.set(tc,this.n.xa);return this.v.send(a,b)};var R=function(){};R.Lb=function(){return R.nb?R.nb:R.nb=new R};R.prototype.send=function(){return jc()};var U=function(a,b){D.call(this,"a");this.Sb=a;this.Ob=b};t(U,D);U.prototype.$b=function(){return this.Sb};U.prototype.Zb=function(){return this.Ob.jb()};var Ec=function(a,b){this.Ha=a;this.v=b};Ec.prototype.send=function(a,b){this.Ha.dispatchEvent(new U(a,b));return this.v.send(a,b)};var Fc=function(a){this.v=a};Fc.prototype.send=function(a,b){Gc(b);Hc(b);return this.v.send(a,b)};var Gc=function(a){pc(a,function(b,c){void 0!==b.maxLength&&"text"==b.valueType&&0<b.maxLength&&c.length>b.maxLength&&a.set(b,c.substring(0,b.maxLength))})},Hc=function(a){pc(a,function(b,c){void 0!==b.defaultValue&&c==b.defaultValue&&a.remove(b)})};var kc={status:"device-offline",pa:void 0},Ic={status:"rate-limited",pa:void 0},Jc={status:"sampled-out",pa:void 0},Kc={status:"sent",pa:void 0};var Lc=function(a,b){this.yb=a;this.v=b};Lc.prototype.send=function(a,b){var c;c=this.yb;var d=c.Ua(),e=Math.floor((d-c.Ta)*c.vb);0<e&&(c.T=Math.min(c.T+e,c.wb),c.Ta=d);1>c.T?c=!1:(c.T-=1,c=!0);return c||"item"==a||"transaction"==a?this.v.send(a,b):jc(Ic)};var Mc=function(a){this.Pb=a};Mc.prototype.send=function(a,b){this.Pb.push({tb:a,ub:b});return jc()};var Nc=function(a,b,c){this.n=a;this.ka=[];this.I={enabled:new Mc(this.ka),disabled:c};this.P=this.I.enabled;O(fc(this.n.aa),ha(this.sb,b),this.rb,this)};Nc.prototype.sb=function(a){this.I.enabled=a(this.n);Oc(this);ka(this.ka,function(a){this.send(a.tb,a.ub)},this);this.ka=null;Pc(this.n,q(this.xb,this))};Nc.prototype.rb=function(){this.P=this.I.enabled=this.I.disabled;this.ka=null};Nc.prototype.send=function(a,b){return this.P.send(a,b)};var Oc=function(a){a.P=a.n.ab()?a.I.enabled:a.I.disabled};
Nc.prototype.xb=function(a){switch(a){case "analytics.tracking-permitted":Oc(this)}};var V=function(a,b){this.P=a;this.Ha=b;this.cb=new Q;this.Ia=!1};h=V.prototype;h.set=function(a,b){var c=Cc(a);this.cb.set(c,b)};h.send=function(a,b){var c=this.cb.B();b&&ta(b,function(a,b){null!=a&&c.set(Cc(b),a)},this);this.Ia&&(this.Ia=!1,c.set(Ba,"start"));return this.P.send(a,c)};h.bc=function(a){var b={description:a};this.set(Ca,a);return this.send("appview",b)};h.cc=function(a,b,c,d){return this.send("event",{eventCategory:a,eventAction:b,eventLabel:c,eventValue:d})};
h.ec=function(a,b,c){return this.send("social",{socialNetwork:a,socialAction:b,socialTarget:c})};h.dc=function(a,b){return this.send("exception",{exDescription:a,exFatal:b})};h.hb=function(a,b,c,d){return this.send("timing",{timingCategory:a,timingVar:b,timingLabel:d,timingValue:c})};h.Wb=function(){this.Ia=!0};h.hc=function(a,b,c){return new Qc(this,a,b,c)};h.Yb=function(){return this.Ha};var Qc=function(a,b,c,d){this.eb=a;this.Gb=b;this.Kb=c;this.Hb=d;this.Jb=r()};
Qc.prototype.send=function(){var a=this.eb.hb(this.Gb,this.Kb,r()-this.Jb,this.Hb);this.eb=null;return a};var Rc=function(){this.T=60;this.wb=500;this.vb=5E-4;this.Ua=function(){return(new Date).getTime()};this.Ta=this.Ua()};var Sc=function(a,b){this.n=a;this.v=b};Sc.prototype.send=function(a,b){var c=b.get(tc);return parseInt(c.split("-")[1],16)<655.36*this.n.Sa?this.v.send(a,b):jc(Jc)};var Tc=RegExp("^(?:([^:/?#.]+):)?(?://(?:([^/?#]*)@)?([^/#?]*?)(?::([0-9]+))?(?=[/#?]|$))?([^?#]+)?(?:\\?([^#]*))?(?:#(.*))?$"),Uc=B,Vc=function(a,b){if(Uc){Uc=!1;var c=k.location;if(c){var d=c.href;if(d&&(d=(d=Vc(3,d))&&decodeURIComponent(d))&&d!=c.hostname)throw Uc=!0,Error();}}return b.match(Tc)[a]||null};var Wc=function(){};Wc.prototype.kb=null;var Yc=function(a){var b;(b=a.kb)||(b={},Xc(a)&&(b[0]=!0,b[1]=!0),b=a.kb=b);return b};var Zc,$c=function(){};t($c,Wc);var ad=function(a){return(a=Xc(a))?new ActiveXObject(a):new XMLHttpRequest},Xc=function(a){if(!a.mb&&"undefined"==typeof XMLHttpRequest&&"undefined"!=typeof ActiveXObject){for(var b=["MSXML2.XMLHTTP.6.0","MSXML2.XMLHTTP.3.0","MSXML2.XMLHTTP","Microsoft.XMLHTTP"],c=0;c<b.length;c++){var d=b[c];try{return new ActiveXObject(d),a.mb=d}catch(e){}}throw Error("Could not create ActiveXObject. ActiveX might be disabled, or MSXML might not be installed");}return a.mb};Zc=new $c;var W=function(a){H.call(this);this.headers=new w;this.Z=a||null;this.w=!1;this.W=this.a=null;this.V=this.ra="";this.D=this.qa=this.U=this.ta=!1;this.Y=0;this.X=null;this.Ka="";this.sa=this.pb=!1};t(W,H);var bd=/^https?$/i,cd=["POST","PUT"],dd=[],ed=function(a,b,c){var d=new W;dd.push(d);b&&d.listen("complete",b);d.Ja("ready",d.Qb);d.send(a,"POST",c,void 0)};W.prototype.Qb=function(){this.na();oa(dd,this)};
W.prototype.send=function(a,b,c,d){if(this.a)throw Error("[goog.net.XhrIo] Object is active with another request="+this.ra+"; newUri="+a);b=b?b.toUpperCase():"GET";this.ra=a;this.V="";this.ta=!1;this.w=!0;this.a=this.Z?ad(this.Z):ad(Zc);this.W=this.Z?Yc(this.Z):Yc(Zc);this.a.onreadystatechange=q(this.La,this);try{this.qa=!0,this.a.open(b,String(a),!0),this.qa=!1}catch(e){this.$(5,e);return}a=c||"";var f=this.headers.B();d&&oc(d,function(a,b){f.set(b,a)});d=na(f.C());c=k.FormData&&a instanceof k.FormData;
!(0<=ja(cd,b))||d||c||f.set("Content-Type","application/x-www-form-urlencoded;charset=utf-8");oc(f,function(a,b){this.a.setRequestHeader(b,a)},this);this.Ka&&(this.a.responseType=this.Ka);"withCredentials"in this.a&&(this.a.withCredentials=this.pb);try{fd(this),0<this.Y&&((this.sa=gd(this.a))?(this.a.timeout=this.Y,this.a.ontimeout=q(this.Ma,this)):this.X=Cb(this.Ma,this.Y,this)),this.U=!0,this.a.send(a),this.U=!1}catch(g){this.$(5,g)}};
var gd=function(a){return z&&C(9)&&"number"==typeof a.timeout&&void 0!==a.ontimeout},ma=function(a){return"content-type"==a.toLowerCase()};W.prototype.Ma=function(){"undefined"!=typeof aa&&this.a&&(this.V="Timed out after "+this.Y+"ms, aborting",this.dispatchEvent("timeout"),this.abort(8))};W.prototype.$=function(a,b){this.w=!1;this.a&&(this.D=!0,this.a.abort(),this.D=!1);this.V=b;hd(this);id(this)};var hd=function(a){a.ta||(a.ta=!0,a.dispatchEvent("complete"),a.dispatchEvent("error"))};
W.prototype.abort=function(){this.a&&this.w&&(this.w=!1,this.D=!0,this.a.abort(),this.D=!1,this.dispatchEvent("complete"),this.dispatchEvent("abort"),id(this))};W.prototype.j=function(){this.a&&(this.w&&(this.w=!1,this.D=!0,this.a.abort(),this.D=!1),id(this,!0));W.J.j.call(this)};W.prototype.La=function(){this.Ga||(this.qa||this.U||this.D?jd(this):this.Bb())};W.prototype.Bb=function(){jd(this)};
var jd=function(a){if(a.w&&"undefined"!=typeof aa&&(!a.W[1]||4!=kd(a)||2!=ld(a)))if(a.U&&4==kd(a))Cb(a.La,0,a);else if(a.dispatchEvent("readystatechange"),4==kd(a)){a.w=!1;try{var b=ld(a),c,d;t:switch(b){case 200:case 201:case 202:case 204:case 206:case 304:case 1223:d=!0;break t;default:d=!1}if(!(c=d)){var e;if(e=0===b){var f=Vc(1,String(a.ra));if(!f&&self.location)var g=self.location.protocol,f=g.substr(0,g.length-1);e=!bd.test(f?f.toLowerCase():"")}c=e}if(c)a.dispatchEvent("complete"),a.dispatchEvent("success");
else{var l;try{l=2<kd(a)?a.a.statusText:""}catch(I){l=""}a.V=l+" ["+ld(a)+"]";hd(a)}}finally{id(a)}}},id=function(a,b){if(a.a){fd(a);var c=a.a,d=a.W[0]?ba:null;a.a=null;a.W=null;b||a.dispatchEvent("ready");try{c.onreadystatechange=d}catch(e){}}},fd=function(a){a.a&&a.sa&&(a.a.ontimeout=null);"number"==typeof a.X&&(k.clearTimeout(a.X),a.X=null)},kd=function(a){return a.a?a.a.readyState:0},ld=function(a){try{return 2<kd(a)?a.a.status:-1}catch(b){return-1}};var md=function(a,b,c){this.p=a||null;this.Mb=!!c},Y=function(a){if(!a.c&&(a.c=new w,a.g=0,a.p))for(var b=a.p.split("&"),c=0;c<b.length;c++){var d=b[c].indexOf("="),e=null,f=null;0<=d?(e=b[c].substring(0,d),f=b[c].substring(d+1)):e=b[c];e=decodeURIComponent(e.replace(/\+/g," "));e=X(a,e);a.add(e,f?decodeURIComponent(f.replace(/\+/g," ")):"")}};h=md.prototype;h.c=null;h.g=null;h.add=function(a,b){Y(this);this.p=null;a=X(this,a);var c=this.c.get(a);c||this.c.set(a,c=[]);c.push(b);this.g++;return this};
h.remove=function(a){Y(this);a=X(this,a);return this.c.Q(a)?(this.p=null,this.g-=this.c.get(a).length,this.c.remove(a)):!1};h.Q=function(a){Y(this);a=X(this,a);return this.c.Q(a)};h.C=function(){Y(this);for(var a=this.c.q(),b=this.c.C(),c=[],d=0;d<b.length;d++)for(var e=a[d],f=0;f<e.length;f++)c.push(b[d]);return c};h.q=function(a){Y(this);var b=[];if(n(a))this.Q(a)&&(b=pa(b,this.c.get(X(this,a))));else{a=this.c.q();for(var c=0;c<a.length;c++)b=pa(b,a[c])}return b};
h.set=function(a,b){Y(this);this.p=null;a=X(this,a);this.Q(a)&&(this.g-=this.c.get(a).length);this.c.set(a,[b]);this.g++;return this};h.get=function(a,b){var c=a?this.q(a):[];return 0<c.length?String(c[0]):b};h.toString=function(){if(this.p)return this.p;if(!this.c)return"";for(var a=[],b=this.c.C(),c=0;c<b.length;c++)for(var d=b[c],e=encodeURIComponent(String(d)),d=this.q(d),f=0;f<d.length;f++){var g=e;""!==d[f]&&(g+="="+encodeURIComponent(String(d[f])));a.push(g)}return this.p=a.join("&")};
h.B=function(){var a=new md;a.p=this.p;this.c&&(a.c=this.c.B(),a.g=this.g);return a};var X=function(a,b){var c=String(b);a.Mb&&(c=c.toLowerCase());return c};var nd=function(a,b){this.Ab=a;this.bb=b};nd.prototype.send=function(a,b){if(ab&&!navigator.onLine)return lc();var c=new N,d=od(a,b);d.length>this.bb?c.A({status:"payload-too-big",pa:mc("Encoded hit length == %s, but should be <= %s.",d.length,this.bb)}):ed(this.Ab,function(){c.G(Kc)},d);return c};var od=function(a,b){var c=new md;c.add(Aa.name,a);pc(b,function(a,b){c.add(a.name,b.toString())});return c.toString()};var Z=function(a,b,c,d){this.Ib=a;this.Eb=b;this.Fb=c;this.n=d},pd;Z.prototype.ac=function(a){var b=new H,b=new V(qd(this,b),b);b.set(vc,this.Ib);b.set(qc,1);b.set(rc,this.Eb);b.set(sc,this.Fb);b.set(yc,a);a=window.navigator.language;b.set(uc,a);a=screen.colorDepth+"-bit";b.set(wc,a);a=[screen.width,screen.height].join("x");b.set(xc,a);a=window.document;a="CSS1Compat"==a.compatMode?a.documentElement:a.body;a=new Ea(a.clientWidth,a.clientHeight);a=[a.width,a.height].join("x");b.set(zc,a);return b};
var qd=function(a,b){return new Nc(a.n,function(a){if(!pd){new K;var d=new Fc(new nd("https://www.google-analytics.com/collect",8192)),e=new Rc;pd=new Dc(a,new Sc(a,new Lc(e,d)))}return new Ec(b,pd)},R.Lb())};Z.prototype.Xb=function(){return fc(this.n.aa)};var $=function(a){this.F=a;this.Sa=100;this.Va=[];this.aa=new N;this.ma=this.xa=null;rd(this)},rd=function(a){O(a.F.get("analytics.tracking-permitted"),function(a){this.ma=void 0!==a?a:!0;this.fb()},a.gb,a);O(sd(a),a.fb,a.gb,a)};$.prototype.gb=function(a){this.aa.A(a)};$.prototype.fb=function(){null===this.ma||null===this.xa||this.aa.G(this)};var Pc=function(a,b){a.Va.push(b)};
$.prototype.gc=function(a){ec(this.F.set("analytics.tracking-permitted",a),function(){this.ma=a;ka(this.Va,function(a){a("analytics.tracking-permitted")})},this)};$.prototype.ab=function(){var a;if(a=this.ma)a=k._gaUserPrefs,a=!(a&&a.ioo&&a.ioo());return a};$.prototype.fc=function(a){this.Sa=a};
var sd=function(a){var b=new N;O(a.F.get("analytics.user-id"),function(a){if(!a){a="xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".split("");for(var d=0,e=a.length;d<e;d++)switch(a[d]){case "x":a[d]=Math.floor(16*Math.random()).toString(16);break;case "y":a[d]=(Math.floor(4*Math.random())+8).toString(16)}a=a.join("");this.F.set("analytics.user-id",a)}this.xa=a;b.G()},function(a){b.A(a)},a);return b};var td=function(a,b){if(!ea(a))throw Error("'storage' argument must be defined and not null.");this.F=a;this.Xa=b||""};td.prototype.get=function(a){var b=new N,c=this.Xa+"."+a;this.F.get(c,function(a){var e=chrome.runtime.lastError;e?b.A(e):b.G(a[c])});return b};td.prototype.set=function(a,b){var c=new N,d={};d[this.Xa+"."+a]=b;this.F.set(d,function(){var a=chrome.runtime.lastError;a?c.A(a):c.G()});return c};var ud=new w;s("goog.async.Deferred",N);s("goog.async.Deferred.prototype.addCallback",N.prototype.Ub);s("goog.events.EventTarget",H);s("goog.events.EventTarget.prototype.listen",H.prototype.listen);s("analytics.getService",function(a){var b=ud.get(a,null);if(null===b){var b=chrome.runtime.getManifest().version,c;c=new td(chrome.storage.local,"google-analytics");c=new $(c);b=new Z("ca3",a,b,c);ud.set(a,b)}return b});s("analytics.internal.GoogleAnalyticsService",Z);
s("analytics.internal.GoogleAnalyticsService.prototype.getTracker",Z.prototype.ac);s("analytics.internal.GoogleAnalyticsService.prototype.getConfig",Z.prototype.Xb);s("analytics.internal.ServiceSettings",$);s("analytics.internal.ServiceSettings.prototype.setTrackingPermitted",$.prototype.gc);s("analytics.internal.ServiceSettings.prototype.isTrackingPermitted",$.prototype.ab);s("analytics.internal.ServiceSettings.prototype.setSampleRate",$.prototype.fc);s("analytics.internal.ServiceTracker",V);
s("analytics.internal.ServiceTracker.prototype.send",V.prototype.send);s("analytics.internal.ServiceTracker.prototype.sendAppView",V.prototype.bc);s("analytics.internal.ServiceTracker.prototype.sendEvent",V.prototype.cc);s("analytics.internal.ServiceTracker.prototype.sendSocial",V.prototype.ec);s("analytics.internal.ServiceTracker.prototype.sendException",V.prototype.dc);s("analytics.internal.ServiceTracker.prototype.sendTiming",V.prototype.hb);
s("analytics.internal.ServiceTracker.prototype.startTiming",V.prototype.hc);s("analytics.internal.ServiceTracker.Timing",Qc);s("analytics.internal.ServiceTracker.Timing.prototype.send",Qc.prototype.send);s("analytics.internal.ServiceTracker.prototype.forceSessionStart",V.prototype.Wb);s("analytics.internal.ServiceTracker.prototype.getEventTarget",V.prototype.Yb);s("analytics.HitTypes.APPVIEW","appview");s("analytics.HitTypes.EVENT","event");s("analytics.HitTypes.SOCIAL","social");
s("analytics.HitTypes.TRANSACTION","transaction");s("analytics.HitTypes.ITEM","item");s("analytics.HitTypes.TIMING","timing");s("analytics.HitTypes.EXCEPTION","exception");s("analytics.Tracker.HitEvent",U);s("analytics.Tracker.HitEvent.EVENT_TYPE","a");s("analytics.Tracker.HitEvent.prototype.getHitType",U.prototype.$b);s("analytics.Tracker.HitEvent.prototype.getHit",U.prototype.Zb);ta(Da,function(a){var b=a.id.replace(/[A-Z]/,"_$&").toUpperCase();s("analytics.Parameters."+b,a)}); })()
