/*
 * Copyright (C) 2015, Nicola Spanti (also known as RyDroid) <dev@nicola-spanti.info>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */


/**
 * @constructor
 */
function NearRunnerPathsGroup(nb)
{
	"use strict";
	
	this.paths = [];
	if(typeof(nb) != 'undefined')
	{
		if(isNaN(nb))
		{
			throw new TypeError('nb is not a number');
		}
		else
		{
			this.paths = new Array(nb);
			for(var i=0; i < nb; ++i)
			{
				this.paths[i] = new RunnerPath();
			}
		}
	}
	
	this.distance = 0;
}

NearRunnerPathsGroup.prototype.getPercentageReady = function()
{
	"use strict";
	
	var percentage = 0;
	for(var i=0; i < this.paths.length; ++i)
		percentage += this.paths[i].getPercentageReady();
	return percentage / this.paths.length;
};

NearRunnerPathsGroup.prototype.isReady = function()
{
	"use strict";
	
	for(var i=0; i < this.paths.length; ++i)
	{
		if(!this.paths[i].isReady())
			return false;
	}
	return true;
};

NearRunnerPathsGroup.prototype.setDrawingInformation = function(drawingInformation)
{
	if(!(drawingInformation instanceof ObjectDrawing2DInformation))
		throw new TypeError('drawingInformation must be an instance of ObjectDrawing2DInformation');
	this.drawingInformation = drawingInformation;
	
	var pathDrawingInformation = this.drawingInformation.getCopy();
	pathDrawingInformation.setWidth(this.drawingInformation.width / this.paths.length);
	for(var i=0; i < this.paths.length; ++i)
	{
		var pathDrawingInformation = this.drawingInformation.getCopy();
		pathDrawingInformation.setWidth(this.drawingInformation.width / this.paths.length);
		pathDrawingInformation.position.setX(pathDrawingInformation.position.x + i * pathDrawingInformation.width);
		this.paths[i].setDrawingInformation(pathDrawingInformation);
	}
};

/**
 * @return {Number} number of items
 */
NearRunnerPathsGroup.prototype.getNbItems = function()
{
	"use strict";
	
	var nb = 0;
	for(var i=0; i < this.paths.length; ++i)
		nb += this.paths[i].items.length;
	return nb;
};

NearRunnerPathsGroup.prototype.update = function(clock, distancePerMicroseconds)
{
	"use strict";
	
	this.distance += clock.deltaTime * distancePerMicroseconds;
	
	if(this.paths.length > 0)
	{
		for(var i=0; i < this.paths.length; ++i)
		{
			this.paths[i].update(clock, distancePerMicroseconds);
		}
		
		var lastItemYValue = this.drawingInformation.height;
		for(var i=0; i < this.paths.length; ++i)
		{
			if(this.paths[i].items.length > 0 && lastItemYValue > this.paths[i].items[this.paths[i].items.length -1].drawingInformation.position.y)
			{
				lastItemYValue = this.paths[i].items[this.paths[i].items.length -1].drawingInformation.position.y;
			}
		}
		
		if(
			lastItemYValue > (2.5 * this.paths[0].getNormalItemHeight()) &&
			Math.random() < 0.06
		)
		{
			var lastItemPath = this.paths[parseInt(Math.random() * this.paths.length, 10)];
			var item = new RunnerPathItem();
			lastItemPath.setItemDrawingInformation(item);
			lastItemPath.items.push(item);
		}
	}
};

NearRunnerPathsGroup.prototype.changeRunnerToAnOtherPath = function()
{
	"use strict";
	
	var oldRunnerPathNumber = -1;
	
	for(var i=0; i < this.paths.length; ++i)
	{
		if(this.paths[i].runner instanceof GnuRunner)
			oldRunnerPathNumber = i;
	}
	
	if(oldRunnerPathNumber < 0)
		return false;
	
	var newRunnerPathNumber = (oldRunnerPathNumber + 1) % this.paths.length;
	this.paths[newRunnerPathNumber].runner = this.paths[oldRunnerPathNumber].runner;
	this.paths[oldRunnerPathNumber].runner = null;
	this.paths[newRunnerPathNumber].setRunnerDrawingInformation();
	return true;
};

/**
 * @return {Number} score
 */
NearRunnerPathsGroup.prototype.getScore = function()
{
	"use strict";
	
	var score = 0;
	for(var i=0; i < this.paths.length; ++i)
		score += this.paths[i].getScore();
	return score;
};
