/*
 * Copyright (C) 2015, Nicola Spanti (also known as RyDroid) <dev@nicola-spanti.info>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */


/* General settings and functions */

if(typeof(screen.lockOrientation) == 'function')
	screen.lockOrientation("portrait-primary");

var menu = new GameSimpleMainMenu([
	'play',
	'lose',
	'how-to-play',
	'privacy',
	'license',
	'contribute',
	'gnu-project',
	'scores'
]);

function setMenuSectionStyle(menu)
{
	'use strict';
	
	var menuSection = menu.getElement();
	var buttons = menuSection.querySelectorAll("button");
	var buttonHeight = parseInt(0.2 * window.innerHeight, 10);
	var buttonImages = null;
	for(var i=0; i < buttons.length; ++i)
	{
		buttons[i].style.minHeight = buttonHeight +'px';
		buttons[i].style.fontSize = parseInt(buttonHeight * 0.5, 10) +'px';
		
		buttonImages = buttons[i].querySelectorAll('button img[src]');
		for(var j=0; j < buttonImages.length; ++j)
		{
			buttonImages[j].style.maxHeight = buttons[i].style.minHeight;
		}
	}
	menu.show();
}
setMenuSectionStyle(menu);

function setLoseSectionStyle()
{
	var loseSection = menu.getActivity('lose');
	var i;
	
	var buttons = loseSection.querySelectorAll('button');
	var buttonHeight    = parseInt(0.2 * window.innerHeight, 10);
	var buttonMinWidth  = parseInt(0.2 * window.innerWidth,  10);
	for(i=0; i < buttons.length; ++i)
	{
		buttons[i].style.minHeight = buttonHeight   +'px';
		buttons[i].style.minWidth  = buttonMinWidth +'px';
	}
	
	var buttonImages = loseSection.querySelectorAll('button img[src]');
	if(buttonImages != null && buttonImages.length > 0)
	{
		var buttonFontSize = parseInt(getComputedStyle(buttons[0], null).getPropertyValue('font-size'), 10);
		var buttonImageRatio;
		var buttonImageHeight;
		for(i=0; i < buttonImages.length; ++i)
		{
			buttonImageHeight = buttonHeight - buttonFontSize;
			buttonImages[i].style.height = buttonImageHeight + 'px';
			
			buttonImageRatio =
				parseInt(getComputedStyle(buttonImages[i], null).getPropertyValue('width'),  10) /
				parseInt(getComputedStyle(buttonImages[i], null).getPropertyValue('height'), 10);
			buttonImages[i].style.width  = parseInt(buttonImageHeight * buttonImageRatio, 10) + 'px';
		}
	}
	
	menu.showActivity('lose');
}

var scores = new ScoresStatistics();

function startPlaying()
{
	var game = new GnuRunnersGame(afterLosingFunction);
	
	var loseInfoElement = document.getElementById('lose-infos');
	if(loseInfoElement)
		loseInfoElement.innerHTML = '';
	
	menu.hide();
	menu.hideActivity('lose');
	game.canvas.style.display = 'inline';
	game.setWindowSizeOnly();
	window.addEventListener('resize', game.setWindowSize.bind(game));
	
	game.start();
}

function afterLosingFunction(game)
{
	window.removeEventListener('resize', game.setWindowSize.bind(game));
	
	scores.push(new FinalScore(game.getScoreValue()));
	if(scores.size() > 1 && scores.isLastInsertedTheBest())
	{
		var loseInfoElement = document.getElementById('lose-infos');
		if(loseInfoElement)
			loseInfoElement.innerHTML += '<li>New high score</li>';
	}
	menu.showButton('scores');
	
	var loseSection = menu.getActivity('lose');
	loseSection.querySelector('.last-score-value').innerHTML = new String(scores.getLast().value);
	loseSection.querySelector('.best-score-value').innerHTML = new String(scores.getBest().value);
	
	setLoseSectionStyle();
}


/* Events */
(function()
{
	document.getElementById("button_play").addEventListener ('click', startPlaying, false);
	document.getElementById("button_retry").addEventListener('click', startPlaying, false);
	
	menu.addDefaultEventListenersToButton('how-to-play');
	
	document.getElementById("button_how-to-play_ok").addEventListener('click', function()
	{
		menu.onClickGoToMenu("how-to-play");
	}, false);
	
	document.getElementById("button_lose_go-menu").addEventListener('click', function()
	{
		menu.show();
		menu.hideActivity('lose');
		menu.hideActivity('play');
	}, false);
	
	menu.addDefaultEventListenersToButton('privacy');
	
	document.getElementById("button_privacy_ok").addEventListener('click', function()
	{
		menu.onClickGoToMenu('privacy');
	}, false);
	
	menu.addDefaultEventListenersToButton('license');
	
	document.getElementById("button_license_ok").addEventListener('click', function()
	{
		menu.onClickGoToMenu('license');
	}, false);
	
	menu.addDefaultEventListenersToButton('contribute');
	
	document.getElementById("button_contribute_ok").addEventListener('click', function()
	{
		menu.onClickGoToMenu('contribute');
	}, false);
	
	menu.addDefaultEventListenersToButton('gnu-project');
	
	document.getElementById("button_gnu-project_ok").addEventListener('click', function()
	{
		menu.onClickGoToMenu('gnu-project');
	}, false);
	
	document.getElementById("button_scores").addEventListener('click', function()
	{
		var scoresSection = menu.getActivity('scores');
		
		scoresSection.querySelector('.last-score-value').innerHTML = new String(scores.getLast().value);
		scoresSection.querySelector('.best-score-value').innerHTML = new String(scores.getBest().value);
		
		menu.onClickGoToActivity('scores');
		
		(function()
		{
			var tableOfScoresBody = scoresSection.querySelector('tbody');
			scores.sortByDate();
			var scoresArray = scores.toArray();
			for(var i=0; i < scoresArray.length; ++i)
			{
				tableOfScoresBody.innerHTML += '<tr><td>'+ new String(scoresArray[i].value) +'</td><td>'+ scoresArray[i].date.toLocaleString() +'</td></tr>';
			}
		})();
		
		(function()
		{
			var addItemToStatsList = (function(htmlOfItem)
			{
				var statsList = document.querySelector('#scores_statistics + ul');
				return function(htmlOfItem)
				{
					if(!isString(htmlOfItem))
						throw new TypeError('htmlOfItem must be a string');
					statsList.innerHTML += '<li>'+ htmlOfItem +'</li>';
				}
			})();
			
			addItemToStatsList('Number of scores = '+ new String(scores.size()));
			addItemToStatsList('Sum of scores = '   + new String(scores.getSumValue()));
			addItemToStatsList('Average score = '   + new String(Math.round10(scores.getAverageValue(), -2)));
			addItemToStatsList('Median score = '    + new String(Math.round10(scores.getMedianValue(),  -2)));
		})();
	}, false);
	
	function scoresSectionToMenu()
	{
		menu.onClickGoToMenu("scores");
		document.querySelector('#scores tbody').innerHTML = '';
		document.querySelector('#scores_statistics + ul').innerHTML = '';
	}
	
	document.getElementById("button_scores_ok").addEventListener('click', scoresSectionToMenu, false);
	
	document.getElementById("button_scores_clear").addEventListener('click', function()
	{
		if(window.confirm("This will remove your entire history of scores."))
		{
			menu.hideButton('scores');
			scoresSectionToMenu();
			scores.clear();
		}
	}, false);
})();
