'use strict';

/**
 *
 * @returns {Array}
 */
function getVoted() {
    var voted = localStorage.getItem('voted');
    if (!voted) {
        voted = [];
    } else {
        voted = JSON.parse(voted);
    }
    return voted;
}

/**
 *
 * @param id
 * @returns {boolean}
 */
function hasVoted(id) {
    var voted = getVoted();
    return (voted.indexOf(id) >= 0);
}

/**
 *
 * @param id
 */
function addVote(id) {
    var voted = getVoted();
    voted.push(id);
    localStorage.setItem('voted', JSON.stringify(voted));
}

/**
 * Handle blob activity
 *
 * @param blob
 */
function shareBlob(blob) {
    if (typeof MozActivity !== 'function') {
        return;
    }
    var activity = new MozActivity({
        // Ask for the "pick" activity
        name: "share",

        // Provide de data required by the filters of the activity
        data: {
            type: "image/*",
            blobs: [blob]
        }
    });

    activity.onerror = function() {
        console.log(this.error);
    };
}

/**
 * Default controller
 * @param $scope
 * @param $location
 * @param $http
 * @param Config
 * @param Storage
 * @constructor
 */
function Default($scope, $location, $http, Config, Storage) {

    $scope.$location = $location;

    $scope.img = null;

    /**
     *
     * @type {number}
     */
    $scope.after = 0;

    $scope.storage = Storage;
    //Init storage
    Storage.init();

    $scope.setAfter = function(after) {
        $scope.after = after;
    };

    $scope.resetAfter = function() {
        $scope.after = 0;
    };

    $scope.previewImage = function(img) {
        $location.path('/preview');
    };


    /**
     * Load image and share it.
     * @param imgUrl
     * @param $event
     */
    $scope.shareImageByURL = function(imgUrl, $event) {
        //Check lock state
        if ($scope.lock) {
            return;
        }
        var xhr = new XMLHttpRequest({
            mozSystem: true
        });
        xhr.open('GET', imgUrl, true);

        xhr.responseType = 'blob';

        xhr.onload = function(e) {
            if (this.status == 200) {
                shareBlob(xhr.response);
            }
        };
        xhr.send();
    };

    /**
     * Download image by URL
     * @param imgUrl
     * @param $event
     */
    $scope.downloadImageByURL = function(imgUrl, $event) {
        //Check lock state
        if ($scope.lock) {
            return;
        }
        if (!navigator.getDeviceStorage) {
            console.log("No storage info");
            return;
        }
        var deviceStorage = navigator.getDeviceStorage('pictures');
        console.log("Downloading image");
        var xhr = new XMLHttpRequest({
            mozSystem: true
        });
        xhr.open('GET', imgUrl, true);

        xhr.responseType = 'blob';
        var parts = imgUrl.split('/');
        xhr.onload = function(e) {
            if (this.status == 200) {
                (function(data) {
                    $scope.storage.store(data, 'graffito/'+Date.now()+parts[parts.length-1]);
                })(xhr.response);
            }
        };
        xhr.send();
    };

    /**
     *
     * @param img
     * @param $event
     */
    $scope.vote = function(img, $event) {
        //Check lock state
        if ($scope.lock) {
            return;
        }
        img.voted = true;
        if (hasVoted(img._id)) {
            return;
        }
        $http.get(Config.url+"api/images/vote/"+img._id+"?up=true")
            .success(function(data) {
                img.voted = true;
                img.rating++;
                addVote(img._id);
            })
            .error(function() {
                img.voted = false;
            });
    };

    /**
     * Set background image
     *
     * @param url
     * @returns {{background-image: string}}
     */
    $scope.getBackground = function(url) {
        return { 'background-image': "url('" + url + "')"  };
    };

    /**
     * Preview image
     * @param img
     */
    $scope.preview = function(img) {
        $scope.img = img;
        $scope.navigate("/preview");

    };
};
Default.$inject = ['$scope', '$location', '$http', 'Config', 'Storage'];

/**
 * Main screen
 * @param $scope
 * @param $http
 * @param Config
 * @constructor
 */
function Main($scope, $http, Config) {
    $scope.images = [];

    $scope.setHeader("Graffito");

    /**
     * Show refresh button
     * @type {boolean}
     */
    $scope.defaults.refresh = true;

    /**
     * Refresh current screen
     */
    $scope.refresh = function() {
        //Check lock state
        if ($scope.lock) {
            return;
        }
        $scope.resetAfter();
        //set loading
        $scope.setLoading(true);
        //clear images
        $scope.images = [];
        //make server call
        $http.get(Config.url+"api/images/last").success(function(data) {
            $scope.images = data;
        });
    };
    /**
     * Run loading images
     */
    $scope.refresh();

    /**
     * handle unlock
     */
    $scope.$on('unlock', function() {
        $scope.refresh();
    });

    /**
     * Handle refresh button click
     */
    $scope.$on('refresh', function(event, args) {
        if (!args.url || args.url != '/') {
            return;
        }
        $scope.refresh();
    });

    $scope.nextPage = function() {
        //Check lock state
        if ($scope.lock) {
            return;
        }
        if ($scope.loading || $scope.menuShow || !$scope.online) return;
        $scope.setLoading(true);
        $scope.setAfter($scope.after++);
        var url = Config.url+"api/images/last?page="+$scope.after;
        $http.get(url).success(function(data) {
            $scope.setLoading(false);
            var items = data;
            for (var i = 0; i < items.length; i++) {
                $scope.images.push(items[i]);
            }
        });
    };
};
Main.$inject = ['$scope', '$http', 'Config'];

/**
 * Main screen
 * @param $scope
 * @param $http
 * @param Config
 * @constructor
 */
function Best($scope, $http, Config) {
    $scope.images = [];
    $scope.page = 0;
    $scope.stop = false;

    $scope.setHeader("Best");

    $scope.nextPage = function(reset) {
        if ($scope.lock) {
            return;
        }
        if (reset) {
            $scope.page = 0;
            $scope.stop = false;
        }
        if ($scope.loading || $scope.menuShow || !$scope.online || $scope.stop) return;
        $scope.setLoading(true);
        var url = Config.url+"api/images/best?page="+$scope.page;
        $http.get(url).success(function(data) {
            $scope.setLoading(false);
            if (data.length == 0) {
                $scope.stop = true;
            }
            for (var i = 0; i < data.length; i++) {
                $scope.images.push(data[i]);
            }
            $scope.page++;
        });
    };
    //Call last images
    $scope.nextPage(true);

    /**
     * handle unlock
     */
    $scope.$on('unlock', function() {
        $scope.nextPage(true);
    });
};
Best.$inject = ['$scope', '$http', 'Config'];

function Help($scope) {};
Help.$inject = ['$scope'];

function Preview($scope, $location) {
    if (!$scope.img) {
        $location.path('/');
    }
    if ($scope.img) {
        $scope.img.voted = hasVoted($scope.img._id);
    }
};
Preview.$inject = ['$scope', '$location'];

/**
 * Categories controller
 * @param $scope
 * @param Config
 * @param $http
 * @param $location
 * @constructor
 */
function Categories($scope, Config, $http, $location) {
    $scope.setHeader("Categories");
    $scope.cats = [];

    $scope.setLoading(true);
    $http.get(Config.url+'api/category').success(function(cats) {
        $scope.setLoading(false);
        $scope.cats = cats;
    }).error(function() {
        $scope.setLoading(false);
    });

    $scope.showCategory = function(cat) {
        $scope.navigate('/category/'+cat._id+"/"+cat.title);
    };
};
Categories.$inject = ['$scope', 'Config', '$http', '$location'];


/**
 * items
 *
 * @param $scope
 * @param $routeParams
 * @param $http
 * @param Config
 * @constructor
 */
function Category($scope, $routeParams, $http, Config) {
    $scope.images = [];
    $scope.page = 0;
    $scope.stop = false;

    $scope.setHeader($routeParams.name);

    $scope.nextPage = function(reset) {
        //Check lock state
        if ($scope.lock) {
            return;
        }
        if (reset) {
            $scope.page = 0;
            $scope.stop = false;
        }
        if ($scope.loading || $scope.stop) {
            return;
        }
        $scope.setLoading(true);
        $http.get(Config.url+"api/images/"+$routeParams.cat_id+'?page='+$scope.page).success(function(data) {
            $scope.setLoading(false);
            if (data.length == 0) {
                $scope.stop = true;
            }
            for(var i = 0; i < data.length; i++) {
                $scope.images.push(data[i]);
            }
            $scope.page++;
        }).error(function() {
            $scope.setLoading(false);
        });
    };
    $scope.nextPage(true);

    /**
     * handle unlock
     */
    $scope.$on('unlock', function() {
        $scope.nextPage(true);
    });
}
Category.$inject = ['$scope', '$routeParams', '$http', 'Config'];

