// Copyright (c) 2015-2016, Masahiko Imanaka. All rights reserved.
/* jshint moz:true, esnext:true */
/* global Swiper */

(function() {
'use strict';

var L10n = document.l10n;

var gCardData = null,
    //gModes = ['CARD', 'LIST', 'READ'],
    gViewMode = null,
    gBackView = null,
    gIndexes = ['NUMBER', 'POEM1', 'POEM2', 'POET', 'CATEGORY'],
    gIndexType = gIndexes[0],
    gCurrentNum = 1,
    gReadingList = shuffleList(100),
    gReadCounter = 1;

var navLine = document.getElementById('navLine'),
    jumpLink = document.getElementById('jumpLink');


function CardData(cards) {
  this._data = cards;
  this.imgSrc = this._data.images.src || "Hyakunin-isshu.jpg";
  this.cardSize = {
    w: parseInt(this._data.images.width) || 200,
    h: parseInt(this._data.images.height) || 280,
  };
  this.imageTable = {
    cols: parseInt(this._data.images.cols) || 10,
    rows: parseInt(this._data.images.rows) || 10
  };
  this.smallRatio = Math.min(window.screen.width * 0.3, this.cardSize.w) / this.cardSize.w;
}
CardData.prototype = {
  get data() {
    return this._data;
  },
  set data(cards) {
    console.error('You cannot set data to CardData.', cards);
  },
  getCard: function (num) {
    if (num < 1 || 100 < num) {
      num = 1;
    }
    return (this._data) ? this._data[(num + '')] : null;
  },
  getCardClipCoordinates: function(num, ratio) {
    var r = ratio || 1.0,
        col = ((num - 1) % this.imageTable.cols),
        row = Math.floor((num - 1) / this.imageTable.rows),
        posX = Math.round(col * this.cardSize.w * r),
        posY = Math.round(row * this.cardSize.h * r);
    //console.log([posX, posY], r);
    return [posX, posY];
  }
};


// Focus manager for keyboard navigations.
var focusManager = {
  _maxRange: 99,
  _posRotation: true,
  _position: {
    'NUMBER': 0, 'POEM1': 0, 'POEM2': 0, 'POET': 0, 'CATEGORY': 0
  },
  get: function getPosition() {
    return this._position[gIndexType];
  },
  set: function setPosition(pos) {
    this._position[gIndexType] = pos;
  },
  nextPos: function moveNextPosition() {
    this.set(this.get() + 1);
    if (this.get() > this._maxRange) {
      this.set((this._posRotation ? 0 : this._maxRange));
    }
    return this.get();
  },
  prevPos: function movePreviousPosition() {
    this.set(this.get() - 1);
    if (this.get() < 0) {
      this.set((this._posRotation ? this._maxRange : 0));
    }
    return this.get();
  },
  firstPos: function() {
    this.set(0);
    return 0;
  },
  lastPos: function() {
    this.set(this._maxRange);
    return this._maxRange;
  },
  toPos: function(pos) {
    if (0 <= pos && pos <= this._maxRange) {
      this.set(pos);
    } else if (pos < this.get()) {
      this.firstPos();
    } else {
      this.lastPos();
    }
    return this.get();
  }
};


function headerKeyHandler(ev) {
  return ev.keyCode != 13 || headerClickHandler(ev);
}

function headerClickHandler(ev) {
  ev.preventDefault();
  var mode = ev.target.getAttribute('mode');
  switch (mode) {
    case 'MENU':
      switchSection(mode);
      break;
    case 'back':
      if (gBackView) {
        switchView(gBackView);
        switchSection(gBackView);
        gBackView = gViewMode;
      } else {
        switchSection('MENU');
      }
      break;
    case 'LIST':
    case 'CARD':
    case 'READ':
      switchView(mode);
      switchSection(mode);
      break;
    default:
      //console.log('Clicked target is ' + ev.target.id);
  }
  toggleMenuBox();
  return;
}

function toggleMenuBox() {
  var menuBox = document.getElementById('menuBox'),
      menuBoxState = menuBox.getAttribute('collapse');
  if (menuBoxState === 'true') {
    menuBox.style.display = 'block';
    menuBox.setAttribute('aria-hidden', 'false');
    menuBox.setAttribute('collapse', 'false');
  } else {
    menuBox.style.display = 'none';
    menuBox.setAttribute('aria-hidden', 'true');
    menuBox.setAttribute('collapse', 'true');
  }
}


function menuKeyHandler(ev) {
  return ev.keyCode != 13 || menuClickHandler(ev);
}

function menuClickHandler(ev) {
  ev.preventDefault();
  var mode = ev.target.getAttribute('mode');

  if (mode !== gViewMode) {
    switchView(mode);
  }
  switch (mode) {
    case 'CARD':
    case 'LIST':
    case 'READ':
      switchSection(mode);
      break;
    default:
      switchSection('MENU');
      //console.log('Clicked target is ' + ev.target.id);
  }
}


function switchSection(sectionName) {
  var menuSection = document.getElementById('menuSection'),
      browsingSection = document.getElementById('browsingSection');

  switch (sectionName) {
    case 'LIST':
    case 'CARD':
    case 'READ':
      menuSection.style.display = 'none';
      browsingSection.style.display = 'block';
      break;
    case 'MENU':
      browsingSection.style.display = 'none';
      menuSection.style.display = 'block';
      break;
    default:
  }
}


// Switch main view mode.
function switchView(mode) {
  var listView = document.getElementById('listContainer'),
      listNav = document.getElementById('listNav'),
      cardView = document.getElementById('cardContainer'),
      cardNav = document.getElementById('cardNav'),
      readView = document.getElementById('readContainer'),
      readNav = document.getElementById('readNav');

  switch (mode) {
    case 'LIST':
      readView.style.display = readNav.style.display = 'none';
      readView.setAttribute('aria-hidden', 'true');
      readNav.setAttribute('aria-hidden', 'true');
      cardView.style.display = cardNav.style.display = 'none';
      cardView.setAttribute('aria-hidden', 'true');
      cardNav.setAttribute('aria-hidden', 'true');
      listView.style.display = listNav.style.display = 'block';
      listView.setAttribute('aria-hidden', 'false');
      listNav.setAttribute('aria-hidden', 'false');
      break;
    case 'CARD':
      listView.style.display = listNav.style.display = 'none';
      listView.setAttribute('aria-hidden', 'true');
      listNav.setAttribute('aria-hidden', 'true');
      readView.style.display = readNav.style.display = 'none';
      readView.setAttribute('aria-hidden', 'true');
      readNav.setAttribute('aria-hidden', 'true');
      cardView.style.display = cardNav.style.display = 'block';
      cardView.setAttribute('aria-hidden', 'false');
      cardNav.setAttribute('aria-hidden', 'false');
      break;
    case 'READ':
      listView.style.display = listNav.style.display = 'none';
      listView.setAttribute('aria-hidden', 'true');
      listNav.setAttribute('aria-hidden', 'true');
      cardView.style.display = cardNav.style.display = 'none';
      cardView.setAttribute('aria-hidden', 'true');
      cardNav.setAttribute('aria-hidden', 'true');
      readView.style.display = readNav.style.display = 'block';
      readView.setAttribute('aria-hidden', 'false');
      readNav.setAttribute('aria-hidden', 'false');
      break;
    default:
      console.error('Incorrect view mode:', mode);
      return;
  }
  gBackView = gViewMode;
  gViewMode = mode;
}


// Navigation toolbar.
function navKeyHandler(ev) {
  if (ev.keyCode === 13) { // Enter
    navClickHandler(ev);
    return;
  }
  switch (gViewMode) {
    case 'CARD':
      var cardNum = gCurrentNum;
      switch (ev.keyCode) {
        case 37: // ←
          cardNum--;
          if (cardNum < 1) {
            cardNum = 100;
          }
          drawCardView(cardNum);
          break;
        case 39: // →
          cardNum++;
          if (cardNum > 100) {
            cardNum = 1;
          }
          drawCardView(cardNum);
          break;
        case 32: // space
          cardNum = Math.ceil(Math.random() * 100);
          drawCardView(cardNum);
          break;
        default:
          //console.log('keyCode: ', ev.keyCode);
      }
      break;
    case 'LIST':
      switch (ev.keyCode) {
        case 8:  // BackSpace
          switchView(gBackView);
          break;
        default:
          //console.log('keyCode: ', ev.keyCode);
      }
      break;
    case 'READ':
      var counter = gReadCounter;
      switch (ev.keyCode) {
        case 37: // ←
          if (counter > 1) {
            counter--;
          }
          gReadCounter = counter;
          drawReadingView(counter);
          break;
        case 39: // →
        case 32: // space
          if (counter < 100) {
            counter++;
          }
          gReadCounter = counter;
          drawReadingView(counter);
          break;
        default:
          //console.log('keyCode: ', ev.keyCode);
      }
      break;
    default:
  }
  return;
}

function navClickHandler(ev) {
  ev.preventDefault();
  var navItem = ev.target.getAttribute('mode');
  switch (gViewMode) {
    case 'CARD':
      var cardNum = gCurrentNum;
      switch (navItem) {
        case 'prev':
          cardNum--;
          if (cardNum < 1) {
            cardNum = 100;
          }
          drawCardView(cardNum);
          break;
        case 'next':
          cardNum++;
          if (cardNum > 100) {
            cardNum = 1;
          }
          drawCardView(cardNum);
          break;
        case 'list':
          switchListType(gIndexType);
          switchView('LIST');
          break;
        default:
      }
      break;
    case 'LIST':
      if (navItem !== gIndexType) {
        switchListType(navItem);
      }
      break;
    case 'READ':
      var counter = gReadCounter;
      switch (navItem) {
        case 'prev':
          if (counter > 1) {
            counter--;
          }
          break;
        case 'next':
          if (counter < 100) {
            counter++;
          }
          break;
        case 'detail':
          cardNum = gReadingList[counter - 1];
          drawCardView(cardNum);
          switchView('CARD');
          break;
        default:
            //console.log('Clicked target is ' + ev.target.id);
      }
      gReadCounter = counter;
      drawReadingView(counter);
      break;
    default:
  }
}


// Swipe Handler.
var gSwipeDetector = null;
function cardViewSwipeHandler(ev) {
  var swipe_threshold = 50;
  if (Math.abs(ev.dx) < swipe_threshold) {
    return;
  }
  //console.log(gViewMode, ev.direction);
  switch (gViewMode) {
    case 'CARD':
      var cardNum = gCurrentNum;
      switch(ev.direction) {
        case 'left':
          cardNum--;
          if (cardNum < 1) {
            cardNum = 100;
          }
          drawCardView(cardNum);
          break;
        case 'right':
          cardNum++;
          if (cardNum > 100) {
            cardNum = 1;
          }
          drawCardView(cardNum);
          break;
        default:
      }
      break;
    case 'READ':
      var counter = gReadCounter;
      switch (ev.direction) {
        case 'left':
          if (counter > 1) {
            counter--;
          }
          break;
        case 'right':
          if (counter < 100) {
            counter++;
          }
          break;
        default:
      }
      gReadCounter = counter;
      drawReadingView(counter);
      break;
    default:
  }
}

function cardViewKeyHandler(ev) {
  var cardNum = gCurrentNum;
  switch (ev.keyCode) {
    case 37: // ←
      cardNum = (ev.ctrlKey ? cardNum - 10 : cardNum - 1);
      if (cardNum < 1) {
        cardNum = 1;
      }
      drawCardView(cardNum);
      break;
    case 39: // →
      cardNum = (ev.ctrlKey ? cardNum + 10 : cardNum + 1);
      if (cardNum > 100) {
        cardNum = 100;
      }
      drawCardView(cardNum);
      break;
    default:
      //console.log('keyCode: ', ev.keyCode);
  }
  return;
}


function listKeyHandler(ev) {
  ev.preventDefault();

  var cardList = document.getElementById('cardList'),
      li_items = cardList.getElementsByTagName('li'),
      idx = focusManager.get();

  switch (ev.keyCode) {
    case 40: // ↓
      idx = focusManager.nextPos();
      li_items[idx].focus();
      break;
    case 38: // ↑
      idx = focusManager.prevPos();
      li_items[idx].focus();
      break;
    case 13: // Enter
      listItemHandler(ev);
      break;
    /*
    case 8:  // BackSpace
      switchView(gBackView);
      break;
    */
    case 9:  // Tab
      (ev.shiftKey ? navLine : jumpLink).focus();
      break;
    case 32: // space
      var dir = (ev.shiftKey ?  -1 : 1);
      window.scrollBy(0, dir * window.innerHeight);
      break;
    case 33: // pageUp
      idx = focusManager.toPos(focusManager.get() - 10);
      cardList.childNodes[idx].focus();
      break;
    case 34: // pageDown
      idx = focusManager.toPos(focusManager.get() + 10);
      cardList.childNodes[idx].focus();
      break;
    case 35: // End
      idx = focusManager.lastPos();
      cardList.childNodes[idx].focus();
      break;
    case 36: // Home
      idx = focusManager.firstPos();
      cardList.childNodes[idx].focus();
      break;
    default:
      //console.log('keyCode: ', ev.keyCode);
  }
  return;
}

function listItemHandler(ev) {
  ev.preventDefault();
  var targetId = ev.target.id;

  // 'targetId' must be 'pNN'. (NN: 1 - 100)
  if (targetId[0] !== 'p') {
    if (targetId === 'jumpLink') {
      navLine.focus();
    }
    return;
  }
  var cardNum = targetId.slice(1) - 0;
  drawCardView(cardNum);
  gCurrentNum = cardNum;
  switchView('CARD');
}

function jumplinkKeyHandler(ev) {
  return ev.keyCode != 13 || jumplinkClickHandler(ev);
}

function jumplinkClickHandler(ev) {
  ev.preventDefault();
  navLine.focus();
}


function readOptionClickHandler(ev) {
  ev.preventDefault();
  var option = ev.target.getAttribute('option'),
      counter = gReadCounter;

  switch(option) {
    case 'renew':
      L10n.formatValue('shuffle_confirm').then(confirm_text => {
        if (confirm(confirm_text)) {
          gReadingList = shuffleList(100);
          gReadCounter = counter = 1;
          drawReadingView(counter);
        }
      });
      break;
    case 'fake':
      var fake_threshold = 10;
      if (counter >= fake_threshold) {
        var fake = Math.floor(Math.random() * counter);
        drawReadingView(fake);
        //***
        return;
      } else {
        console.log('You have to read ' + (fake_threshold - counter) + ' more cards to fake.');
      }
      drawReadingView(counter);
      break;
    default:
  }
}


// Draw Card view.
function drawCardView(poemNum) {
  var card = {};
  if (!poemNum || poemNum < 1 || poemNum > 100) {
    poemNum = gCurrentNum;
  } else {
    gCurrentNum = poemNum;
  }
  card = gCardData.getCard(poemNum);
  if (!card) {
    console.error('Card data is not defined.');
    return;
  }
  var currentNumBlock = document.getElementById('currentNum'),
      categoryBlock = document.getElementById('category'),
      poetBlock = document.getElementById('poet'),
      poemBlock1 = document.getElementById('poem_first'),
      poemBlock2 = document.getElementById('poem_second'),
      cardImage = document.getElementById('cardImage'),
      poemReadingBlock = document.getElementById('poem_pronounce'),
      poetReadingBlock = document.getElementById('poet_pronounce');

  L10n.setAttributes(currentNumBlock, 'poemNumber', {'n': poemNum});
  poetBlock.textContent = card.poet;
  poetBlock.title = card.poet_kana;
  L10n.setAttributes(categoryBlock, 'category_' + card.poem_category);

  poemBlock1.textContent = card.poem[0];
  poemBlock1.title = card.poem_kana[0];
  poemBlock2.textContent = card.poem[1];
  poemBlock2.title = card.poem_kana[1];
  var [img_posX, img_posY] = gCardData.getCardClipCoordinates(poemNum);
  // clip: rect(t, r, b, l)  : distances from top and left border.
  cardImage.style.clip = 'rect(' +
    img_posY + 'px,' + (img_posX + gCardData.cardSize.w) + 'px,' +
    (img_posY + gCardData.cardSize.h) + 'px,' + img_posX + 'px)';
  cardImage.style.transform = 'translate(' + (-img_posX) + 'px,' + (-img_posY) +'px)';
  poemReadingBlock.innerHTML = card.poem_pronounce[0] + '<br>'+ card.poem_pronounce[1];
  poetReadingBlock.textContent = card.poet_pronounce;
}


// Draw List view.
function drawListView(idxType) {
  var cardData = gCardData.data,
      cardListBlock = document.getElementById('cardList'),
      cardItems = document.createDocumentFragment();

  if (!idxType) {
    idxType = gIndexType;
  }

  if (idxType === 'NUMBER') {
    for (var n = 1; n <= 100; n++) {
      var card = cardData[n+''],
          item = document.createElement('li');
      item.setAttribute('id', 'p'+n);
      item.setAttribute('role', 'option');
      item.setAttribute('tabindex','-1');
      //item.setAttribute('class', 'carditem');
      item.innerHTML = card.poem[0] + ' ' + card.poem[1] + ' —<em>' + card.poet + '</em>';
      item.title = card.poem_kana[0] + ' ' + card.poem_kana[1];
      cardItems.appendChild(item);
    }
    cardListBlock.setAttribute('class', 'decimal');
  } else {
    IndexData[idxType].forEach(function (idxItem, n) {
      var itemHead = idxItem[0],
          itemList = idxItem[1],
          subheader = document.createElement('h3');
      // Set subheader.
      if (idxType === 'CATEGORY') {
        L10n.setAttributes(subheader, 'category_' + itemHead);
      } else {
        subheader.textContent = itemHead;
      }
      subheader.setAttribute('id', 'sub'+n);
      cardItems.appendChild(subheader);
      // Set items.
      itemList.forEach(function (num) {
        var item = document.createElement('li'),
            card = cardData[num+''],
            indexTarget, indexTargetSub, indexTargetTitle;
        switch (idxType) {
          case 'POEM1':
            indexTarget = card.poem_kana[0];
            indexTargetSub = '&rarr; ' + _getFirstWord(card.poem_kana[1]);
            indexTargetTitle = card.poem[0] + ' ' + card.poem[1];
            break;
          case 'POEM2':
            indexTarget = card.poem_kana[1];
            indexTargetSub = '&larr; ' + _getFirstWord(card.poem_kana[0]);
            indexTargetTitle = card.poem[0] + ' ' + card.poem[1];
            break;
          case 'POET':
            indexTarget = '<ruby><rb>' + card.poet + '</rb><rp>（</rp><rt>' + card.poet_kana+ '</rt><rp>)</rp></ruby>';
            indexTargetSub = '&#x300c;' + _getFirstWord(card.poem_kana[0]) + '&#x300d;';
            indexTargetTitle = card.poet_kana;
            break;
          case 'CATEGORY':
            indexTarget = card.poem[0] + ' ' + card.poem[1];
            indexTargetSub = ' —<em>' + card.poet;
            indexTargetTitle = card.poem_kana[0] + ' ' + card.poem_kana[1];
            break;
          default:
            console.error('Incorrect Index type:', idxType);
            return;
        }
        item.setAttribute('id', 'p'+num);
        item.setAttribute('role', 'option');
        item.setAttribute('tabindex','-1');
        //item.setAttribute('class', 'carditem');
        item.innerHTML = indexTarget + ' <em>' + indexTargetSub + '</em>';
        item.title = indexTargetTitle;
        cardItems.appendChild(item);
      });
    });
    cardListBlock.setAttribute('class', 'circle');
  }
  cardListBlock.textContent = '';
  cardListBlock.appendChild(cardItems);

  return;

  function _getFirstWord(str) {
    // "word1 word2 word3" -> "word1…"
    return str.split(' ')[0] + '&hellip;';
  }
}


// Switch index list type.
function switchListType(type) {
  var listViewTitle = document.getElementById('listViewTitle');

  switch (type) {
    case 'NUMBER':
      L10n.setAttributes(listViewTitle, 'indexNumber');
      break;
    case 'POEM1':
      L10n.setAttributes(listViewTitle, 'indexPoem1');
      break;
    case 'POEM2':
      L10n.setAttributes(listViewTitle, 'indexPoem2');
      break;
    case 'POET':
      L10n.setAttributes(listViewTitle, 'indexPoet');
      break;
    case 'CATEGORY':
      L10n.setAttributes(listViewTitle, 'indexCategory');
      break;
    default:
      console.error('Incorrect Index type:', type);
      return;
  }
  drawListView(type);
  
  gIndexType = type;
  return;
}


// Draw Reading view.
function drawReadingView(readCounter) {
  var readCount = document.getElementById('readCount'),
      poemNumBlock = document.getElementById('readNum'),
      poemCard = document.getElementById('readCard'),
      poemRead1 = document.getElementById('poemRead_first'),
      poemRead2 = document.getElementById('poemRead_second'),
      poemTake2 = document.getElementById('poemTake_second');

  if (!readCounter || readCounter < 1) {
    readCounter = gReadCounter;
  }

  var poemNum = gReadingList[readCounter - 1],
      card = gCardData.getCard(poemNum);
  if (!card) {
    console.log('card not found:', poemNum);
    return;
  }

  L10n.setAttributes(readCount, 'readCount', {'n': (gReadCounter)});
  L10n.setAttributes(poemNumBlock, 'poemNumber', {'n': poemNum});
  var [img_posX, img_posY] = gCardData.getCardClipCoordinates(poemNum, gCardData.smallRatio);
  poemCard.style.clip = 'rect(' +
    img_posY + 'px,' + (img_posX + gCardData.cardSize.w * gCardData.smallRatio) + 'px,' +
    (img_posY + gCardData.cardSize.h * gCardData.smallRatio) + 'px,' + img_posX + 'px)';
  poemCard.style.transform = 'translate(' + (-img_posX) + 'px,' + (-img_posY) +'px)';

  poemCard.alt = card.poet;
  poemRead1.textContent = card.poem_yomi[0];
  poemRead1.title = card.poem[0];
  poemRead2.textContent = card.poem_yomi[1];
  poemRead2.title = card.poem[1];
  poemTake2.textContent = card.poem_kana[1];
  poemTake2.title = card.poem[1];

  return;
}


// Shuffle numbers by Fisher-Yates method.
function shuffleList(num) {
  var arr = [];
  for (var i = 0; i < num; i++) {
    arr[i] = i + 1;
  }
  var n = arr.length;
  while(n){
    var j = Math.floor(Math.random() * n),
        t = arr[--n];
    arr[n] = arr[j];
    arr[j] = t;
  }
  return arr;
}


// Load JSON File.
function loadJsonFile() {
  var url = 'Hyakunin-isshu.json';
  return new Promise(function(resolve, reject) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', url, true);
    xhr.responseType = 'json';
    xhr.overrideMimeType('application/json; charset=utf-8');
    xhr.onload = function() {
      if (xhr.readyState === 4) {
        if (xhr.status !== 404 && xhr.response) {
          resolve(xhr.response);
        } else {
          reject(xhr.statusText);
        }
      }
    };
    xhr.send();
  });
}


// Initialize.
function init() {
  loadJsonFile().then(function(cards) {
    try {
      if (Object.keys(cards).length < 1) {
        console.error('No card data.');
        return;
      } else {
        gCardData = new CardData(cards);
        if (cards.notes) {
          console.log(cards.notes.appname, cards.notes.version, cards.notes.modified);
        }
        var cardImage = document.getElementById('cardImage'),
            poemCard = document.getElementById('readCard');
        cardImage.src = poemCard.src = gCardData.imgSrc;
        poemCard.style.width = gCardData.cardSize.w * gCardData.imageTable.cols * gCardData.smallRatio + 'px';
        poemCard.style.height = gCardData.cardSize.h * gCardData.imageTable.rows * gCardData.smallRatio + 'px';
      }
    } catch(e) {
      console.error('ERROR: Loading card data:', e);
    }
    
    var headerButtons = document.getElementById('headerButtons'),
        //selectMenu = document.getElementById('selectMenu'),
        menuButtons = document.getElementById('mainMenu'),
        cardContainer = document.getElementById('cardContainer'),
        cardListBlock = document.getElementById('cardList'),
        readOptions = document.getElementById('readOption');

    // Register click and keydown event handlers.
    headerButtons.addEventListener('click', headerClickHandler, false);
    headerButtons.addEventListener('keydown', headerKeyHandler, false);
    menuButtons.addEventListener('click', menuClickHandler, false);
    menuButtons.addEventListener('keydown', menuKeyHandler, false);
    navLine.addEventListener('click', navClickHandler, false);
    navLine.addEventListener('keydown', navKeyHandler, false);
    cardContainer.addEventListener('keydown', cardViewKeyHandler, false);
    cardListBlock.addEventListener('click', listItemHandler, false);
    cardListBlock.addEventListener('keydown', listKeyHandler, false);
    jumpLink.addEventListener('click', jumplinkClickHandler, false);
    jumpLink.addEventListener('keydown', jumplinkKeyHandler, false);
    readOptions.addEventListener('click', readOptionClickHandler, false);

    // Register swipe handlers by swiper.js.
    var browsingSection = document.getElementById('browsingSection');
    gSwipeDetector = new Swiper(browsingSection);
    gSwipeDetector.start();
    gSwipeDetector.swipeHandler = cardViewSwipeHandler;

    // Register focus event handlers.
    navLine.onfocus = function () {
      var forcusTarget;
      if (gViewMode === 'LIST') {
        forcusTarget = this.getElementsByTagName('span')[(gIndexes.indexOf(gIndexType))];
      } else {
        forcusTarget = this.getElementsByTagName('span')[2];
      }
      forcusTarget.focus();
    };
    cardListBlock.onfocus = function() {
      var idx = focusManager.get(),
          li_items = this.getElementsByTagName('li');
      li_items[idx].focus();
    };

    // Draw initial pages.
    drawListView();
    drawCardView();
    drawReadingView();
  }).catch(function (e) {
    console.error(e);
  });
}

// Terminate.
function quit() {
  var headerButtons = document.getElementById('headerButtons'),
      menuButtons = document.getElementById('mainMenu'),
      cardContainer = document.getElementById('cardContainer'),
      cardListBlock = document.getElementById('cardList'),
      readOptions = document.getElementById('readOption');

  gCardData = null;
  // Unregister event handlers.
  headerButtons.removeEventListener('click', headerClickHandler, false);
  headerButtons.removeEventListener('keydown', headerKeyHandler, false);
  menuButtons.removeEventListener('click', menuClickHandler, false);
  menuButtons.removeEventListener('keydown', menuKeyHandler, false);
  navLine.removeEventListener('click', navClickHandler, false);
  navLine.removeEventListener('keydown', navKeyHandler, false);
  cardContainer.removeEventListener('keydown', cardViewKeyHandler, false);
  cardListBlock.removeEventListener('click', listItemHandler, false);
  cardListBlock.removeEventListener('keydown', listKeyHandler, false);
  jumpLink.removeEventListener('click', jumplinkClickHandler, false);
  jumpLink.removeEventListener('keydown', jumplinkKeyHandler, false);
  readOptions.removeEventListener('click', readOptionClickHandler, false);

  if (gSwipeDetector) {
    gSwipeDetector.stop();
  }

  window.removeEventListener('load', init, false);
}

window.addEventListener('load', init, false);
window.addEventListener('unload', quit, false);



var IndexData = {
  'POEM1': [
    ['あ',[79,1,52,39,31,64,3,78,45,43,44,12,7,56,69,30,58]],
    ['い',[61,21,63]], ['う',[74,65]], ['お',[5,72,60,95,82]],
    ['か',[51,6,98,48]], ['き',[50,15,91]],
    ['こ',[29,68,97,24,41,10]], ['さ',[70]], ['し',[40,37]],
    ['す',[18]], ['せ',[77]], ['た',[73,55,4,16,89,34]],
    ['ち',[75,42,17]], ['つ',[23,13]],
    ['な',[80,84,53,86,36,25,88,19]], ['は',[96,9,2,67]],
    ['ひ',[33,35,99]], ['ふ',[22]], ['ほ',[81]],
    ['み',[49,27,90,14,94]], ['む',[87]], ['め',[57]],
    ['も',[100,66]], ['や',[59,47,32,28]], ['ゆ',[71,46]],
    ['よ',[93,83,85,62]], ['わ',[8,92,38,54,76,11,20]], ['を',[26]]
  ],
  'POEM2': [
    ['あ',[30,71,19,75,93,39,64,21]], ['い',[53,78,70,27,58,56,26]],
    ['う',[82,84]], ['お',[29]], ['か',[72,86,59,67,17]], ['き',[87]],
    ['く',[48,57,36,76]], ['け',[61,54]], ['こ',[68,13,65,91,2,5]],
    ['さ',[51]], ['し',[33,89,10,6]], ['す',[42]], ['た',[81,69]],
    ['つ',[37]], ['と',[73]], ['な',[50,3,32,55,100,52]], ['ぬ',[90]],
    ['ね',[85]], ['は',[74,35,66]], ['ひ',[92,47,41,63,25,11,38,28,44,49]],
    ['ふ',[4,96,94]], ['ま',[60,16,34]], ['み',[7,98,14,80,45,20,88]],
    ['む',[43,22]], ['も',[40,24,79]], ['や',[97,83]], ['ゆ',[46, 18]],
    ['よ',[31,62,8,99]], ['わ',[15,1,95,23,9,77]], ['を',[12]]
  ],
  'POET': [
    ['あ',[59,7,17]], ['い',[56,19,61,90]], ['う',[38,53]],
    ['え',[47]], ['お',[23,29,49,9]],
    ['か',[3,93,14,24]], ['き',[8,54,35,33,36,42]],
    ['け',[45]], ['こ',[88, 15, 83, 91, 60, 81, 99, 43, 64, 97]],
    ['さ',[86,31,65,66,95,73,79,63,5,11,39,94,68,25]],
    ['し',[2,87,98,85,100,89]], ['す',[67,77]],
    ['せ',[62,10]], ['そ',[12,21,46]],
    ['た',[80,55,71,58,40]], ['ち',[44,27,6,16]],
    ['て',[26,1]], ['と',[82]], ['に',[92,96]], ['の',[69]],
    ['は',[32]], ['ふ',[34,84,51,18,52,75,50,37,22]],
    ['ほ',[76]], ['み',[78,48,74,28,41,30]], ['む',[57]], 
    ['も',[20]], ['や',[4]], ['ゆ',[72]], ['よ',[13]], ['り',[70]]
  ],
  'CATEGORY': [
    ['spring',[9,15,33,35,61,73]],
    ['summer',[2,36,81,98]],
    ['autumn',[1,5,17,22,23,26,29,32,37,47,69,70,71,79,87,91,94]],
    ['winter',[4,6,28,31,64,78]],
    ['casual',[8,10,12,34,55,57,60,62,66,67,68,75,76,83,84,95,96,99,100]],
    ['love',[3,13,14,18,19,20,21,25,27,30,38,39,40,41,42,43,44,45,46,48,49,50,
             51,52,53,54,56,58,59,63,65,72,74,77,80,82,85,86,88,89,90,92,97]],
    ['parting',[16]],
    ['travel',[7,11,24,93]]
  ]
};


})(window);
