// Helper functions for WebGL
"use strict";

var mIsLandscape = false;
var gl;
var glViewportWidth;
var glViewportHeight;
var mvMatrix;
var mvMatrixStack = [];
var pMatrix;
var triangleVertexPositionBuffer;
var triangleVertexColorBuffer;
var tessVertexPositionBuffer;
var tessVertexColorBuffer;
var tessVertexIndexBuffer;
var rTri = 0;
var mAttributesPerVertice;
var mStrideBytes;
var mPositionOffset;
var mPositionDataSize;
var mColorOffset;
var mColorDataSize;
var mVerticeCount;
var lastTime = 0;


 function initGL(canvas) {
   var canvasElement = canvas[0];
   try {  
     gl = canvasElement.getContext("webgl");
     glViewportWidth = canvasElement.width;
     glViewportHeight = canvasElement.height;
   } catch(e1) {
   }
   if (!gl) {
   try {
	     gl = canvasElement.getContext('3d', {alpha:false} );
	     glViewportWidth = canvasElement.width;
	     glViewportHeight = canvasElement.height;
	   } catch(e2) {
	   }
   if (!gl) {
	   try {
		     gl = canvasElement.getContext("experimental-webgl", { premultipliedAlpha: false } );
		     glViewportWidth = canvasElement.width;
		     glViewportHeight = canvasElement.height;
		   } catch(e3) {
		   }
   	   }
   }
   
   if (!gl) {
	   // Bar chart will be rendered instead of WebGL - pie chart
   }
   else{
	   initShaders();
       mvMatrix = mat4.create();
       pMatrix = mat4.create();
       gl.clearColor(0.0, 0.0, 0.0, 0.0)   // Last param is alpha
	   gl.enable(gl.DEPTH_TEST);	   
   }
 }

 
 function getShader(gl, id) {
     var shaderScript = document.getElementById(id);
     if (!shaderScript) {
         return null;
     }

     var str = "";
     var k = shaderScript.firstChild;
     while (k) {
         if (k.nodeType == 3)
             str += k.textContent;
         k = k.nextSibling;
     }

     var shader;
     if (shaderScript.type == "x-shader/x-fragment") {
         shader = gl.createShader(gl.FRAGMENT_SHADER);
     } else if (shaderScript.type == "x-shader/x-vertex") {
         shader = gl.createShader(gl.VERTEX_SHADER);
     } else {
         return null;
     }

     gl.shaderSource(shader, str);
     gl.compileShader(shader);

     if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {
         alert(gl.getShaderInfoLog(shader));
         return null;
     }

     return shader;
 }
 

 var shaderProgram;
 
 function mvPushMatrix() {
     var copy = mat4.create();
     mat4.set(mvMatrix, copy);
     mvMatrixStack.push(copy);
 }

 function mvPopMatrix() {
     if (mvMatrixStack.length == 0) {
         throw "Invalid popMatrix!";
     }
     mvMatrix = mvMatrixStack.pop();
 }

 function setMatrixUniforms() {
    gl.uniformMatrix4fv(shaderProgram.pMatrixUniform, false, pMatrix);
    gl.uniformMatrix4fv(shaderProgram.mvMatrixUniform, false, mvMatrix);
 }

 function degToRad(degrees) {
    return degrees * Math.PI / 180;
 }
  
 function initShaders() {
	 
     var fragmentShader = getShader(gl, "shader-fs");
     var vertexShader = getShader(gl, "shader-vs");

     shaderProgram = gl.createProgram();
     gl.attachShader(shaderProgram, vertexShader);
     gl.attachShader(shaderProgram, fragmentShader);
     gl.linkProgram(shaderProgram);

     if (!gl.getProgramParameter(shaderProgram, gl.LINK_STATUS)) {
         alert("Could not initialise shaders");
     }

     gl.useProgram(shaderProgram);

     shaderProgram.vertexPositionAttribute = gl.getAttribLocation(shaderProgram, "aVertexPosition");
     gl.enableVertexAttribArray(shaderProgram.vertexPositionAttribute);

     shaderProgram.vertexColorAttribute = gl.getAttribLocation(shaderProgram, "aVertexColor");
     gl.enableVertexAttribArray(shaderProgram.vertexColorAttribute);
     
     shaderProgram.pMatrixUniform = gl.getUniformLocation(shaderProgram, "uPMatrix");
     shaderProgram.mvMatrixUniform = gl.getUniformLocation(shaderProgram, "uMVMatrix");
 } 
 
 function initBuffers( ratioSection1, ratioSection2 ) {

     // Vertex (triangle)	 
	 triangleVertexPositionBuffer = gl.createBuffer();
     gl.bindBuffer(gl.ARRAY_BUFFER, triangleVertexPositionBuffer);

     var vertices = setPieData( false, ratioSection1, ratioSection2 );
     triangleVertexPositionBuffer.itemSize = mPositionDataSize;
     triangleVertexPositionBuffer.numItems = (4*mVerticeCount*mPositionDataSize)/3;  // four triangles per three points
     gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(vertices), gl.STATIC_DRAW);

     // Colors (triangle)
     triangleVertexColorBuffer = gl.createBuffer();
     gl.bindBuffer(gl.ARRAY_BUFFER, triangleVertexColorBuffer);
   	 var colors = setPieData( true, ratioSection1, ratioSection2 );
     triangleVertexColorBuffer.itemSize = 4;
     triangleVertexColorBuffer.numItems = (4*mVerticeCount*4)/3;  // four triangles per three points
     gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(colors), gl.STATIC_DRAW);     
 }
  
 function drawChart() {
	 if(glViewportWidth>glViewportHeight){
		 mIsLandscape = true;		 
	 }
	 else{
		 mIsLandscape = false;		 
	 }
	 
     gl.viewport(0, 0, glViewportWidth, glViewportHeight);
     gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

     var ratio = 1
//     ratio = glViewportWidth / glViewportHeight;
     mat4.perspective(50, ratio, 0.1, 20.0, pMatrix);

     mat4.identity(mvMatrix);

     // Set perspective first, afterwards adjust result via translation
     mat4.translate(mvMatrix, [ 0, 0, -8.0]);
     
     mvPushMatrix();
     mat4.rotate(mvMatrix, degToRad(rTri), [1,1,1]);

     // Vertex (triangle)
     gl.bindBuffer(gl.ARRAY_BUFFER, triangleVertexPositionBuffer);
     gl.vertexAttribPointer(shaderProgram.vertexPositionAttribute, triangleVertexPositionBuffer.itemSize, gl.FLOAT, false, 0, 0);
     // Colors (triangle)
     gl.bindBuffer(gl.ARRAY_BUFFER, triangleVertexColorBuffer);
     gl.vertexAttribPointer(shaderProgram.vertexColorAttribute, triangleVertexColorBuffer.itemSize, gl.FLOAT, false, 0, 0);
     
     setMatrixUniforms();
     gl.drawArrays(gl.TRIANGLES, 0, triangleVertexPositionBuffer.numItems);
  
     mvPopMatrix();
 }
 
 function setPieData( getColor, ratioSection1, ratioSection2 ){
	    var verticeOrColorData=[];
	    // Circle
	    if(getColor){
			mAttributesPerVertice = 7;	    	
	    }
	    else{
			mAttributesPerVertice = 3;	    	
	    }
    	mPositionOffset = 0;
    	mPositionDataSize = 3;
    	mColorOffset = 3;
    	mColorDataSize = 4;
    	mVerticeCount = 360;
    	verticeOrColorData = MakePie2d(mVerticeCount, 0.0, 0.0, 0.0, 1.0, ratioSection1, ratioSection2, getColor);
    	
        return verticeOrColorData;
 }

 
 function MakePie2d(points, x, y, z, alpha, ratioSection1, ratioSection2, getColor)
 {
    var vertexCount=0;
    var radius;

	var rad;
	var red=0.0;
	var green=0.0;
	var blue=0.0;
	var redBright=0.0;
	var greenBright=0.0;
	var blueBright=0.0;
	var brightDelta=24.0;
    var verts = [];
    var c=0;
    var zLevel1;
    var zLevel2;
    
    if(mIsLandscape){
 	   radius=2.7;
       zLevel1=z;
       zLevel2=z+0.60;
    }
    else{
 	   radius=2.7;    	   
       zLevel1=z;
       zLevel2=z+0.85;
    }
    
    // side triangles
    // - side triangle top (level2)
    for (var i = 0; i < points; i+=3){    // three because one section is made of three points
        if((i*1.00)/(points*1.00)<ratioSection1){
     	   red=3.00/255.00;
     	   green=168.00/255.00;
     	   blue=158.00/255.00;
        }
        else if((i*1.00)/(points*1.00)<ratioSection2){
     	   red=230.0/255.00;
     	   green=165.0/255.00;
     	   blue=175.0/255.00;
        }
        else{
     	   red=165.0/255.00;
     	   green=165.0/255.00;
     	   blue=165.0/255.00;
        }
        redBright=red+brightDelta/255.00;
        greenBright=green+brightDelta/255.00;
        blueBright=blue+brightDelta/255.00;
        c=vertexCount;
        if(getColor){
        	verts[c]=red;    // Red
        	verts[c+1]=green;  // Green
        	verts[c+2]=blue;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
     	    rad=(i*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.0 + zLevel1;                    // Z        	
            vertexCount+=mAttributesPerVertice;
        }
        
        c=vertexCount;
        if(getColor){
        	verts[c]=red;    // Red
        	verts[c+1]=green;  // Green
        	verts[c+2]=blue;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
            rad=((i+3)*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.0 + zLevel1;                    // Z        	
            vertexCount+=mAttributesPerVertice;
        }

        c=vertexCount;
        if(getColor){
        	verts[c]=redBright;    // Red
        	verts[c+1]=greenBright;  // Green
        	verts[c+2]=blueBright;   // Blue
        	verts[c+3]=alpha;        // Alpha
            vertexCount+=4;
        }
        else{
            rad=((i+0)*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.0 + zLevel2;   // Z        	
            vertexCount+=mAttributesPerVertice;           
        }
    }
    // - side triangle bottom (level1)
    for (var i = 0; i < points; i+=3){    // three because one section is made of three points
        if((i*100.00)/(points*1.00)<ratioSection1*100.00){
     	   red=3.00/255.00;
     	   green=168.00/255.00;
     	   blue=158.00/255.00;
        }
        else if((i*100.00)/(points*1.00)<ratioSection2*100.00){
     	   red=230.0/255.00;
     	   green=165.0/255.00;
     	   blue=175.0/255.00;
        }
        else{
     	   red=165.0/255.00;
     	   green=165.0/255.00;
     	   blue=165.0/255.00;
        }
        redBright=red+brightDelta/255.00;
        greenBright=green+brightDelta/255.00;
        blueBright=blue+brightDelta/255.00;
        c=vertexCount;
        if(getColor){
        	verts[c]=redBright;    // Red
        	verts[c+1]=greenBright;  // Green
        	verts[c+2]=blueBright;   // Blue
        	verts[c+3]=alpha;        // Alpha
            vertexCount+=4;
        }
        else{
     	    rad=(i*360.00/(points*1.00))*(2.0*3.141592626/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.0 + zLevel2;                    // Z        	
            vertexCount+=mAttributesPerVertice;
        }
        
        c=vertexCount;
        if(getColor){
        	verts[c]=redBright;    // Red
        	verts[c+1]=greenBright;  // Green
        	verts[c+2]=blueBright;   // Blue
        	verts[c+3]=alpha;        // Alpha
            vertexCount+=4;
        }
        else{
            rad=((i+3)*360.00/(points*1.00))*(2.0*3.141592626/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.0 + zLevel2;                    // Z        	
            vertexCount+=mAttributesPerVertice;
        }

        c=vertexCount;
        if(getColor){
        	verts[c]=red;    // Red
        	verts[c+1]=green;  // Green
        	verts[c+2]=blue;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
            rad=((i+3)*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.0 + zLevel1;   // Z
            vertexCount+=mAttributesPerVertice;        	
        }
    }
    
    // top triangles
    for (var i = 0; i < points; i+=3)    // three because one section is made of three points
    {
        // top triangle
        if((i*100.00)/(points*1.00)<ratioSection1*100.00){
     	   red=102.00/255.00;
     	   green=205.00/255.00;
     	   blue=170.00/255.00;
        }
        else if((i*100.00)/(points*1.00)<ratioSection2*100.00){
     	   red=255.00/255.00;
     	   green=182.00/255.00;
     	   blue=193.00/255.00;
        }
        else{
     	   red=180.00/255.00;
     	   green=180.00/255.00;
     	   blue=180.00/255.00;
        }
        red=red-0.25*brightDelta/255.00;
        green=green-0.25*brightDelta/255.00;
        blue=blue-0.25*brightDelta/255.00;
        redBright=red+brightDelta/255.00;
        greenBright=green+brightDelta/255.00;
        blueBright=blue+brightDelta/255.00;

        c=vertexCount;
        if(getColor){
        	verts[c]=redBright;    // Red
        	verts[c+1]=greenBright;  // Green
        	verts[c+2]=blueBright;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
     	    rad=(i*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.00 + zLevel2;                    // Z
            vertexCount+=mAttributesPerVertice;
        }
        
        c=vertexCount;
        if(getColor){
        	verts[c]=redBright;    // Red
        	verts[c+1]=greenBright;  // Green
        	verts[c+2]=blueBright;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
            rad=((i+3)*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.00 + zLevel2;                   // Z
            vertexCount+=mAttributesPerVertice;
        }
 
        // center
        c=vertexCount;
        if(getColor){
        	verts[c]=red;    // Red
        	verts[c+1]=green;  // Green
        	verts[c+2]=blue;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
            verts[c]  =0.00 + x;   // X
            verts[c+1]=0.00 + y;   // Y
            verts[c+2]=0.00 + zLevel2;    // Z        	
            vertexCount+=mAttributesPerVertice;
        }
    }

    // bottom triangles
    for (var i = 0; i < points; i+=3)    // three because one section is made of three points
    {
        // bottom triangle
        if((i*100.00)/(points*1.00)<ratioSection1*100.00){
     	   red=102.00/255.00;
     	   green=205.00/255.00;
     	   blue=170.00/255.00;
        }
        else if((i*100.00)/(points*1.00)<ratioSection2*100.00){
     	   red=255.00/255.00;
     	   green=182.00/255.00;
     	   blue=193.00/255.00;
        }
        else{
     	   red=180.00/255.00;
     	   green=180.00/255.00;
     	   blue=180.00/255.00;
        }
        red=red-0.25*brightDelta/255.00;
        green=green-0.25*brightDelta/255.00;
        blue=blue-0.25*brightDelta/255.00;
        redBright=red+brightDelta/255.00;
        greenBright=green+brightDelta/255.00;
        blueBright=blue+brightDelta/255.00;

        c=vertexCount;
        if(getColor){
        	verts[c]=redBright;    // Red
        	verts[c+1]=greenBright;  // Green
        	verts[c+2]=blueBright;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
     	    rad=(i*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.00 + zLevel1;                    // Z
            vertexCount+=mAttributesPerVertice;
        }
        
        c=vertexCount;
        if(getColor){
        	verts[c]=redBright;    // Red
        	verts[c+1]=greenBright;  // Green
        	verts[c+2]=blueBright;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
            rad=((i+3)*360.00/(points*1.00))*(2.00*3.1415926/360.0);
            verts[c]  = (Math.cos(rad) * radius +x);   // X
            verts[c+1]= (Math.sin(rad) * radius +y);   // Y
            verts[c+2]=0.00 + zLevel1;                   // Z
            vertexCount+=mAttributesPerVertice;
        }
 
        // center
        c=vertexCount;
        if(getColor){
        	verts[c]=red;    // Red
        	verts[c+1]=green;  // Green
        	verts[c+2]=blue;   // Blue
        	verts[c+3]=alpha;  // Alpha
            vertexCount+=4;
        }
        else{
            verts[c]  =0.00 + x;   // X
            verts[c+1]=0.00 + y;   // Y
            verts[c+2]=0.00 + zLevel1;    // Z        	
            vertexCount+=mAttributesPerVertice;
        }
    }

    return verts;
 }
  
 function animate() {
     var timeNow = new Date().getTime();
     if (lastTime != 0) {
         var elapsed = timeNow - lastTime;

         rTri += (90 * elapsed) / 2800.0;
     }
     lastTime = timeNow;
 }

 function tick() {
     requestAnimFrame(tick);
     drawChart();
     animate();
 }

