'use strict';

var jorteUtils = (function() {

    function string2array(val) {
        var output = new Array();
        var kv = val.split('&');
        for (var i = 0; i < kv.length; i++) {
            var key = kv[i].split('=');
            output[key[0]] = key[1];
        }
        return output;
    }

    var htmlEntities = /[&<>"'\/]/g;
    var htmlEscapes = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#x27;',
        '/': '&#x2F;'
    };

    function replacer(match) {
        return htmlEscapes[match];
    }

    function htmlEscape(str) {
        return ('' + str).replace(htmlEntities, replacer);
    }

    var HOUR_MS = 3600000;
    function fixDate(d, check) { // force d to be on check's YMD, for daylight savings purposes
        if (+d) { // prevent infinite looping on invalid dates
            while (d.getDate() !== check.getDate()) {
                d.setTime(+d + (d < check ? 1 : -1) * HOUR_MS);
            }
        }
    }

    function parseISO8601(s, ignoreTimezone) { // ignoreTimezone defaults to false
        // derived from http://delete.me.uk/2005/03/iso8601.html
        // TODO: for a know glitch/feature, read tests/issue_206_parseDate_dst.html
        var m = s.match(/^([0-9]{4})(-([0-9]{2})(-([0-9]{2})([T ]([0-9]{2}):([0-9]{2})(:([0-9]{2})(\.([0-9]+))?)?(Z|(([-+])([0-9]{2})(:?([0-9]{2}))?))?)?)?)?$/);
        if (!m) {
            return null;
        }
        var date = new Date(m[1], 0, 1);
        if (ignoreTimezone || !m[13]) {
            var check = new Date(m[1], 0, 1, 9, 0);
            if (m[3]) {
                date.setMonth(m[3] - 1);
                check.setMonth(m[3] - 1);
            }
            if (m[5]) {
                date.setDate(m[5]);
                check.setDate(m[5]);
            }
            fixDate(date, check);
            if (m[7]) {
                date.setHours(m[7]);
            }
            if (m[8]) {
                date.setMinutes(m[8]);
            }
            if (m[10]) {
                date.setSeconds(m[10]);
            }
            if (m[12]) {
                date.setMilliseconds(Number("0." + m[12]) * 1000);
            }
            fixDate(date, check);
        } else {
            date.setUTCFullYear(
                    m[1],
                    m[3] ? m[3] - 1 : 0,
                    m[5] || 1
                    );
            date.setUTCHours(
                    m[7] || 0,
                    m[8] || 0,
                    m[10] || 0,
                    m[12] ? Number("0." + m[12]) * 1000 : 0
                    );
            if (m[14]) {
                var offset = Number(m[16]) * 60 + (m[18] ? Number(m[18]) : 0);
                offset *= m[15] === '-' ? 1 : -1;
                date = new Date(+date + (offset * 60 * 1000));
            }
        }
        return date;
    }

    function clearTime(d) {
        d.setHours(0);
        d.setMinutes(0);
        d.setSeconds(0);
        d.setMilliseconds(0);
        return d;
    }

    function cloneDate(d, dontKeepTime) {
        if (dontKeepTime) {
            return clearTime(new Date(+d));
        }
        return new Date(+d);
    }

    function addDays(d, n, keepTime) { // deals with daylight savings
        if (+d) {
            var dd = d.getDate() + n;
            var check = cloneDate(d);
            check.setHours(9); // set to middle of day
            check.setDate(dd);
            d.setDate(dd);
            if (!keepTime) {
                clearTime(d);
            }
            fixDate(d, check);
        }
        return d;
    }

    function zeroPad(n) {
        return (n < 10 ? '0' : '') + n;
    }

    function iso8601Week(date) {
        var time;
        var checkDate = new Date(date.getTime());

        // Find Thursday of this week starting on Monday
        checkDate.setDate(checkDate.getDate() + 4 - (checkDate.getDay() || 7));

        time = checkDate.getTime();
        checkDate.setMonth(0); // Compare with Jan 1
        checkDate.setDate(1);
        return Math.floor(Math.round((time - checkDate) / 86400000) / 7) + 1;
    }

    var dateFormatters = {
        s: function(d) {
            return d.getSeconds();
        },
        ss: function(d) {
            return zeroPad(d.getSeconds());
        },
        m: function(d) {
            return d.getMinutes();
        },
        mm: function(d) {
            return zeroPad(d.getMinutes());
        },
        h: function(d) {
            return d.getHours() % 12 || 12;
        },
        hh: function(d) {
            return zeroPad(d.getHours() % 12 || 12);
        },
        H: function(d) {
            return d.getHours();
        },
        HH: function(d) {
            return zeroPad(d.getHours());
        },
        d: function(d) {
            return d.getDate();
        },
        dd: function(d) {
            return zeroPad(d.getDate());
        },
        ddd: function(d, o) {
            return o.dayNamesShort[d.getDay()];
        },
        dddd: function(d, o) {
            return o.dayNames[d.getDay()];
        },
        M: function(d) {
            return d.getMonth() + 1;
        },
        MM: function(d) {
            return zeroPad(d.getMonth() + 1);
        },
        MMM: function(d, o) {
            return o.monthNamesShort[d.getMonth()];
        },
        MMMM: function(d, o) {
            return o.monthNames[d.getMonth()];
        },
        yy: function(d) {
            return (d.getFullYear() + '').substring(2);
        },
        yyyy: function(d) {
            return d.getFullYear();
        },
        t: function(d) {
            return d.getHours() < 12 ? 'a' : 'p';
        },
        tt: function(d) {
            return d.getHours() < 12 ? 'am' : 'pm';
        },
        T: function(d) {
            return d.getHours() < 12 ? 'A' : 'P';
        },
        TT: function(d) {
            return d.getHours() < 12 ? 'AM' : 'PM';
        },
        u: function(d) {
            return formatDate(d, "yyyy-MM-dd'T'HH:mm:ss'Z'");
        },
        S: function(d) {
            var date = d.getDate();
            if (date > 10 && date < 20) {
                return 'th';
            }
            return ['st', 'nd', 'rd'][date % 10 - 1] || 'th';
        },
        w: function(d, o) { // local
            return o.weekNumberCalculation(d);
        },
        W: function(d) { // ISO
            return iso8601Week(d);
        },
        WW: function(d) { // ISO
            return zeroPad(iso8601Week(d));
        }
    };
    var defaults = {
        // display
        defaultView: 'month',
        aspectRatio: 1.35,
        header: {
            left: 'title',
            center: '',
            right: 'today prev,next'
        },
        weekends: true,
        weekNumbers: false,
        weekNumberCalculation: 'iso',
        weekNumberTitle: 'W',
        allDayDefault: true,
        ignoreTimezone: true,
        // event ajax
        lazyFetching: true,
        startParam: 'start',
        endParam: 'end',
        // time formats
        titleFormat: {
            month: 'MMMM yyyy',
            week: "MMM d[ yyyy]{ '&#8212;'[ MMM] d yyyy}",
            day: 'dddd, MMM d, yyyy'
        },
        columnFormat: {
            month: 'ddd',
            week: 'ddd M/d',
            day: 'dddd M/d'
        },
        timeFormat: {// for event elements
            '': 'h(:mm)t' // default
        },
        // locale
        isRTL: false,
        firstDay: 0,
        monthNames: ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'],
        monthNamesShort: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
        dayNames: ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'],
        dayNamesShort: ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],
        buttonText: {
            prev: "<span class='fc-text-arrow'>&lsaquo;</span>",
            next: "<span class='fc-text-arrow'>&rsaquo;</span>",
            prevYear: "<span class='fc-text-arrow'>&laquo;</span>",
            nextYear: "<span class='fc-text-arrow'>&raquo;</span>",
            today: 'today',
            month: 'month',
            week: 'week',
            day: 'day'
        },
        // jquery-ui theming
        theme: false,
        buttonIcons: {
            prev: 'circle-triangle-w',
            next: 'circle-triangle-e'
        },
        //selectable: false,
        unselectAuto: true,
        dropAccept: '*'
    };
    function formatDate(date, format, options) {
        return formatDates(date, null, format, options);
    }
    function formatDates(date1, date2, format, options) {
        options = options || defaults;
        var date = date1,
                otherDate = date2,
                i, len = format.length, c,
                i2, formatter,
                res = '';
        for (i = 0; i < len; i++) {
            c = format.charAt(i);
            if (c === "'") {
                for (i2 = i + 1; i2 < len; i2++) {
                    if (format.charAt(i2) === "'") {
                        if (date) {
                            if (i2 === i + 1) {
                                res += "'";
                            } else {
                                res += format.substring(i + 1, i2);
                            }
                            i = i2;
                        }
                        break;
                    }
                }
            }
            else if (c === '(') {
                for (i2 = i + 1; i2 < len; i2++) {
                    if (format.charAt(i2) === ')') {
                        var subres = formatDate(date, format.substring(i + 1, i2), options);
                        if (parseInt(subres.replace(/\D/, ''), 10)) {
                            res += subres;
                        }
                        i = i2;
                        break;
                    }
                }
            }
            else if (c === '[') {
                for (i2 = i + 1; i2 < len; i2++) {
                    if (format.charAt(i2) === ']') {
                        var subformat = format.substring(i + 1, i2);
                        var subres = formatDate(date, subformat, options);
                        if (subres !== formatDate(otherDate, subformat, options)) {
                            res += subres;
                        }
                        i = i2;
                        break;
                    }
                }
            }
            else if (c === '{') {
                date = date2;
                otherDate = date1;
            }
            else if (c === '}') {
                date = date1;
                otherDate = date2;
            }
            else {
                for (i2 = len; i2 > i; i2--) {
                    var formatter = dateFormatters[format.substring(i, i2)];
                    if (formatter) {
                        if (date) {
                            res += formatter(date, options);
                        }
                        i = i2 - 1;
                        break;
                    }
                }
                if (i2 === i) {
                    if (date) {
                        res += c;
                    }
                }
            }
        }
        return res;
    }

    return {
        /*
         * To get url hash key value
         */
        getUrlHash: function(key) {
            var hash = document.URL.split('#')[1];
            return hash ? ((key ? (string2array(hash))[key] : null)) : null;
        },
        /*
         * Html escaping
         */
        htmlEscape: htmlEscape,
        /*
         * To add days
         */
        addDays: addDays,
        /*
         * Parse date in ISO standard
         */
        parseISO8601: parseISO8601,
        /*
         * To get formatted date
         */
        formatDate: formatDate,
        /*
         * To clone date
         */
        cloneDate: cloneDate,
        /*
         * To clear date
         */
        clearTime: clearTime,
        /*
         * To add year
         */
        addYears: function(d, n, keepTime) {
            d.setFullYear(d.getFullYear() + n);
            if (!keepTime) {
                clearTime(d);
            }
            return d;
        },
        /*
         * To add month
         */
        addMonths: function(d, n, keepTime) { // prevents day overflow/underflow
            if (+d) { // prevent infinite looping on invalid dates
                var m = d.getMonth() + n;
                var check = cloneDate(d);
                check.setDate(1);
                check.setMonth(m);
                d.setMonth(m);
                if (!keepTime) {
                    clearTime(d);
                }
                while (d.getMonth() !== check.getMonth()) {
                    d.setDate(d.getDate() + (d < check ? 1 : -1));
                }
            }
            return d;
        },
        /*
         * To get UTC date to saveein to DB
         */
        getSqlDateUTC: function(date) {
            var jDate;
            switch ($.type(date)) {
                case 'string':
                case 'number':
                    jDate = new Date(date);
                    break;
                case 'date':
                    jDate = date;
                    break;
                default:
                    jDate = new Date();
            }
            jDate.setMilliseconds(0);
            return jDate.getTime();
        },
        
        toRFP3339: function (d) {
            d = typeof d === 'object' ? d : new Date(d || null);
            function pad(n) {
                return n < 10 ? '0' + n : n;
            }
            return d.getUTCFullYear() + '-' + pad(d.getUTCMonth() + 1) + '-' + pad(d.getUTCDate()) + 'T' + pad(d.getUTCHours()) + ':' + pad(d.getUTCMinutes()) + ':' + pad(d.getUTCSeconds()) + 'Z';
        },
        _toRFP3339: function (d) {
            d = typeof d === 'object' ? d : new Date(d || null);
            function pad(n) {
                return n < 10 ? '0' + n : n;
            }
            return d.getFullYear() + '-' + pad(d.getMonth() + 1) + '-' + pad(d.getDate()) + 'T' + pad(d.getHours()) + ':' + pad(d.getMinutes()) + ':' + pad(d.getSeconds()) + '.000Z';
        },
        toDateOnly: function (d) {
            d = typeof d === 'object' ? d : new Date(d || null);
            function pad(n) {
                return n < 10 ? '0' + n : n;
            }
            return d.getFullYear() + '-' + pad(d.getMonth() + 1) + '-' + pad(d.getDate());
        },
        _toDateOnly: function (d) {
            d = typeof d === 'object' ? d : new Date(d || null);
            function pad(n) {
                return n < 10 ? '0' + n : n;
            }
            return d.getFullYear() + '-' + pad(d.getMonth() + 1) + '-' + pad(d.getDate() + 1);
        },
        toDateGlobal: function (d) {
            d = typeof d === 'object' ? d : new Date(d || null);
            function pad(n) {
                return n < 10 ? '0' + n : n;
            }
            return d.getFullYear() + '' + pad(d.getMonth() + 1) + '' + pad(d.getDate());
        },
        toRFP3339Global: function (d) {
            d = typeof d === 'object' ? d : new Date(d || null);
            function pad(n) {
                return n < 10 ? '0' + n : n;
            }
            return d.getUTCFullYear() + '' + pad(d.getUTCMonth() + 1) + '' + pad(d.getUTCDate()) + 'T' + pad(d.getUTCHours()) + '' + pad(d.getUTCMinutes()) + '' + pad(d.getUTCSeconds()) + 'Z';
        },
        /*
         * To set day head
         */
        setDayHead: function(elm, day, mustClass) {
            if (day === undefined) {
                $(elm).attr('class', 'header_todo ' + mustClass);
            } else if (day.getDay() === 0) {
                $(elm).attr('class', 'fc-day-header fc-sun ' + mustClass);
            } else if (day.getDay() === 6) {
                $(elm).attr('class', 'fc-day-header fc-sat ' + mustClass);
            } else {
                $(elm).attr('class', 'fc-day-header ' + mustClass);
            }
        },
        /*
         * To convert date format based on the passing format
         */
        TimeConvertOption: function(event_date, date_format) {
            var converted_time = '';
            if (event_date) {
                var date_obj = new Date(event_date);
                converted_time = formatDate(date_obj, date_format);
            }
            return converted_time;
        },
        /*
         * To compare between two dates
         */
        compare: function(item1, item2, prop) {
            if (!item1[prop] && item2[prop]) {
                return -1;
            } else if (item1[prop] && !item2[prop]) {
                return 1;
            } else if (item1[prop] === item2[prop]) {
                return 0;
            } else if (item1[prop] < item2[prop]) {
                return -1;
            } else if (item1[prop] > item2[prop]) {
                return 1;
            }
        },
        /*
         * To sort events/tasks and event calendar
         */
        sortEvents: function(events) {
            var dispOrder = jorteConfig.getEventDispOrder();
            if (dispOrder === 0) {//order by start-date-time
                for (var i = 0; i < events.length; i++) {
                    for (var j = 0; j < events.length - i - 1; j++) {
                        if (jorteUtils.compare(events[j], events[j + 1], 'begin') > 0) {
                            var temp = events[j];
                            events[j] = events[j + 1];
                            events[j + 1] = temp;
                        } else if (events[j].begin === events[j + 1].begin) {
                            if (jorteUtils.compare(events[j], events[j + 1], 'id') > 0) {
                                var temp = events[j];
                                events[j] = events[j + 1];
                                events[j + 1] = temp;
                            }
                        }

                        if (jorteUtils.compare(events[j], events[j + 1], 'all_day') < 0) {
                            var temp = events[j];
                            events[j] = events[j + 1];
                            events[j + 1] = temp;
                        }

                    }
                }
            } else { //order by importance, date time
                for (var i = 0; i < events.length; i++) {
                    for (var j = 0; j < events.length - i - 1; j++) {

                        if (jorteUtils.compare(events[j], events[j + 1], 'begin') > 0) {
                            var temp = events[j];
                            events[j] = events[j + 1];
                            events[j + 1] = temp;
                        } else if (events[j].begin === events[j + 1].begin) {
                            if (jorteUtils.compare(events[j], events[j + 1], 'id') > 0) {
                                var temp = events[j];
                                events[j] = events[j + 1];
                                events[j + 1] = temp;
                            }
                        }

                        if (jorteUtils.compare(events[j], events[j + 1], 'all_day') < 0) {
                            var temp = events[j];
                            events[j] = events[j + 1];
                            events[j + 1] = temp;
                        } else if (events[j].all_day === events[j + 1].all_day) {
                            if (jorteUtils.compare(events[j], events[j + 1], 'importance') < 0) {
                                var temp = events[j];
                                events[j] = events[j + 1];
                                events[j + 1] = temp;
                            }
                        }
                    }
                }
            }
        },
        sortTasks: function(res) {
            for (var i = 0; i < res.length; i++) {
                for (var j = 0; j < res.length - 1; j++) {
                    if (jorteUtils.compare(res[j], res[j + 1], 'importance') < 0) {
                        var t = res[j];
                        res[j] = res[j + 1];
                        res[j + 1] = t;
                    }
                    if (jorteUtils.compare(res[j], res[j + 1], 'start') > 0) {
                        var t = res[j];
                        res[j] = res[j + 1];
                        res[j + 1] = t;
                    } else if (res[j].start === res[j + 1].start) {
                        if ((res[j].start_time > res[j + 1].start_time && res[j + 1].start_time)|| (!res[j].start_time && res[j+1].start_time)) {
                            var t = res[j];
                            res[j] = res[j + 1];
                            res[j + 1] = t;
                        } else if (res[j].start_time === res[j + 1].start_time) {
                            if (jorteUtils.compare(res[j], res[j + 1], 'end') > 0) {
                                var t = res[j];
                                res[j] = res[j + 1];
                                res[j + 1] = t;
                            } else if (res[j].end === res[j + 1].end) {
                                if ((res[j].due_time > res[j + 1].due_time && res[j+1].due_time )|| (!res[j].due_time && res[j+1].due_time)) {
                                    var t = res[j];
                                    res[j] = res[j + 1];
                                    res[j + 1] = t;
                                }
                            }
                        }
                    }

                    if (jorteUtils.compare(res[j], res[j + 1], 'completed') > 0) {
                        var t = res[j];
                        res[j] = res[j + 1];
                        res[j + 1] = t;
                    }

                }
            }
        },
        splitArray: function(list, chunk) {
            var p, q, temparray;
            var newArray = [];

            for (p = 0, q = list.length; p < q; p += chunk) {
                temparray = list.slice(p, p + chunk);
                newArray.push(temparray);
            }
            return newArray;
        },
        sortEventsByDate: function(events) {
            for (var i = 0; i < events.length; i++) {
                for (var j = 0; j < events.length - i - 1; j++) {
                    if (events[j].all_day < events[j + 1].all_day) {
                        var temp = events[j + 1];
                        events[j + 1] = events[j];
                        events[j] = temp;
                    }
                    if (events[j].begin > events[j + 1].begin) {
                        var temp = events[j];
                        events[j] = events[j + 1];
                        events[j + 1] = temp;
                    } else if (events[j].begin === events[j + 1].begin) {
                        if (events[j].id > events[j + 1].id && events[j+1].all_day===1) {
                            var temp = events[j];
                            events[j] = events[j + 1];
                            events[j + 1] = temp;
                        }
                    }

                }
            }
        },
        sortById: function(tasks) {
            for (var i = 0; i < tasks.length; i++) {
                for (var j = 0; j < tasks.length - i - 1; j++) {
                    if (jorteUtils.compare(tasks[j], tasks[j + 1], 'id') < 0) {
                        var temp = tasks[j];
                        tasks[j] = tasks[j + 1];
                        tasks[j + 1] = temp;
                    }
                    if (jorteUtils.compare(tasks[j], tasks[j + 1], 'seq') > 0) {
                        var t = tasks[j];
                        tasks[j] = tasks[j + 1];
                        tasks[j + 1] = t;
                    }
                }
            }
        },
        parseMinutes: function(time) {
            var reminder;
            var format_time = parseInt(time / 60);
            if (format_time < 10) {
                format_time = "0" + format_time;
            }
            reminder = parseInt((time % 60));
            reminder < 10 ? format_time += ":0" + reminder : format_time += ":" + reminder;
            return format_time;
        },
        parseDateString: function(s) {
            s = typeof s === 'number' ? s.toString() : s;
            var m = s.match(/^([0-9]{4})(([0-9]{2})(([0-9]{2})(([0-9]{2})([0-9]{2})(([0-9]{2})(\.([0-9]+))?)?)?)?)?$/);
            if (!m) {
                return null;
            }
            var date = new Date(m[1], 0, 1);
            var check = new Date(m[1], 0, 1, 9, 0);
            if (m[3]) {
                date.setMonth(m[3] - 1);
                check.setMonth(m[3] - 1);
            }
            if (m[5]) {
                date.setDate(m[5]);
                check.setDate(m[5]);
            }
            fixDate(date, check);
            if (m[7]) {
                date.setHours(m[7]);
            }
            if (m[8]) {
                date.setMinutes(m[8]);
            }
            if (m[10]) {
                date.setSeconds(m[10]);
            }
            if (m[12]) {
                date.setMilliseconds(Number("0." + m[12]) * 1000);
            }
            fixDate(date, check);
            return date;
        },
        dateConvert: function(dateStr) {
            var parts = dateStr.split('-');
            var y = parts[0];
            var m = parts[1];
            var d = parts[2];
            var new_date = new Date(y, m - 1, d);
            var date_str = formatDate(new_date, getMSG('LongDateFormat5'));
            return date_str;
        },
        numberOfdaysCalc: function(date, currentDate) {
            var date_obj = new Date(date);
            var endDate = new Date(date_obj.getFullYear(), date_obj.getMonth(), date_obj.getDate(), 0, 0, 0);
            var days = null;
            if (currentDate && endDate) {
                var diff = endDate - currentDate;
                days = diff / 1000 / 60 / 60 / 24;
            }
            return days;
        },
        dayTimefinder: function(val) {
            if (val === 0) {
                return getMSG('NoneHead');
            } else if (val === 1) {
                return getMSG('AllDayHead');
            } else if (val === 2) {
                return getMSG('EarlymorningHead');
            } else if (val === 3) {
                return getMSG('MorningHead');
            } else if (val === 4) {
                return getMSG('NoonHead');
            } else if (val === 5) {
                return getMSG('AfternoonHead');
            } else if (val === 6) {
                return getMSG('EveningHead');
            } else if (val === 7) {
                return getMSG('NightHead');
            } else if (val === 8) {
                return getMSG('MidnightHead');
            }
            return '';
        },
        getReminderText: function(val) {
            if (val === 0) {
                return getMSG('AtStartTimeHead');
            } else if (val === 5) {
                return getMSG('FiveMinutesHead');
            } else if (val === 10) {
                return getMSG('TenMinutesHead');
            } else if (val === 15) {
                return getMSG('FifteenMinutesHead');
            } else if (val === 20) {
                return getMSG('TwentyMinutesHead');
            } else if (val === 25) {
                return getMSG('TwentyFiveMinutesHead');
            } else if (val === 30) {
                return getMSG('ThirtyMinutesHead');
            } else if (val === 45) {
                return getMSG('FourtyFiveMinutesHead');
            } else if (val === 60) {
                return getMSG('OneHourHead');
            } else if (val === 120) {
                return getMSG('TwoHourHead');
            } else if (val === 180) {
                return getMSG('ThreeHourHead');
            } else if (val === 720) {
                return getMSG('TwelveHourHead');
            } else if (val === 1440) {
                return getMSG('TwentyFourHourHead');
            } else if (val === 2880) {
                return getMSG('TwoDayHead');
            } else if (val === 10080) {
                return getMSG('OneWeekHead');
            }
            return '';
        },
        getRepeatText: function(val) {
            if (val === 0) {
                return getMSG('OneTimeEventHead');
            } else if (val === 1) {
                return getMSG('DailyHead');
            } else if (val === 2) {
                return getMSG('EveryWeekdayHead');
            } else if (val === 3) {

            } else if (val === 4) {
                return getMSG('WeeklyHead');
            } else if (val === 5) {
                return getMSG('MonthlyHead');
            } else if (val === 6) {
                return getMSG('YearlyHead');
            }
            return '';
        },
        getDayText: function(val) {
            if (val === 0) {
                return getMSG('SundayHead');
            } else if (val === 1) {
                return getMSG('MondayHead');
            } else if (val === 2) {
                return getMSG('TuesdayHead');
            } else if (val === 3) {
                return getMSG('WensdayHead');
            } else if (val === 4) {
                return getMSG('ThursdayHead');
            } else if (val === 5) {
                return getMSG('FridayHead');
            } else if (val === 6) {
                return getMSG('SaturdayHead');
            }
            return '';
        },
        getIconSizeText: function(val) {
            if (val === 3) {
                return getMSG('LargeHead');
            } else if (val === 2) {
                return getMSG('MediumHead');
            } else if (val === 1) {
                return getMSG('SmallHead');
            }
            return '';
        },
        findUrl: function(content) {
            var searchText = content, urls = searchText.match(/\b(http|https)?(:\/\/)?(\S*)\.(\w{2,4})([\/\-\?\_\w])*\b/ig), r = '';
            if (urls) {
                var il = urls.length;
                for (var i = 0; i < il; i++) {
                    r = (urls.length > 1) ? urls[i] + ',' : urls[i];
                }
            }
            return r;
        },
        calcTime: function(city, dateTime) {
            var locOffset = timezones[jorteConfig.TIME_ZONE];
            var newOffset = timezones[city];
            var new_date = dateTime;
            if (locOffset !== newOffset) {
                new_date = dateTime - ((newOffset - locOffset) * 60 * 1000);
            }
            return new Date(new_date);

        },
        calcTimeLocal: function(city, dateTime) {
            var locOffset = timezones[jorteConfig.TIME_ZONE];
            var newOffset = timezones[city];
            var new_date = dateTime;
            if (locOffset !== newOffset) {
                new_date = dateTime - ((locOffset - newOffset) * 60 * 1000);
            }
            return new Date(new_date);
        },
        //Event color
        getColor: function(item) {
            var color;
            if ((item.importance && !item.completed) || item.holiday || item.provider === 200) {
                color = jorteConfig.COLORS.red;
            } else if (item.completed === 1 && jorteConfig.getDispCompletedEvent() === 1) {
                color = jorteConfig.COLORS.completed;
            } else if (item.color) {
                color = item.color;
            } else {
                var eventColor = jorteConfig.JORTE_THEME.title_color[jorteConfig.getCurrentTheme() - 1];
                color = item.cal_color ? item.cal_color : (item.type !== 1 ? eventColor : jorteConfig.COLORS.taskDefault);
            }
            return color;
        }
    };

}());
String.prototype.format = String.prototype.f = function() {
    var s = this, i = arguments.length;
    while (i--) {
        s = s.replace(new RegExp('\\{' + i + '\\}', 'gm'), arguments[i]);
    }
    return s;
};