/*	Kronometro Vulpa - Pomodoro-style Javascript Timer
*	Copyright (C) 2013  Kyle Alexander Thompson (Kyle.Alexander.Thompson@gmail.com)
*
*	This program is free software: you can redistribute it and/or modify
*	it under the terms of the GNU General Public License as published by
*	the Free Software Foundation, either version 3 of the License, or
*	(at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*	GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

(function(){
$(document).ready(function() {

	//Load the settings defined at the end of the file
	loadSettings();

	//Set variables for each time setting to the values held in local storage
	setpom = localStorage.getItem("pomodoro") * 60;
	shrtbreak = localStorage.getItem("shortbreak") * 60;
	lngbreak = localStorage.getItem("longbreak") * 60;

	//Set switch to be used in pausing the timer with keystroke
	timerswitch = false;

	//Load alarm tone
	alarm = new Audio("./audio/alarm.wav");

	//Check if timerstorageseconds is set to a value
	if(localStorage.getItem("timerstorageseconds")) {
		//Set the Pomodoro counter to its current value so it is properly displayed
		$("#pomodorocounter").text(
			"Pomodoro Count: " + (localStorage.pomodorocount = parseInt(localStorage.pomodorocount, 10))
		);
		restartTimer();
	}
	//Otherwise set timerdisplay to the normal pomodoro duration with autostart set to false
	else {
		handleTimer(setpom, false);
	}

	//Disable the start button if the program just started to make sure the user 
	//presses the pomodoro button
	if(localStorage.getItem("disableflag") === "0") {
		$('#startbutton').addClass("disabled");
	}

	//Initializes and starts the Pomodoro
	$("#pomodorobutton").click(function(){
		//Reenable disabled start button
		reenablestart();
		//Pass setpom to handleTimer
		handleTimer(setpom, true);
		hidedropdownbox();
	});

	//Initializes and starts the short break
	$("#shortbreakbutton").click(function(){
		//Reenable disabled start button
		reenablestart();
		//Pass shrtbreak to handleTimer
		handleTimer(shrtbreak, true);
		hidedropdownbox();
	});

	//Initializes and starts the long break
	$("#longbreakbutton").click(function(){
		//Reenable disabled start button
		reenablestart();
		//Pass lngbreak to handleTimer
		handleTimer(lngbreak, true);
		hidedropdownbox();
	});

	//Starts the timer if paused
	$("#startbutton").click(function(){
		restartTimer();
	});

	//Pauses the timer
	$("#pausebutton").click(function(){
		$("#timerdisplay").pauseTimer();
	});

	//Reset pomodoro counter
	$("#resetcountbutton").click(function(){
		localStorage.pomodorocount = 0;
		$("#pomodorocounter").text("Pomodoro Count: 0");
	});

	//Resets the timer to top of countdown, gets "top of countdown" value from timerinput
	$("#resetbutton").click(function(){
		executeReset();
	});

});

//Sets the various times in minutes to be used at startup
function loadSettings() {
  if (localStorage["pomflag"] != 1) {
    localStorage.setItem("pomodoro", 25);
    localStorage.setItem("shortbreak", 5);
    localStorage.setItem("longbreak", 30);
    localStorage.setItem("pomflag", 1);
    localStorage.setItem("pomodorocount", 0);
    localStorage.setItem("disableflag", 0);
  }
}

function handleTimer(timertype, autostartflag) {
	//Set timerinput to the pomodoro duration value 
	//stored in appropriate time variable and include in local storage
	localStorage.setItem("timerinput", timertype);
	$("#timerdisplay").createTimer({
		autostart: autostartflag,
		time_in_seconds: localStorage.getItem("timerinput"),
		tick: function(){
			//On each tick of the timer store the seconds value 
			//(taken from tick arguments[1]) and adjust the progress bar
			localStorage.setItem("timerstorageseconds", arguments[1]);
			$("span").css("width",((arguments[1] / localStorage.getItem("timerinput")) * 100 + "%"));
			document.title = "Kronometro Tomata (" + $("#timerdisplay").text() + ")";
		},
		//Call buzzer function defined in the final countdown, 
		//play alarm tone, add to pomodoro count and output count text 
		//if pomodoro-type timer is active
		buzzer: function(){
			alarm.play();
			document.title = "Time's up!";
			if(timertype === setpom) {
				$("#pomodorocounter").text(
					"Pomodoro Count: " + (localStorage.pomodorocount = (parseInt(localStorage.pomodorocount, 10) + 1))
				);
			}
		}
	});
}

function restartTimer() {
	//Check that timerstorageseconds is initialized
	//and greater than zero to avoid unwanted activation.
	if(localStorage.timerstorageseconds && parseInt(localStorage.timerstorageseconds, 10) > 0){
		//Set progress bar to proper length 
		//(As it would normally reset to the default HTML defined value unless the timer was fired)
		$("span").css("width",((localStorage.getItem("timerstorageseconds") / localStorage.getItem("timerinput")) * 100 + "%"));
		//If it is, set timerdisplay to the value stored in timerstorage seconds, 
		//so the user can continue timer if they close a tab by accident
		$("#timerdisplay").createTimer({
			time_in_seconds: localStorage.getItem("timerstorageseconds"),
			autostart: true,
			tick: function(){
				//On each tick of the timer store the seconds value 
				//(taken from tick arguments[1]) and adjust the progress bar
				localStorage.setItem("timerstorageseconds", arguments[1]);
				$("span").css("width",((arguments[1] / localStorage.getItem("timerinput")) * 100 + "%"));
				document.title = "Kronometro Tomata (" + $("#timerdisplay").text() + ")";
			},
			//Call buzzer function defined in the final countdown, 
			//play alarm tone, add to pomodoro count and output count text 
			//if pomodoro-type timer is active
			buzzer: function(){
				alarm.play();
				document.title = "Time's up!";
				if(parseInt(localStorage.getItem("timerinput"), 10) === setpom) {
					$("#pomodorocounter").text(
						"Pomodoro Count: " + (localStorage.pomodorocount = (parseInt(localStorage.pomodorocount, 10) + 1))
					);
				}
			}
		});
	}
	//Set timer to "00:00" on reload if timer is equal to zero to avoid unwanted default to "Loading..." text
	else if(localStorage.timerstorageseconds && parseInt(localStorage.timerstorageseconds, 10) === 0){
		$("#timerdisplay").text("00:00");
		$("span").css("width", 0);
	}
}

function executeReset() {
	//Set timerstorage seconds to the maximum timer value and reset the progress bar
	localStorage.timerstorageseconds = localStorage.timerinput;
	$("span").css("width",("100%"));
	$("#timerdisplay").createTimer({
		time_in_seconds: localStorage.getItem("timerinput"),
		autostart: false,
		tick: function(){
			localStorage.setItem("timerstorage", arguments[2]);
			localStorage.setItem("timerstorageseconds", arguments[1]);
			$("span").css("width",((arguments[1] / localStorage.getItem("timerinput")) * 100 + "%"));
			document.title = "Kronometro Tomata (" + $("#timerdisplay").text() + ")";
		},
		//Call buzzer function defined in the final countdown, play alarm tone
		buzzer: function(){
			alarm.play();
			if(parseInt(localStorage.getItem("timerinput"), 10) === setpom) {
				$("#pomodorocounter").text(
					"Pomodoro Count: " + (localStorage.pomodorocount = (parseInt(localStorage.pomodorocount, 10) + 1))
				);
			}
		}
	});
	//Reset the title timer even if the timer isn't started
	document.title = "Kronometro Tomata (" + $("#timerdisplay").text() + ")";
}

//Enables the start button if disabled
function reenablestart() {
	if(localStorage.getItem("disableflag") === "0") {
		$('#startbutton').removeClass("disabled");
		localStorage.disableflag = "1";
	}
}

//Hides dropdown box when triggered
function hidedropdownbox() {
		$('#timeselection').css(Foundation.rtl ? 'right':'left', '-99999px');
		$('#timeselection').removeClass(Foundation.libs.dropdown.settings.activeClass);
		$('#timeselection').trigger('closed');
}

})();
