/* global Status */
/**
 * ErrorManager
 * Controls what we do for each error detected after an action
 */
(function (exports) {
    'use strict';

    var _errorList = {
        'offline': {
            'code': 1,
            'message': 'No network availabe'
        },
        'parameter': {
            'code': 2,
            'message': 'Missing parameter'
        },
        'server': {
            'code': 50,
            'message': 'Server error' // + statusCode
        },
        'noResponse': {
            'code': 51,
            'message': 'Response not received'
        },
        'authentication': {
            'code': 52,
            'message': 'Unauthorized'
        }
    };

    var outdatedErrorShown = false;
    function _checkErrorType(error, params) {
        if (!error) {
            return true;
        }

        var keepWorking = false;

        switch (error.code) {
            // Internal
            case 1: // no network
                Status.show(navigator.mozL10n.get('no_network'));
                console.error('No network available');
                break;
            case 2:
                Status.show(navigator.mozL10n.get('server_error'));
                console.error('Missing parameter');
                break;
            case 50: // server error
                Status.show(navigator.mozL10n.get('server_error'));
                console.error('Server error: ' + (params && params[0] || 'Unknown'));
                break;
            case 51:
                Status.show(navigator.mozL10n.get('server_error'));
                console.error('No response');
                break;
            case 52:
                Status.show(navigator.mozL10n.get('not_authorized'));
                console.error('Not authorized');
                break;
                // Client
            case 105: // invalid credentials
                Status.show(navigator.mozL10n.get('login_invalid_auth'));
                console.error('Missing auth credentials');
                Proxy.logout();
                UIManager.load('signin');
                break;
            case 106: // outdated client
                keepWorking = true; // this error doesn't stop the flow
                if (!outdatedErrorShown) {
                    Status.show(navigator.mozL10n.get('client_outdated'));
                    outdatedErrorShown = true;
                    console.warn('Client outdated');
                }
                break;
            case 107: // unsupported client
                Status.show(navigator.mozL10n.get('client_unsupported'));
                console.error('Client not supported');
                keepWorking = false;
                break;
                // Operations
            case 201: // account not paired
                console.error('Account not paired');
                break;
            case 202: // invalid account name
                console.error('Invalid account name');
                break;
            case 301: // application or operation not found
                console.error('Operation not found');
                break;
            case 401: // missing parameter
                console.error('Missing parameter');
                break;
            case 501: // no OTP available
                console.error('There isn\'t a token generated for this operation and user account');
                Status.show(navigator.mozL10n.get('otp-no-available'));
                break;
            case 404: // Server not found
                console.error('Status: 404 Not found');
                break;
            default: // no idea
                if (error && error.code && error.message) {
                    Status.show(navigator.mozL10n.get('unknown_error'));
                    console.error('Unknown error: ' + error.code);
                    console.error('message: ' + error.message);
                    keepWorking = false;
                } else {
                    Status.show(navigator.mozL10n.get('unknown_error'));
                    console.error('Unknown error - ' + error);
                    keepWorking = false;
                }
        }

        return keepWorking;
    }

    var ErrorManager = {
        init: function () {
            outdatedErrorShown = false;
        },
        get errorList() {
            return _errorList;
        },
        checkError: _checkErrorType
    };

    exports.ErrorManager = ErrorManager;
}(this));

