if (!Array.isArray) {
    Array.isArray = function (arg) {
        return Object.prototype.toString.call(arg) === '[object Array]';
    };
}

var app = {

    domain: 'leeds-gigs.co.uk',

    googleMapsApiKey: 'AIzaSyAHpAsytfy29sveXm0BIvsw47r9oBzLN9w',

    googleAnalyticsKey: 'UA-209086-16',

    debug: false,

    forceReloadGigs: false,

    indexed_saved_gigs: null,
    indexed_gigs: null,
    indexed_saved_venues: null,
    indexed_venues: null,
    indexed_gigs_by_venue_id: null,

    saved_gigs: [],
    saved_venues: [],
    gigs: [],
    news: [],
    venues: [],

    gig_details: {},
    geocode: {},

    // Application Constructor
    initialize: function () {

        document.addEventListener('deviceready', app.onDeviceReady, false);
        document.addEventListener('backbutton', app.backKeyDown, false);
        document.addEventListener('gigsReady', app.getGigsForView, false);
        document.addEventListener('savedGigsReady', app.getSavedGigsForView, false);
        document.addEventListener('savedVenuesReady', app.getSavedVenuesForView, false);
        document.addEventListener('venuesReady', app.getVenuesForView, false);
        document.addEventListener('newsReady', app.getNewsForView, false);
        document.addEventListener('online', app.deviceCameOnline, false);
        document.addEventListener('offline', app.deviceWentOffline, false);
        document.addEventListener('pause', app.onPause, false);
        document.addEventListener('resume', app.onResume, false);

        $(function () {

            view.start();

            app.start();

        });
    },

    // Event creator
    createEvent: function (event) {
        var evt = document.createEvent('Event');
        evt.initEvent(event, true, true);
        document.dispatchEvent(evt); // Dispatch the Event
    },

    // deviceready Event Handler
    onDeviceReady: function () {
        app.receivedEvent('deviceready');

        if (typeof analytics !== 'undefined') {
            try {
                analytics.startTrackerWithId(app.googleAnalyticsKey);
                app.trackEvent('App', 'Started');
            } catch (err) {
                console.log(err);
            }
        } else {
            console.log('Google Analytics plugin not available');
        }
    },

    // deviceWentOffline Event Handler
    deviceWentOffline: function () {
        app.receivedEvent('deviceWentOffline');
        view.notOnlineWarning();
    },

    // deviceCameOnline Event Handler
    deviceCameOnline: function () {
        app.receivedEvent('deviceCameOnline');
    },

    // start Event Handler
    start: function () {
        app.receivedEvent('start');

        var gigs = app.getGigsList();
        if (Array.isArray(gigs)) app.gigs = gigs;

        var saved_gigs = app.getSavedGigsList();
        if (Array.isArray(saved_gigs)) app.saved_gigs = saved_gigs;

        var saved_venues = app.getSavedVenuesList();
        if (Array.isArray(saved_venues)) app.saved_venues = saved_venues;

        var venues = app.getVenuesList();
        if (Array.isArray(venues)) app.venues = venues;

        var news = app.getNewsList();
        if (Array.isArray(news)) app.news = news;

        app.downloadGigs(); // Download the gigs
        app.downloadVenues(); // Download the venues
        app.downloadNews(); // Download the latest news

        if (typeof navigator.splashscreen !== 'undefined') {
            console.log('Closing splash screen');
            navigator.splashscreen.hide();
        }

    },

    // cancel / exit the application
    stop: function () {
        console.log('Stopping application');
        app.trackEvent('App', 'Stopped');
        if (typeof navigator.app !== 'undefined') {
            navigator.app.exitApp();
        }
    },

    onPause: function (e) {
        app.receivedEvent('pause');
    },

    onResume: function (e) {
        app.receivedEvent('resume');
    },

    // backbutton Event Handler
    backKeyDown: function (e) {
        app.receivedEvent('backbutton');
        if ($.mobile.activePage.is('#homepage')) {
            e.preventDefault();
            app.stop(); // Quit the application
        } else {
            if (!$.mobile.activePage.is('#venue')) {
                view.$selector.venuemap.empty().hide();
            }
            navigator.app.backHistory();
        }
    },

    trackEvent: function (category, action) {
        if (typeof analytics !== 'undefined') {
            try {
                analytics.trackEvent(category, action);
            } catch (err) {
                console.log(err);
            }
        } else {
            console.log('Google Analytics plugin not available');
        }
    },

    trackView: function (screen) {
        if (typeof analytics !== 'undefined') {
            try {
                analytics.trackView(screen);
            } catch (err) {
                console.log(err);
            }
        } else {
            console.log('Google Analytics plugin not available');
        }
    },

    isOnline: function () {
        if (typeof navigator.connection === 'undefined' || typeof navigator.connection.type === 'undefined' || typeof Connection === 'undefined') {
            return navigator.onLine;
        } else {
            return (navigator.connection.type !== Connection.NONE);
        }
    },

    getFromLocalStorage: function (key) {
        return localStorage.getItem(key);
    },
    saveToLocalStorage: function (key, item) {
        localStorage.setItem(key, item);
    },
    getGigsList: function () {
        return JSON.parse(app.getFromLocalStorage('gigs-list'));
    },
    setGigsList: function (gigs) {
        return app.saveToLocalStorage('gigs-list', JSON.stringify(gigs));
    },
    getSavedGigsList: function () {
        return JSON.parse(app.getFromLocalStorage('saved-gigs-list'));
    },
    setSavedGigsList: function (gigs) {
        return app.saveToLocalStorage('saved-gigs-list', JSON.stringify(gigs));
    },
    getSavedVenuesList: function () {
        return JSON.parse(app.getFromLocalStorage('saved-venues-list'));
    },
    setSavedVenuesList: function (venues) {
        return app.saveToLocalStorage('saved-venues-list', JSON.stringify(venues));
    },
    getVenuesList: function () {
        return JSON.parse(app.getFromLocalStorage('venues-list'));
    },
    setVenuesList: function (venues) {
        return app.saveToLocalStorage('venues-list', JSON.stringify(venues));
    },
    getNewsList: function () {
        return JSON.parse(app.getFromLocalStorage('news-list'));
    },
    setNewsList: function (news) {
        return app.saveToLocalStorage('news-list', JSON.stringify(news));
    },
    getLatestGigsDownloadDate: function () {
        return app.getFromLocalStorage('gigs-download-date');
    },
    setLatestGigsDownloadDate: function (date) {
        app.saveToLocalStorage('gigs-download-date', date);
    },
    getLatestVenuesDownloadDate: function () {
        return app.getFromLocalStorage('venues-download-date');
    },
    setLatestVenuesDownloadDate: function (date) {
        app.saveToLocalStorage('venues-download-date', date);
    },
    getLatestNewsDownloadDate: function () {
        return app.getFromLocalStorage('news-download-date');
    },
    setLatestNewsDownloadDate: function (date) {
        app.saveToLocalStorage('news-download-date', date);
    },

    // Update the Saved Gig details based on the latest download content.
    // Remove the Saved Gig if it no longer exists in the latest download content.
    updateSavedGigs: function () {
        var len = app.saved_gigs.length, updated_saved_gigs = [], removed = false;
        for (var i = 0; i < len; ++i) {
            var saved_gig = app.saved_gigs[i], current_gig = app.getGigById(saved_gig.id);
            if (typeof current_gig !== 'undefined') {
                updated_saved_gigs.push(current_gig);
            } else {
                removed = true;
            }
        }
        if (removed) {
            app.indexed_saved_gigs = null;
            app.setSavedGigsList(updated_saved_gigs);
            app.saved_gigs = updated_saved_gigs;
        }
    },

    // Update the Saved Venues details based on the latest download content.
    // Don't remove Saved Venues that don't exist any more.
    updateSavedVenues: function () {
        var len = app.saved_venues.length, updated_saved_venues = [];
        for (var i = 0; i < len; ++i) {
            var saved_venue = app.saved_venues[i], current_venue = app.getVenueById(saved_venue.id);
            if (typeof current_venue !== 'undefined') {
                updated_saved_venues.push(current_venue);
            } else {
                updated_saved_venues.push(saved_venue);
            }
        }
        app.index_saved_venues = null;
        app.setSavedVenuesList(updated_saved_venues);
        app.saved_venues = updated_saved_venues;
    },

    // Download gigs
    downloadGigs: function () {
        var now = moment().format("YYYY-MM-DD");
        var then = app.getLatestGigsDownloadDate();
        if (!then) then = '';
        if (app.isOnline() && (app.gigs.length == 0 || app.forceReloadGigs || app.debug || then < now)) {
            console.log('Downloading the latest gigs from server');
            $.ajax({
                url: 'http://' + app.domain + '/ajax/getEventsForApp.js',
                dataType: 'json',
                cache: false,
                crossDomain: true
            }).done(function (data) {
                var len = data.events.length;
                console.log(len + ' gigs retrieved from server');
                if (len > 0) {
                    app.setLatestGigsDownloadDate(now); // Save the date we ran this download
                    app.gigs = data.events;
                    app.indexed_gigs = null;
                    app.indexed_gigs_by_venue_id = null;
                    app.setGigsList(data.events);
                    app.updateSavedGigs();
                }
            }).always(function () {
                app.createEvent('gigsReady');
                app.createEvent('savedGigsReady');
            });
        } else {
            app.createEvent('gigsReady');
            app.createEvent('savedGigsReady');
        }

    },

    truncateSavedGigsList: function () {
        // The first gig should be >= view.gigStartDate, if it's not, let's delete some old gigs
        if (app.saved_gigs.length > 0 && app.saved_gigs[0].event_date < view.gigStartDate) {
            var temp_gigs = [], len = app.saved_gigs.length;
            console.log('Truncating gigs from saved gig list. Start count = ' + len);
            for (var i = 0; i < len; ++i) {
                var gig = app.saved_gigs[i];
                if (gig.event_date >= view.gigStartDate) {
                    temp_gigs.push(gig);
                }
            }
            app.setSavedGigsList(temp_gigs); // Save to storage
            app.saved_gigs = temp_gigs; // Save internally
            len = app.saved_gigs.length;
            console.log('Truncated gigs from saved gig list. End count = ' + len);
        }
    },

    truncateGigsList: function () {
        // The first gig should be >= view.gigStartDate, if it's not, let's delete some old gigs
        if (app.gigs.length > 0 && app.gigs[0].event_date < view.gigStartDate) {
            var temp_gigs = [], len = app.gigs.length;
            console.log('Truncating gigs from gig list. Start count = ' + len);
            for (var i = 0; i < len; ++i) {
                var gig = app.gigs[i];
                if (gig.event_date >= view.gigStartDate) {
                    temp_gigs.push(gig);
                }
            }
            app.setGigsList(temp_gigs); // Save to storage
            app.gigs = temp_gigs; // Save internally
            len = app.gigs.length;
            console.log('Truncated gigs from gig list. End count = ' + len);
        }
    },

    // Read the saved gigs for the saved listings view
    getSavedGigsForView: function () {
        app.truncateSavedGigsList(); // Make sure we don't have any old saved gigs hanging around
        view.updateSavedGigsCount(app.saved_gigs.length);
        view.showSavedGigs(app.saved_gigs);
    },

    // Get the saved gig by its unique identifier
    getSavedGigById: function (gig_id) {
        console.log('Reading the saved gigs for gig = ' + gig_id);
        if (app.indexed_saved_gigs == null) {
            console.log('Indexing saved gig list');
            app.truncateSavedGigsList(); // Make sure we don't have any old saved gigs hanging around
            app.indexed_saved_gigs = _.indexBy(app.saved_gigs, 'id'); // Create an indexed version of the gigs list
        }
        return app.indexed_saved_gigs[gig_id];
    },

    // Read the gigs for the listings view
    getGigsForView: function () {
        console.log("Reading up to " + view.gigsPerPage + " gigs where date >= " + view.gigStartDate + " with offset = " + view.gigOffset);
        app.truncateGigsList(); // Make sure we don't have any old gigs hanging around
        view.showGigs(app.gigs.slice(view.gigOffset, view.gigOffset + view.gigsPerPage));
    },

    // Read the gigs for the specified venue
    getGigsForVenue: function () {
        console.log('Reading the gigs for venue = ' + view.venue_id);
        if (app.indexed_gigs_by_venue_id == null) {
            console.log('Indexing gigs by venue_id list');
            app.truncateGigsList(); // Make sure we don't have any old gigs hanging around
            app.indexed_gigs_by_venue_id = _.groupBy(app.gigs, 'venue_id'); // Index gigs by venue_id
        }
        view.showVenueGigs(app.indexed_gigs_by_venue_id[view.venue_id]);
    },

    // Get the gig by its unique identifier
    getGigById: function (gig_id) {
        console.log('Reading the gigs for gig = ' + gig_id);
        if (app.indexed_gigs == null) {
            console.log('Indexing gig list');
            app.truncateGigsList(); // Make sure we don't have any old gigs hanging around
            app.indexed_gigs = _.indexBy(app.gigs, 'id'); // Create an indexed version of the gigs list
        }
        return app.indexed_gigs[gig_id];
    },

    // Retrieve details of the gig from the server (or cache, if we've already asked for this data in this session)
    getGigDetails: function (gig_id) {
        var gig = app.gig_details[gig_id];
        if (typeof gig === 'undefined') {
            if (app.isOnline()) {
                $.ajax({
                    url: 'http://' + app.domain + '/ajax/getEvent.js',
                    data: {
                        event_id: gig_id
                    },
                    dataType: 'json',
                    cache: true,
                    crossDomain: true
                }).done(function (gig) {
                    app.gig_details[gig_id] = gig;
                    view.gigDetailsCallback(gig);
                });
            } else {
                view.notOnlineWarning();
                view.gigDetailsCallbackError();
            }
        } else {
            view.gigDetailsCallback(gig);
        }
    },

    // Read the saved venues for the saved venues view
    getSavedVenuesForView: function () {
        view.updateSavedVenuesCount(app.saved_venues.length);
        view.showSavedVenues(app.saved_venues);
    },

    // Get the saved venue by its unique identifier
    getSavedVenueById: function (venue_id) {
        console.log('Reading the saved venues for venue = ' + venue_id);
        if (app.indexed_saved_venues == null) {
            console.log('Indexing saved venue list');
            app.indexed_saved_venues = _.indexBy(app.saved_venues, 'id'); // Create an indexed version of the venues list
        }
        return app.indexed_saved_venues[venue_id];
    },

    downloadVenues: function () {
        var now = moment().format("YYYY-MM-DD");
        var then = app.getLatestVenuesDownloadDate();
        if (!then) then = '';
        if (app.isOnline() && (app.venues.length == 0 || app.debug || then < now)) {
            console.log('Downloading the latest venues from the server');
            $.ajax({
                url: 'http://' + app.domain + '/ajax/getVenuesForApp.js',
                dataType: 'json',
                crossDomain: true,
                cache: false
            }).done(function (data) {
                var len = data.venues.length;
                console.log(len + ' venues retrieved from server');
                if (len > 0) {
                    app.setLatestVenuesDownloadDate(now); // Save the date we downloaded
                    app.venues = data.venues;
                    app.indexed_venues = null;
                    app.setVenuesList(data.venues);
                    app.updateSavedVenues();
                }
            }).always(function () {
                app.createEvent('venuesReady');
                app.createEvent('savedVenuesReady');
            });
        } else {
            app.createEvent('venuesReady');
            app.createEvent('savedVenuesReady');
        }
    },

    getVenuesForView: function () {
        view.showVenues(app.venues);
    },

    // Get the venue by its unique identifier
    getVenueById: function (venue_id) {
        console.log('Reading the venue for venue = ' + venue_id);
        if (app.indexed_venues == null) {
            console.log('Indexing venues list');
            app.indexed_venues = _.indexBy(app.venues, 'id'); // Create an indexed version of the venues list
        }
        return app.indexed_venues[venue_id];
    },

    getGeoByVenue: function (venue_id) {
        return app.geocode[venue_id];
    },
    saveGeo: function (venue_id, geo) {
        app.geocode[venue_id] = geo;
    },

    // Download news
    downloadNews: function () {
        var now = moment().format("YYYY-MM-DD");
        var then = app.getLatestNewsDownloadDate();
        if (!then) then = '';
        if (app.isOnline() && (app.news.length == 0 || app.debug || then < now)) {
            console.log('Downloading the latest news from the server');
            $.ajax({
                url: 'http://' + app.domain + '/ajax/getBlogPosts.js',
                dataType: 'json',
                crossDomain: true
            }).done(function (data) {
                var len = data.length;
                console.log(len + ' news records retrieved from server');
                if (len > 0) {
                    app.setLatestNewsDownloadDate(now); // Save the date we downloaded
                    var master_list = app.news;
                    if (master_list !== null && master_list.length > 0) {
                        var max_news_id = master_list[0].id;
                        for (var i = 0; i < len; ++i) {
                            if (data[i].id > max_news_id) {
                                master_list.push(data[i]);
                            }
                        }
                    } else {
                        master_list = data;
                    }
                    // Sort the news into news id order descending
                    master_list.sort(function (a, b) {
                        if (a.id > b.id) return -1;
                        if (a.id < b.id) return 1;
                        return 0;
                    });
                    app.news = master_list;
                    app.setNewsList(master_list);
                }
            }).always(function () {
                app.createEvent('newsReady');
            });
        } else {
            app.createEvent('newsReady');
        }
    },

    getNewsForView: function () {
        var news_limit = 10, counter = 0, news = [], len = app.news.length;
        console.log("Reading up to " + news_limit + " posts from news where id >= " + view.maxNewsIdDisplayed);
        // We are assuming the app.news data is already stored in descending news.id order
        for (var i = 0; i < len && counter < news_limit; ++i) {
            var post = app.news[i];
            if (post.id >= view.maxNewsIdDisplayed) {
                news.push(post);
                counter++;
            }
        }
        view.showNews(news);
    },

    getMoreNewsForView: function () {
        var news_limit = 6, counter = 0, news = [], len = app.news.length;
        console.log("Reading up to " + news_limit + " posts from news where id < " + view.minNewsIdDisplayed);
        // We are assuming the app.news data is already stored in descending news.id order
        for (var i = 0; i < len && counter < news_limit; ++i) {
            var post = app.news[i];
            if (post.id < view.minNewsIdDisplayed) {
                news.push(post);
                counter++;
            }
        }
        view.showNews(news);
    },
    getNewsById: function (post_id) {
        console.log('Reading the news for post = ' + post_id);
        var len = app.news.length;
        for (var i = 0; i < len; ++i) {
            var post = app.news[i];
            if (post.id == post_id) {
                return post;
            }
        }
        return {};
    },

    receivedEvent: function (id) {
        console.log('Received Event: ' + id);
    }
};

var view = {

    maxNewsIdDisplayed: 0,
    minNewsIdDisplayed: 99999999,
    gigStartDate: moment().format("YYYY-MM-DD"),
    gigDateFormat: 'dddd, Do MMMM YYYY',
    gigOffset: 0,
    newsTimeout: 0,
    newsTimoutPeriod: 1500,
    gigsTimeout: 0,
    gigsTimoutPeriod: 1500,
    onlineWarning: false,
    monthDivider: '',
    venue_id: 0,
    gig_id: 0,
    googleApiLoaded: false,
    venueMapCount: 0,
    gigMapCount: 0,
    gigsPerPage: 20,

    $selector: {},

    start: function () {

        console.log("View initializing");

        view.$selector.navpanel = $('#nav-panel');
        view.$selector.navpanel.panel();

        view.$selector.header = $('#header');

        $('#nav-panel-listview').listview();
        view.$selector.header.toolbar();

        $('#event-nav-popup-listview').listview();

        view.$selector.gigspage = $('#homepage');
        view.$selector.tournewspage = $('#tour-news');
        view.$selector.venuespage = $('#venues');
        view.$selector.individualvenuepage = $('#venue');

        view.$selector.pageTitle = view.$selector.header.find('h1');

        view.$selector.newsinfinitescroller = $('#show-me-more-news');
        view.$selector.gigsinfinitescroller = $('#show-me-more-events');

        view.$selector.savedgigscount = $('#saved-gigs-count');
        view.$selector.savedvenuescount = $('#saved-venues-count');

        view.$selector.listingsbtn = $('#listings-btn');
        view.$selector.savedbtn = $('#saved-btn');
        view.$selector.newsbtn = $('#news-btn');
        view.$selector.venuesbtn = $('#venues-btn');
        view.$selector.aboutbtn = $('#about-btn');
        view.$selector.favouritesbtn = $('#favourites-btn');

        view.$selector.savegigbtn = $('#event-save-btn');
        view.$selector.savevenuebtn = $('#venue-save-btn');
        view.$selector.venuetwitterbtn = $('#venue-twitter-btn');

        view.$selector.savegigbtn.off('tap').on('tap', view.saveGigClickHandler);
        view.$selector.savevenuebtn.off('tap').on('tap', view.saveVenueClickHandler);
        view.$selector.venuetwitterbtn.off('tap').on('tap', view.navigateToTwitter);

        view.$selector.gigslist = $('#events-listview');
        view.$selector.savedgigslist = $('#saved-gigs-listview');
        view.$selector.savedvenueslist = $('#saved-venues-listview');
        view.$selector.venueslist = $('#venues-listview');
        view.$selector.newslist = $('#news-listview');
        view.$selector.venuegigslist = $('#venue-gigs-listview');
        view.$selector.ticketslist = $('#event-tickets-listview');

        view.$selector.postheading = $('#tour-news-post-heading');
        view.$selector.postexcerpt = $('#tour-news-post-excerpt');
        view.$selector.postcontent = $('#tour-news-post-content');
        view.$selector.postimage = $('#tour-news-post-image');

        view.$selector.venuemap = $('#venue-map');
        view.$selector.venueaddress = $('.venue-address');
        view.$selector.venuename = $('.venue-name');
        view.$selector.venuetwitter = $('#venue-twitter-div');

        view.$selector.eventtitle = $('#event-title');
        view.$selector.eventdate = $('#event-date');
        view.$selector.eventvenuemap = $('#event-venue-map');
        view.$selector.eventvenueaddress = $('#event-venue-address');
        view.$selector.eventvenuename = $('.event-venue-name');
        view.$selector.eventvenuebtn = $('#event-venue-listings-btn');
        view.$selector.eventinformation = $('#event-information');
        view.$selector.eventinformationloading = $('#event-information-loading');
        view.$selector.eventnoinformation = $('#event-no-more-information');

        $("#event-venue-listings-div").off('tap').on('tap', view.venueClickHandler);

        view.$selector.listingsbtn.addClass('ui-btn-selected');

        $(document).on('pagecontainerbeforeshow', function (event, ui) {
            try {
                view.$selector.navpanel.find('.ui-btn-selected').removeClass('ui-btn-selected');
                var activePage = $.mobile.pageContainer.pagecontainer('getActivePage'), activePageId = activePage[0].id;
                app.trackView(activePageId);
                switch (activePageId) {
                    case 'tour-news':
                        view.$selector.pageTitle.html('Tour News');
                        view.refreshNewsList();
                        view.$selector.newsbtn.addClass('ui-btn-selected');
                        break;
                    case 'tour-news-post':
                        view.$selector.pageTitle.html('Tour News');
                        view.$selector.newsbtn.addClass('ui-btn-selected');
                        break;
                    case 'saved':
                        view.$selector.pageTitle.html('Saved Events');
                        view.refreshSavedGigsList();
                        view.$selector.savedbtn.addClass('ui-btn-selected');
                        break;
                    case 'favourites':
                        view.$selector.pageTitle.html('Favourite Venues');
                        view.refreshSavedVenuesList();
                        view.$selector.favouritesbtn.addClass('ui-btn-selected');
                        break;
                    case 'homepage':
                        view.$selector.pageTitle.html('Events');
                        view.refreshGigsList();
                        view.$selector.listingsbtn.addClass('ui-btn-selected');
                        break;
                    case 'event':
                        view.$selector.pageTitle.html('Events');
                        view.refreshTicketsList();
                        view.$selector.listingsbtn.addClass('ui-btn-selected');
                        break;
                    case 'venues':
                        view.$selector.pageTitle.html('Venues');
                        view.refreshVenueList();
                        view.$selector.venuesbtn.addClass('ui-btn-selected');
                        break;
                    case 'venue':
                        view.$selector.pageTitle.html('Venues');
                        view.refreshVenueGigsList();
                        view.$selector.venuesbtn.addClass('ui-btn-selected');
                        break;
                    case 'about':
                        view.$selector.pageTitle.html('About');
                        view.$selector.aboutbtn.addClass('ui-btn-selected');
                        break;
                    default:
                        break;
                }
            } catch (err) {
                console.error(err);
            }
        });

        console.log("View initialized");
    },

    loadGoogleMapsApiReady: function () {
        if (typeof google !== 'undefined' && typeof google.maps !== 'undefined') {
            console.log('Google Maps API loaded');
            view.googleApiLoaded = true;
        }
    },

    notOnlineWarning: function () {
        if (!view.onlineWarning) {
            view.onlineWarning = true;
            $.fn.dpToast('No connection detected, some content may not be loaded.');
        }
    },

    googleMapsError: function () {
        $.fn.dpToast('Problem detected while loading Google Map.');
    },

    infiniteScrollNews: function () {
        if (view.$selector.newsinfinitescroller.isOnScreen()) {
            // console.log('Infinite scroll detected');
            app.getMoreNewsForView();
        } else {
            // console.log('Infinite scroll not detected');
            view.newsTimeout = setTimeout(view.infiniteScrollNews, view.newsTimoutPeriod);
        }
    },

    infiniteScrollGigs: function () {
        if (view.$selector.gigsinfinitescroller.isOnScreen()) {
            app.getGigsForView();
        } else {
            view.gigsTimeout = setTimeout(view.infiniteScrollGigs, view.gigsTimoutPeriod);
        }
    },

    resetGigView: function () {
        view.$selector.gigslist.empty();
        view.gigOffset = 0;
    },

    updateSavedGigsCount: function (count) {
        view.$selector.savedgigscount.html(count);
    },

    updateSavedVenuesCount: function (count) {
        view.$selector.savedvenuescount.html(count);
    },

    showSavedGigs: function (gigs) {
        var len = gigs.length, dom_append = '', monthDivider = '';
        console.log(len + ' saved gigs read');
        view.$selector.savedgigslist.empty();
        for (var i = 0; i < len; ++i) {
            var gig = gigs[i], gigdate = moment(gig.event_date), day = gigdate.format("D"), month = gigdate.format("MMMM YYYY");
            if (monthDivider != month) {
                monthDivider = month;
                dom_append += '<li data-role="list-divider">' + month + '</li>';
            }
            dom_append += '<li class="ui-li-has-thumb"><a href="#event" data-gig-id="' + gig.id + '"><img src="img/calendar/' + day + '.png" /><h2>' + gig.event_title + '</h2><p>' + gig.venue_name + ', ' + gigdate.format(view.gigDateFormat) + '</p></a></li>';
        }
        if (len == 0) {
            dom_append = '<li>No saved gigs found.</li>';
        }
        view.$selector.savedgigslist.append(dom_append);
        view.refreshSavedGigsList();
    },

    showGigs: function (gigs) {
        var len = gigs.length, dom_append = '';
        console.log(len + ' gigs read');
        for (var i = 0; i < len; ++i) {
            var gig = gigs[i], gigdate = moment(gig.event_date), day = gigdate.format("D"), month = gigdate.format("MMMM YYYY");
            if (view.monthDivider != month) {
                view.monthDivider = month;
                dom_append += '<li data-role="list-divider">' + month + '</li>';
            }
            dom_append += '<li class="ui-li-has-thumb"><a href="#event" data-gig-id="' + gig.id + '"><img src="img/calendar/' + day + '.png" /><h2>' + gig.event_title + '</h2><p>' + gig.venue_name + ', ' + gigdate.format(view.gigDateFormat) + '</p></a></li>';
            view.gigOffset++;
        }
        view.$selector.gigslist.append(dom_append);
        view.refreshGigsList();
        if (len > 0) {
            view.$selector.gigsinfinitescroller.show();
            view.gigsTimeout = setTimeout(view.infiniteScrollGigs, view.gigsTimoutPeriod);
        } else if (view.gigOffset == 0) {
            // Set infinite scroller
            view.$selector.gigsinfinitescroller.show();
            view.gigsTimeout = setTimeout(view.infiniteScrollGigs, view.gigsTimoutPeriod);
        } else {
            console.log('No more gigs - removing timer');
            view.$selector.gigsinfinitescroller.hide();
            clearTimeout(view.gigsTimeout);
        }
    },

    showNews: function (news) {
        var len = news.length, dom_append = '';
        console.log(len + ' news records read');
        for (var i = 0; i < len; ++i) {
            var post = news[i];
            dom_append += '<li><a data-post-id="' + post.id + '" href="#tour-news-post"><img src="' + post.image + '" /><h2>' + post.title + '</h2><p>' + post.excerpt + '</p></a></li>';
            if (view.maxNewsIdDisplayed < post.id) view.maxNewsIdDisplayed = post.id;
            if (view.minNewsIdDisplayed > post.id) view.minNewsIdDisplayed = post.id;
        }
        view.$selector.newslist.append(dom_append);
        view.refreshNewsList();
        if (len > 0) {
            view.$selector.newsinfinitescroller.show();
            view.newsTimeout = setTimeout(view.infiniteScrollNews, view.newsTimoutPeriod);
        } else {
            console.log('No more news - removing timer');
            view.$selector.newsinfinitescroller.remove();
            clearTimeout(view.newsTimeout);
        }
    },

    setMapLocation: function (selector, geocode, mapId, venue) {
        var scale = (window.devicePixelRatio > "1.0") ? 2 : 1, height, width = $('body').outerWidth();
        if (width <= 360) {
            width = 308;
            height = 200;
        } else if (width <= 640) {
            width = 400;
            height = 300;
        } else {
            width = 640;
            height = 300;
        }
        selector.html('<div id="' + mapId + '" style="width:100%px;overflow:hidden;"><img data-venue="' + venue + '" data-lat="' + geocode.lat() + '" data-lng="' + geocode.lng() + '" style="width:100%;" src="http://maps.googleapis.com/maps/api/staticmap?&markers=color:blue%7Clabel:V%7C' + geocode.lat() + ',' + geocode.lng() + '&center=' + geocode.lat() + ',' + geocode.lng() + '&zoom=16&size=' + width + 'x' + height + '&scale=' + scale + '&key=' + app.googleMapsApiKey + '" /> </div>').show();
        $('#' + mapId).on('tap', view.navigateToMap);
    },

    showVenueMap: function (selector, venue_id, mapId) {
        selector.empty().hide();
        var venue = app.getVenueById(venue_id);
        if (app.isOnline() && view.googleApiLoaded && typeof venue !== 'undefined' && venue.address > ' ') {
            try {
                console.log('Showing map for venue ' + venue_id + ', address = ' + venue.address);
                var geocode = app.getGeoByVenue(venue_id);
                if (typeof geocode === 'undefined') {
                    var geocoder = new google.maps.Geocoder();
                    geocoder.geocode({ 'region': 'uk', 'address': venue.address }, function (results, status) {
                        if (status == google.maps.GeocoderStatus.OK) {
                            var geocode = results[0].geometry.location;
                            console.log('Google returned location ' + geocode.toString() + ' for ' + venue.address);
                            app.saveGeo(venue_id, geocode);
                            view.setMapLocation(selector, geocode, mapId, venue.name);
                        } else {
                            view.googleMapsError();
                            selector.empty().hide();
                        }
                    });
                } else {
                    console.log('Using saved location ' + geocode.toString() + ' for ' + venue.address);
                    app.saveGeo(venue_id, geocode);
                    view.setMapLocation(selector, geocode, mapId, venue.name);
                }
            } catch (err) {
                view.googleMapsError();
                selector.empty().hide();
                console.error(err);
            }
        } else if (!app.isOnline()) {
            view.notOnlineWarning();
        } else if (!view.googleApiLoaded) {
            view.googleMapsError();
        }
    },

    showVenueGigs: function (gigs) {
        var len = gigs.length, dom_append = '';
        view.$selector.venuegigslist.empty();
        console.log(len + ' gigs read for venue = ' + view.venue_id);
        for (var i = 0; i < len; ++i) {
            var gig = gigs[i], gigdate = moment(gig.event_date);
            dom_append += '<li><a href="#event" data-gig-id="' + gig.id + '"><h2>' + gig.event_title + '</h2><p>' + gigdate.format(view.gigDateFormat) + '</p></a></li>';
        }
        if (len == 0) {
            dom_append = '<li>No future gigs found.</li>';
        }
        view.$selector.venuegigslist.append(dom_append);
        view.refreshVenueGigsList();
    },

    showVenues: function (venues) {
        var len = venues.length, preChar = '', preCharIndex = 0, dom_append = '';
        console.log(len + ' venues read for view');
        for (var i = 0; i < len; ++i) {
            var venue = venues[i], img_source = '';
            if (venue.image > ' ') {
                img_source = venue.image;
            } else {
                var char = venue.name.substring(0, 1).toLowerCase();
                if (char >= 'a' && char <= 'z') {
                    if (char !== preChar) {
                        preChar = char;
                        preCharIndex++;
                        if (preCharIndex > 7) preCharIndex = 0;
                    }
                    img_source = 'img/alphabet/' + char + preCharIndex + '.jpg';
                } else {
                    img_source = 'img/150x150.png';
                }
            }
            dom_append += '<li><a href="#venue" data-venue-id="' + venue.id + '"><img src="' + img_source + '" /><h2>' + venue.name + '</h2><p>' + venue.address + '</p></a></li>';
        }
        view.$selector.venueslist.append(dom_append);
        view.refreshVenueList();
    },

    showSavedVenues: function (venues) {
        var len = venues.length, preChar = '', preCharIndex = 0, dom_append = '';
        console.log(len + ' saved venues read for view');
        view.$selector.savedvenueslist.empty();
        for (var i = 0; i < len; ++i) {
            var venue = venues[i], img_source = '';
            if (venue.image > ' ') {
                img_source = venue.image;
            } else {
                var char = venue.name.substring(0, 1).toLowerCase();
                if (char >= 'a' && char <= 'z') {
                    if (char !== preChar) {
                        preChar = char;
                        preCharIndex++;
                        if (preCharIndex > 7) preCharIndex = 0;
                    }
                    img_source = 'img/alphabet/' + char + preCharIndex + '.jpg';
                } else {
                    img_source = 'img/150x150.png';
                }
            }
            dom_append += '<li><a href="#venue" data-venue-id="' + venue.id + '"><img src="' + img_source + '" /><h2>' + venue.name + '</h2><p>' + venue.address + '</p></a></li>';
        }
        if (len == 0) {
            dom_append = '<li>No saved venues found.</li>';
        }
        view.$selector.savedvenueslist.append(dom_append);
        view.refreshSavedVenuesList();
    },

    // Refresh ListView methods

    refreshVenueGigsList: function () {
        view.$selector.venuegigslist.listview().listview('refresh');
        view.$selector.venuegigslist.off('tap').on('tap', 'li', view.gigClickHandler);
    },

    refreshVenueList: function () {
        view.$selector.venueslist.listview().listview('refresh');
        view.$selector.venueslist.off('tap').on('tap', 'li', view.venueClickHandler);
    },

    refreshNewsList: function () {
        view.$selector.newslist.listview().listview('refresh');
        view.$selector.newslist.off('tap').on('tap', 'li', view.newsClickHandler);
    },

    refreshSavedGigsList: function () {
        view.$selector.savedgigslist.listview().listview('refresh');
        view.$selector.savedgigslist.off('tap').on('tap', 'li', view.gigClickHandler);
    },

    refreshSavedVenuesList: function () {
        view.$selector.savedvenueslist.listview().listview('refresh');
        view.$selector.savedvenueslist.off('taphold').on('taphold', 'li', view.saveVenueTapHoldHandler);
        view.$selector.savedvenueslist.off('tap').on('tap', 'li', view.venueClickHandler);
    },

    refreshGigsList: function () {
        view.$selector.gigslist.listview().listview('refresh');
        view.$selector.gigslist.off('tap').on('tap', 'li', view.gigClickHandler);
    },

    refreshTicketsList: function () {
        view.$selector.ticketslist.listview().listview('refresh');
        view.$selector.ticketslist.off('tap').on('tap', 'li', view.ticketClickHandler);
    },

    // Click Handlers

    newsClickHandler: function (e) {
        e.preventDefault();
        var id = $(this).find('a').data('post-id');
        console.log('Displaying post = ' + id);
        var post = app.getNewsById(id);
        view.$selector.postheading.html(post.title);
        view.$selector.postexcerpt.html('<p>' + post.excerpt + '</p>');
        view.$selector.postimage.html('<img src="' + post.image + '" />');
        view.$selector.postcontent.html(post.description).find('iframe').css('maxWidth', '100%');
        $.mobile.pageContainer.pagecontainer("change", "#tour-news-post");
        return false;
    },

    venueClickHandler: function (e) {
        e.preventDefault();
        view.venue_id = $(this).find('a').data('venue-id');
        var venue = app.getVenueById(view.venue_id);
        console.log('Displaying venue = ' + view.venue_id);

        view.$selector.venuename.html(venue.name);
        view.$selector.venueaddress.html(venue.address);

        var saved_venue = app.getSavedVenueById(view.venue_id);
        if (typeof saved_venue === 'undefined') {
            view.$selector.savevenuebtn.removeClass('saved');
        } else {
            view.$selector.savevenuebtn.addClass('saved');
        }
        view.$selector.savevenuebtn.data('venue-id', venue.id);

        view.$selector.venuegigslist.empty(); // Clear any old gigs
        app.getGigsForVenue(); // Load the new gigs

        // Load the new map
        view.venueMapCount++;
        view.showVenueMap(view.$selector.venuemap, view.venue_id, 'venue-canvas-map-' + view.venueMapCount);

        $.mobile.pageContainer.pagecontainer("change", "#venue");
        return false;
    },

    confirmSavedVenueDelete: function (venue_id) {
        console.log('Removing saved venue with id = ' + venue_id);
        var output_list = [];
        for (var i = 0; i < app.saved_venues.length; ++i) {
            var old_venue = app.saved_venues[i];
            if (old_venue['id'] != venue_id) {
                output_list.push(old_venue);
            }
        }
        app.indexed_saved_venues = null;
        app.setSavedVenuesList(output_list);
        app.saved_venues = output_list;
        app.getSavedVenuesForView();
        $.fn.dpToast('Venue removed from your Favourites.');
    },

    saveVenueTapHoldHandler: function (e) {
        e.preventDefault();
        var venue_id = $(this).find('a').data('venue-id'), venue = app.getVenueById(venue_id);
        if (typeof venue !== 'undefined') {
            if (typeof navigator.notification !== 'undefined') {
                navigator.notification.confirm(
                    'Remove venue from your favourites?',
                    function (button) {
                        if (button === 1) {
                            view.confirmSavedVenueDelete(venue_id);
                        }
                    },
                    'Delete venue',
                    ['Remove', 'Cancel']
                );
            } else {
                var r = confirm("Remove venue from your favourites?");
                if (r == true) {
                    view.confirmSavedVenueDelete(venue_id);
                }
            }
        }
        return false;
    },

    saveVenueClickHandler: function (e) {
        e.preventDefault();
        var venue_id = $(this).data('venue-id'), venue = app.getVenueById(venue_id);
        if (typeof venue !== 'undefined') {
            var currently_saved = $(this).hasClass('saved'), output_list = [];
            if (currently_saved) {
                console.log('Removing saved venue with id = ' + venue_id);
                output_list = [];
                for (var i = 0; i < app.saved_venues.length; ++i) {
                    var old_venue = app.saved_venues[i];
                    if (old_venue['id'] != venue_id) {
                        output_list.push(old_venue);
                    }
                }
                $(this).removeClass('saved');
                $.fn.dpToast('Venue removed from your Favourites.');
            } else {
                console.log('Saving venue with id = ' + venue_id);
                output_list = app.saved_venues;
                output_list.push(venue);
                output_list.sort(function (a, b) {
                    if (a.name < b.name) return -1;
                    if (a.name > b.name) return 1;
                    return 0;
                });
                $(this).addClass('saved');
                $.fn.dpToast('Venue added to your Favourites.');
            }
            app.indexed_saved_venues = null;
            app.setSavedVenuesList(output_list);
            app.saved_venues = output_list;
            app.getSavedVenuesForView();
        }
        return false;
    },

    saveGigClickHandler: function (e) {
        e.preventDefault();
        var gig_id = $(this).data('gig-id'), gig = app.getGigById(gig_id);
        if (typeof gig !== 'undefined') {
            var currently_saved = $(this).hasClass('saved'), output_list = [];
            if (currently_saved) {
                console.log('Removing saved gig with id = ' + gig_id);
                output_list = [];
                for (var i = 0; i < app.saved_gigs.length; ++i) {
                    var old_gig = app.saved_gigs[i];
                    if (old_gig['id'] != gig_id) {
                        output_list.push(old_gig);
                    }
                }
                $(this).removeClass('saved');
                $.fn.dpToast('Gig removed from your Saved Events.');
            } else {
                console.log('Saving gig with id = ' + gig_id);
                output_list = app.saved_gigs;
                output_list.push(gig);
                output_list.sort(function (a, b) {
                    if (a.event_date < b.event_date) return -1;
                    if (a.event_date > b.event_date) return 1;
                    return 0;
                });
                $(this).addClass('saved');
                $.fn.dpToast('Gig added to your Saved Events.');
            }
            app.indexed_saved_gigs = null;
            app.setSavedGigsList(output_list);
            app.saved_gigs = output_list;
            app.getSavedGigsForView();
        }
        return false;
    },

    gigClickHandler: function (e) {
        e.preventDefault();
        view.gig_id = $(this).find('a').data('gig-id');
        var gig = app.getGigById(view.gig_id), venue = app.getVenueById(gig.venue_id), gigdate = moment(gig.event_date);
        if (typeof venue === 'undefined') {
            $.fn.dpToast('Problem loading venue for this gig.');
            return false;
        }
        console.log('Displaying gig = ' + view.gig_id + ' at venue = ' + venue.id);
        view.$selector.eventtitle.html(gig.event_title);
        var saved_gig = app.getSavedGigById(view.gig_id);
        if (typeof saved_gig === 'undefined') {
            view.$selector.savegigbtn.removeClass('saved');
            view.$selector.savegigbtn.data('gig-id', gig.id);
        } else {
            view.$selector.savegigbtn.addClass('saved');
            view.$selector.savegigbtn.data('gig-id', gig.id);
        }
        view.$selector.eventdate.html(gigdate.format(view.gigDateFormat));
        view.$selector.eventvenueaddress.html(venue.address);
        view.$selector.eventvenuename.html(venue.name);
        view.$selector.eventvenuebtn.data('venue-id', venue.id);

        view.$selector.ticketslist.empty();

        view.$selector.eventinformation.empty().hide();
        view.$selector.eventinformationloading.show();
        view.$selector.eventnoinformation.hide();

        app.getGigDetails(gig.id);

        // Load the new map
        view.gigMapCount++;
        view.showVenueMap(view.$selector.eventvenuemap, gig.venue_id, 'event-canvas-map-' + view.gigMapCount);

        $.mobile.pageContainer.pagecontainer("change", "#event");
        return false;
    },

    // Gig Details callback

    gigDetailsCallback: function (gig) {

        var len = gig.tickets.length;
        console.log(len + ' tickets read for gig = ' + gig.id);
        for (var i = 0; i < len; ++i) {
            var ticket = gig.tickets[i];
            view.$selector.ticketslist.append('<li><a href="#" data-ticket-url="' + ticket.url + '"><img src="img/agency/' + ticket.agency_id + '.jpg" /><h2>' + ticket.agency_name + '</h2><p>Tickets are on-sale now.</p></a></li>');
        }
        view.refreshTicketsList();

        view.$selector.eventinformationloading.hide();
        if (typeof gig.description !== 'undefined') {
            view.$selector.eventtitle.html(gig.lineup);
            view.$selector.eventinformation.html(gig.description).show();
        } else {
            view.$selector.eventnoinformation.show();
        }
    },

    gigDetailsCallbackError: function () {
        view.$selector.eventinformationloading.hide();
        view.$selector.eventnoinformation.show();
    },

    // Intents

    navigateUsingBrowser: function (url) {
        if (typeof navigator !== 'undefined' && typeof navigator.app !== 'undefined' && typeof navigator.app.loadUrl !== 'undefined') {
            // Mobile device.
            navigator.app.loadUrl(url, {openExternal: true});
        } else {
            // Possible web browser
            window.open(url, "_blank", "location=yes");
        }
    },

    navigateUsingMozActivity: function (url) {
        var activity = new MozActivity({
            name: "view",
            data: {
                type: "url",
                url: url
            }
        });
        activity.onsuccess = function () {
            console.log("Successfully navigated to " + url + " using MozActivity");
        };
        activity.onerror = function () {
            console.log('Failed to navigate to ' + url);
            console.log(this.error);
        };
    },

    ticketClickHandler: function (e) {
        e.preventDefault();
        var url = $(this).find('a').data('ticket-url');
        app.trackEvent('App', 'Ticket Navigation');
        if (typeof window.plugins !== 'undefined' && typeof window.plugins.webintent !== 'undefined') {
            console.log('Navigating to ' + url + ' via intent');
            try {
                window.plugins.webintent.startActivity({
                        action: window.plugins.webintent.ACTION_VIEW,
                        url: 'http://' + url},
                    function () {
                        console.log('Successfully navigated to ' + url);
                    },
                    function () {
                        if (typeof MozActivity !== 'undefined') {
                            view.navigateUsingMozActivity('http://' + url);
                        } else {
                            view.navigateUsingBrowser('http://' + url);
                        }
                    }
                );
            } catch (error) {
                console.log(error);
            }
        } else {
            console.log('Web Intents plugin not available');
            if (typeof MozActivity !== 'undefined') {
                view.navigateUsingMozActivity('http://' + url);
            } else {
                view.navigateUsingBrowser('http://' + url);
            }
        }
        return false;
    },

    navigateToTwitter: function (e) {
        app.trackEvent('App', 'Twitter Navigation');
        e.preventDefault();
        var username = $(this).data('twitter-user');
        if (typeof window.plugins !== 'undefined' && typeof window.plugins.webintent !== 'undefined') {
            console.log('Navigating to Twitter via intent');
            try {
                window.plugins.webintent.startActivity({
                        action: window.plugins.webintent.ACTION_VIEW,
                        url: 'twitter://user?screen_name=' + escape(username)},
                    function () {
                        console.log('Successfully navigated to Twitter');
                    },
                    function () {
                        console.log('Failed to navigate to Twitter - navigating using browser');
                        window.plugins.webintent.startActivity({
                                action: window.plugins.webintent.ACTION_VIEW,
                                url: 'https://twitter.com/' + username},
                            function () {
                                console.log('Successfully navigated to Twitter');
                            },
                            function () {
                                if (typeof MozActivity !== 'undefined') {
                                    view.navigateUsingMozActivity('https://twitter.com/' + username);
                                } else {
                                    view.navigateUsingBrowser('https://twitter.com/' + username);
                                }
                            }
                        );
                    }
                );
            } catch (error) {
                console.log(error);
            }
        } else {
            console.log('Web Intents plugin not available');
            if (typeof MozActivity !== 'undefined') {
                view.navigateUsingMozActivity('https://twitter.com/' + username);
            } else {
                view.navigateUsingBrowser('https://twitter.com/' + username);
            }
        }
        return false;
    },

    navigateToMap: function (e) {
        app.trackEvent('App', 'Map Navigation');
        e.preventDefault();
        var img = $(this).find('img'), lat = img.data('lat'), lng = img.data('lng'), venue = img.data('venue');
        if (typeof window.plugins !== 'undefined' && typeof window.plugins.webintent !== 'undefined') {
            console.log('Navigating to map display via intent');
            try {
                window.plugins.webintent.startActivity({
                        action: window.plugins.webintent.ACTION_VIEW,
                        url: 'geo:' + lat + ',' + lng + '?z=18&q=' + escape(venue)},
                    function () {
                        console.log('Successfully navigated to map display');
                    },
                    function () {
                        console.log('Failed to navigate to map display - navigating using browser');
                        window.plugins.webintent.startActivity({
                                action: window.plugins.webintent.ACTION_VIEW,
                                url: 'https://maps.google.com/?q=' + escape(venue) + '&ll=' + lat + ',' + lng},
                            function () {
                                console.log('Successfully navigated to map display');
                            },
                            function () {
                                if (typeof MozActivity !== 'undefined') {
                                    view.navigateUsingMozActivity('https://maps.google.com/?q=' + escape(venue) + '&ll=' + lat + ',' + lng);
                                } else {
                                    view.navigateUsingBrowser('https://maps.google.com/?q=' + escape(venue) + '&ll=' + lat + ',' + lng);
                                }
                            }
                        );
                    }
                );
            } catch (error) {
                console.log(error);
            }
        } else {
            console.log('Web Intents plugin not available');
            if (typeof MozActivity !== 'undefined') {
                view.navigateUsingMozActivity('https://maps.google.com/?q=' + escape(venue) + '&ll=' + lat + ',' + lng);
            } else {
                view.navigateUsingBrowser('https://maps.google.com/?q=' + escape(venue) + '&ll=' + lat + ',' + lng);
            }
        }
        return false;
    }
};
