/* jshint camelcase: false */
/* global jQuery, Handlebars, store, utils, L */
'use strict';

(function ($, document, undefined) {
  var map,
      currentPosition,
      webWatchHost = 'http://ltcapi.herokuapp.com',
      templates = {
      'routes-list': Handlebars.compile($('#routes-list').html()),
      'favourites-list': Handlebars.compile($('#favourites-list').html()),
      'directions': Handlebars.compile($('#directions').html()),
      'stops-list': Handlebars.compile($('#stops-list').html()),
      'detail': Handlebars.compile($('#detail').html()),
      'detail-header': Handlebars.compile($('#detail-header').html())
    };
  $.support.cors = true;

  function getCountdown(time) {
    var dateStart, timeStart, dateEnd, timeEnd, diffInMinutes;
    dateStart = new Date();
    timeStart = dateStart.getHours() * 60 + dateStart.getMinutes();
    dateEnd = new Date(dateStart.toDateString() + ' ' + time);
    timeEnd = dateEnd.getHours() * 60 + dateEnd.getMinutes();
    diffInMinutes = timeEnd - timeStart;
    if (diffInMinutes < 0) {
      diffInMinutes += 24 * 60;
    }
    return diffInMinutes;
  }

  function round(number) {
    return parseInt(number, 10);
  }

  function getPosition(position) {
    currentPosition = position.coords;
  }

  function getPositionError(e) {
    if (e.code === 3) {
      utils.status.show('Geolocation error: timeout.');
    } else {
      utils.status.show('Geolocation error.');
    }
  }

  function routeName(number) {
    var routes = store.get('routes');
    return routes[number].name;
  }

  function showLoading() {
    $('#loading').removeClass('fade-out');
    $('#loading').addClass('fade-in');
  }

  function hideLoading() {
    $('#loading').removeClass('fade-in');
    $('#loading').addClass('fade-out');
  }

  function onLocationFound(e) {
    var radius = e.accuracy / 2;
    L.marker(e.latlng).addTo(map)
      .bindPopup('You').openPopup();
    L.circle(e.latlng, radius).addTo(map);
  }

  function onLocationError(e) {
    utils.status.show(e.message);
  }

  function initMap(latlng) {
    if (map !== undefined) {
      map.remove();
    }
    map = L.map('map', {
      center: [latlng.lat, latlng.lng],
      zoom: 17,
    });
    L.tileLayer('http://{s}.tile.osm.org/{z}/{x}/{y}.png', {
        maxZoom: 20,
        attribution: '<a href="http://osm.org/copyright">OpenStreetMap</a>',
      }).addTo(map);
    map.on('locationfound', onLocationFound);
    map.on('locationerror', onLocationError);
  }

  function init() {
    if ('geolocation' in navigator) {
      navigator.geolocation.getCurrentPosition(
        getPosition, getPositionError, {timeout: 10000}
      );
    }

    Handlebars.registerHelper('round', round);
    Handlebars.registerHelper('routeName', routeName);
    Handlebars.registerHelper('countdown', getCountdown);
    Handlebars.registerHelper('lower', String.toLowerCase);
    Handlebars.registerHelper('ifFavourited', function (number, options) {
      var favourites = store.get('favourites') || {};
      if (favourites[number] !== undefined) {
        return options.fn(this);
      } else {
        return options.inverse(this);
      }
    });

    utils.status.init();
    loadRoutes();
    showFavourites();
    updateEventDetail();

    $('#search-stop').on('focusout', searchStop);
    $('#search-stop').on('keyup', searchStop);
    $('#search-stop-reset').on('click', function () {
      $('#search-stop').val('');
      searchStop();
    });

    $('#list-refresh').on('click', function () {
      loadStops($(this).data('route'));
    });

    $('#btn-lists-back').on('click', function () {
      $('#lists').attr('class', 'right');
      $('[data-position="current"]').attr('class', 'current');
      showFavourites();
      updateEventDetail();
    });

  }

  function updateEventDetail() {
    $('[data-action="detail"]').on('click', function () {
      var elem = $(this);
      loadStop(elem.data('route'), elem.data('stop-number'));
    });
  }

  function favStop(stopTag) {
    var stopNumber = stopTag.data('stop-number'),
        favourites = store.get('favourites') || {};
    if (favourites[stopNumber] === undefined) {
      favourites[stopNumber] = {
        'number': stopNumber,
        'name': stopTag.data('name'),
        'route': stopTag.data('route')
      };
      stopTag.removeClass('bookmark');
      stopTag.addClass('bookmarked');
      stopTag.data('favourited', 'true');
    } else {
      delete favourites[stopNumber];
      stopTag.removeClass('bookmarked');
      stopTag.addClass('bookmark');
      stopTag.data('favourited', 'false');
    }
    store.set('favourites', favourites);
  }

  function loadRoutes() {
    showLoading();
    var jqxhr,
        webWatchURL,
        routes = {};
    webWatchURL = webWatchHost + '/routes/';
    jqxhr = $.ajax({
      url: webWatchURL,
      dataType: 'json',
      timeout: 10000
    })
    .done(
      function (data) {
        hideLoading();
        showRoutes(data);
        $(data).each(function () {
          routes[this.route] = this;
        });
        store.set('routes', routes);
      }
    )
    .fail(
      function () {
        hideLoading();
        utils.status.show('LTC Online seems down, sorry');
        showRoutes([]);
      }
    );
  }

  function showRoutes(routes) {
    var source = $('#routes-list'),
        html = templates['routes-list']({
        'routes': routes
      });
    source.empty();
    source.append(html);
    $('.lane').on('click', function () {
      loadStops($(this).data('lane'));
    });

  }

  function showFavourites() {
    var source = $('#favourites-list'),
        html = templates['favourites-list']({
        'favourites': store.get('favourites'),
        'favouritesNotEmpty': !($.isEmptyObject(store.get('favourites')))
      });
    source.empty();
    source.append(html);
  }

  function loadStops(routeNumber) {
    showLoading();
    var jqxhr,
        params,
        webWatchURL;
    webWatchURL = webWatchHost + '/routes/' + routeNumber;
    params = {
      'timestamp': new Date().getTime()
    };
    if (currentPosition) {
      params.latitude = currentPosition.latitude;
      params.longitude = currentPosition.longitude;
    }
    jqxhr = $.ajax({
      url: webWatchURL,
      data: params,
      dataType: 'json',
      timeout: 60000
    })
    .done(
      function (data) {
        hideLoading();
        showDirections(data);
        showStops(routeNumber, data);
      }
    )
    .fail(
      function () {
        hideLoading();
        utils.status.show('LTC Online seems down, sorry');
      }
    );
  }

  function showDirections(stops) {
    var source = $('#directions'),
        context, html, directions = [];
    $(stops).each(function (i, stop) {
      var direction = stop.direction;
      if (directions.indexOf(direction) < 0) {
        directions.push(direction);
      }
    });
    context = {'directions': directions};
    html = templates.directions(context);
    source.empty();
    source.append(html);
    $('[data-direction]').on('click', function () {
      filterByDirection(this);
    });
  }

  function showStops(routeNumber, stops) {
    var source = $('#stops-list'),
        context = {'stops': stops},
        html = templates['stops-list'](context);
    source.empty();
    source.append(html);
    $('#routeName').html(store.get('routes')[routeNumber].name);
    $('#list-refresh').data('route', routeNumber);
    $('#lists').attr('class', 'current');
    try {
      $('[data-position="current"]').attr('class', 'left');
    } catch (ex) {
      console.error('Unable to find the elemet', ex);
    }
    source.scrollTop(0);
    $('[data-favourited]').on('click', function () {
      favStop($(this));
    });
  }

  function filterByDirection(elem) {
    var direction = $(elem).data('direction');
    if (direction === 'all') {
      $('#stops-list header, #stops-list ul').show();
    } else {
      $('#stops-list header, #stops-list ul').hide();
      $('.' + direction).show();
    }
  }

  function searchStop() {
    var searchTerm = $('#search-stop').val().trim().toLowerCase();
    if (searchTerm.length === 0) {
      $('#stops-list header, #stops-list ul').show();
    } else {
      $('#stops-list header, #stops-list ul').hide();
      $('[data-stop-search*="' + searchTerm + '"]').show();
    }
  }

  function loadStop(routeNumber, stopNumber) {
    showLoading();
    var jqxhr,
        params,
        webWatchURL;
    webWatchURL = webWatchHost + '/routes/' + routeNumber;
    params = {
      'timestamp': new Date().getTime(),
      'stop': stopNumber
    };
    if (currentPosition) {
      params.latitude = currentPosition.latitude;
      params.longitude = currentPosition.longitude;
    }
    jqxhr = $.ajax({
      url: webWatchURL,
      data: params,
      dataType: 'json'
    })
    .done(
      function (data) {
        hideLoading();
        showStop(routeNumber, stopNumber, data[0]);
      }
    )
    .fail(
      function () {
        hideLoading();
        utils.status.show('LTC Online seems down, sorry');
      }
    );
  }

  function showStop(routeNumber, stopNumber, stop) {
    stop.route = routeNumber;
    var source = $('#detail'),
        sourceHeader = $('#detail-header'),
        context = stop,
        html = templates.detail(context),
        htmlHeader = templates['detail-header'](context),
        currentLatLng,
        latlng = new L.LatLng(stop.latitude, stop.longitude);
    sourceHeader.empty();
    sourceHeader.append(htmlHeader);
    source.empty();
    source.append(html);
    $('#details').attr('class', 'current');
    try {
      $('[data-position="current"]').attr('class', 'left');
    } catch (ex) {
      console.error('Unable to find the elemet', ex);
    }
    $('#details-refresh').on('click', function () {
      var elem = $(this);
      loadStop(elem.data('route'), elem.data('stop-number'));
    });
    initMap(latlng);
    L.marker(latlng).addTo(map)
      .bindPopup('#' + stopNumber);
    if (currentPosition) {
      currentLatLng = new L.LatLng(currentPosition.latitude,
                                   currentPosition.longitude);
      L.marker(currentLatLng).addTo(map)
        .bindPopup('You').openPopup();
    }
    $('#btn-details-back').on('click', function () {
      $('#details').attr('class', 'right');
      $('[data-position="current"]').attr('class', 'current');
      showFavourites();
      updateEventDetail();
    });
  }

  $(document).ready(init);

})(jQuery, document);

