var dbName = "manana",
    dbVersion = 1,
    db = undefined;


function getDB(onsuccess) {
    if (typeof(db) != "undefined") {
        console.debug("DB::Reusing connection");
        onsuccess(db);
        return;
    }

    var request = indexedDB.open(dbName, dbVersion);

    request.onerror = function(e) {
        console.error("DB::Can't open IndexedDB!!!", e);
    };

    request.onsuccess = function(e) {
        console.debug("DB::OK");
        onsuccess(e.target.result);
    };

    request.onupgradeneeded = function(e) {
        console.debug("DB::onUpgradeNeeded");

        db = e.target.result;

        if (!db.objectStoreNames.contains("links")) {
            console.debug("DB::Creating objectStore for links");

            var objectStore = db.createObjectStore("links", {
                keyPath: "id",
                autoIncrement: true
            });
        }

        if (!db.objectStoreNames.contains("kv")) {
            console.debug("DB::Creating objectStore for kv");

            var objectStore = db.createObjectStore("kv", {
                keyPath: "key"
            });
        }
    };
}

var DB = {
    AllTags: function(onsuccess) {
        getDB(function(db) {
            var objectStore = db.transaction("links").objectStore("links");
            console.debug("DB::Listing tags");

            var allTags = {},
                tagPush = function(newTag) {
                    if (typeof(allTags[newTag]) != "undefined") {
                        return;
                    }

                    allTags[newTag] = true;
                };

            objectStore.openCursor().onsuccess = function(e) {
                var cursor = e.target.result;

                if (cursor) {
                    var tags = cursor.value.tags;

                    if (typeof(tags) == "undefined") {
                        cursor.continue();
                    }

                    for (var i = 0; i < tags.length; i++) {
                        var tag = tags[i];

                        tagPush(tag);
                    }

                    cursor.continue();
                } else {
                    onsuccess(Object.keys(allTags));
                }
            };
        });
    },

    Add: function(data, onsuccess) {
        getDB(function(db) {
            console.debug("DB::Saving link");

            if (typeof(data.tags) != "undefined") {
                data.tags = normalizeTags(data.tags);
            }

            var transaction = db.transaction(["links"], "readwrite"),
                objectStore = transaction.objectStore("links"),
                request = objectStore.put(data);

            request.onsuccess = function(e) {
                console.debug("DB::Link saved with id: " + e.target.result);
                onsuccess(e.target.result);
            };
        });
    },

    Get: function(ids, onsuccess) {
        getDB(function(db) {
            var objectStore = db.transaction("links").objectStore("links"),
                singleValue = !isArray(ids),
                rs          = {},
                finish      = function (e) {
                    var obj = e.target.result;

                    // assign the current object to the record set
                    if (obj)
                        rs[obj.id] = obj;

                    // call `onsuccess` if we have finished processing the ids
                    // (i.e. total is 0). If the request was with a single value,
                    // unpack the record set.
                    !--total && onsuccess(singleValue ? rs[ids[0]] : rs);
                },
                request, total;


            if (singleValue)
                ids = [ids];

            total = ids.length;

            for (var i = 0; i < ids.length; i++) {
                id = ids[i];
                console.debug("DB::Getting link with id " + id);
                request = objectStore.get(parseInt(id));
                request.onsuccess = finish;
                request.onerror   = finish;
            }
        });
    },

    Delete: function(id, onsuccess) {
        getDB(function(db) {
            var objectStore = db.transaction("links", "readwrite").objectStore("links");
            console.debug("DB::Deleting link with id " + id);

            objectStore.delete(parseInt(id)).onsuccess = function(e) {
                onsuccess();
            };
        });
    },

    All: function(onsuccess) {
        getDB(function(db) {
            var objectStore = db.transaction("links").objectStore("links");
            console.debug("DB::Listing links");

            objectStore.openCursor().onsuccess = function(e) {
                var cursor = e.target.result;

                if (cursor) {
                    console.debug("DB::Found link: " + cursor.value.title);

                    onsuccess(cursor.value);
                    cursor.continue();
                }
            };
        });
    },

    LinksByTag: function(tag, onsuccess) {
        getDB(function(db) {
            var objectStore = db.transaction("links").objectStore("links");
            console.debug("DB::Listing links");

            objectStore.openCursor().onsuccess = function(e) {
                var cursor = e.target.result;

                if (cursor) {
                    var link = cursor.value;
                    console.debug("DB::Found link: " + link.title);

                    if (link.tags.indexOf(tag) != -1) {
                        onsuccess(cursor.value);
                    }

                    cursor.continue();
                }
            };
        });
    },

    Count: function(onsuccess) {
        getDB(function(db) {
            var objectStore = db.transaction("links").objectStore("links");
            console.debug("DB::Count");

            objectStore.count().onsuccess = function(e) {
                onsuccess(e.target.result);
            };
        });
    },


    // Key Value API
    ksetValue: function(key, value, onsuccess) {
        getDB(function(db) {
            var transaction = db.transaction(["kv"], "readwrite"),
                objectStore = transaction.objectStore("kv"),
                request = objectStore.put({key: key, value: value});

            request.onsuccess = function(e) {
                onsuccess && onsuccess(value);
            };
        });
    },

    ksetFunc: function(key, func, onsuccess) {
        getDB(function(db) {
            DB.kget(key, function (value) {
                var transaction = db.transaction(["kv"], "readwrite"),
                    objectStore = transaction.objectStore("kv"),
                    newValue    = func(value),
                    request     = objectStore.put({ key: key, value: newValue });

                request.onsuccess = function(e) {
                    onsuccess && onsuccess(newValue, value);
                };
            });

        });
    },

    kset: function(key, value, onsuccess) {
        getDB(function(db) {
            if (isFunction(value)) {
                DB.ksetFunc(key, value, onsuccess);
            } else {
                DB.ksetValue(key, value, onsuccess);
            }
        });
    },

    kget: function(key, onsuccess) {
        getDB(function(db) {
            var objectStore = db.transaction("kv").objectStore("kv");
            objectStore.get(key).onsuccess = function(e) {
                onsuccess(e.target.result && e.target.result.value);
            };
        });
    },

};

function NewLink(data) {
    if (typeof(data) == "undefined") {
        data = {};
    }

    var link = {
        url     : data.url     || "",
        title   : data.title   || "",
        text    : data.text    || "",
        created : data.created || Date.now(),
        readat  : data.readat  || null,
        tags    : data.tags    || []
    };

    return link;
}

function normalizeTags(tags) {
    var xs = [];

    for (var i = 0; i < tags.length; i++) {
        var x = tags[i],
            cleaned = x.trim().toLowerCase().replace(/[\s,\_]/g, "-");

        xs.push(cleaned);
    }

    return xs;
}
