/*
 *  Resources:
 *
 *    http://www.html5rocks.com/en/tutorials/audio/scheduling/
 *    http://www.html5rocks.com/en/tutorials/speed/rendering/
 *    https://developer.mozilla.org/en/docs/Web/API/window.requestAnimationFrame
 *    https://developer.mozilla.org/en-US/docs/Web/Guide/CSS/Using_CSS_transitions
 */

var metroApp;

var INITIAL_BPM = 90;
var TAP_MAX_DELAY = 3.0 * 1000;
var BPM_UPDATED_DELAY = 0.5 * 1000;
var MAX_BPM = 500;
var MIN_BPM = 10;

/* Latency between audio and video: positive value means audio is output X
 * seconds after something happens on the screen.  There's no way to
 * determine this accurately, current value seems to be about right for
 * ZTE Open C.
 */
var AUDIO_TO_VIDEO_LATENCY = 0.1;

var touchCapable = ("ontouchstart" in window || navigator.msMaxTouchPoints);
var reqAnimFrame = window.requestAnimationFrame || window.mozRequestAnimationFrame || window.webkitRequestAnimationFrame;

/*
 *  Metronome application class.
 *
 *  Provides the audio part but also currently assumes concrete DOM
 *  identifiers so this only works as a singleton.
 */

function MetronomeApplication() {
}
MetronomeApplication.prototype.createDomElements = function () {
	var i, elem, button, br;
	var metro = this;

	// Separate functions used for scope hack
	function getNthHandler(x) {
		return function () { metro.handleNth(x); };
	}
	function getBpmHandler(x) {
		return function () { metro.handleBpm(x); };
	}
	function getBpmStepHandler(x) {
		return function () { metro.handleBpmStep(x); };
	}
	elem = document.getElementById('emph-buttons');
	for (i = 1; i <= 8; i++) {
		button = document.createElement('button');
		button.className = 'nth';
		button.innerHTML = (i == 1 ? 'None' : String(i));
		button.addEventListener('click', getNthHandler(i));
		elem.appendChild(button);
	}
	elem = document.getElementById('bpm-step-buttons');
	[ '/2', '-10', '-1', '+1', '+10', '*2' ].forEach(function (step) {
		var button = document.createElement('button');
		button.className = 'bpm-step';
		button.innerHTML = step;
		button.addEventListener('click', getBpmStepHandler(step));
		elem.appendChild(button);
	});
	br = document.createElement('br');
	elem = document.getElementById('bpm-buttons');
	for (i = 60; i <= 200; i += 10) {
		button = document.createElement('button');
		button.className = 'bpm';
		button.innerHTML = String(i);
		button.addEventListener('click', getBpmHandler(i));
		elem.appendChild(button);
	}

	/* Prefer ontouchstart to onclick when device supports
	 * it, because it most probably matches what's expected
	 * of a tap button.
	 */
	elem = document.getElementById('tap-button');
	if (touchCapable) {
		elem.addEventListener('touchstart', function () { metro.handleTap(); });
	} else {
		elem.addEventListener('click', function () { metro.handleTap(); });
	}
	elem = document.getElementById('run-button');
	elem.addEventListener('click', function () { metro.handleStopStart(); });

	bpmNumberElement = document.getElementById('bpm-number');  // global, for minimal requestAnimationFrame handler
	if (!reqAnimFrame) {
		alert('No support for requestAnimationFrame, animations disabled.');
	} else {
		reqAnimFrame(function () { metro.animFrameCallback(); });
	}
};
MetronomeApplication.prototype.init = function () {
	var metro = this;

	this.createDomElements();

	this.bpmUpdatedTimeout = null;
	this.animCurrBeatCount = null;
	this.bpmTapFirst = null;
	this.bpmTapCount = null;
	this.bpmTapTimeout = null;

	this.elemRunButton = document.getElementById('run-button');
	this.elemBpmDiv = document.getElementById('bpm-div');
	this.elemBpmNumber = document.getElementById('bpm-number');
	this.elemTapDiv = document.getElementById('tap-div');

	window.AudioContext = window.AudioContext || window.webkitAudioContext;
	if (!window.AudioContext) {
		alert('No audio API support (AudioContext), application will be unusable.');
	}
	this.audioContext = new AudioContext();
	this.sampleRate = this.audioContext.sampleRate;

	// background audio (no audio break if screen is locked etc)
	// - https://runawaydev.wordpress.com/2013/05/11/firefox-os-how-to-prevent-musicvideo-to-stop-playing-when-the-app-goes-into-the-background/comment-page-1/
	// - http://dxr.mozilla.org/mozilla-central/source/content/media/webaudio/test/test_mozaudiochannel.html
	this.audioContext.mozAudioChannelType = 'content';
	if (this.audioContext.mozAudioChannelType !== 'content') {
		alert("Failed to set AudioContext mozAudioChannelType, " +
		      "audio may break when screen becomes locked.");
	}

	this.gainNode = this.audioContext.createGain();
	this.gainNode.gain.value = 0.0;
	this.gainNode.connect(this.audioContext.destination);

	// The sample scheduler runs regardless of whether or not the metronome
	// is running.  When the metronome is started, the scheduler can be called
	// manually to ensure a quick start.
	this.sampleSchedulerInterval = 1000;  // msec
	this.sampleSchedulerMargin = 3.000;   // sec
	this.sampleSchedulerTime = this.audioContext.currentTime;
	setInterval(function () { metro.sampleScheduler(); }, this.sampleSchedulerInterval);

	this.nth = 1;
	this.running = false;
	this.setBpm(INITIAL_BPM);
	this.startMetronome();
	this.updateNthActive();
	this.updateBpmNumber();
};
MetronomeApplication.prototype.setBpm = function (bpm) {
	this.bpm = Math.max(MIN_BPM, Math.min(MAX_BPM, Math.round(bpm)));
};
MetronomeApplication.prototype.getBpm = function (bpm) {
	return this.bpm;
};
MetronomeApplication.prototype.isBeatCountEmphasized = function (beatCount) {
	return (this.nth <= 1 ? false : ((beatCount % this.nth) == 0));
};
MetronomeApplication.prototype.sampleScheduler = function () {
	/* Samples have been scheduled in ]-inf,this.sampleSchedulerTime[.
	 * Schedule all samples from this.sampleSchedulerTime up to (but
	 * excluding) current time + a suitable safety window.
	 */
	var metro = this;
	var start = this.sampleSchedulerTime;
	var now = this.audioContext.currentTime;
	var end = now + this.sampleSchedulerMargin;
	var clickNumber, clickTime;
	var secPerBeat = 60 / this.bpm;
	var src;
	var emph;
	var sanity;

	if (this.running) {
		clickNumber = Math.floor((start - this.metroStartTime) / secPerBeat) - 1;
		for (sanity = 1000; sanity >= 0; sanity--, clickNumber++) {
			clickTime = clickNumber * secPerBeat + this.metroStartTime;
			if (clickTime < start) { continue; }
			if (clickTime >= end) { break; }
			//alert(JSON.stringify({start:start,now:now,end:end,clickTime:clickTime}));

			emph = this.isBeatCountEmphasized(clickNumber);
			src = this.createSingleClickSource(emph);
			src.start(clickTime);

			/* We could track active sources by using the 'onended' property
			 * and a callback.  This would allow us to know which samples are
			 * still active and stop them when the metronome is stopped.
			 * However, we can just connect the samples to a gain node which
			 * is muted/disconnected when the metronome stops.  This is much	
			 * easier and imposes less of a load with high BPM.
			 */
		}

		if (sanity < 0) {
			alert('Internal error in sample scheduler: loop sanity reached.');
		}
	}

	this.sampleSchedulerTime = end;
};
MetronomeApplication.prototype.createSingleClickSource = function (emph) {
	var clickLength;
	var clickBuffer;
	var clickSource;

	clickLength = Math.floor(this.sampleRate / 1000);
	clickBuffer = this.audioContext.createBuffer(1 /*channels*/, clickLength /*length*/, this.sampleRate);
	this.fillMetroClick(clickBuffer.getChannelData(0),
	                    0,
	                    clickLength,
	                    (emph ? 1500 : 1000) /*freq*/,
	                    this.sampleRate);
	clickSource = this.audioContext.createBufferSource();
	clickSource.buffer = clickBuffer;
	clickSource.loop = false;
	clickSource.connect(this.gainNode);
	return clickSource;
};
MetronomeApplication.prototype.fillMetroClick = function (buf, offset, nsamples, freq, sampleRate) {
	var i, t, v;

	for (i = 0; i < nsamples; i++) {
		t = i / sampleRate;  // seconds
		v = Math.sin(t * 2.0 * Math.PI * freq);
		buf[offset + i] = v;
	}
};
MetronomeApplication.prototype.startMetronome = function () {
	// Must be set early so that startMetronomeScheduled() can trigger
	// sample scheduling.
	this.running = true;
	this.elemRunButton.innerHTML = '\u275a\u275a';

	// Create a new GainNode for this metronome activation; when the metronome
	// is stopped, we can just mute/disconnect the gain so that we don't have
	// to cancel samples scheduled into the future (which is quite awkward).
	this.gainNode = this.audioContext.createGain();
	this.gainNode.gain.value = 1.0;
	this.gainNode.connect(this.audioContext.destination);

	// Add a short period to start time so that we have time to schedule the
	// first sample without missing its start time.
	this.metroStartTime = this.audioContext.currentTime + 0.1;
	this.sampleSchedulerTime = this.metroStartTime;
	this.sampleScheduler();  // trigger manually for a quick start
};
MetronomeApplication.prototype.stopMetronome = function () {
	this.running = false;
	this.elemRunButton.innerHTML = '\u25ba';

	this.gainNode.gain.value = 0.0;
	this.gainNode.disconnect();
};
MetronomeApplication.prototype.updateBpmNumber = function () {
	var elem = this.elemBpmDiv;
	this.elemBpmNumber.innerHTML = '' + this.getBpm();
	elem.className = 'updated';
	if (this.bpmUpdatedTimeout) {
		clearTimeout(this.bpmUpdatedTimeout);
		this.bpmUpdatedTimeout = null;
	}
	this.bpmUpdatedTimeout = setTimeout(function () {
		this.bpmUpdatedTimeout = null;
		elem.className = '';
	}, BPM_UPDATED_DELAY);
}
MetronomeApplication.prototype.updateNthActive = function () {
	var metro = this;
	var elems = document.getElementsByClassName('nth');
	Array.prototype.forEach.call(elems, function (elem) {
		if ((String(elem.innerHTML) == 'None' && metro.nth === 1) ||
		    (String(elem.innerHTML) == String(metro.nth))) {
			elem.className = 'nth active';
		} else {
			elem.className = 'nth';
		}
	});
}
MetronomeApplication.prototype.handleStop = function () {
	this.stopMetronome();
};
MetronomeApplication.prototype.handleBpm = function (bpm) {
	this.setBpm(bpm);
	this.updateBpmNumber();
	this.stopMetronome();
	this.startMetronome();
	this.animCurrBeatCount = null;
}
MetronomeApplication.prototype.handleBpmStep = function (step) {
	var bpm = this.getBpm();
	var val = +step.substring(1);
	switch (step[0]) {
	case '*': bpm *= val; break;
	case '/': bpm /= val; break;
	case '+': bpm += val; break;
	case '-': bpm -= val; break;
	}
	this.setBpm(bpm);
	this.updateBpmNumber();
	this.stopMetronome();
	this.startMetronome();
	this.animCurrBeatCount = null;
}
MetronomeApplication.prototype.handleNth = function (nth) {
	this.nth = null;
	if (nth > 0) {
		this.nth = nth;
	}
	this.updateNthActive();
	this.stopMetronome();
	this.startMetronome();
	this.animCurrBeatCount = null;
}
MetronomeApplication.prototype.handleTap = function () {
	var metro = this;
	var now = Date.now();
	var diff;
	var elem;

	if (this.bpmTapTimeout != null) {
		clearTimeout(this.bpmTapTimeout);
		this.bpmTapTimeout = null;
	}

	elem = this.elemTapDiv;

	/* Average the BPM over the last N taps.  This makes it easier
	 * to sync to music.  If the user waits for long enough, the
	 * tap mode resets, which is indicated through the background
	 * of the TAP button.
	 */

	if (this.bpmTapFirst != null) {
		this.bpmTapCount++;
		diff = ((now - this.bpmTapFirst) / 1000) / this.bpmTapCount;  // secs / beat
		this.setBpm(60 / diff);
		this.updateBpmNumber();
	} else {
		this.bpmTapFirst = now;
		this.bpmTapCount = 0;

		// Stop the metronome for the duration of the tapping
		// sequence.
		this.stopMetronome();
	}

	elem.className = 'active';
	this.bpmTapTimeout = setTimeout(function () {
		elem.className = '';
		metro.bpmTapFirst = null;
		metro.bpmTapCount = null;
		metro.bpmTapTimeout = null;

		// Start metronome only when tapping is complete so as
		// to not confuse the poor user
		metro.stopMetronome();
		metro.startMetronome();
		metro.animCurrBeatCount = null;
	}, TAP_MAX_DELAY);
}
MetronomeApplication.prototype.handleStopStart = function () {
	if (this.running) {
		this.stopMetronome();
	} else {
		this.startMetronome();
		this.animCurrBeatCount = null;
	}
}
MetronomeApplication.prototype.animFrameCallback = function () {
	var metro = this;
	var beatCount;
	var secPerBeat;

	/*
	 *  Animation frame callback, triggers a simple CSS animation for every beat.
	 *  This callback is needed to couple audio and video as closely as possible.
	 *  Because this runs every frame, it should exit very quickly if nothing
	 *  needs to be done.
	 */

	if (this.running) {
		secPerBeat = 60 / this.bpm;
		beatCount = Math.floor((this.audioContext.currentTime - this.metroStartTime - AUDIO_TO_VIDEO_LATENCY) / secPerBeat);
		if (this.animCurrBeatCount !== beatCount) {
			this.animCurrBeatCount = beatCount;
			this.elemBpmNumber.className = (this.isBeatCountEmphasized(beatCount) ? 'beatemph' : 'beat');
		} else {
			this.elemBpmNumber.className = '';
		}
	}
	reqAnimFrame(function () { metro.animFrameCallback(); });
}

window.onload = function () {
	try {
		metroApp = new MetronomeApplication();
		metroApp.init();
	} catch (e) {
		alert(e);
	}
};
