function onBeforeGameWindowOpen(win) {
	var grid = win.findChildByName("ui-grid", true);

	var rows = grid.getRows();
	var cols = grid.getCols();

	grid.initData = function() {
		var n = rows * cols;
		var children = this.getChildren();
		var m = n - children.length;

		for(var i = 0; i < m; i++) {
			this.addChild(this.dupTemplateChild());
		}
		children.length = n;

		this.data = [];
		for(var i = 0; i < rows; i ++) {
			this.data.push([]);
			for(var k = 0; k < cols; k++) {
				var item = {};
				this.data[i].push(item);
			}
		}
		this.relayoutChildren();

		return;
	}

	var n = rows * cols;
	var children = grid.getChildren();
	var bombNr = window.bombNr ? window.bombNr : 15;

	var ITEM_VALUE_MAX_BOMBS = 8;
	var ITEM_VALUE_ASK = 9;
	var ITEM_VALUE_BOMB = 10;
	var ITEM_VALUE_TRIGGERED_BOMB = 11;

	var STATE_STOPPED = 0;
	var STATE_STARTED = 1;
	var STATE_TERMINATED = 2;

	grid.replay = function() {
		this.reset();

		return;
	}

	grid.reset = function() {
		var row = null;
		var item = null;
		var prevRow = null;
		var nextRow = null;
		
		this.ellapsedTime = 0;
		this.gameState = STATE_STOPPED;

		//reset all items
		for(var i = 0; i < rows; i ++) {
			row = this.data[i];
			for(var k = 0; k < cols; k++) {
				var item = row[k];;
				item.uncovered = false;
				item.flagged = false;
				item.value = 0;
			}
		}

		//layout bomb
		var arr = makeUniqRandArray(0, n-1);
		for(var i = 0; i < bombNr; i++) {
			var index = arr[i];
			row = this.data[Math.floor(index/cols)];
			item = row[index%cols];
			item.value = ITEM_VALUE_BOMB;
		}

		//count neighbor bombs;
		for(var i = 0; i < rows; i ++) {
			row = this.data[i];
			prevRow = i > 0 ? this.data[i-1] : null;
			nextRow = i < (rows - 1) ? this.data[i+1] : null;

			for(var k = 0; k < cols; k++) {
				var item = row[k];
				if(item.value === ITEM_VALUE_BOMB) {
					continue;
				}

				item.value = 0;
				if(prevRow) {
					if(prevRow[k].value === ITEM_VALUE_BOMB) {
						item.value = item.value + 1;
					}

					if(k > 0 && prevRow[k-1].value === ITEM_VALUE_BOMB) {
						item.value = item.value + 1;
					}
					
					if(k < (cols - 1) && prevRow[k+1].value === ITEM_VALUE_BOMB) {
						item.value = item.value + 1;
					}
				}

				if(nextRow) {
					if(nextRow[k].value === ITEM_VALUE_BOMB) {
						item.value = item.value + 1;
					}
					if(k > 0 && nextRow[k-1].value === ITEM_VALUE_BOMB) {
						item.value = item.value + 1;
					}
					
					if(k < (cols - 1) && nextRow[k+1].value === ITEM_VALUE_BOMB) {
						item.value = item.value + 1;
					}
				}

				if(k > 0 && row[k-1].value === ITEM_VALUE_BOMB) {
					item.value = item.value + 1;
				}
					
				if(k < (cols - 1) && row[k+1].value === ITEM_VALUE_BOMB) {
					item.value = item.value + 1;
				}
			}
		}

		grid.startTime = null;
		this.getWindow().findChildByName("ui-led-digits-time", true).setText("000");
		this.getWindow().findChildByName("ui-led-digits-bomb", true).setText("0" + bombNr);

		return;
	}

	grid.expandEmpty = function(r, c) {
		var row = this.data[r];
		var prevRow = r > 0 ? this.data[r-1] : null;
		var nextRow = r < (rows - 1) ? this.data[r+1] : null;

		var item = row[c];
		if(item.value || item.uncovered) {
			return;
		}

		item.uncovered = true;
		if(prevRow) {
			if(!prevRow[c].value) {
				this.expandEmpty(r-1, c);
			}

			if(c > 0 && !prevRow[c-1].value) {
				this.expandEmpty(r-1, c-1);
			}
			
			if(c < (cols - 1) && !prevRow[c+1].value) {
				this.expandEmpty(r-1, c+1);
			}
		}

		if(nextRow) {
			if(!nextRow[c].value) {
				this.expandEmpty(r+1, c);
			}
			if(c > 0 && !nextRow[c-1]) {
				this.expandEmpty(r+1, c-1);
			}
			
			if(c < (cols - 1) && !nextRow[c+1].value) {
				this.expandEmpty(r+1, c+1);
			}
		}

		if(c > 0 && !row[c-1].value) {
			this.expandEmpty(r, c-1);
		}
			
		if(c < (cols - 1) && !row[c+1].value) {
			this.expandEmpty(r, c+1);
		}

		return;
	}

	grid.uncoverAllBombs = function() {
		for(var i = 0; i < rows; i++) {
			for(var j = 0; j < cols; j++) {
				var item = this.data[i][j];
				if(item.value === ITEM_VALUE_BOMB) {
					item.uncovered = true;
				}
			}
		}

		return;
	}

	grid.isDone = function() {
		var n = 0;
		for(var row = 0; row < rows; row++) {
			for(var col = 0; col < cols; col++) {
				var item = this.data[row][col];
				if(!item.uncovered) {
					n = n+1;
				}
			}
		}

		return n == bombNr;
	}

	grid.onChildClicked = function(child) {
		if(child.longPress || this.gameState == STATE_TERMINATED) {
			return;
		}

		if(!grid.startTime) {
			grid.startTime = new Date();
		}

		var index = children.indexOf(child);
		var row = Math.floor(index/cols);
		var col = Math.floor(index%cols);
		var item = this.data[row][col];

		if(item.flagged) {
			return;
		}

		if(!item.value) {
			this.expandEmpty(row, col);
		}
		else if(item.value === ITEM_VALUE_BOMB){
			item.uncovered = true;
			item.value = ITEM_VALUE_TRIGGERED_BOMB;
			grid.gameState = STATE_TERMINATED;
			this.uncoverAllBombs();
			
			var tips = "Lost! You triggered a bomb!";
			grid.openWindow("win_done", 
				function (retCode) {
					if(retCode) {
						grid.reset();
					}
					else {
						setTimeout(function() {
							grid.closeWindow(0);
						}, 1000);
					}	
			}, false, tips);		
		}
		else {
			item.uncovered = true;
		}

		if(grid.gameState != STATE_TERMINATED && grid.isDone()) {
			grid.gameState = STATE_TERMINATED;
			var now = new Date();
			var cost = (now.getTime() - grid.startTime.getTime())/1000;
			var tips = "You Won! " + cost + " seconds";
			grid.openWindow("win_done", 
				function (retCode) {
					if(retCode) {
						grid.reset();
					}
					else {
						setTimeout(function() {
							grid.closeWindow(0);
						}, 1000);
					}	
			}, false, tips);	

			HighScores.add(window.gameLevel, "player", cost);
		}

		return;
	}

	grid.onChildPointerDown = function(child) {
		var grid = this;
		child.longPress = false;

		setTimeout(function() {
			grid.onChildLongPress(child);
		}, 600);

		if(this.gameState == STATE_STOPPED) {
			this.gameState = STATE_STARTED;;
			this.ellapsedTime = 0;
			var ellapsedElement = this.getWindow().findChildByName("ui-led-digits-time", true);

			var grid = this;
			function formatTime(n) {
				if(n < 10) {
					return "00" + n;
				}
				else if( n < 100) {
					return "0" + n;
				}
				else if(n < 100) {
					return "0" + n;
				}
				
				return n+"";
			}

			function updateTime() {
				grid.ellapsedTime = grid.ellapsedTime + 1;
				ellapsedElement.setText(formatTime(grid.ellapsedTime));
				grid.postRedraw();

				if(grid.gameState == STATE_STARTED) {
					setTimeout(updateTime, 1000);
				}
			}
			
			setTimeout(updateTime, 1000);
		}
	}

	grid.onChildLongPress = function(child) {
		if(!child.pointerDown || this.gameState != STATE_STARTED) {
			return;
		}

		var index = children.indexOf(child);
		var row = Math.floor(index/cols);
		var col = Math.floor(index%cols);
		var item = this.data[row][col];

		item.flagged = !item.flagged;

		child.longPress = true;
		this.postRedraw();

		return;
	}

	grid.paintChild = function(child, canvas) {
		var x = child.x;
		var y = child.y;
		var w = child.w;
		var h = child.h;
		var image = null;
		var fontSize = Math.floor(16 * w/40)
		var index = children.indexOf(child);
		var row = Math.floor(index/cols);
		var col = Math.floor(index%cols);
		var item = this.data[row][col];
		var textColors = ["white", "Blue", "Green", "Red", "Red", "Red", "Red", "Red", "Red"];

		canvas.rect(0, 0, w, h);	
		if(item.uncovered) {
			canvas.fillStyle = item.value === ITEM_VALUE_TRIGGERED_BOMB ? "Red" :"#c0c0c0";
			canvas.fill();
			canvas.stroke();
		}
		else {
			canvas.fillStyle = "#cccccc";
			canvas.fill();
		}
		
		if(item.uncovered) {
			canvas.strokeStyle = "#818181";
			canvas.stroke();

			if(item.value <= ITEM_VALUE_MAX_BOMBS && item.value >= 1) {
				var str = item.value + "";

				canvas.fillStyle =  textColors[item.value];
				this.style.setFontSize(fontSize);
				canvas.font = this.style.getFont();
				canvas.textAlign = "center";
				canvas.textBaseline = "middle";
				canvas.fillText(str, w/2, h/2);
			}

			if(item.value === ITEM_VALUE_BOMB || item.value === ITEM_VALUE_TRIGGERED_BOMB) {
				image = child.getHtmlImageByType("option_image_0");
			}
		}
		else {
			canvas.beginPath();
			canvas.strokeStyle = "#f9f9f9";
			canvas.moveTo(0, 0);
			canvas.lineTo(w, 0);
			canvas.stroke();

			canvas.beginPath();
			canvas.moveTo(w, 0);
			canvas.strokeStyle = "#888888";
			canvas.lineTo(w, h);
			canvas.stroke();		

			canvas.beginPath();
			canvas.moveTo(w, h);
			canvas.strokeStyle = "#8c8c8c";
			canvas.lineTo(0, h);
			canvas.stroke();		

			canvas.beginPath();
			canvas.moveTo(0, h);
			canvas.strokeStyle = "#fefefe";
			canvas.lineTo(0, 0);
			canvas.stroke();	

			if(child.pointerDown) {		
				canvas.beginPath();
				canvas.rect(1, 1, w-2, h-2);
				canvas.fillStyle = "Yellow";
				canvas.fill();
			}

			if(item.flagged) {
				image = child.getHtmlImageByType("option_image_1");
			}
		}

		if(image) {
			child.drawImageAtCenter(canvas, image, 0, 0, w, h, true);
		}

		return;
	}

	grid.initData();
	grid.reset();

	HighScores.setGameName("mine");
	HighScores.setUnit("seconds");
    HighScores.setSortInc(true);
}

