/**
 * Strong Cocoa
 * 
 * @version 1.0.0.0
 * @copyright Copyright (C) 2013 TagDam. All rights reserved.
 */
(function(window, undefined) {
	
	/**
	 * Strong Cocoa: Abstract
	 */
	var StrongCocoa_Abstract = function() {};
	
	/**
	 * Strong Cocoa
	 * 
	 * @extends {StrongCocoa_Abstract}
	 */
	var StrongCocoa = function() {};
	StrongCocoa.prototype = new StrongCocoa_Abstract();
	
	/**
	 * Strong Cocoa: Classes
	 * 
	 * @extends {StrongCocoa_Abstract}
	 */
	var StrongCocoa_Classes = function() {};
	StrongCocoa_Classes.prototype = new StrongCocoa_Abstract();
	StrongCocoa_Classes.prototype.StrongCocoa_Classes = StrongCocoa_Classes;
	StrongCocoa_Classes.prototype.StrongCocoa_Abstract = StrongCocoa_Abstract;
	StrongCocoa_Classes.prototype.StrongCocoa = StrongCocoa;
	
	/**
	 * Strong Cocoa: Utility
	 * 
	 * @extends {StrongCocoa_Abstract}
	 */
	var StrongCocoa_Utility = function() {};
	StrongCocoa_Utility.prototype = new StrongCocoa_Abstract();
	StrongCocoa_Classes.prototype.StrongCocoa_Utility = StrongCocoa_Utility;
	var utility = new StrongCocoa_Utility();
	
	/**
	 * Strong Cocoa: Option
	 * 
	 * @param {Any} [val]
	 * @property {Any} [_val]
	 * @extends {StrongCocoa_Abstract}
	 */
	var StrongCocoa_Option = function(val) {
		this._val = val;
	};
	StrongCocoa_Option.prototype = new StrongCocoa_Abstract();
	StrongCocoa_Classes.prototype.StrongCocoa_Option = StrongCocoa_Option;
	
	/**
	 * Strong Cocoa: Local Storage
	 * 
	 * @extends {StrongCocoa_Abstract}
	 */
	var StrongCocoa_LocalStorage = function() {};
	StrongCocoa_LocalStorage.prototype = new StrongCocoa_Abstract();
	StrongCocoa_Classes.prototype.StrongCocoa_LocalStorage = StrongCocoa_LocalStorage;
	
	/**
	 * Strong Cocoa: Translate
	 * 
	 * @param {Any} [backend]
	 * @property {Any} [_backend]
	 * @property {Any} [defaultBackend]
	 * @extends {StrongCocoa_Abstract}
	 */
	var StrongCocoa_Translate = function(backend) {
		this._backend = backend;
		this.defaultBackend = StrongCocoa_Translate_Backend_Globalize;
	};
	StrongCocoa_Translate.prototype = new StrongCocoa_Abstract();
	StrongCocoa_Classes.prototype.StrongCocoa_Translate = StrongCocoa_Translate;
	
	/**
	 * Strong Cocoa: Translate: Backend: Abstract
	 * 
	 * @property {Any} [_engine]
	 * @extends {StrongCocoa_Abstract}
	 */
	var StrongCocoa_Translate_Backend_Abstract = function() {
		this._engine;
	};
	StrongCocoa_Translate_Backend_Abstract.prototype = new StrongCocoa_Abstract();
	StrongCocoa_Classes.prototype.StrongCocoa_Translate_Backend_Abstract = StrongCocoa_Translate_Backend_Abstract;
	
	/**
	 * Strong Cocoa: Translate: Backend: Globalize
	 * 
	 * @property {Any} [_engine]
	 * @extends {StrongCocoa_Translate_Backend_Abstract}
	 */
	var StrongCocoa_Translate_Backend_Globalize = function() {
		this._engine = Globalize;
	};
	StrongCocoa_Translate_Backend_Globalize.prototype = new StrongCocoa_Translate_Backend_Abstract();
	StrongCocoa_Classes.prototype.StrongCocoa_Translate_Backend_Globalize = StrongCocoa_Translate_Backend_Globalize;
	
	/**
	 * Classes
	 * 
	 * @var {StrongCocoa_Utility}
	 */
	StrongCocoa_Abstract.prototype.classes = new StrongCocoa_Classes();;
	
	/**
	 * Utility
	 * 
	 * @var {StrongCocoa_Utility}
	 */
	StrongCocoa_Abstract.prototype.utility = utility;
	
	/**
	 * Local Storage
	 * 
	 * @var {StrongCocoa_LocalStorage}
	 */
	StrongCocoa_Abstract.prototype.localStorage = new StrongCocoa_LocalStorage();
	
	/**
	 * Translate
	 * 
	 * @var {StrongCocoa_Translate}
	 */
	StrongCocoa_Abstract.prototype.translate = new StrongCocoa_Translate();
	
	/**
	 * Is Null
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isNull = function(val) {
		return (null === val);
	};
	
	/**
	 * Is Undefined
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isUndefined = function(val) {
		return (typeof val == 'undefined');
	};
	
	/**
	 * Is Object
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isObject = function(val) {
		return (((val instanceof Object) && (Object.getPrototypeOf(val) === Object.prototype)) || (Object.prototype.toString.call(val) === '[object Object]'));
	};
	
	/**
	 * Is Boolean
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isBoolean = function(val) {
		return (typeof val == 'boolean');
	};
	
	/**
	 * Is String
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isString = function(val) {
		return (typeof val == 'string');
	};
	
	/**
	 * Is Numeric
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isNumeric = function(val) {
		return this.notEmpty(val) && !this.isArray(val) && isFinite(val);
	};
	
	/**
	 * Is Array
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isArray = function(val) {
		return ((val instanceof Array) || (Object.prototype.toString.call(val) === '[object Array]'));
	};
	
	/**
	 * Is RegExp
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isRegExp = function(val) {
		return ((val instanceof RegExp) || (Object.prototype.toString.call(val) === '[object RegExp]'));
	};
	
	/**
	 * Is Date
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isDate = function(val) {
		return (val instanceof Date);
	};
	
	/**
	 * Is Function
	 * 
	 * @param {any} [val]
	 * @return boolean
	 */
	StrongCocoa_Utility.prototype.isFunction = function(val) {
		return (typeof val == 'function');
	};
	
	/**
	 * Is the Empty
	 * 
	 * @param {Any} [val]
	 * @returns {Boolean}
	 */
	StrongCocoa_Utility.prototype.isEmpty = function(val) {
		return (undefined === val) || (null === val) || ('' === val);
	};
	
	/**
	 * Not the Empty
	 * 
	 * @param {Any} [val]
	 * @returns {Boolean}
	 */
	StrongCocoa_Utility.prototype.notEmpty = function(val) {
		return !this.isEmpty(val);
	};
	
	/**
	 * Set the Global Variable
	 * 
	 * @param {Any} [val]
	 * @param {String} [name]
	 * @returns void
	 */
	StrongCocoa_Utility.prototype.setGlobalVariable = function(value, name) {
		if ((typeof require !== 'undefined')
				&& (typeof exports !== 'undefined')
				&& (typeof module !== 'undefined')) {
			module.exports = value;
		} else {
			window[name] = value;
		}
	};
	
//	/**
//	 * In Array
//	 * 
//	 * @param {any} [val]
//	 * @param {Array} [ary]
//	 * @return {Boolean}
//	 */
//	StrongCocoa_Utility.prototype.inArray = function(val, ary) {
//		var result = false;
//		if (this.isArray(ary)) {
//			for (var i = 0, __len = ary.length; i < __len; i++) {
//				if (val == ary[i]) {
//					result = true;
//					break;
//				}
//			}
//		}
//		return result;
//	};
	/**
	 * In Array
	 * 
	 * @param {any} [val]
	 * @param {Array} [ary]
	 * @return {Number} Not in to "-1".
	 */
	StrongCocoa_Utility.prototype.inArray = function(val, ary) {
		if (this.isArray(ary)) {
			for (var i = 0, __len = ary.length; i < __len; i++) {
				if (val === ary[i]) {
					return i;
				}
			}
		}
		return -1;
	};
	
	/**
	 * Clone
	 * 
	 * @param {Any} [val]
	 * @param {Boolean} [deep] true is deep
	 * @return {Any}
	 */
	StrongCocoa_Utility.prototype.clone = function(val, deep) {
		var res;
		var that = this;
		var callback = function(key) {
			return (false === deep) ? val[key] : that.clone(val[key]);
		};
		if (this.isArray(val)) {
			res = [];
			for (var i = 0, __len = val.length; i < __len; i++) {
				res[res.length] = callback(i);
			}
		} else if (this.isObject(val)) {
			res = {};
			for (var key in val) {
				res[key] = callback(key);
			}
		} else {
			res = val;
		}
		return res;
	};
	
	/**
	 * Trim
	 * 
	 * @param {string} [str]
	 * @return string
	 */
	StrongCocoa_Utility.prototype.trim = function(str) {
		return this.isString(str) ? str.replace(/^[\s　]+|[\s　]+$/ig, '') : str;
	};
	
	/**
	 * To the Lower Camel Case
	 * 
	 * @param {string} [str]
	 * @return string
	 */
	StrongCocoa_Utility.prototype.toLowerCamelCase = function(str) {
		var result;
		if (!this.isString(str)) {
			result = str;
		} else {
			result = str.substring(0, 1).toLowerCase() + str.substring(1);
		}
		return result;
	};
	
	/**
	 * To the Upper Camel Case
	 * 
	 * @param {string} [str]
	 * @return string
	 */
	StrongCocoa_Utility.prototype.toUpperCamelCase = function(str) {
		var result;
		if (!this.isString(str)) {
			result = str;
		} else {
			result = str.substring(0, 1).toUpperCase() + str.substring(1);
		}
		return result;
	};
	
	/**
	 * To the Number Format
	 * 
	 * @param {String} [val]
	 * @returns {String}
	 */
	StrongCocoa_Utility.prototype.toNumberFormat = function(val) {
		return String(val).replace(/^(-?\d+)(\d{3})/, '$1,$2');
	};
	
	/**
	 * Get
	 * 
	 * @returns {Any}
	 */
	StrongCocoa_Option.prototype.get = function() {
		return this._val;
	};
	
	/**
	 * Get or Else
	 * 
	 * @param {Any} [elseValue]
	 * @returns {Any}
	 */
	StrongCocoa_Option.prototype.getOrElse = function(elseValue) {
		var val = this.get();
		return this.utility.notEmpty(val) ? val : elseValue;
	};
	
	/**
	 * Save
	 * 
	 * @param {String} [key]
	 * @param {Any} [value]
	 * @returns void
	 */
	StrongCocoa_LocalStorage.prototype.save = function(key, value) {
		localStorage.setItem(key, (this.utility.isEmpty(value) ? value : JSON.stringify(value)));
	};
	
	/**
	 * Get
	 * 
	 * @param {String} [key]
	 * @returns {Any}
	 */
	StrongCocoa_LocalStorage.prototype.get = function(key) {
		var data = localStorage.getItem(key);
		if (this.utility.notEmpty(data)) {
			data = JSON.parse(data);
		}
		return data;
	};
	
	/**
	 * Remove
	 * 
	 * @param {String} [key]
	 * @returns void
	 */
	StrongCocoa_LocalStorage.prototype.remove = function(key) {
		localStorage.removeItem(key);
	};
	
	/**
	 * Clear
	 * 
	 * @returns void
	 */
	StrongCocoa_LocalStorage.prototype.clear = function() {
		localStorage.clear();
	};
	
	/**
	 * Culture
	 * 
	 * @var {String}
	 */
	StrongCocoa_Translate.prototype._culture = 'en_US';
	
	/**
	 * Valid
	 * 
	 * @reutnr {Boolean}
	 */
	StrongCocoa_Translate.prototype.valid = function() {
		return this.utility.notEmpty();
	};
	
	/**
	 * Backend
	 * 
	 * @param {Any} [backend] (Option)
	 * @returns {Any}
	 */
	StrongCocoa_Translate.prototype.backend = function(backend) {
		if (this.utility.notEmpty(backend)) {
			this._backend = backend;
		} else if (this.utility.isEmpty(this._backend)) {
			this._backend = new this.defaultBackend();
		}
		return this._backend;
	};
	
	/**
	 * Translate
	 * 
	 * @param {String} [id]
	 * @param {String} [culture]
	 * @returns {String}
	 */
	StrongCocoa_Translate.prototype.translate = StrongCocoa_Translate.prototype._ = function(id, culture) {
		return this.backend()._(id, culture);
	};
	
	/**
	 * Engine
	 * 
	 * @return {Any}
	 */
	StrongCocoa_Translate_Backend_Globalize.prototype.engine = function() {
		return this._engine;
	};
	
	/**
	 * Translate
	 * 
	 * @param {String} [id]
	 * @param {String} [culture] (Option)
	 * @returns {Any}
	 */
	StrongCocoa_Translate_Backend_Globalize.prototype.translate = StrongCocoa_Translate_Backend_Globalize.prototype._ = function(id, culture) {
		if (this.utility.notEmpty(culture)) {
			return this.engine().localize(id, culture);
		}
		return this.engine().localize(id);
	};
	
	var strongCocoa = new StrongCocoa();
	strongCocoa.utility.setGlobalVariable(strongCocoa, 'strongCocoa');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa, 'StrongCocoa');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa_Abstract, 'StrongCocoa_Abstract');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa_Utility, 'StrongCocoa_Utility');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa_Option, 'StrongCocoa_Option');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa_LocalStorage, 'StrongCocoa_LocalStorage');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa_Translate, 'StrongCocoa_Translate');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa_Translate_Backend_Abstract, 'StrongCocoa_Translate_Backend_Abstract');
//	strongCocoa.utility.setGlobalVariable(StrongCocoa_Translate_Backend_Globalize, 'StrongCocoa_Translate_Backend_Globalize');
})(this);
