/**
 * The memo app stores its data using indexedDB.
 *
 * This file holds the indexedDB routines and the routines for
 * working with the memos.
 *
 * First we handle the indexedDB setup then we implement the memo
 * handling routines.
 */

const dbName = "memos";
const dbVersion = 1;

var db;
var request = indexedDB.open(dbName, dbVersion);

request.onerror = function (event) {
    console.error("Can't open indexedDB!!!", event);
};
request.onsuccess = function (event) {
    console.log("Database opened ok");
    db = event.target.result;
};

request.onupgradeneeded = function (event) {

    console.log("Running onUpgradeNeeded");

    db = event.target.result;

    if (!db.objectStoreNames.contains("memos")) {

        console.log("Creating objectStore for memos");

        var objectStore = db.createObjectStore("memos", {
            keyPath: "id",
            autoIncrement: true
        });
        objectStore.createIndex("title", "title", {
            unique: false
        });

        console.log("Adding sample memo");
        var sampleMemo1 = new Memo();
        sampleMemo1.title = "Bem-Vindo!";
        sampleMemo1.content = "Insira um conteudo aqui.";

        objectStore.add(sampleMemo1);
    }
}

/**
 * This memo function is used to create new memos.
 * @constructor
 */

function Memo() {
    var today = new Date();
    var dd = today.getDate();
    var mm = today.getMonth()+1; //January is 0!
    var hh = today.getHours();
    var mh = today.getMinutes();
    
    if(dd<10) {
    dd='0'+dd
    } 

    if(mm<10) {
    mm='0'+mm
    } 
    
    this.title = "";
    this.content = "";
    this.created = dd+'/'+mm;
    this.modified = mh+'h'+hh;
}

/**
 * This function will load all memos and fire its callback routine for each
 * memo loaded.
 * @param inCallback
 */
function listAllMemoTitles(inCallback) {
    var objectStore = db.transaction("memos").objectStore("memos");
    console.log("Listing memos...");

    objectStore.openCursor().onsuccess = function (event) {
        var cursor = event.target.result;
        if (cursor) {
            console.log("Found memo #" + cursor.value.id + " - " + cursor.value.title);
            inCallback(null, cursor.value);
            cursor.continue();
        }
    };
}

/**
 * This function is used to save a memo into the indexedDB database. It is called
 * on the 'change' event of the text inputs, so it is very aggressive. The idea behind
 * this is that the user never needs to save a memo for it is always in the saved state.
 * @param inMemo
 * @param inCallback
 */
function saveMemo(inMemo, inCallback) {
    var transaction = db.transaction(["memos"], "readwrite");
    console.log("Saving memo");

    transaction.oncomplete = function (event) {
        console.log("All done");
    };

    transaction.onerror = function (event) {
        console.error("Error saving memo:", event);
        inCallback({
            error: event
        }, null);

    };

    var objectStore = transaction.objectStore("memos");

    inMemo.modified = Date.now();

    var request = objectStore.put(inMemo);
    request.onsuccess = function (event) {
        console.log("Memo saved with id: " + request.result);
        inCallback(null, request.result);

    }
}

/**
 * This function is used to remove a memo from the database. The only way to delete
 * a memo in this app is by using the trash button in the memo editing screen.
 * @param inId
 * @param inCallback
 */
function deleteMemo(inId, inCallback) {
    console.log("Deletando nota...");
    var request = db.transaction(["memos"], "readwrite").objectStore("memos").delete(inId);

    request.onsuccess = function (event) {
        console.log("Nota deletada!");
        inCallback();
    };
}
