/*
 *  Offtrail application class
 *
 *  Works as a singleton and expects the index.html to contain specific
 *  elements for the UI.
 *
 *  Geolocation API: http://dev.w3.org/geo/api/spec-source.html
 */

var OFFTRAIL_VERSION = '1.2';
var OFFTRAIL_GITDESCRIBE = 'v1.2';
var OFFTRAIL_GITCOMMIT = 'abd2ed583b334d417502e0759668697cabebe5f9';

OfftrailApplication = function () {
	// settings object, serialized to local storage
	this.settings = undefined;

	// audio
	this.audioContext = undefined;
	this.gainNode = undefined;
	this.buffer_alarm = undefined;

	// geolocation handle
	this.watchId = undefined;

	// current fix, updated on every received fix
	this.currLon = undefined;
	this.currLat = undefined;
	this.currAlt = undefined;
	this.currHea = undefined;
	this.currSpd = undefined;
	this.currAccuracy = undefined;
	this.currAltAccuracy = undefined;
	this.currTime = undefined;

	// previous fix accepted into the active trail segment,
	// only updated if a set of conditions is met
	this.prevLon = undefined;
	this.prevLat = undefined;
	this.prevAlt = undefined;
	this.prevHea = undefined;
	this.prevSpd = undefined;
	this.prevAccuracy = undefined;
	this.prevAltAccuracy = undefined;
	this.prevTime = undefined;

	// current (only) trail, with multiple segments
	this.trail = undefined;

	// number of fixes received from platform
	this.fixCountRaw = 0;

	// number of fixes added to active trail segment (before
	// simplification)
	this.fixCountTrail = 0;

	// log buffer, fixed size ring buffer
	this.logLines = [];
	while (this.logLines.length < this.LOG_LINE_COUNT) {
		this.logLines.push('');
	}
	this.logNext = 0;
};

OfftrailApplication.prototype.LOG_LINE_COUNT = 1000;

// Minimum distance or time (either one alone is enough) required to add
// a raw fix to the trail.  These are applied for raw fixes added to the
// active trail segment prior to simplification, so these limits are mainly
// useful to control intermediate storage size and run time of the simplifier.
OfftrailApplication.prototype.DIFF_DIST_SUFFICIENT = 3.0;  /* meters */
OfftrailApplication.prototype.DIFF_TIME_SUFFICIENT = 60.0 * 1000.0;  /* seconds */

// Minimum position accuracy (meters) required to add a fix to the trail.
// Fixes less accurate than this are updated to "current" fix but are not
// added to the active trail segment.
OfftrailApplication.prototype.POSITION_ACCURACY_LIMIT = 50.0;

// Lon/lat accuracy limits for update flash colors.
OfftrailApplication.prototype.UPDATED1_ACCURACY_LIMIT = 10.0;
OfftrailApplication.prototype.UPDATED2_ACCURACY_LIMIT = 50.0;

// Minimum timestamp difference to add raw fix to trail.  This may be useful
// because there's no way to control the watchPosition() callback interval
// and at least Firefox OS 1.3 will call once per second.  This is just a
// sanity limit to avoid processing a flood of callbacks (perhaps caused by
// a platform bug).
OfftrailApplication.prototype.DIFF_TIME_IGNORE_LIMIT = 0.5 * 1000.0;

// Maximum age for the -first- fix given by watchTimeout().
OfftrailApplication.prototype.FIRST_FIX_MAXIMUM_AGE = 60 * 1000.0;

// Timeout for the -first- fix given by watchTimeout().
OfftrailApplication.prototype.FIRST_FIX_TIMEOUT = 10 * 1000.0;

// Autosave interval.  Saves current trail to the local storage, so that
// trail data loss is limited if the application crashes or device runs
// out of battery.  Trail data is not simplified for autosave, but the
// app will close and simplify the active trail segment on restart.
OfftrailApplication.prototype.AUTO_SAVE_INTERVAL = 60 * 1000.0;

// Number of fixes to ignore since startup.  At least N9 will sometimes
// return a few spurious fixes initially.
OfftrailApplication.prototype.IGNORE_INITIAL_FIX_COUNT = 2;

// Automatically close a track segment if it contains more fixes than this
// limit.  This is a basic sanity limit to ensure that track segments don't
// become so large that simplification would be prohibitively slow, or that
// the track segment wouldn't fit into a local storage key/value pair.
OfftrailApplication.prototype.AUTO_CLOSE_FIX_LIMIT = 1000;

// To safeguard against accidental trail clearing, the clear button needs
// to be pressed twice in quick succession.  The button stays "armed" for
// this amount of milliseconds.
OfftrailApplication.prototype.CLEAR_ARMED_WINDOW = 500;

// Interval to update trail length.  This is not done on every fix because
// the length calculation requires somewhat heavy computation.
OfftrailApplication.prototype.TRAIL_LENGTH_UPDATE_INTERVAL = 60 * 1000.0;

OfftrailApplication.prototype.formatUiTime = function (timeDate, millis) {
	function pad2(n) {
		var tmp = '00' + String(n);
		return tmp.substring(tmp.length - 2);
	}
	function pad3(n) {
		var tmp = '000' + String(n);
		return tmp.substring(tmp.length - 3);
	}
	return pad2(timeDate.getUTCHours()) + ':' +
	       pad2(timeDate.getUTCMinutes()) + ':' +
	       pad2(timeDate.getUTCSeconds()) +
	       (millis ? '.' + pad3(timeDate.getUTCMilliseconds()) : '');
};

OfftrailApplication.prototype.logUi = function (msg) {
	var idx = this.logNext;
	var i, n;
	var now = new Date();

	this.logNext = (idx + 1) % this.logLines.length;
	this.logLines[idx] = this.formatUiTime(now, true) + ': ' + String(msg);

	n = 3;  // fixed line count
	while (n > 0) {
		idx = (this.logNext - n + this.logLines.length) % this.logLines.length;
		DomUtil.setTextToId('log-line' + n, this.logLines[idx]);
		n--;
	}
};

OfftrailApplication.prototype.debugLogUi = function (msg) {
	//this.logUi(msg);
};

OfftrailApplication.prototype.getFullLogText = function () {
	var res = [];
	var line;
	var i, n;
	for (i = 0, n = this.logLines.length; i < n; i++) {
		line = this.logLines[(this.logNext + i) % this.logLines.length];
		if (line !== '') {
			res.push(line);
		}
	}
	return res.join('\n');
};

/* Full screen window (e.g. about, log view).  Window is a <div> with
 * 'fswindow' class.
 */
OfftrailApplication.prototype.showFullScreenWindow = function (args) {
	var elem;

	elem = document.getElementById(args.windowId);
	elem.classList.add('active');

	elem = document.getElementById(args.closeId);
	elem.addEventListener('click', function (event) {
		var elem;
		event.stopPropagation();
		elem = document.getElementById(args.windowId);
		elem.classList.remove('active');
		if (typeof args.closeCallback === 'function') {
			args.closeCallback();
		}
	});
};

/* Simple timer/CSS animation for flashing the screen a bit when the fix
 * is updated.
 *
 * XXX: should be ignored if page is not visible.
 */
OfftrailApplication.prototype.animatePositionUpdate = function () {
	var elem;
	var style = 'updated3';  // default to worst

	// Different colors for a few different accuracies
	if (typeof this.currAccuracy === 'number') {
		if (this.currAccuracy < this.UPDATED1_ACCURACY_LIMIT) {
			style = 'updated1';
		} else if (this.currAccuracy < this.UPDATED2_ACCURACY_LIMIT) {
			style = 'updated2';
		}
	}

	elem = document.getElementById('curr-fix');
	elem.classList.add(style);
	setTimeout(function () {
		elem.classList.remove('updated1');
		elem.classList.remove('updated2');
		elem.classList.remove('updated3');
	}, 200);
};

/* Update fix UI to unknown. */
OfftrailApplication.prototype.updateFixToUnknown = function () {
	DomUtil.setTextToId('curr-lon', '?');
	DomUtil.setTextToId('curr-lon-accuracy', '?');
	DomUtil.setTextToId('curr-lat', '?');
	DomUtil.setTextToId('curr-lat-accuracy', '?');
	DomUtil.setTextToId('curr-alt', '?');
	DomUtil.setTextToId('curr-alt-accuracy', '?');
	DomUtil.setTextToId('curr-hea', '?');
	DomUtil.setTextToId('curr-spd', '?');
	DomUtil.setTextToId('curr-time', '?');
};

/* Update fix UI from 'current' fix. */
OfftrailApplication.prototype.updateFixFromCurrent = function () {
	var elem;
	var timeDate = new Date(this.currTime);
	var timeIso = timeDate.toISOString();
	var speed_kmh = this.currSpd * 3.6;  // * 3600 / 1000

	elem = document.getElementById('curr-fix');
	elem.classList.remove('nofix');
	// FIXME: update log when fix becomes active
	DomUtil.setTextToId('curr-lon', this.currLon.toFixed(6));
	DomUtil.setTextToId('curr-lon-accuracy', this.currAccuracy.toFixed(0));
	DomUtil.setTextToId('curr-lat', this.currLat.toFixed(6));
	DomUtil.setTextToId('curr-lat-accuracy', this.currAccuracy.toFixed(0));
	DomUtil.setTextToId('curr-alt', this.currAlt.toFixed(0));
	DomUtil.setTextToId('curr-alt-accuracy', this.currAltAccuracy.toFixed(0));
	DomUtil.setTextToId('curr-hea', this.currHea.toFixed(0));
	DomUtil.setTextToId('curr-spd', speed_kmh.toFixed(1));
	DomUtil.setTextToId('curr-time', this.formatUiTime(timeDate));
};

/* Update fix stats UI parts. */
OfftrailApplication.prototype.updateFixStats = function () {
	DomUtil.setTextToId('fix-count-info',
	                    String(this.fixCountRaw) + ' raw fixes, ' +
	                    String(this.fixCountTrail) + ' trail fixes, ' +
	                    'trail id: ' + this.trail.trailId);
};

/* Clear previous fix state.  Needed when trail segment is closed, cleared,
 * etc.
 */
OfftrailApplication.prototype.clearPreviousFix = function () {
	this.prevLon = undefined;
	this.prevLat = undefined;
	this.prevAlt = undefined;
	this.prevHea = undefined;
	this.prevSpd = undefined;
	this.prevAccuracy = undefined;
	this.prevAltAccuracy = undefined;
	this.prevTime = undefined;
};

OfftrailApplication.prototype.updatePositionToUi = function () {
	this.updateFixFromCurrent();
	this.updateFixStats();
};

OfftrailApplication.prototype.updateAverageSpeedToUi = function () {
	var speed_kmh = this.trail.getAverageSpeedForLastSegment() * 3.6;  // * 3600 / 1000

	DomUtil.setTextToId('avg-spd', speed_kmh.toFixed(1));
};

var notify_test = -1;  // FIXME: audio 5km notify test
OfftrailApplication.prototype.updateTrailLengthToUi = function () {
	var len2d, len3d;
	var tmp;

	len2d = this.trail.getLength2D() / 1000;
	len3d = this.trail.getLength3D() / 1000;
	DomUtil.setTextToId('curr-len', len3d.toFixed(3));
	//alert('2D: ' + len2d + ', 3D: ' + len3d);

	// FIXME: audio notify test
	if (false) {
		tmp = Math.floor(len3d / 5);  // 5km
		if (tmp > notify_test) {
			notify_test = tmp;
			this.playBuffer(this.buffer_alarm);
		}
	}
};

/* Success callback for watchPosition().  This is called inside a try-catch
 * wrapper to convert errors to alerts.
 *
 * NOTE: There is no way to detect reliably that a fix has been lost other
 * than getting an error callback from the platform.  Note in particular that
 * getting no fix success callbacks at all does -not- mean that the fix has
 * been lost: it can simply mean that the user is not moving at all (there is
 * no guaranteed callback interval).
 */
OfftrailApplication.prototype.updatePosition = function (pos) {
	var _this = this;
	var lat, lon, alt, time, accuracy, altAccuracy, hea, spd;
	var diffDist, diffTime;
	var addToTrail;

	/*
	 *  Current fix is updated every time, even if the fix does not
	 *  contribute to the trail.  Tolerate both string and number
	 *  values, at least (old) Chromium 'Manual GeoLocation' gives
	 *  the coordinate values as strings instead of numbers.
	 *
	 *  parseFloat() will return NaN for invalid values as well as
	 *  for undefined ones.
	 */

	// http://dev.w3.org/geo/api/spec-source.html#position_interface

	this.fixCountRaw += 1;

	lon = parseFloat(pos.coords.longitude);
	lat = parseFloat(pos.coords.latitude);
	alt = parseFloat(pos.coords.altitude);
	hea = parseFloat(pos.coords.heading);
	spd = parseFloat(pos.coords.speed);
	accuracy = parseFloat(pos.coords.accuracy);
	altAccuracy = parseFloat(pos.coords.altitudeAccuracy);
	time = parseFloat(pos.timestamp || Date.now());

	/* This can be used during debugging to create a 'random walk' */
/*
	lon += Math.random() * 0.01;
	lat += Math.random() * 0.01;
	accuracy = altAccuracy = 1;
*/

	this.currLon = lon;
	this.currLat = lat;
	this.currAlt = alt;
	this.currHea = hea;
	this.currSpd = spd;
	this.currAccuracy = accuracy;
	this.currAltAccuracy = altAccuracy;
	this.currTime = time;

	/*
	 *  Decide whether to include the new fix in the active trail
	 *  segment with multiple successive criteria.
	 */

	addToTrail = false;
	if (this.prevLat === undefined || this.prevLon === undefined) {
		// No previous fix, always add to trail.
		addToTrail = true;
	} else {
		// Compare current fix to previous fix added to the trail.
		// Distance is estimated using a spherical Earth model.
		diffDist = GeoUtil.pointDist(this.prevLon, this.prevLat, this.prevAlt || 0, this.prevTime,
		                             this.currLon, this.currLat, this.currAlt || 0, this.currTime);
		diffTime = Math.abs((this.prevTime - this.currTime) || 0);

		if (diffDist > this.DIFF_DIST_SUFFICIENT ||
		    diffTime > this.DIFF_TIME_SUFFICIENT) {
			// If either of the criteria are exceeded, it's sufficient
			// to add the fix to the trail.
			addToTrail = true;
		}
		if (diffTime < this.DIFF_TIME_IGNORE_LIMIT) {
			// Even if the other criteria are met, if the fix is
			// too new, ignore it anyway.  This reduces the fixes
			// added in high speed motion when the platform gives
			// new fixes much more frequently than requested.
			addToTrail = false;
		}
	}
	if (this.fixCountRaw < this.IGNORE_INITIAL_FIX_COUNT) {
		// Initial fixes may be spurious - at least N9 will return a
		// spurious first fix occasionally.  Update the GPS info but
		// don't add to the trail even if everything else checks out.
		addToTrail = false;
	}
	if (!isNaN(accuracy) && accuracy > this.POSITION_ACCURACY_LIMIT) {
		// Too inaccurate to be included in the trail.
		addToTrail = false;
	}

	/*
	 *  If fix is added to the trail, update previous fix information
	 *  and add fix to the active trail segment.
	 */

	if (addToTrail) {
		this.prevLon = this.currLon;
		this.prevLat = this.currLat;
		this.prevAlt = this.currAlt;
		this.prevHea = this.currHea;
		this.prevSpd = this.currSpd;
		this.prevAccuracy = this.currAccuracy;
		this.prevAltAccuracy = this.currAltAccuracy;
		this.prevTime = this.currTime;

		this.trail.addFix(lon, lat, alt, time);

		this.fixCountTrail++;

		if (this.trail.getActiveSegmentFixCount() >= OfftrailApplication.prototype.AUTO_CLOSE_FIX_LIMIT) {
			// If a trail segment becomes too large, close it
			// forcibly to ensure it can be simplified and stored
			// in a reasonable manner.  Add a new trail segment and
			// add the same fix to it to avoid gaps.

			this.trail.closeActive();
			this.trail.addFix(lon, lat, alt, time);
		}
	}

	/*
	 *  Update the UI fix information, regardless or whether or not the
	 *  fix is added to the trail.
	 */

	//this.logUi('page visible: ' + isVisible());
	if (isVisible()) {
		this.updateTrailLengthToUi();
		this.updateAverageSpeedToUi();
		this.updatePositionToUi();
		this.animatePositionUpdate();  /* flash only when fix received, not when redrawing on pagevisible */
	}
};

/* Error callback for watchPosition().  This is called inside a try-catch
 * wrapper to convert errors to alerts.
 */
OfftrailApplication.prototype.errorPosition = function (err) {
	var code2text = {
		'1': 'PERMISSION_DENIED',
		'2': 'POSITION_UNAVAILABLE',
		'3': 'TIMEOUT'
	};

	this.logUi((code2text[err.code] || 'UNKNOWN') +
	           ' (' + err.code + ')' + ': ' + err.message);

	/* If trail has an active segment, close it and autosave. */
	if (this.trail.hasActiveSegment()) {
		this.trail.closeActive();
		this.saveCurrentTrail();
		this.clearPreviousFix();
	}

	/* Indicate no fix status. */
	elem = document.getElementById('curr-fix');
	elem.classList.add('nofix');
};

/* Setup geolocation using watchPosition().  We rely on the API to be
 * reliable so that we only need to do this once and can then rely on
 * watchPosition() providing success/error callbacks forever.
 */
OfftrailApplication.prototype.setupGeolocation = function () {
	var _this = this;

	if (typeof navigator !== 'object' ||
	    typeof navigator.geolocation !== 'object' ||
	    typeof navigator.geolocation.watchPosition === 'undefined') {
		alert('No positioning support in browser.');
		return;
	}

	this.watchId = navigator.geolocation.watchPosition(
		function(pos) {
			try {
				_this.updatePosition(pos);
			} catch (e) {
				alert('updatePosition failed: ' + e);
			}
		},
		function(err) {
			try {
				_this.errorPosition(err);
			} catch (e) {
				alert('errorPosition failed: ' + e);
			}
		},
		{
			// We want/need GPS level accuracy
			enableHighAccuracy: true,

			// With watchPosition() this only applies to the first
			// fix; it can be a bit stale and will still be better
			// than showing nothing.
			maximumAge: this.FIRST_FIX_MAXIMUM_AGE,

			// With watchPosition() this timeout applies to the
			// acquisition of the first fix.
			timeout: this.FIRST_FIX_TIMEOUT
		}
	);
};

/* Save current trail.  The active trail segment is not closed/simplified. */
OfftrailApplication.prototype.saveCurrentTrail = function () {
	var st;

	this.trail.exportToLocalStorage(window.localStorage);

	st = this.trail.getStats();
	this.logUi('Autosave: ' + st.numSegments + ' segments' +
	           ', ' + st.numPoints + ' points');
};

/* Setup periodic autosave handling.
 *
 * XXX: this is currently not very intelligent: if the trail is saved for
 * some other reason, an autosave can happen right after the other save.
 */
OfftrailApplication.prototype.setupAutosave = function () {
	var _this = this;
	setInterval(function () {
		try {
			_this.saveCurrentTrail();
		} catch (e) {
			alert('Autosave failed: ' + e);
		}
	}, this.AUTO_SAVE_INTERVAL);
};

/* Setup close button handling. */
OfftrailApplication.prototype.setupCloseSegmentButton = function () {
	var _this = this;
	var elem;

	elem = document.getElementById('close-segment');
	elem.addEventListener('click', function (event) {
		event.stopPropagation();
		try {
			_this.trail.closeActive();
			_this.saveCurrentTrail();
			_this.clearPreviousFix();
		} catch (e) {
			alert(e);
		}
	});
};

/* Build an e-mail body for trail e-mail (assumes KML format now). */
OfftrailApplication.prototype.buildEmailBody = function (timestamp) {
	var res = [];
	res.push('Trail data sent by Offtrail ' + OFFTRAIL_VERSION + ' on ' + timestamp.toISOString());
	res.push('');
	res.push('You can view KML (KMZ) and GPX files e.g. with http://www.gpsvisualizer.com/ as follows:');
	res.push('');
	res.push('1. Save trail file (attached)');
	res.push('');
	res.push('2. Open http://www.gpsvisualizer.com/ and click "Browse" near "Get started now!"');
	res.push('');
	res.push('3. Select the trail file you saved in step 1 and click "Go"');
	res.push('');
	res.push('The GPX attachment is ZIP compressed. GPX files can be posted to e.g. http://www.openstreetmap.org/');
	res.push('');
	res.push('Offtrail ' + OFFTRAIL_VERSION + ' (' + OFFTRAIL_GITDESCRIBE + ': ' + OFFTRAIL_GITCOMMIT + ')');
	res.push('');
	return res.join('\n');
};

/* Setup e-mail button handling.
 *
 * XXX: for now, the active trail segment is not simplified for sending
 * (even temporarily) so the active trail segment appears different than
 * when it is closed and simplified.  The trail data is also larger than
 * would be after close/simplify.  Perhaps fix this such that the active
 * trail segment is simplified (temporarily) for sending only?
 */
OfftrailApplication.prototype.setupEmailSendButton = function () {
	var _this = this;
	var elem;

	elem = document.getElementById('email-trail');
	elem.addEventListener('click', function (event) {
		// https://developer.mozilla.org/en-US/docs/Web/API/Blob
		var kmlExporter, gpxExporter, emailSender, blobs = [], filenames = [];
		var to, subject, body;
		var now = new Date();
		event.stopPropagation();
		try {
			kmlExporter = new KmlExporter();
			gpxExporter = new GpxExporter();
			emailSender = new EmailSender();
			if (_this.settings.useKml) {
				blobs.push(kmlExporter.trailToKmlBlob(_this.trail));
				filenames.push('offtrail-' + _this.trail.trailId + '-' + now.getTime() + '.kml');
			}
			if (_this.settings.useKmz) {
				blobs.push(kmlExporter.trailToKmzBlob(_this.trail));
				filenames.push('offtrail-' + _this.trail.trailId + '-' + now.getTime() + '.kmz');
			}
			if (_this.settings.useGpx) {
				blobs.push(gpxExporter.trailToGpxBlob(_this.trail));
				filenames.push('offtrail-' + _this.trail.trailId + '-' + now.getTime() + '.gpx');
			}
			if (_this.settings.useGpxZip) {
				blobs.push(gpxExporter.trailToGpxZipBlob(_this.trail));
				filenames.push('offtrail-' + _this.trail.trailId + '-' + now.getTime() + '.gpx.zip');
			}
			to = '';  // empty: user fills (e.g. address book)
			subject = 'Offtrail: ' + _this.trail.trailId + ' ('  + now.toISOString() + ')';
			body = _this.buildEmailBody(now);
			emailSender.send(to, subject, body, blobs, filenames);
		} catch (e) {
			alert(e);
		}
	});
};

OfftrailApplication.prototype.setupSaveTrailButton = function () {
	var _this = this;
	var elem;

	elem = document.getElementById('save-trail');
	elem.addEventListener('click', function (event) {
		event.stopPropagation();
		try {
			_this.saveCurrentTrail();
		} catch (e) {
			alert(e);
		}
	});
};

/* Setup clear trail button handling.
 *
 * XXX: This needs a confirm screen.  For now, you need to press it twice.
 */
OfftrailApplication.prototype.setupClearTrailButton = function () {
	var _this = this;
	var elem;
	var clearArmed = false;

	elem = document.getElementById('clear-trail-button');
	elem.addEventListener('click', function (event) {
		event.stopPropagation();
		try {
			if (!clearArmed) {
				clearArmed = true;
				elem.classList.add('armed');
				setTimeout(function () {
					clearArmed = false;
					elem.classList.remove('armed');
				}, _this.CLEAR_ARMED_WINDOW);
			} else {
				clearArmed = false;
				elem.classList.remove('armed');

				// Avoid localStorage.clear() because it also deletes
				// non-trail data like "first launch" indicator

				_this.logUi('Clearing trail');
				Trail.clearAllTrailsFromLocalStorage(window.localStorage);
				_this.trail = new Trail();
				_this.saveCurrentTrail();
				_this.clearPreviousFix();
				_this.fixCountTrail = 0;
				_this.updateFixToUnknown();
				_this.updateFixStats();
				_this.updateTrailLengthToUi();
				_this.updateAverageSpeedToUi();
			}
		} catch (e) {
			alert(e);
		}
	});
};

/* Setup app exit button handling. */
OfftrailApplication.prototype.setupExitButton = function () {
	var _this = this;
	var elem;

	elem = document.getElementById('exit-button');
	elem.addEventListener('click', function (event) {
		event.stopPropagation();
		try {
			/* Close active segment, save current trail, and
			 * try to exit before any further fixes are received.
			 */
			_this.trail.closeActive();
			_this.saveCurrentTrail();
			_this.clearPreviousFix();
		} catch (e) {
			alert(e);
		}
		window.close();
	});
};

/* Setup about button handling. */
OfftrailApplication.prototype.setupAboutButton = function () {
	var _this = this;
	var elem;

	elem = document.getElementById('about-button');
	elem.addEventListener('click', function (event) {
		event.stopPropagation();
		try {
			_this.showFullScreenWindow({
				windowId: 'aboutview',
				closeId: 'about-close'
			});
		} catch (e) {
			alert(e);
		}
	});
};

/* Setup settings button handling. */
OfftrailApplication.prototype.setupSettingsButton = function () {
	var _this = this;
	var elem;

	elem = document.getElementById('settings-button');
	elem.addEventListener('click', function (event) {
		event.stopPropagation();
		try {
			_this.showFullScreenWindow({
				windowId: 'settingsview',
				closeId: 'settings-close'
			});
		} catch (e) {
			alert(e);
		}
	});
};

/* Setup log viewer button. */
OfftrailApplication.prototype.setupLogViewButton = function () {
	var _this = this;
	var elem;

	elem = document.getElementById('log-lines');
	elem.addEventListener('click', function (event) {
		event.stopPropagation();
		try {
			DomUtil.setTextToId('logview-text', _this.getFullLogText());
			_this.showFullScreenWindow({
				windowId: 'logview',
				closeId: 'logview-close'
			});
		} catch (e) {
			alert(e);
		}
	});
};

// FIXME: prototyping
OfftrailApplication.prototype.postCurrentFix = function () {
	var _this = this;

	//this.logUi('Posting current fix');
	var doc = {
		lon: this.currLon,
		lat: this.currLat,
		alt: this.currAlt,
		hea: this.currHea,
		spd: this.currSpd,
		accuracy: this.currAccuracy,
		altAccuracy: this.currAltAccuracy,
		time: this.currTime
	};
	$.ajax({
		type: 'POST',
		url: _this.settings.fixPostUri,
		crossDomain: true,
		async: true,
		cache: false,
		data: JSON.stringify(doc),
		contentType: 'application/json; charset=utf-8',
		processData: false,
		success: function (data, textStatus, jqxhr) {
			_this.logUi('Fix POST successful');
		},
		error: function (jqxhr, textStatus, errorThrown) {
			_this.logUi('Fix POST failed: ' + errorThrown + ', ' + textStatus);
		}
	});
};

OfftrailApplication.prototype.setupFixPost = function () {
	var _this = this;
	setInterval(function () {
		try {
			_this.postCurrentFix();
		} catch (e) {
			_this.logUi('Fix POST failed: ' + e);
		}
	}, _this.settings.fixPostInterval);
};

OfftrailApplication.prototype.playBuffer = function (buffer) {
	var source;

	if (!this.audioContext || !this.gainNode || buffer == null) {
		return;
	}
	source = this.audioContext.createBufferSource();
	source.buffer = buffer;
	source.connect(this.gainNode);
	source.start(0);
};

OfftrailApplication.prototype.loadBuffer = function (uri, callback) {
	var _this = this;
	var request = new XMLHttpRequest();
	request.open('GET', uri, true);
	request.responseType = 'arraybuffer';
	request.onload = function () {
		_this.audioContext.decodeAudioData(request.response, function (buffer) {
			//console.log('buffer ' + uri + ':', buffer.sampleRate, buffer.length, buffer.duration, buffer.numberOfChannels);
			callback(buffer);
		}, function () {
			alert('audio decode failed');
		});
	};
	request.send();
}

OfftrailApplication.prototype.audioInit = function () {
	var _this = this

	// http://www.html5rocks.com/en/tutorials/webaudio/intro/
	// http://stackoverflow.com/questions/17988630/audiocontext-of-web-audio-api-is-not-exist-in-chrome-beta-29-of-android-4-0-tabl
	var audioCtxConstructor =
		window.AudioContext ||
		window.webkitAudioContext ||
		window.mozAudioContext ||
		window.oAudioContext ||
		window.msAudioContext;
	if (!audioCtxConstructor) {
		alert('No Audio API support, audio disabled.');
		return;
	}

	this.audioContext = new audioCtxConstructor();

	// background audio (no audio break if screen is locked etc)
	// - https://runawaydev.wordpress.com/2013/05/11/firefox-os-how-to-prevent-musicvideo-to-stop-playing-when-the-app-goes-into-the-background/comment-page-1/
	// - http://dxr.mozilla.org/mozilla-central/source/content/media/webaudio/test/test_mozaudiochannel.html
	this.audioContext.mozAudioChannelType = 'content';
	if (this.audioContext.mozAudioChannelType !== 'content') {
		alert("Failed to set AudioContext mozAudioChannelType, " +
		      "audio may break when screen is locked.");
	}

	this.gainNode = this.audioContext.createGain();
	this.gainNode.gain.value = 1.0;
	this.gainNode.connect(this.audioContext.destination);

	// Load sounds asynchronously

	this.loadBuffer('alarm.ogg', function (buffer) {
		_this.buffer_alarm = buffer;
	});
};

OfftrailApplication.prototype.getDefaultSettings = function () {
	return {
		// email attachment types
		useKml: false,
		useKmz: true,
		useGpx: false,
		useGpxZip: true,

		// fix posting
		fixPostUri: 'http://127.0.0.1:8080/fix',
		fixPostInterval: 60 * 1000,  /* 1 min, times 1 kB => ~60kB/hour */

		// trail posting
		trailPostUri: 'http://127.0.0.1:8080/trail',
		trailPostInterval: 30 * 60 * 1000,  /* 30 mins, times 10 kB => ~20kB/hour */
	};
};
OfftrailApplication.prototype.loadSettings = function () {
	var settings = this.getDefaultSettings();
	var tmp;
	var k;

	/* Start from default settings and copy any settings from
	 * local storage over the defaults.  If a key has no default
	 * value it is not relevant to the current version, and is
	 * not copied.  This behavior ensures that in a software update
	 * (a) new keys get default values and (b) obsolete keys are
	 * removed.
	 */

	try {
		tmp = window.localStorage.getItem('settings');
		if (typeof tmp === 'string') {
			tmp = JSON.parse(tmp);
			for (k in tmp) {
				if (k in settings) {
					settings[k] = tmp[k];
				}
			}
		}
	} catch (e) {
		alert('Failed to load settings, falling back to defaults: ' + e);
	}
	this.settings = settings;
};
OfftrailApplication.prototype.saveSettings = function () {
	var data = JSON.stringify(this.settings);
	//alert('Saving settings: ' + data);
	window.localStorage.setItem('settings', data);
};

OfftrailApplication.prototype.actualInit = function () {
	var _this = this;
	var trailIds;

	this.loadSettings();
	this.saveSettings();

	// Disabled for 1.2
	//this.audioInit();

	onVisibilityChange(function () {
		if (isVisible()) {
			/* Page became visible, update all visible items
			 * up-to-date to make the UI seamless.
			 */
			_this.debugLogUi('Page became visible');
			this.updateTrailLengthToUi();
			this.updateAverageSpeedToUi();
			this.updatePositionToUi();
		} else {
			_this.debugLogUi('Page became hidden');
		}
	});

	trailIds = Trail.scanTrailIdsFromLocalStorage(window.localStorage);
	//alert('Found trails: ' + trailIds.join(','));
	this.trail = new Trail();
	if (trailIds.length > 1) {
		alert('There are multiple trails in the store: ' +
		      JSON.stringify(trailIds) + ', using the first one.');
	}
	if (trailIds.length > 0) {
		//alert('Import trail from local storage: ' + trailIds[0]);
		this.trail.importFromLocalStorage(window.localStorage, trailIds[0]);
		this.trail.closeActive();  // always close active track on load (simplifies it too)
		this.logUi('Loaded trail ' + trailIds[0] + ' from local storage');
	}

	this.updateFixToUnknown();
	this.saveCurrentTrail();
	//this.setupCloseSegmentButton();
	//this.setupSaveTrailButton();
	this.setupEmailSendButton();
	this.setupClearTrailButton();
	this.setupExitButton();
	this.setupAboutButton();
	//this.setupSettingsButton();
	this.setupLogViewButton();
	this.setupGeolocation();
	this.setupAutosave();

	// FIXME: prototyping
	this.setupFixPost();
};
OfftrailApplication.prototype.init = function () {
	var _this = this;
	var elemAbout;
	var elemAboutClose;
	var initDone = false;

	// Put version info in its place before about text
	DomUtil.setTextToId('offtrail-info-detailed',
	                    'Offtrail ' + OFFTRAIL_VERSION + ' (' +
	                    OFFTRAIL_GITDESCRIBE + ': ' +
	                    OFFTRAIL_GITCOMMIT + ')');
	this.logUi('Offtrail ' + OFFTRAIL_VERSION);

	if (window.localStorage.getItem('first-launch-shown')) {
		this.actualInit();
		initDone = true;
		return;
	}

	_this.showFullScreenWindow({
		windowId: 'aboutview',
		closeId: 'about-close',
		closeCallback: function () {
			if (!initDone) {
				try {
					_this.actualInit();
				} catch (e) {
					alert(e);
				}
				initDone = true;
				window.localStorage.setItem('first-launch-shown', 'true');
			}
		}
	});
};
