/*
 *  Miscellaneous geo utilities
 *
 *  NOTE: the primitives here carry a time component which is currently
 *  ignored.  It might be added into distance calculations for e.g.
 *  simplification, but it needs a seconds-to-meters conversion ratio
 *  which depends on the call site.  It might be cleanest to remove the
 *  time component from these helpers.
 */

GeoUtil = {};

/* Spherical Earth radius estimate used for distance calculations
 * (see http://en.wikipedia.org/wiki/Earth_radius).
 */
GeoUtil.SPHERICAL_EARTH_RADIUS = 6371000;

/* Convert a lon/lat/alt triple to ECEF coordinates assuming a spherical
 * Earth (see http://en.wikipedia.org/wiki/ECEF):
 * http://stackoverflow.com/questions/1185408/converting-from-longitude-latitude-to-cartesian-coordinates
 *
 * The WGS84 conversion would be more complicated:
 * http://en.wikipedia.org/wiki/Geodetic_system#From_geodetic_to_ECEF
 */
GeoUtil.geoToEcef = function(lon, lat, alt, out) {
	var R = GeoUtil.SPHERICAL_EARTH_RADIUS + alt;  // radius corrected by altitude
	lat = lat / 180 * Math.PI;
	lon = lon / 180 * Math.PI;
	out[0] = R * Math.cos(lat) * Math.cos(lon);
	out[1] = R * Math.cos(lat) * Math.sin(lon);
	out[2] = R * Math.sin(lat);
};

/* Distance between two points in meters.  Does not account for curvature.
 *
 * The time dimension is currently ignored, but if it weren't, it would be
 * converted to "meter equivalents" with some user specified weight (in other
 * words: X seconds of time error is equivalent to 1m of distance error from
 * a simplification standpoint).
 */
GeoUtil.pointDist = function (lon1, lat1, alt1, time1, lon2, lat2, alt2, time2) {
	var xyz1 = [];
	var xyz2 = [];
	var diff = [];
	GeoUtil.geoToEcef(lon1, lat1, alt1, xyz1);
	GeoUtil.geoToEcef(lon2, lat2, alt2, xyz2);
	diff[0] = xyz1[0] - xyz2[0];
	diff[1] = xyz1[1] - xyz2[1];
	diff[2] = xyz1[2] - xyz2[2];
	return Math.sqrt(diff[0] * diff[0] + diff[1] * diff[1] + diff[2] * diff[2]);
};

/* Compute the distance of point P3 (lon3, lat3, alt3, time3) from a line
 * segment running between points P1 (lon1, lat1, alt1, time1) and P2
 * (lon2, lat2, alt2, time2).  Note the difference between computing a
 * distance between a point and an infinite line vs. a point and a line
 * segment.
 *
 * http://stackoverflow.com/questions/849211/shortest-distance-between-a-point-and-a-line-segment
 */
GeoUtil.lineSegmentDist = function (lon1, lat1, alt1, time1, lon2, lat2, alt2, time2, lon3, lat3, alt3, time3) {
	var xyz1 = [];
	var xyz2 = [];
	var xyz3 = [];
	var x1, y1, z1;
	var x2, y2, z2;
	var x3, y3, z3;
	var x4, y4, z4;

	GeoUtil.geoToEcef(lon1, lat1, alt1, xyz1);
	GeoUtil.geoToEcef(lon2, lat2, alt2, xyz2);
	GeoUtil.geoToEcef(lon3, lat3, alt3, xyz3);
	x1 = xyz1[0]; y1 = xyz1[1]; z1 = xyz1[2];
	x2 = xyz2[0]; y2 = xyz2[1]; z2 = xyz2[2];
	x3 = xyz3[0]; y3 = xyz3[1]; z3 = xyz3[2];

	//print('p1', lon1, lat1, alt1, '->', x1, y1, z1);
	//print('p2', lon2, lat2, alt2, '->', x2, y2, z2);
	//print('p3', lon3, lat3, alt3, '->', x3, y3, z3);

	var dx = x2 - x1;    // vector P1->P2
	var dy = y2 - y1;
	var dz = z2 - z1;
	var dpx = x3 - x1;   // vector P1->P3
	var dpy = y3 - y1;
	var dpz = z3 - z1;
	var segLenSquared = dx * dx + dy * dy + dz * dz;

	//print(segLenSquared);
	if (segLenSquared < 1e-3) {
		/* Segment is very short, we can just take a point distance
		 * to avoid other inaccuracies.
		 */
		//print('line is very short -> use point distance to P1');
		return Math.sqrt(dpx * dpx + dpy * dpy + dpz * dpz);
	}

	/* Shortest distance to infinite line going through P1->P2,
	 * t=[0,1] when closest point is inside the line segment.
	 */
	var t = (dpx * dx + dpy * dy + dpz * dz) / segLenSquared;
	if (t <= 0) {
		/* P1 is closest to P3, fall through. */
		//print('P1 is closest to P3');
	} else if (t >= 1) {
		/* P2 is closest to P3. */
		//print('P2 is closest to P3');
		dpx = x3 - x2;
		dpy = y3 - y2;
		dpz = z3 - z2;
	} else {
		/* Closest point is between P1 and P2. */
		//print('closest point to P3 is between ]P1,P2[');
		x4 = t * dx + x1;
		y4 = t * dy + y1;
		z4 = t * dz + z1;
		//print('p4, t:', t, '->', x4, y4, z4);
		dpx = x3 - x4;
		dpy = y3 - y4;
		dpz = z3 - z4;
	}
	//print('final', dpx, dpy, dpz);
	return Math.sqrt(dpx * dpx + dpy * dpy + dpz * dpz);
};
