/*
 *  GPX export
 *
 *  http://en.wikipedia.org/wiki/GPS_Exchange_Format
 *  http://www.topografix.com/gpx.asp
 *  https://github.com/cgeo/cgeo/issues/527
 */

function GpxExporter() {
}

GpxExporter.prototype._writeSegment = function (res, seg) {
	var i, n;
	var lon, lat, alt, time;
	var nelem = seg.ELEMS_PER_FIX;

	res.push('<trkseg>');
	for (i = 0, n = seg.data.length; i < n; i += nelem) {
		lon = seg.data[i];
		lat = seg.data[i + 1];
		alt = seg.data[i + 2];
		time = seg.data[i + 3];
		res.push('<trkpt lat="' + lat.toFixed(7) + '" lon="' + lon.toFixed(7) + '">' +
		         '<ele>' + alt.toFixed(0) + '</ele>' +
		         '<time>' + XmlUtil.formatTime(new Date(time)) + '</time>' +
		         '</trkpt>');
	}
	res.push('</trkseg>');
};

GpxExporter.prototype.trailToGpx = function (trail) {
	var res = [];
	var i, n;
	var trailName = 'Offtrail ' + trail.trailId + ' ' + new Date().toISOString();
	res.push('<?xml version="1.0" encoding="UTF-8"?>');
	res.push('<gpx xmlns="http://www.topografix.com/GPX/1/1"')
	res.push('     xmlns:gpxx="http://www.garmin.com/xmlschemas/GpxExtensions/v3"');
	res.push('     xmlns:gpxtpx="http://www.garmin.com/xmlschemas/TrackPointExtension/v1"');
	res.push('     creator="Offtrail ' + XmlUtil.xmlEscape(OFFTRAIL_VERSION) + '"');
	res.push('     version="1.1"');
	res.push('     xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"');
	res.push('     xsi:schemaLocation="http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd http://www.garmin.com/xmlschemas/GpxExtensions/v3 http://www.garmin.com/xmlschemas/GpxExtensionsv3.xsd http://www.garmin.com/xmlschemas/TrackPointExtension/v1 http://www.garmin.com/xmlschemas/TrackPointExtensionv1.xsd">');
	res.push('<!--');
	res.push(XmlUtil.xmlEscape('Created with Offtrail ' + OFFTRAIL_VERSION + ' ' + OFFTRAIL_GITDESCRIBE + ' ' + OFFTRAIL_GITCOMMIT));
	res.push('-->');
	res.push('<metadata>');
	res.push('<link href="https://marketplace.firefox.com/app/offtrail">');
	res.push('<text>' + XmlUtil.xmlEscape('Offtrail ' + OFFTRAIL_VERSION) + '</text>');
	res.push('</link>');
	res.push('</metadata>');
	res.push('<trk>');
	res.push('<name>' + XmlUtil.xmlEscape(trailName) + '</name>');
	for (i = 0, n = trail.segments.length; i < n; i++) {
		this._writeSegment(res, trail.segments[i]);
	}
	res.push('</trk>');
	res.push('</gpx>');
	return res.join('\n');
};

GpxExporter.prototype.trailToGpxBlob = function (trail) {
	return new Blob([
		this.trailToGpx(trail)
	], {
		type: 'application/gpx+xml'
	});
};

GpxExporter.prototype.trailToGpxZipBlob = function (trail) {
	var xmlData = this.trailToGpx(trail);
	var zip = new JSZip();
	// The 'main document' naming matches KMZ at the moment.  It might
	// be better for the name to include the trail ID, so that if you
	// unpack multiple files the GPX files don't overwrite each other.
	// (On the other hand this happens with KMZ anyway.)
	zip.file('doc.gpx', xmlData);

	// Important to specify 'compression', otherwise 'STORE' is used
	// http://stuk.github.io/jszip/documentation/api_jszip/generate.html
	var content = zip.generate({ type: 'blob', compression: 'DEFLATE' });
	return content;
};
