/*
 *  KML export
 *
 *  http://en.wikipedia.org/wiki/Keyhole_Markup_Language
 *  https://developers.google.com/kml/documentation/
 *  https://developers.google.com/kml/documentation/kmzarchives
 */

function KmlExporter() {
}

KmlExporter.prototype._writeSegment = function (res, seg) {
	var i, n;
	var lon, lat, alt, time;
	var nelem = seg.ELEMS_PER_FIX;

	res.push('<LineString>');
	res.push('<gx:altitudeOffset>0</gx:altitudeOffset>');
	res.push('<extrude>false</extrude>');
	res.push('<tessellate>false</tessellate>');
	res.push('<altitudeMode>clampToGround</altitudeMode>');  // XXX: configurable (flight paths)?
	res.push('<coordinates>');
	for (i = 0, n = seg.data.length; i < n; i += nelem) {
		lon = seg.data[i];
		lat = seg.data[i + 1];
		alt = seg.data[i + 2];
		time = seg.data[i + 3];
		res.push(lon.toFixed(7) + ',' + lat.toFixed(7) + ',' + alt.toFixed(0));
	}
	res.push('</coordinates>');
	res.push('</LineString>');
};

KmlExporter.prototype.trailToKml = function (trail) {
	var res = [];
	var i, n;
	var trailName = 'Offtrail ' + trail.trailId + ' ' + new Date().toISOString();
	res.push('<?xml version="1.0" encoding="UTF-8"?>');
	res.push('<!-- Offtrail ' + OFFTRAIL_VERSION + ' (' + OFFTRAIL_GITDESCRIBE + ')' + ' -->');
	res.push('<kml xmlns="http://www.opengis.net/kml/2.2" xmlns:gx="http://www.google.com/kml/ext/2.2">');
	res.push('<Document>');
	res.push('<name>' + XmlUtil.xmlEscape(trailName) + '</name>');
	res.push('<Placemark>');
	res.push('<name>' + XmlUtil.xmlEscape(trailName) + '</name>');
	for (i = 0, n = trail.segments.length; i < n; i++) {
		this._writeSegment(res, trail.segments[i]);
	}
	res.push('</Placemark>');
	res.push('</Document>');
	res.push('</kml>');
	return res.join('\n');
};

KmlExporter.prototype.trailToKmlBlob = function (trail) {
	return new Blob([
		this.trailToKml(trail)
	], {
		type: 'application/vnd.google-earth.kml+xml'
	});
};

KmlExporter.prototype.trailToKmzBlob = function (trail) {
	var xmlData = this.trailToKml(trail);
	var metaData = JSON.stringify({
		trailId: trail.trailId,
		offtrailVersion: OFFTRAIL_VERSION,
		offtrailGitDescribe: OFFTRAIL_GITDESCRIBE,
		offtrailGitCommit: OFFTRAIL_GITCOMMIT
	});
	var zip = new JSZip();
	zip.file('doc.kml', xmlData);
	var files = zip.folder('files');  // extra files
	files.file('metadata.json', metaData);

	// Important to specify 'compression', otherwise 'STORE' is used
	// http://stuk.github.io/jszip/documentation/api_jszip/generate.html
	var content = zip.generate({ type: 'blob', compression: 'DEFLATE' });
	return content;
};
