/*
 *  Trail
 */

function Trail() {
	this.segments = [];                            // TrailSegment objects
	this.trailId = Trail.generateRandomTrailId();  // Trail ID (string); may change when loading
}

/* Douglas-Peucker simplification tolerance in meters. */
Trail.prototype.SIMPLIFY_TOLERANCE_METERS = 4.0;

Trail.generateRandomTrailId = function () {
	var num = (Date.now() * 1000) + Math.floor(Math.random() * 1000);
	return num.toString(36);  // base-36
};

/* Clear all trails from the local storage. */
Trail.clearAllTrailsFromLocalStorage = function (ls) {
	var toDelete = [];
	var i, n, k;
	var trPrefix = 'tr-';

	for (i = 0, n = ls.length; i < n; i++) {
		k = ls.key(i);
		if (k.substring(0, trPrefix.length) === trPrefix) {
			toDelete.push(k);
		}
	}

	toDelete.forEach(function (k) {
		ls.removeItem(k);
	});
};

/* 'Class' method for scanning all trail IDs from local storage. */
Trail.scanTrailIdsFromLocalStorage = function (ls) {
	var res = {};
	var i, n;

	for (i = 0, n = ls.length; i < n; i++) {
		m = /^tr-([^-]+)-.*$/.exec(ls.key(i));
		if (m) {
			res[m[1]] = true;
		}
	}

	res = Object.keys(res);
	res.sort();
	return res;
};

/* 'Class' method for clearing all local storage entries for a certain
 * trail ID.  Use two passes to avoid mutation while iterating.
 */
Trail.clearLocalStorageForTrail = function(ls, trailId) {
	var toDelete = [];
	var i, n, k;
	var trPrefix = 'tr-' + trailId + '-';

	for (i = 0, n = ls.length; i < n; i++) {
		k = ls.key(i);
		if (k.substring(0, trPrefix.length) === trPrefix) {
			toDelete.push(k);
		}
	}

	toDelete.forEach(function (k) {
		ls.removeItem(k);
	});
};

/* Import trail from local storage.  Also overwrites this.trailId. */
Trail.prototype.importFromLocalStorage = function(ls, trailId) {
	var i, tmp, seg;
	var trPrefix = 'tr-' + trailId + '-';

	this.segments = [];
	this.trailId = trailId;

	for (i = 0; ; i++) {
		tmp = ls.getItem(trPrefix + i);
		if (typeof tmp !== 'string') {
			/* Stop scanning when no more segments available.
			 * Also handles gaps and corruption by simply
			 * ignoring them.
			 */
			break;
		}
		seg = new TrailSegment();
		seg.importFromString(tmp);
		this.segments.push(seg);
	}
};

/* Export trail to local storage, overwriting any trail with the same
 * trailId.
 */
Trail.prototype.exportToLocalStorage = function(ls) {
	var trPrefix = 'tr-' + this.trailId + '-';
	var i, n;
	var seg;

	// Clear local storage of any previous entries related to this trail.
	// This is more robust (but slower) than simply continuing to remove
	// any higher indices that are present in the local storage (an approach
	// which fails if there are numbering gaps).
	Trail.clearLocalStorageForTrail(ls, this.trailId);

	for (i = 0, n = this.segments.length; i < n; i++) {
		seg = this.segments[i];
		ls.setItem(trPrefix + i, seg.exportToString());
	}
};

/* Add a fix to the trail, creating a new trail segment if necessary. */
Trail.prototype.addFix = function(lon, lat, alt, time) {
	var seg;
	if (this.segments.length > 0 && this.segments[this.segments.length - 1].active) {
		seg = this.segments[this.segments.length - 1];
	} else {
		seg = new TrailSegment();
		seg.active = true;
		this.segments.push(seg);
	}

	seg.addFix(lon, lat, alt, time);

	// XXX: implement live saving of fixes for the active segment to minimize data
	// loss compared to the current autosave mechanim?  Perhaps use a different
	// format where fixes are saved in chunks instead of a single long string,
	// so that the incremental write is reasonable (say <1kB).  One option would also
	// just save the raw fixes accumulated between autosaves as a kind of a journal
	// which could be replayed on init.
};

Trail.prototype.hasActiveSegment = function () {
	return (this.segments.length > 0) &&
	       (this.segments[this.segments.length - 1].active);
};
Trail.prototype.getActiveSegmentFixCount = function () {
	var seg;

	if (this.segments.length > 0) {
		seg = this.segments[this.segments.length - 1];
		if (seg.active) {
			return Math.floor(seg.data.length / seg.ELEMS_PER_FIX);
		}
	}
	return 0;
};

Trail.prototype.closeActive = function() {
	var seg;
	var startTime, endTime;

	if (this.segments.length == 0) {
		return;
	}
	seg = this.segments[this.segments.length - 1];
	if (!seg.active) {
		return;
	}
	seg.active = false;

	// Simplify closed segment once when it is closed.  If a segment
	// is serialized as active and the application exits, the segment
	// is loaded back in active state and the init code will close it,
	// simplifying it in the process.

	startTime = Date.now();
	TrailSegmentSimplifier.simplifyTrailSegment(seg, this.SIMPLIFY_TOLERANCE_METERS);
	endTime = Date.now();

	// FIXME: any trail segment caches should be cleared here (e.g.
	// for partial lengths and such)

	//alert('simplify took ' + (endTime - startTime) + ' ms');
};

Trail.prototype.getLength3D = function() {
	var i, n;
	var len = 0;
	for (i = 0, n = this.segments.length; i < n; i++) {
		len += this.segments[i].getLength3D();
	}
	return len;
};

Trail.prototype.getLength2D = function() {
	var i, n;
	var len = 0;
	for (i = 0, n = this.segments.length; i < n; i++) {
		len += this.segments[i].getLength2D();
	}
	return len;
};

Trail.prototype.getAverageSpeedForLastSegment = function () {
	var seg;

	if (this.segments.length === 0) {
		return NaN;
	}
	seg = this.segments[this.segments.length - 1];
	return seg.getLength3D() / (seg.getDuration() / 1000);
};

Trail.prototype.getStats = function() {
	var nseg = this.segments.length;
	var npts = 0;
	var i, n, seg;

	for (i = 0, n = this.segments.length; i < n; i++) {
		seg = this.segments[i];
		npts += Math.floor(seg.data.length / seg.ELEMS_PER_FIX);
	}

	return {
		numSegments: nseg,
		numPoints: npts
	};
};
