/*
 *  Trail segment representation
 *
 *  A segment is serialized to a string as a space-separated string
 *  with the following format.  The first element is 'T' or 'F',
 *  indicating whether the segment is active or not.  Trail data
 *  follows the first element, in 4-tuples of (lon, lat, alt, time).
 *  Each value is encoded either as an absolute value or as a value
 *  relative to the previous value (e.g. longitude is encoded as a
 *  difference to a previous longitude value).
 *
 *  The absolute values are represented as follows:
 * 
 *    - Longitude and latitude: float values are converted to integers
 *      as: round(x * LONLAT_SCALE).
 *
 *    - Altitude: float value is converted to integer directly (meters)
 *
 *    - Timestamp: encoded directly as an integer (seconds).
 *
 *  A '!' character precedes an absolute integer value, in a base-36
 *  encoding.
 *
 *  An absolute value may also be 'N', indicating 'not available'.
 *  A relative value cannot be used if the previous value does not
 *  exist or is 'N'.
 *
 *  Relative values are represented by taking the integer difference:
 *
 *    D = new - old
 *
 *  and then converting D to a positive integer as follows:
 *
 *    if D >= 0 -> 2*D
 *    else      -> 2*(-D) - 1
 *
 *  For instance:
 *
 *    0 -> 0
 *    1 -> 2
 *    2 -> 4
 *   -1 -> 1
 *   -2 -> 3
 *
 *  The relative value is then encoded in a base-36 encoding.
 */

function TrailSegment() {
	// timestamp etc
	this.data = [];   /* lon, lat, alt, time quadruples; fixes = this.data.length / ELEMS_PER_FIX */
	this.active = false;
	this.encoded = undefined;   /* set when segment is closed */
	this.lenIndex = 1;          /* exclusive fix index up to which length is precomputed (must start at 1, not 0, so that there is a previous point) */
	this.len2d = 0;             /* cached length up to lenIndex */
	this.len3d = 0;             /* cached length up to lenIndex */
}

TrailSegment.prototype.ELEMS_PER_FIX = 4;

TrailSegment.prototype.NUM_LON_DIGITS = 6;
TrailSegment.prototype.NUM_LAT_DIGITS = 6;
TrailSegment.prototype.NUM_ALT_DIGITS = 0;
TrailSegment.prototype.NUM_TIMESTAMP_DIGITS = 0;  /* after conversion to seconds */

TrailSegment.prototype.LONLAT_SCALE = 100000.0;
TrailSegment.prototype.TIME_SCALE = 1.0 / 1000.0;

TrailSegment.prototype.addFix = function(lon, lat, alt, time) {
	if (!this.active) {
		throw new Error('attempt to modify a closed segment');
	}
	this.data.push(lon);
	this.data.push(lat);
	this.data.push(alt);
	this.data.push(time);
}

TrailSegment.prototype.decodeValue = function(x, prev) {
	var c, i, d;

	c = x[0];
	if (c === 'N') {
		return NaN;
	} else if (c === '!') {
		return parseInt(x.substring(1), 36);
	} else {
		i = parseInt(x, 36);
		if ((i % 2) === 0) {
			d = i / 2;
		} else {
			d = (i + 1) / -2;
		}

		return prev + d;
	}
}

TrailSegment.prototype.encodeValue = function(x_new, x_old) {
	var t;

	if (x_new === undefined || isNaN(x_new)) {
		return 'N';
	}

	if (x_old === undefined || isNaN(x_old)) {
		return '!' + x_new.toString(36);
	}

	t = x_new - x_old;
	if (t >= 0) {
		return (2 * t).toString(36);
	} else {
		return ((-2 * t) - 1).toString(36);
	}
}

TrailSegment.prototype.decodeLonLat = function(x) {
	return x / this.LONLAT_SCALE;
}

TrailSegment.prototype.decodeAlt = function(x) {
	return x;
}

TrailSegment.prototype.decodeTime = function(x) {
	return x / this.TIME_SCALE;
}

TrailSegment.prototype.encodeLonLat = function(x) {
	return Math.round(x * this.LONLAT_SCALE);
}

TrailSegment.prototype.encodeAlt = function(x) {
	return Math.round(x);
}

TrailSegment.prototype.encodeTime = function(x) {
	return Math.round(x * this.TIME_SCALE);
}

TrailSegment.prototype.importFromString = function(x) {
	var i, n;
	var t = x.split(' ');
	var prev_lon, prev_lat, prev_alt, prev_time;
	var nelem = this.ELEMS_PER_FIX;

	this.data = [];
	this.active = (t[0] == 'T');

	for (i = 1, n = t.length; i < n; i += nelem) {
		prev_lon = this.decodeValue(t[i], prev_lon);
		prev_lat = this.decodeValue(t[i + 1], prev_lat);
		prev_alt = this.decodeValue(t[i + 2], prev_alt);
		prev_time = this.decodeValue(t[i + 3], prev_time);

		this.data.push(this.decodeLonLat(prev_lon));
		this.data.push(this.decodeLonLat(prev_lat));
		this.data.push(this.decodeAlt(prev_alt));
		this.data.push(this.decodeTime(prev_time));
	}
}

TrailSegment.prototype.exportToString = function() {
	var x = [];
	var i, n;
	var t;
	var prev_lon, prev_lat, prev_alt, prev_time;
	var res;
	var nelem = this.ELEMS_PER_FIX;

	if (this.encoded !== undefined) {
		return this.encoded;
	}

	x.push(this.active ? 'T' : 'F');

	n = this.data.length;
	for (i = 0; i < n; i += nelem) {
		t = this.encodeLonLat(this.data[i]);
		x.push(this.encodeValue(t, prev_lon));
		prev_lon = t;

		t = this.encodeLonLat(this.data[i + 1]);
		x.push(this.encodeValue(t, prev_lat));
		prev_lat = t;

		t = this.encodeAlt(this.data[i + 2]);
		x.push(this.encodeValue(t, prev_alt));
		prev_alt = t;

		t = this.encodeTime(this.data[i + 3]);
		x.push(this.encodeValue(t, prev_time));
		prev_time = t;
	}

	res = x.join(' ');

	if (!this.active) {
		// Cache the result for closed segments to speed up
		// trail writing.
		this.encoded = res;
	}

	return res;
}

/* Internal helper for length calculation.  Currently uses a spherical Earth
 * approximation and doesn't take into account Earth curvature.  These should
 * be quite meaningless issues in practice.
 */
TrailSegment.prototype.computeLengthRaw = function (fixedAlt, startIdx, endIdx) {
	var len = 0;
	var data = this.data;
	var i;
	var xyz1, xyz2;
	var alt1, alt2;
	var nelem = this.ELEMS_PER_FIX;

	for (i = startIdx; i < endIdx; i++) {
		if (fixedAlt !== null) {
			alt1 = fixedAlt;
			alt2 = fixedAlt;
		} else {
			alt1 = data[(i - 1) * nelem + 2] || 0;
			alt2 = data[i * nelem + 2] || 0;
		}
		len += GeoUtil.pointDist(data[(i - 1) * nelem],
		                         data[(i - 1) * nelem + 1],
		                         alt1,
		                         0,  // XXX: time
		                         data[i * nelem],
		                         data[i * nelem + 1],
		                         alt2,
		                         0);  // XXX: time
	}

	//console.log('incremental length [' + startIdx + ',' + endIdx + '[ -> ' + len);
	return len;
};

/* Update length caches to current trail segment length. */
TrailSegment.prototype.updateLengthCaches = function () {
	var nelem = this.ELEMS_PER_FIX;

	if (this.lenIndex * nelem === this.data.length) {
		//console.log('length cache up-to-date');
		return;
	}

	//console.log('update length cache: ' + this.lenIndex + ' -> ' + this.data.length / nelem);
	this.len2d += this.computeLengthRaw(0, this.lenIndex, this.data.length / nelem);
	this.len3d += this.computeLengthRaw(null, this.lenIndex, this.data.length / nelem);
	this.lenIndex = this.data.length / nelem;
	//console.log('len2d=' + this.len2d + ', len3d=' + this.len3d);
};

/* Get trail segment length (estimate) taking into account altitude changes
 * so that the trail is considered in 3D space.
 */
TrailSegment.prototype.getLength3D = function () {
	this.updateLengthCaches();
	return this.len3d;
};

/* Get trail segment length (estimate) assuming the trail is flat, at fixed
 * zero altitude.  This matches typical length measurements made with a
 * paper map better than a 3D distance.
 */
TrailSegment.prototype.getLength2D = function () {
	this.updateLengthCaches();
	return this.len2d;
};

TrailSegment.prototype.getDuration = function () {
	var nelem = this.ELEMS_PER_FIX;
	var npts = this.data.length / nelem;

	if (npts <= 1) {
		return 0;
	}

	return this.data[(npts - 1) * nelem + 3] - this.data[0 * nelem + 3];
};
