/*
 *  Trail segment simplification.
 *
 *  Trail segments are simplified with the Douglas-Peucker algorithm, which
 *  selects an optimal subset of input points so that the output polyline
 *  is within a specified tolerance of the input polyline.
 *
 *  There are several limitations:
 *
 *    - Trail segments are assumed to be reasonably small so that there
 *      is no need to account for the curvature of the Earth.
 *
 *    - The point distance conversion uses a spherical Earth model for
 *      simplicity; this is only used in the simplification decisions
 *      so should be quite adequate.
 *
 *    - The time parameter is passed around but is not used in the distance
 *      computations at the moment.  It could be included in the process
 *      by providing a seconds-to-meters conversion factor which weights
 *      the importance of time vs. distance in simplification.
 *
 *  http://en.wikipedia.org/wiki/Ramer%E2%80%93Douglas%E2%80%93Peucker_algorithm
 *  http://wiki.openstreetmap.org/wiki/Douglas-Peucker
 *  http://karthaus.nl/rdp/
 */

TrailSegmentSimplifier = {};

/* Recursive Douglas-Peucker helper.  'data' is a set of input points which
 * is processed.  The function returns a simplified set of points for 'data'
 * but always excludes the final point in 'data' (this happens throughout the
 * algorithm).  The 'tolerance' value is in (approximate) meters.
 */
TrailSegmentSimplifier.douglasPeuckerRaw = function (data, tolerance, nelem) {
	var npts = data.length / nelem;  /* elements are: lon, lat, alt, time */
	var idx, idxPx;
	var dist, distPx;
	var sub1, sub2;

	//print('Douglas-Peucker: ' + (data.length / nelem) + ' points: ' + JSON.stringify(data));

	if (npts == 0) {
		/* Should not happen, return empty */
		return [];
	}
	if (npts == 1) {
		/* Single point case: return empty */
		return [];
	}
	if (npts == 2) {
		/* Two points case: return the initial point */
		return data.slice(0, nelem);
	}

	/*
	 *  For points P0,P1,...,Pn-1 imagine a line between P0 and Pn-1,
	 *  and find the point Px in P1,...,Pn-2 with the largest distance
	 *  from the line.
	 */

	distPx = 0;
	idxPx = 1;
	for (idx = 1; idx <= npts - 2; idx++) {
		dist = GeoUtil.lineSegmentDist(
			data[0],
			data[1],
			data[2],
			data[3],
			data[(npts - 1) * nelem + 0],
			data[(npts - 1) * nelem + 1],
			data[(npts - 1) * nelem + 2],
			data[(npts - 1) * nelem + 3],
			data[idx * nelem + 0],
			data[idx * nelem + 1],
			data[idx * nelem + 2],
			data[idx * nelem + 3]
		);
		if (dist >= distPx) {
			distPx = dist;
			idxPx = idx;
		}
	}
	//print('selected idxPx:', idxPx, '->', distPx);

	/*
	 *  If the largest distance is smaller than the tolerance, replace
	 *  the entire polyline with the line P0 -> Pn-1.  Otherwise keep
	 *  Px and simplify P0,P1,...,Px and Px,...,Pn-1 recursively.
	 */

	if (distPx < tolerance) {
		/* Replace the entire sequence with one line P0 -> Pn-1.
		 * Because the endpoint is exclusive, just return P0.
		 */
		return data.slice(0, nelem);
	} else {
		/* Keep P0, Px, and Pn-1.  Process [P0,Px[ and [Px,Pn-1[
		 * recursively.  Note that Px participates in the first
		 * sub-case as an exclusive endpoint, and in the second
		 * sub-case as an inclusive start point.
		 */
		sub1 = TrailSegmentSimplifier.douglasPeuckerRaw(data.slice(0, (idxPx + 1) * nelem), tolerance, nelem);
		sub2 = TrailSegmentSimplifier.douglasPeuckerRaw(data.slice(idxPx * nelem), tolerance, nelem);
		return sub1.concat(sub2);
	}
};

/* Simplify a trail segment.  The 'data' element of the trail segment will be
 * replaced with the simplified version.  The trail segment should be closed
 * because it's not generally a good idea to simplify a trail multiple times.
 */
TrailSegmentSimplifier.simplifyTrailSegment = function (seg, tolerance) {
	var simplified;
	var nelem = seg.ELEMS_PER_FIX;

	// Get a simplified version of seg.data, excluding the final point.
	simplified = TrailSegmentSimplifier.douglasPeuckerRaw(seg.data.slice(0), tolerance, nelem);

	// Add the final point into the result.
	if (seg.data.length >= nelem) {
		simplified = simplified.concat(seg.data.slice(seg.data.length - nelem));
	}

	// Replace segment's data points.
	seg.data = simplified;
};

/* Manual testing. */
if (false) {
	var lon1 = -1;
	var lat1 = 0;
	var alt1 = 0;

	var lon2 = 1;
	var lat2 = 0;
	var alt2 = 0;

	var lon3 = 0;
	var lat3 = 0;
	var alt3 = 0;
	
	var out = [];

	print('geo-to-ecef');
	GeoUtil.geoToEcef(lon1, lat1, alt1, out);
	print(lon1, lat1, alt1, '->', out[0], out[1], out[2]);

	print('point dist');
	print(GeoUtil.pointDist(lon1, lat1, alt1, 0, lon2, lat2, alt2, 0));

	print('line segment dist');
	print(GeoUtil.lineSegmentDist(lon1, lat1, alt1, 0, lon2, lat2, alt2, 0, lon3, lat3, alt3, 0));

	var data = [
		0, 0, 0, 0,
		1, 0, 0, 0,
		2, 0, 0, 0,
		3, 0, 0, 0,
		4, 0, 0, 0,
		5, 0, 0, 0,
		5.01, 0, 0, 0,
		5.02, 0, 0, 0,
		5.03, 0, 0, 0,
		6, 0, 0, 0,
		7, 0, 0, 0,
		7, 0, 10, 0,
		7, 0, 0, 0,
		7, 0, 0, 0,
		8, 0, 0, 0,
		9, 0, 0, 0,
	];

	print(JSON.stringify(data));
	print(JSON.stringify(TrailSegmentSimplifier.douglasPeuckerRaw(data, 1, 4)));
}
