
/**
 * One Whale Trip - HTML5 Edition
 * Ported by thp <m@thp.io>; 2013-04-25 + 2013-04-26
 * Original project: http://pyug.at/PyWeek/2012-09
 **/

var KeyCode = {
    SPACE: 32,

    LEFT: 37,
    UP: 38,
    RIGHT: 39,
    DOWN: 40,

    S: 83,
};

var Constants = {
    INITIAL_HEALTH: 9,
    MAX_HEALTH: 30,
    GRAVITY: 1.2,
    BLINKING_FRAMES: 20,

    DEPTH: 15,
    WORLD_DEPTH: 10, // fudge factor... works for DEPTH @ 15

    MAX_SPEEDUP: 4,
    SPEEDUP_STEP: 0.1,

    // keyboard repeat rate (modulo) -> higher value = less repeat
    KEYBOARD_REPEAT_MOD: 7,

    MIN_DEST_X: 0,
    MAX_DEST_X: 4,

    // This is not just enemies, but also pick-ups (for whatever reason)
    ENEMY_NAMES: [
        'coral_a',
        'coral_b',
        'coral_c',
        'coral_d',
        'pearl',
        'fishy_rainbow',
        'fishy_red',
        'fishy_deepsea',
        'diver',
        'seaweed',
        'lanternfish',
        'shell',
        'sandboxtoys',
        'oyster_0_pearl',
        'oyster_1_pearl',
        'oyster_2_pearl',
        'oyster_3_pearl',
        'rock_m',
        'rock_l',
        'rock_s',
        'jellyfish_a',
        'jellyfish_b',
        'starfish',
        'shoe',
    ],

    // level row width
    ROW_WIDTH: 5,

    // default speed in level
    DEFAULT_SPEED: 10,

    // number of animation frames for enemies and pickups
    // some objects have 2 or 3 frames
    NR_FRAMES: {
        "lanternfish": 3,
        "fishy_rainbow": 2,
        "fishy_red": 2,
        "fishy_deepsea": 2,
        "seaweed": 2,
        "diver": 3,
        "pearl": 1,
        "shell": 1,
        "sandboxtoys": 1,
        "oyster_0_pearl": 1,
        "oyster_1_pearl": 1,
        "oyster_2_pearl": 1,
        "oyster_3_pearl": 1,
        'rock_m': 1,
        'rock_l': 1,
        'rock_s': 1,
        'jellyfish_a': 2,
        'jellyfish_b': 2,
        'starfish': 1,
        'shoe': 2,
        "coral_a": 2,
        "coral_b": 1,
        "coral_c": 2,
        "coral_d": 1,
    }
};

var lamemath = {
    center: function(points) {
        // Calculate the center of a list of points
        var sum_x = 0.0;
        var sum_y = 0.0;
        for (var i=0; i<points.length; i++) {
            sum_x += points[i][0];
            sum_y += points[i][1];
        }
        return [sum_x / points.length, sum_y / points.length];
    },
};

function Scene(app) {
    this.app = app;
    this.next_state = null; // holds null or a string with classname of the place to go
};

Scene.prototype.process = function() {
    return this.next_state;
};

Scene.prototype.resume = function() {
    // Called from App when being switched to
    this.next_state = null;
};

Scene.prototype.process_input = function(event) {
    // Original game contains key handler for quit - can't quit a website here
};

Scene.prototype.draw = function() {
    // do nothing by default
};

function Intermission(app) {
    Scene.apply(this, [app]);
    this.init();
};

Intermission.prototype = new Scene();

Intermission.prototype.init = function() {
    //Scene.prototype.init.apply(this, []);
    this.creatures = null;
    this._setup();

    this.story_index = 0;
    this.update();
};

Intermission.prototype._setup = function() {
    // Define the details of this cut scene.
    this.next_scene = 'Start';

    this.background = null;//this.app.resman.get_background('i_normal')[0];

    this.title = 'ONE-WAY ALE-WHAY IP-TRAY';
    this.story = [
        //[this.app.resman.get_creature('whale_story')],
        'a-way ove-lay ory-stay'
    ];
};

Intermission.prototype.resume = function() {
    Scene.prototype.resume.apply(this, []);
    this.init();
};

Intermission.prototype.update = function() {
    var item = this.story[this.story_index];
    this.story_index += 1;

    if (typeof(item) === 'string') {
        this.line = item;
    } else {
        this.creatures = item;
        this.line = this.story[this.story_index];
        this.story_index += 1;
    }
};

Intermission.prototype.process_input = function(event) {
    var type = event[0];
    if (type === 'keydown' && event[1] == KeyCode.S) {
        this.next_state = this.next_scene;
    } else if (type == 'keydown' || type == 'mousedown') {
        if (this.story_index >= this.story.length) {
            this.next_state = this.next_scene;
        } else {
            this.update();
        }
    }
    
    Scene.prototype.process_input.apply(this, [event]);
};

Intermission.prototype.draw = function() {
    this.app.screen.draw_card(this.title, this.line,
            this.background, this.creatures);

    var count = 0;
    for (var i=0; i<this.story.length; i++) {
        if (typeof(this.story[i]) === 'string') {
            count += 1;
        }
    }

    if (count !== 1) {
        this.app.screen.draw_skip();
    }
};

function Start(app) {
    Intermission.apply(this, [app]);
};

Start.prototype = new Intermission();

Start.prototype._setup = function() {
    this.next_scene = "Intro";

    this.background = this.app.resman.get_background("i_normal")[0];

    this.title = "ONE WHALE TRIP";
    this.story = [
        [this.app.resman.get_creature("whale_story")],
        "a love story"
    ];
};

function Intro(app) {
    Intermission.apply(this, [app]);
};

Intro.prototype = new Intermission();

Intro.prototype._setup = function() {
    this.next_scene = "Game";

    this.background = this.app.resman.get_background("i_normal")[0];

    this.title = "A LOVE STORY";
    this.story = [
        [this.app.resman.get_creature("whale_story")],
        "hi there! what a nice whale you are!",

        [this.app.resman.get_creature("submarine")],
        "the moment you saw that lovely creature...",
        "... you knew it was love on first sight",

        [this.app.resman.get_creature("whale_story_heart")],
        "at first your advances were met with...",
        "cold disinterest...",
        "cold disinterest... and distance",
        "a whole ocean of distance, in fact",
        "but that made your resolve only stronger",
        "across the ocean you would follow!",

        [this.app.resman.get_creature("oyster_1_pearl")],
        "maybe a string of pearls would help?",

        [this.app.resman.get_creature("fishy_rainbow")],
        "the journey is long and hard",
        "food will sustain you",

        [this.app.resman.get_creature("diver")],
        "watch out!",
        "there are many dangers to be avoided",
    ];
};

function Enemy(app, name) {
    Sprite.apply(this, []);

    this.app = app;
    if (this.app.resman.get_sprite(name) !== undefined) {
        this.init(name, 1);
    } else {
        this.init(name + '-%d', Constants.NR_FRAMES[name]);
    }
};

Enemy.prototype = new Sprite();

Enemy.prototype.step = function() {
    this.process();
};

Enemy.prototype.draw = function(points, opacity, tint) {
    var sprite_name = this.current_sprite_name();
    var sprite = this.lookup_sprite(sprite_name);

    var w = sprite.naturalWidth;
    var h = sprite.naturalHeight;

    var x_coordinates = [];
    var y_coordinates = [];
    for (var i=0; i<points.length; i++) {
        x_coordinates.push(points[i][0]);
        y_coordinates.push(points[i][1]);
    }

    var left = Math.min.apply(null, x_coordinates);
    var right = Math.max.apply(null, x_coordinates);
    var bottom = Math.max.apply(null, y_coordinates);
    var factor = Math.min(1.0, (right - left) / w);

    var x = left + (right - left) / 2 - (w * factor) / 2;
    var y = bottom - h*factor;

    // align the enemy in the center of the polygon
    // and with the bottom (frontmost) edge of the
    // polygon aligned with the bottom of the enemy
    this.app.renderer.draw(sprite, [x, y], factor, opacity, tint);
};

function LevelItem(name, is_enemy) {
    this.name = name;
    this.is_enemy = is_enemy;
};

LevelItem.prototype.collide = function(player) {
    if (this.is_enemy) {
        player.crashed();
        return 1;
    } else if (this.name) {
        player.picked_up(this.name);
        this.name = '';
        return -1;
    }

    return 0;
};

function LevelRow(level, line) {
    this.level = level;

    // pad line to ROW_WIDTH chars
    while (line.length < Constants.ROW_WIDTH) {
        line += ' ';
    }

    this.items = [];
    for (var i=0; i<line.length; i++) {
        this.items.push(this.level.lookup(line[i]));
    }
};

var LevelSection = {
    ENEMIES: 0,
    PICKUP: 1,
    META: 2,
};

function Level(filename) {
    this.charmap = {};
    this.rows = [];
    this.speed = Constants.DEFAULT_SPEED;
    this.background = 'test';

    // FIXME: open filename here, not static string
    var data = Resources.levels[filename].split('\n');
    var section = LevelSection.ENEMIES;
    for (var i=0; i<data.length; i++) {
        var line = data[i];
        if (line.indexOf(':enemies:') !== -1) {
            section = LevelSection.ENEMIES;
        } else if (line.indexOf(':pickups:') !== -1) {
            section = LevelSection.PICKUP;
        } else if (line.indexOf(':meta:') !== -1) {
            section = LevelSection.META;
        }

        var definition = new RegExp('^# ([^=]+)=(.*)$').exec(line.trim());
        if (definition !== null) {
            var key = definition[1];
            var value = definition[2];
            if (section === LevelSection.ENEMIES || section === LevelSection.PICKUP) {
                this.add_item(key, value, section === LevelSection.ENEMIES);
            } else if (section === LevelSection.META) {
                this.set_meta(key, value);
            }
        }

        if (line.indexOf('#') === 0) {
            continue;
        }

        // XXX: In Python, we have line.rstrip('\n'), but not needed here?
        this.rows.push(new LevelRow(this, line));
    }
};

Level.prototype.exceeds_row = function(y) {
    return (y >= this.rows.length);
};

Level.prototype.add_item = function(char, name, is_enemy) {
    // XXX assert char not in this.charmap
    this.charmap[char] = [name, is_enemy];
};

Level.prototype.set_meta = function(key, value) {
    if (key == 'speed') {
        this.speed = parseInt(value);
    } else if (key == 'background') {
        this.background = value.trim();
    }
};

Level.prototype.lookup = function(char) {
    if (char == ' ') {
        return null;
    }

    var item = this.charmap[char];
    return new LevelItem(item[0], item[1]);
};

function LevelProgress(levels) {
    this.levels = levels;
    this.index = 0;
};

LevelProgress.prototype.next = function() {
    if (this.index >= this.levels.length) {
        return null;
    }
    var result = this.levels[this.index];
    this.index += 1;
    return result;
};

function Game(app) {
    Scene.apply(this, [app]);

    this.enemies = {};

    for (var i=0; i<Constants.ENEMY_NAMES.length; i++) {
        var key = Constants.ENEMY_NAMES[i];
        this.enemies[key] = new Enemy(this.app, key);
    }

    // reset everything
    this.reset(true);
};

Game.prototype = new Scene();

Game.prototype.reset = function(hard) {
    if (hard === undefined) {
        hard = false;
    }

    this.time = 0.0;
    this.i = 0;
    this.direction = 0;
    this.boost = false;
    this.speedup = 0;

    if (hard) {
        this.levels = new LevelProgress(this.app.resman.levels);
        this.level_nr = this.levels.next();
    }

    // XXX: This is ugly - sync up with Python or move to resmgr (here + in Python)
    var filename = 'levels/level-' + this.level_nr[0] + '-' + this.level_nr[1] + '.txt';
    this.level = new Level(this.app.get_filename(filename));

    this.app.player.reset(hard);
};

Game.prototype.process = function() {
    this.i += 1;

    var step = 0.01 * this.level.speed;
    if (this.boost) {
        if (this.speedup < Constants.MAX_SPEEDUP) {
            this.speedup += Constants.SPEEDUP_STEP;
        }
        if (this.speedup > Constants.MAX_SPEEDUP) {
            this.speedup = Constants.MAX_SPEEDUP;
        }
    } else {
        if (this.speedup > 0) {
            this.speedup -= Constants.SPEEDUP_STEP * 2;
        }
        if (this.speedup < 0) {
            this.speedup = 0;
        }
    }

    this.time += step * (1 + this.speedup);

    if (this.time > 1.0) {
        this.time -= 1.0;
        this.app.player.y += 1;
    }

    if (this.level.exceeds_row(this.app.player.y)) {
        // advance a level and reset
        var old_level_nr = this.level_nr;
        var new_level_nr = this.levels.next();
        if (new_level_nr !== null) {
            this.level_nr = new_level_nr;

            // If the major level number changed, show intermission card
            if (old_level_nr[0] !== this.level_nr[0]) {
                // TODO: Check if this works the same as in Python
                this.next_state = 'NextLevelGroup_' + old_level_nr[0] + '_' + old_level_nr[1];
            }
            this.reset();
        } else {
            this.next_state = "Victory";
        }

        // TODO: animate level end
    }

    if ((this.i % Constants.KEYBOARD_REPEAT_MOD) == 0) {
        var next_x = this.app.player.dest_x + this.direction;
        if (next_x >= Constants.MIN_DEST_X && next_x <= Constants.MAX_DEST_X) {
            this.app.player.dest_x += this.direction;
        }
    }

    this.app.player.step();
    for (var i=0; i<this.enemies.length; i++) {
        this.enemies[i].step();
    }

    if (this.app.player.health <= 0) {
        // reset player and game
        this.reset(true);
        this.next_state = "GameOver";
    }

    return Scene.prototype.process.apply(this, []);
};

Game.prototype.process_input = function(event) {
    var game = this;

    function go_left() {
        game.direction = -1;
        game.i = 0;
        if (game.app.player.dest_x > Constants.MIN_DEST_X) {
            game.app.player.dest_x -= 1;
        }
    }

    function go_right() {
        game.direction = 1;
        game.i = 0;
        if (game.app.player.dest_x < Constants.MAX_DEST_X) {
            game.app.player.dest_x += 1;
        }
    }

    var type = event[0];
    if (type == 'mousedown') {
        var pos = event[1];
        var x = (pos[0] - this.app.screen.offset[0]) / this.app.screen.scale;
        var y = (pos[1] - this.app.screen.offset[1]) / this.app.screen.scale;

        if (y < this.app.screen.height / 4) {
            this.boost = true;
        } else if (y > this.app.screen.height * 3 / 4) {
            this.app.player.jump();
        } else if (x < this.app.screen.width / 3) {
            go_left();
        } else if (x > this.app.screen.width * 2 / 3) {
            go_right();
        }
    } else if (type == 'mouseup') {
        this.direction = 0;
        this.boost = false;
    } else if (type == 'keydown') {
        var key = event[1];
        if (key == KeyCode.SPACE) {
            this.app.player.jump();
        } else if (key == KeyCode.LEFT) {
            go_left();
        } else if (key == KeyCode.RIGHT) {
            go_right();
        } else if (key == KeyCode.UP) {
            this.boost = true;
        }
    } else if (type == 'keyup') {
        var key = event[1];
        if (key == KeyCode.LEFT) {
            this.direction = 0;
        } else if (key == KeyCode.RIGHT) {
            this.direction = 0;
        } else if (key == KeyCode.UP) {
            this.boost = false;
        }
    }

    Scene.prototype.process_input.apply(this, [event]);
};

Game.prototype.map_coords = function(lane, jump, distance) {
    /*
     * Translate game coordinates to world coordinates.
     *
     * lane:     0..5
     * jump:     not defined yet
     * distance: 0..DEPTH+1 (number of rows)
     *
     * world coordinates: 1x1xWORLD_DEPTH
     */

    var x = (lane + 0.5) / 5.0;
    var y = (500.0 - jump) / 500.0;
    var z = Constants.WORLD_DEPTH * (distance + 0.5) / Constants.DEPTH;

    return [x, y, z];
};

Game.prototype.draw = function() {
    var backgrounds = this.app.resman.get_background(this.level.background);
    var pos = parseInt(this.time + this.app.player.y) % backgrounds.length;
    this.app.renderer.draw(backgrounds[pos], [0, 0]);

    var x = this.app.player.x;
    var y = this.time;
    var player_points = this.mkpoints(x, y, this.app.player.height);

    // draw queue for back-to-front drawing of enemies
    var draw_queue = [[y, this.app.player, player_points]];

    var view_range = [];
    for (var i=this.app.player.y; i<this.app.player.y+Constants.DEPTH; i++) {
        view_range.push(i);
    }

    for (var yidx=0; yidx<view_range.length; yidx++) {
        var offset = view_range[yidx];
        if (offset >= this.level.rows.length) {
            continue;
        }

        for (var xidx=0; xidx<this.level.rows[offset].items.length; xidx++) {
            var column = this.level.rows[offset].items[xidx];
            if (column === null) {
                continue;
            }

            x = xidx;
            y = yidx;

            if (yidx == 1 && xidx == this.app.player.dest_x && this.app.player.height < 10) {
                var c = column.collide(this.app.player);
                if (c > 0) {
                    // do something when the player collides
                    if ((this.app.player.health % 3) == 0) {
                        // lost a life
                        // reset to beginning of current level
                        this.reset();
                        this.next_state = "LostLife";
                    }
                }
            }

            var points = this.mkpoints(x, y);
            // XXX: Check if the following line works ("in")
            if (column.name in this.enemies) {
                var enemy = this.enemies[column.name];
                draw_queue.push([y, enemy, points]);
            } else if (column.name) {
                console.log('Missing graphic: ' + column.name);
            }
        }
    }

    var tint = [1.0, 1.0, 1.0];
    if (this.level.background == 'surreal') {
        // Special FX for the Surreal level - tint like crazy!
        tint = [
            0.5 + 0.5 * Math.sin(this.i * 0.009),
            0.5 + 0.5 * Math.sin(0.9 + this.i * 0.004),
            0.5 + 0.5 * Math.sin(5.6 + this.i * 0.2)
        ];
    }

    // Draw all enemies (+player), back-to-front for proper stacking order
    draw_queue.sort(); // XXX: This doesn't seem to always work correctly
    draw_queue.reverse();
    for (var i=0; i<draw_queue.length; i++) {
        var item = draw_queue[i];
        var y = item[0];
        var sprite = item[1];
        var points = item[2];

        this.app.screen.draw_sprite(y-this.time, sprite, points, tint);
    }

    this.app.renderer.begin_overlay();
    this.app.screen.draw_stats(this.app.player.coins_collected,
                               this.app.player.health);
};

Game.prototype.mkpoints = function(x, y, height) {
    if (height === undefined) {
        height = 0.0;
    }
    return [
        this.map_coords(x-0.45, height, y-0.45 - this.time),
        this.map_coords(x+0.45, height, y-0.45 - this.time),
        this.map_coords(x+0.45, height, y+0.45 - this.time),
        this.map_coords(x-0.45, height, y+0.45 - this.time),
    ];
};


function NextLevelGroup_1_3(app) {
    Intermission.apply(this, [app]);
};

NextLevelGroup_1_3.prototype = new Intermission();

NextLevelGroup_1_3.prototype._setup = function() {
    this.next_scene = "Game";

    this.background = this.app.resman.get_background("i_beach")[0];

    this.title = "THE BEACH";
    this.story = [
        [this.app.resman.get_creature("jellyfish_a"),
         this.app.resman.get_creature("jellyfish_b")],
        "the first hurdle is cleared...",
        "...but beware of those hairless apes",
        "ruining everything with their trash"
    ];
}

function NextLevelGroup_2_3(app) {
    Intermission.apply(this, [app]);
};

NextLevelGroup_2_3.prototype = new Intermission();

NextLevelGroup_2_3.prototype._setup = function() {
    this.next_scene = "Game";

    this.background = this.app.resman.get_background("i_coralreef")[0];

    this.title = "THE CORAL REEF";
    this.story = [
        [this.app.resman.get_creature("diver")],
        "everything is so colorful down here",
        "you wish you could share it with someone"
    ];
};

function NextLevelGroup_3_1(app) {
    Intermission.apply(this, [app]);
};

NextLevelGroup_3_1.prototype = new Intermission();

NextLevelGroup_3_1.prototype._setup = function() {
    this.next_scene = "Game";

    this.background = this.app.resman.get_background("i_deepsea")[0];

    this.title = "THE DEEP SEA";
    this.story = [
        [this.app.resman.get_creature("lanternfish")],
        "it's getting dark and lonely",
        "you wish you had someone to hold"
    ];
};

function NextLevelGroup_4_1(app) {
    Intermission.apply(this, [app]);
};

NextLevelGroup_4_1.prototype = new Intermission();

NextLevelGroup_4_1.prototype._setup = function() {
    this.next_scene = "Game";

    this.background = this.app.resman.get_background("i_cliff")[0];

    this.title = "THE CLIFFS";
    this.story = [
        [this.app.resman.get_creature("rock_l")],
        "closer, ever closer...",
        "you can feel your destination approaching"
    ];
};

function NextLevelGroup_5_1(app) {
    Intermission.apply(this, [app]);
};

NextLevelGroup_5_1.prototype = new Intermission();

NextLevelGroup_5_1.prototype._setup = function() {
    this.next_scene = "Game";

    this.background = this.app.resman.get_background("i_surreal")[0];

    this.title = "WHAT A TRIP!";
    this.story = [
        [this.app.resman.get_creature("submarine")],
        "almost there"
    ];
};

function LostLife(app) {
    Intermission.apply(this, [app]);
};

LostLife.prototype = new Intermission();

LostLife.prototype._setup = function() {
    this.next_scene = "Game";

    this.background = this.app.resman.get_background("i_deepsea")[0];

    this.title = "YOU LOST A LIFE";
    this.story = [
        [this.app.resman.get_creature("lost_life_whale")],
        "be careful! you have only " + parseInt(this.app.player.health/3) + " left",
    ];
};

function GameOver(app) {
    Intermission.apply(this, [app]);
};

GameOver.prototype = new Intermission();

GameOver.prototype._setup = function() {
    this.next_scene = "Start";

    this.background = this.app.resman.get_background("i_deepsea")[0];

    this.title = "GAME OVER";
    this.story = [
        [this.app.resman.get_creature("game_over_whale")],
        "it wasn't meant to be"
    ];
};

function Victory(app) {
    Intermission.apply(this, [app]);
};

Victory.prototype = new Intermission();

Victory.prototype._setup = function() {
    this.next_scene = "Outro";

    this.background = this.app.resman.get_background("i_surreal")[0];

    this.title = "VICTORY";
    this.story = [
        [this.app.resman.get_creature("whale_story_heart_mirror"),
         this.app.resman.get_creature("victory_submarine")],
        "a happy ending"
    ];
};

function Outro(app) {
    Intermission.apply(this, [app]);
};

Outro.prototype = new Intermission();

Outro.prototype._setup = function() {
    this.next_scene = "GoodBye";

    this.background = this.app.resman.get_background("i_outro")[0];

    this.title = "GOOD BYE!";
    this.story = [
        "the team:",
        "gfx: lobbbe",
        "code: thp, styts, hop",
        "sfx: styts, thp",
        "html5 port: thp",
    ];
};

function CanvasRenderer(app) {
    this.app = app;
    this.global_offset_x = 0;
    this.global_offset_y = 0;
    // The global tint is ignored in this renderer
    this.global_tint = [1.0, 1.0, 1.0];
    this.context = null;
};

CanvasRenderer.prototype.setup = function(size) {
    // Nothing to be done here
};

CanvasRenderer.prototype.register_sprite = function(name, sprite) {
    return sprite;
};

CanvasRenderer.prototype.begin = function() {
    if (this.context === null) {
        this.context = this.app.screen.display.getContext('2d');
    }
};

CanvasRenderer.prototype.begin_overlay = function() {
    // Nothing to do
};

CanvasRenderer.prototype.draw_text = function(text, pos, font, color) {
    this.context.font = font;
    this.context.fillStyle = color;
    this.context.textBaseline = "top";
    this.context.fillText(text, pos[0], pos[1]);
};

CanvasRenderer.prototype.draw = function(sprite, pos, scale, opacity, tint) {
    // Opacity is ignored in this blitting renderer
    // Tint is also ingored in this blitting renderer
    var x = pos[0];
    var y = pos[1];
    x += this.global_offset_x;
    y += this.global_offset_y;
    if (scale !== undefined && scale !== null) {
        var w = sprite.naturalWidth * scale;
        var h = sprite.naturalHeight * scale;
        this.context.drawImage(sprite, x, y, w, h);
    } else {
        this.context.drawImage(sprite, x, y);
    }
};

CanvasRenderer.prototype.finish = function() {
    // Nothing to do - display will flip automatically
};

function SpriteProxy(gl, sprite) {
    this.gl = gl;
    this._sprite = sprite;
    this._texcoords = null;
    this._texture_id = null;

    var proxy = this;
    sprite.addEventListener('load', function(e) {
        proxy._texture_id = gl.createTexture();
        gl.bindTexture(gl.TEXTURE_2D, proxy._texture_id);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
        proxy._load_from(sprite);

        // Forward normal attributes (XXX Kludge, we use __getattr_ in Python)
        proxy.naturalWidth = sprite.naturalWidth;
        proxy.naturalHeight = sprite.naturalHeight;
    });
};

SpriteProxy.prototype._load_from = function(sprite) {
    var w0 = sprite.naturalWidth;
    var h0 = sprite.naturalHeight;

    // XXX: Power-of-2-textures (texImage / texSubImage)
    var w = 1;
    while (w < w0) w *= 2;
    var h = 1;
    while (h < h0) h *= 2;
    
    var wf = w0 / w;
    var hf = h0 / h;

    // Account for the different texture size by making the
    // texture coordinates use only part of the image
    this._texcoords = [
        0, 0,
        0, hf,
        wf, 0,
        wf, hf,
    ];

    var gl = this.gl;
    gl.bindTexture(gl.TEXTURE_2D, this._texture_id);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, w, h, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);
    gl.texSubImage2D(gl.TEXTURE_2D, 0, 0, 0, gl.RGBA, gl.UNSIGNED_BYTE, sprite);
    gl.bindTexture(gl.TEXTURE_2D, null);
};

function Framebuffer(gl, width, height) {
    this.gl = gl;
    this.started = new Date().getTime() / 1000;
    this.width = width;
    this.height = height;
    this.texture_id = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, this.texture_id);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);

    var w = 1;
    while (w < width) w *= 2;
    var h = 1;
    while (h < height) h *= 2;

    this.wf = width / w;
    this.hf = height / h;

    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, w, h, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);
    gl.bindTexture(gl.TEXTURE_2D, null);

    this.framebuffer_id = gl.createFramebuffer();
    this.bind();
    gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0,
            gl.TEXTURE_2D, this.texture_id, 0);
    this.unbind();
};

Framebuffer.prototype.bind = function() {
    var gl = this.gl;
    gl.bindFramebuffer(gl.FRAMEBUFFER, this.framebuffer_id);
};

Framebuffer.prototype.unbind = function() {
    var gl = this.gl;
    gl.bindFramebuffer(gl.FRAMEBUFFER, null);
};

Framebuffer.prototype.rerender = function(effect) {
    // render this.texture_id as full screen quad
    var vertices = [
        // Vertex coordinates
        -1, -1, 0,
        -1, 1, 0,
        1, -1, 0,
        1, 1, 0,

        // Texture coordinates
        0, 0,
        0, this.hf,
        this.wf, 0,
        this.wf, this.hf,
    ];

    var gl = this.gl;

    gl.bindTexture(gl.TEXTURE_2D, this.texture_id);

    effect.use();

    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(vertices), gl.STATIC_DRAW);

    var pos = effect.attrib('position');
    gl.enableVertexAttribArray(pos);
    gl.vertexAttribPointer(pos, 3, gl.FLOAT, gl.FALSE, 0, 0);

    var tex = effect.attrib('texcoord');
    gl.enableVertexAttribArray(tex);
    /* 4 vertices * 3 coordinates * 4 (sizeof float) */
    gl.vertexAttribPointer(tex, 2, gl.FLOAT, gl.FALSE, 0, 4*3*4);

    var dim = effect.uniform('dimensions');
    gl.uniform2f(dim, this.width, this.height);

    var tim = effect.uniform('time');
    var now = new Date().getTime() / 1000;
    gl.uniform1f(tim, now - this.started);

    gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);

    gl.disableVertexAttribArray(pos);
    gl.disableVertexAttribArray(tex);
    // XXX gl.useProgram(null);
};

function build_shader(gl, typ, source) {
    var shader_id = gl.createShader(typ);
    gl.shaderSource(shader_id, source);
    gl.compileShader(shader_id);
    // check_shader_status(shader_id, source)
    return shader_id;
};

function ShaderEffect(gl, vertex_shader, fragment_shader) {
    this.gl = gl;

    var is_gles = true; // WebGL = OpenGL ES 2.0
    if (is_gles) {
        vertex_shader = "precision mediump float;\n" + vertex_shader;
        fragment_shader = "precision mediump float;\n" + fragment_shader;
    }

    this.vertex_shader = build_shader(gl, gl.VERTEX_SHADER, vertex_shader);
    this.fragment_shader = build_shader(gl, gl.FRAGMENT_SHADER, fragment_shader);
    this.program = gl.createProgram();
    gl.attachShader(this.program, this.vertex_shader);
    gl.attachShader(this.program, this.fragment_shader);
    gl.linkProgram(this.program);
    //check_program_status(this.program);
};

ShaderEffect.prototype.use = function() {
    this.gl.useProgram(this.program);
};

ShaderEffect.prototype.attrib = function(name) {
    return this.gl.getAttribLocation(this.program, name);
};

ShaderEffect.prototype.uniform = function(name) {
    return this.gl.getUniformLocation(this.program, name);
};


function WebGLRenderer(app) {
    this.app = app;

    this.tmp_sprite = null;
    this.framebuffer = null;
    this.framebuffer2 = null;
    this.effect_pipeline = [];
    this.postprocessed = false;
    this.global_offset_x = 0;
    this.global_offset_y = 0;
    this.global_tint = [1.0, 1.0, 1.0];

    this.vbo = null;

    this.gl = null;
};

WebGLRenderer.prototype.setup = function(size) {
    this.gl = this.app.screen.display.getContext('experimental-webgl');

    var gl = this.gl;
    gl.clearColor(0.0, 0.0, 0.0, 1.0);

    var width = size[0];
    var height = size[1];
    var offset_x = this.app.screen.offset[0];
    var offset_y = this.app.screen.offset[1];
    var scale = this.app.screen.scale;

    gl.viewport(0, 0, width, height);

    this.framebuffer = new Framebuffer(this.gl, width, height);
    this.framebuffer2 = new Framebuffer(this.gl, width, height);

    this.vbo = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, this.vbo);

    this.draw_sprites = new ShaderEffect(this.gl, Shaders.draw_sprites_vertex_shader,
            Shaders.draw_sprites_fragment_shader);

    this.sepia_effect = new ShaderEffect(this.gl, Shaders.effect_vertex_shader,
            Shaders.sepia_effect_fragment_shader);

    this.blur_effect = new ShaderEffect(this.gl, Shaders.effect_vertex_shader,
            Shaders.blur_effect_fragment_shader);

    this.underwater_effect = new ShaderEffect(this.gl, Shaders.effect_vertex_shader,
            Shaders.underwater_effect_fragment_shader);

    this.gles_combined_effect = new ShaderEffect(this.gl, Shaders.effect_vertex_shader,
            Shaders.gles_combined_effect_fragment_shader);

    // TODO: Use gles_combined_effect on mobile platforms for better performance
    //this.effect_pipeline = [this.gles_combined_effect];
    //this.effect_pipeline = [this.blur_effect, this.underwater_effect];
    this.effect_pipeline = [this.underwater_effect];

    // Configure constant uniforms of draw_sprites
    this.draw_sprites.use();
    var size_loc = this.draw_sprites.uniform('size'); // vec2
    var offset_loc = this.draw_sprites.uniform('offset'); // vec2
    var scale_loc = this.draw_sprites.uniform('scale'); // float
    gl.uniform2f(size_loc, width, height);
    gl.uniform2f(offset_loc, offset_x, offset_y);
    gl.uniform1f(scale_loc, scale);
    // XXX gl.useProgram(null);
    
    gl.enable(gl.BLEND);
    gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
};

WebGLRenderer.prototype.register_sprite = function(name, sprite) {
    // Upload the sprite as a texture
    return new SpriteProxy(this.gl, sprite);
};

WebGLRenderer.prototype.begin = function() {
    if (this.effect_pipeline.length !== 0) {
        this.postprocessed = false;
        this.framebuffer.bind();
    }
    var gl = this.gl;
    gl.clear(gl.COLOR_BUFFER_BIT);
};

WebGLRenderer.prototype.draw_text = function(text, pos, font, color) {
    var canvas = document.createElement('canvas');
    var ctx = canvas.getContext('2d');

    ctx.font = font;
    var textWidth = ctx.measureText(text).width;
    var textHeight = 30 / 2; // XXX measure the height properly

    var w = 1;
    while (w < textWidth) w *= 2;
    var h = 1;
    while (h < textHeight) h *= 2;

    canvas.width = w;
    canvas.height = h;

    ctx.font = font;
    ctx.fillStyle = color;
    ctx.textBaseline = "top";
    ctx.fillText(text, 0, 0);
    ctx = null;

    var gl = this.gl;

    var texture = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, texture);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, canvas);
    gl.bindTexture(gl.TEXTURE_2D, null);

    // This resembles SpriteProxy - at least all things needed in this.draw()
    var sprite = {
        naturalWidth: textWidth, //canvas.width,
        naturalHeight: textHeight, //canvas.height,
        _texture_id: texture,
        _texcoords: [
            0, 0,
            0, textHeight / canvas.height,
            textWidth / canvas.width, 0,
            textWidth / canvas.width, textHeight / canvas.height,
        ]
    };

    this.draw(sprite, pos, 1.0, 1.0);
};

WebGLRenderer.prototype.draw = function(sprite, pos, scale, opacity, tint) {
    if (scale === undefined) {
        scale = 1.0;
    }
    if (opacity === undefined) {
        opacity = 1.0;
    }
    if (tint === undefined) {
        tint = [1.0, 1.0, 1.0];
    }

    var gl = this.gl;

    if (sprite._texture_id == null) {
        // sprite texture not yet loaded
        return;
    }

    this.draw_sprites.use();
    var w = sprite.naturalWidth;
    var h = sprite.naturalHeight;
    var x = pos[0];
    var y = pos[1];
    x += this.global_offset_x;
    y += this.global_offset_y;

    var r = tint[0];
    var g = tint[1];
    var b = tint[2];

    var gr = this.global_tint[0];
    var gg = this.global_tint[1];
    var gb = this.global_tint[2];

    if (true /* XXX is_gles */) {
        // Apply tinting here for performance reasons
        // XXX only useful with this.effect_pipeline == [this.gles_combined_effect]
        gr *= 0.7;
        gg *= 0.9;
    }

    var color_loc = this.draw_sprites.uniform('color'); // vec4
    gl.uniform4f(color_loc, r*gr, g*gg, b*gb, opacity);

    var vertices = [
        x, y, 0.0,
        x, y+h*scale, 0.0,
        x+w*scale, y, 0.0,
        x+w*scale, y+h*scale, 0.0,
    ];
    Array.prototype.push.apply(vertices, sprite._texcoords)
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(vertices), gl.STATIC_DRAW);

    gl.bindTexture(gl.TEXTURE_2D, sprite._texture_id);

    var position_loc = this.draw_sprites.attrib('position');
    gl.enableVertexAttribArray(position_loc);
    gl.vertexAttribPointer(position_loc, 3, gl.FLOAT, gl.FALSE, 0, 0);

    var texcoord = sprite._texcoords;
    var texcoord_loc = this.draw_sprites.attrib('texcoord');
    gl.enableVertexAttribArray(texcoord_loc);
    /* 4 vertices * 3 coordinates * 4 (sizeof float) */
    gl.vertexAttribPointer(texcoord_loc, 2, gl.FLOAT, gl.FALSE, 0, 4*3*4);

    gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);

    gl.disableVertexAttribArray(texcoord_loc);
    gl.disableVertexAttribArray(position_loc);

    // XXX gl.useProgram(null);


};

WebGLRenderer.prototype.begin_overlay = function() {
    // Force postprocessing NOW, so overlays will be drawn as-is
    this.postprocess();
};

WebGLRenderer.prototype.postprocess = function() {
    if (this.effect_pipeline.length === 0) {
        return;
    }

    var gl = this.gl;

    this.framebuffer.unbind();

    // Apply effect by drawing between framebuffers and
    // finally rendering the last effect to the screen
    var a = this.framebuffer;
    var b = this.framebuffer2;

    for (var idx=0; idx<this.effect_pipeline.length; idx++) {
        var effect = this.effect_pipeline[idx];
        if (idx < (this.effect_pipeline.length - 1)) {
            b.bind();
        }
        gl.clear(gl.COLOR_BUFFER_BIT);
        a.rerender(effect);
        if (idx < (this.effect_pipeline.length - 1)) {
            b.unbind();
        }
        var tmp = a;
        a = b;
        b = tmp;
    }

    this.postprocessed = true;
};

WebGLRenderer.prototype.finish = function() {
    if (this.effect_pipeline.length !== 0 && !this.postprocessed) {
        this.postprocess();
    }

    // TODO
};

function TimeAccumulator(fps) {
    this.fps = fps;
    this.step = 1.0 / this.fps;
    this.accumulated = 0;
    this.last_time = new Date().getTime() / 1000;
};

TimeAccumulator.prototype.update = function(callback) {
    var result = null;
    var now = new Date().getTime() / 1000;
    this.accumulated += (now - this.last_time);
    this.last_time = now;

    while (this.accumulated > this.step) {
        this.accumulated -= this.step;
        result = callback();
    }

    return result;
};

function Screen(app, title, width, height, fullscreen) {
    this.app = app;

    // flags, fullscreen, renderer

    this.width = 480;//width;
    this.height = 320;//height;

    this.loading = document.createElement('h1');
    this.loading.id = 'loader';
    this.loading.style.color = 'white';
    document.body.appendChild(this.loading);

    this.display = document.createElement('canvas');
    this.display.style.visibility = 'hidden'; // will be set be resman
    this.display.style.position = 'absolute';
    this.display.style.left = '50%';
    this.display.style.top = '50%';
    this.display.style.marginLeft = '-'+(width/2)+'px';
    this.display.style.marginTop = '-'+(height/2)+'px';
    this.display.width = width;
    this.display.height = height;
    this.display.style.border = '0px';
    this.display.addEventListener('mousedown', function(e) {
        app.post_input_event('mousedown', e);
    }, false);
    this.display.addEventListener('mouseup', function(e) {
        app.post_input_event('mouseup', e);
    }, false);
    document.addEventListener('keydown', function(e) {
        app.post_input_event('keydown', e);
    }, false);
    document.addEventListener('keyup', function(e) {
        app.post_input_event('keyup', e);
    }, false);
    document.body.appendChild(this.display);

    // Real size
    this.size = [width, height];

    // Scale to fill the screen, with letterboxing
    this.scale = Math.min(this.size[0] / this.width,
                          this.size[1] / this.height);

    // Calculate the offset for symmetric letterboxing
    this.offset = [(this.size[0] - this.width*this.scale) / 2,
                   (this.size[1] - this.height*this.scale) / 2];

    document.title = title;

    this.zeye = this.width * 1.2; // Assumed distance from screen
    this.xeye = this.width * 0.5; // Middle of the screen
    this.yeye = this.height * 0.33; // High horizon
};

Screen.prototype.projection = function(x, y, z) {
    // Project world coordinates onto the screen.
    // The world's dimentions are 1x1xWORLD_DEPTH

    x = x * this.width;
    y = y * this.height;
    z = z * this.width;

    // projection
    var xs = (this.zeye * (x - this.xeye)) / (this.zeye + z) + this.xeye;
    var ys = (this.zeye * (y - this.yeye)) / (this.zeye + z) + this.yeye;

    return [xs, ys];
};

Screen.prototype.draw_debug = function() {
    var font = this.app.resman.font(FONT_SMALL);
    // TODO: Render FPS
};

Screen.prototype.draw_sprite = function(y, sprite, points, tint) {
    // Project a sprite onto the screen.
    // Coordinates are given in world coordinates.

    var projected = [];
    for (var i=0; i<points.length; i++) {
        projected.push(this.projection(points[i][0], points[i][1], points[i][2]));
    }

    // Fade in enemy sprites coming from the back
    var opacity = 1.0;
    if (y > 10) {
        opacity = 1.0 - (y-10)/5;
        tint = [tint[0]*opacity, tint[1]*opacity, tint[2]*opacity];
    }

    sprite.draw(projected, opacity, tint);
};

Screen.prototype.draw_stats = function(bonus, health) {
    // Draw bonus and health bar.
    var font = this.app.resman.font(FONT_STD);
    var offset = 10;

    // bonus
    var pos_x = offset;
    var pos_y = offset;
    var icon = this.app.resman.get_sprite('pearlcount_icon-1');
    this.app.renderer.draw(icon, [offset, offset]);

    pos_x += icon.naturalWidth + offset;
    this.app.renderer.draw_text(''+bonus, [pos_x, pos_y-5 /* XXX -10 to fix offset */], font, 'yellow');

    // health
    pos_x = this.width;
    pos_y = offset;

    while (health > 0) {
        var rest = Math.min(health, 3);
        health -= 3;
        var sprite = this.app.resman.get_sprite('whale_ico-' + rest);
        var icon_width = sprite.naturalWidth;
        pos_x -= icon_width + offset;
        this.app.renderer.draw(sprite, [pos_x, pos_y]);
    }
};

Screen.prototype.draw_card = function(message, story, background, creatures) {
    this.app.renderer.draw(background, [0, 0]);

    var font = this.app.resman.font(FONT_STD);
    var color = 'white';

    // main message
    var pos_x = this.width / 15;
    this.app.renderer.draw_text(message, [pos_x, this.height/2 + 50], font, color);

    // additional message
    if (story) {
        this.app.renderer.draw_text(story, [pos_x, this.height/2 + 100], font, color);
    }

    if (creatures) {
        var width = 0;
        for (var i=0; i<creatures.length; i++) {
            width += creatures[i].naturalWidth;
            width += 20;
        }

        var pos_x = 3*this.width/4 - width/2;
        var pos_x = Math.min(pos_x, this.width - width);
        for (var i=0; i<creatures.length; i++) {
            var creature = creatures[i];
            var pos_y = this.height/3 - creature.naturalHeight/2;
            this.app.renderer.draw(creature, [pos_x, pos_y]);
            pos_x += creature.naturalWidth + 20;
        }
    }
};

Screen.prototype.draw_skip = function() {
    var font = this.app.resman.font(FONT_SMALL);
    // FIXME: have to get text width and text height
    var textWidth = 175;
    var textHeight = 25;
    var pos = [
        this.width - textWidth - 10,
        this.height - textHeight - 10,
    ];
    //this.app.renderer.draw_text('[S] ... SKIP INTRO', pos, font, 'white');
};

var FONT_STD = ['visitor2', 38/2];
var FONT_SMALL = ['visitor2', 25/2];

function ResourceManager(app) {
    this.app = app;
    this._surfaces = {};
    this._backgrounds = {};
    this._creatures = {};
    this._sounds = {};
    this._fonts = {};
    this.total = 0;
    this.loaded = 0;
    this._load_all();
};

ResourceManager.prototype.updated = function() {
    if (this.loaded == this.total) {
        this.app.screen.display.style.visibility = 'visible';
        document.body.removeChild(this.app.screen.loading);
    } else {
        var msg = 'Loading: ' + parseInt(this.loaded / this.total * 100) + '%';
        this.app.screen.loading.innerHTML = msg;
    }
};

ResourceManager.prototype._load_all = function() {
    var resman = this;
    function loaded() {
        resman.loaded += 1;
        resman.updated();
    }
    function needsLoading(item) {
        resman.total += 1;
        item.addEventListener('load', loaded);
        return item;
    }

    // load sprites
    var sprites = Resources.sprites;
    for (var i=0; i<sprites.length; i++) {
        var fn = sprites[i];
        var bn = fn.substring(fn.lastIndexOf('/')+1, fn.lastIndexOf('.'));
        var surf = needsLoading(new Image());
        surf.src = fn;
        surf = this.app.renderer.register_sprite(bn, surf);
        this._surfaces[bn] = surf;
    }

    // load backgrounds
    var backgrounds = Resources.backgrounds;
    for (var i=0; i<backgrounds.length; i++) {
        var fn = backgrounds[i];
        var bn = fn.substring(fn.lastIndexOf('/')+1, fn.lastIndexOf('.'));
        var bn_parts = bn.split('-');
        var key = bn_parts[0];
        var frame = bn_parts[1];

        var surf = needsLoading(new Image());
        surf.src = fn;
        surf = this.app.renderer.register_sprite(bn, surf);
        if (key in this._backgrounds) {
            this._backgrounds[key].push(surf);
        } else {
            this._backgrounds[key] = [surf];
        }
    }

    // load intermission assets
    var creatures = Resources.creatures;
    for (var i=0; i<creatures.length; i++) {
        var fn = creatures[i];
        var bn = fn.substring(fn.lastIndexOf('/')+1, fn.lastIndexOf('.'));
        var surf = needsLoading(new Image());
        surf.src = fn;
        surf = this.app.renderer.register_sprite(bn, surf);
        this._creatures[bn] = surf;
    }

    // TODO: load sfx (wav and optionally ogg files if they exist)
    var sounds = Resources.sounds;
    for (var i=0; i<sounds.length; i++) {
        var fn = sounds[i];
        var bn = fn.substring(fn.lastIndexOf('/')+1, fn.lastIndexOf('.'));
        var sfx = new Audio();
        sfx.src = fn;
        this._sounds[bn] = sfx;
    }

    // No need to load fonts - will be loaded via CSS (or generate style elements?

    // load levels
    // XXX: hardcoded list of levels - should read from filesystem?
    this.levels = [
        [1, 1], [1, 2], [1, 3],
        [2, 1], [2, 2], [2, 3],
        [3, 1],
        [4, 1],
        [5, 1],
        [6, 1]
    ];
};

ResourceManager.prototype._path = function(filename) {
    return 'data/' + filename;
};

ResourceManager.prototype.get_sprite = function(name) {
    return this._surfaces[name];
};

ResourceManager.prototype.get_background = function(name) {
    return this._backgrounds[name];
};

ResourceManager.prototype.get_creature = function(name) {
    return this._creatures[name];
};

ResourceManager.prototype.get_sound = function(name) {
    return this._sounds[name];
};

ResourceManager.prototype.font = function(font_spec) {
    return 'normal ' + font_spec[1] + 'px ' + font_spec[0];
};

function AudioManager(app) {
    this.app = app;
};

AudioManager.prototype.sfx = function(name) {
    var sound = this.app.resman.get_sound(name);
    try {
        sound.currentTime = 0;
        sound.play();
    } catch (err) {
        console.log('Error: ' + err);
    }
};

function Sprite() {
};

Sprite.prototype.init = function(format_str, frames, duration) {
    if (duration === undefined) {
        duration = 0.2;
    }

    this.duration = duration;
    if (this.app.resman.get_sprite(format_str) !== undefined) {
        this.sprites = [format_str];
    } else {
        this.sprites = [];
        var sequence = this.make_sequence(frames);
        for (var i=0; i<sequence.length; i++) {
            this.sprites.push(format_str.replace('%d', ''+sequence[i]));
        }
    }
    this.frames_per_sprite = parseInt(duration * this.app.fps);
    this.current_sprite = 0;
    this.current_frame = 0;
};

Sprite.prototype.make_sequence = function(frames) {
    // Make a loopable sequence of frames
    // make_sequence(3) -> [1, 2, 3, 2]
    var result = [];
    for (var i=1; i<=frames; i++) {
        result.push(i);
    }
    for (var i=frames-1; i>1; i--) {
        result.push(i);
    }
    return result;
};

Sprite.prototype.process = function() {
    // Gets called every frame.
    // Updates current sprite image.
    this.current_frame += 1;
    if (this.current_frame >= this.frames_per_sprite) {
        this.current_sprite = (this.current_sprite + 1) % this.sprites.length;
        this.current_frame = 0;
    }
};

Sprite.prototype.current_sprite_name = function() {
    return this.sprites[this.current_sprite];
};

Sprite.prototype.lookup_sprite = function(name) {
    return this.app.resman.get_sprite(name);
};

function Player(app) {
    Sprite.apply(this, []);
    this.app = app;

    this.init('whale_%d', 3);
    this.reset();
};

Player.prototype = new Sprite();

Player.prototype.reset = function(hard) {
    if (hard === undefined) {
        hard = false;
    }

    this.x = 2;
    this.y = 0;
    this.dest_x = 2;
    this.height = 0;
    this.vertical_velocity = 0;
    this.can_jump = true;
    this.blinking = 0;

    if (hard) {
        this.coins_collected = 0;
        this.health = Constants.INITIAL_HEALTH;
        this.max_health = Constants.MAX_HEALTH;
    }
};

Player.prototype.jump = function() {
    if (this.can_jump) {
        this.vertical_velocity = 15;
        this.can_jump = false;
        this.app.audman.sfx("jump");
    }
};

Player.prototype.picked_up = function(thingie) {
    if (thingie == 'pearl') {
        this.coins_collected += 1;
        this.app.audman.sfx('pearl' + parseInt(this.dest_x+1));
    } else if (thingie == 'oyster_1_pearl') {
        this.coins_collected += 1;
        this.app.audman.sfx('pearl');
    } else if (thingie == 'oyster_2_pearl') {
        this.coins_collected += 2;
        this.app.audman.sfx('pearl');
    } else if (thingie == 'oyster_3_pearl') {
        this.coins_collected += 3;
        this.app.audman.sfx('pearl');
    } else if (thingie.indexOf('fishy') === 0) {
        // [11:47pm] lobbbe_: what about that: you get your health back to full
        // if you eat a fishy, and if your health is already full - and only
        // then - you get an extra life?
        this.app.audman.sfx('fishy');
        var lives = parseInt(this.health/3) + 1;
        this.health = Math.min(lives*3, Constants.MAX_HEALTH);
    }
};

Player.prototype.crashed = function() {
    if (!this.blinking) {
        this.health -= 1;
        this.app.audman.sfx('crash');
        this.blinking = Constants.BLINKING_FRAMES;
    }
};

Player.prototype.step = function() {
    this.x = this.x * 0.5 + this.dest_x * 0.5;
    this.height += this.vertical_velocity;
    if (this.height < 0) {
        this.can_jump = true;
        this.height = 0;
        this.vertical_velocity *= -0.5;
        if (Math.abs(this.vertical_velocity) < 2) {
            this.vertical_velocity = 0;
            this.height = 0;
        }
    }
    this.vertical_velocity -= Constants.GRAVITY;
    if (this.blinking) {
        this.blinking -= 1;
    }
    this.process();
};

Player.prototype.draw = function(points, opacity, tint) {
    var xoffset = 0.0;
    var yoffset = 0.0;
    var opacity = 1.0;
    var tint = [1.0, 1.0, 1.0];

    if (this.blinking) {
        var value = 1.0 - Math.abs(Math.sin(this.blinking * 0.2));
        tint = [1.0, value, value];
        yoffset = Math.sin(this.blinking * 0.5) * 5.0;
    }

    var sprite_name = this.current_sprite_name();
    
    // TODO: check if the below expressions works the same as in Python
    sprite_name = sprite_name.replace('whale_', 'whale_' + this.dest_x + '-');
    var sprite = this.lookup_sprite(sprite_name);

    var w = sprite.naturalWidth;
    var h = sprite.naturalHeight;
    var coords = lamemath.center(points);
    coords = [coords[0] - w/2 + xoffset, coords[1] - h/2 + yoffset];
    this.app.renderer.draw(sprite, coords, 1.0, opacity, tint);
};

function App(title, width, height, fullscreen, scenes, entry, debug, opengl) {
    // pygame.init()

    this.debug = debug;

    // self._clock = pygame.time.Clock()
    
    this.fps = 30;

    this.accumulator = new TimeAccumulator(this.fps);

    if (opengl) {
        this.renderer = new WebGLRenderer(this);
    } else {
        this.renderer = new CanvasRenderer(this);
    }

    this.screen = new Screen(this, title, width, height, fullscreen);
    this.renderer.setup(this.screen.size);

    this.resman = new ResourceManager(this);
    this.audman = new AudioManager(this);

    this.player = new Player(this);

    this.scenes = {};
    for (var key in scenes) {
        this.scenes[key] = new scenes[key](this);
    }

    this.scene = this.scenes[entry];
    this.old_scene = null;
    this.scene_transition = 0.0;
    this.quit = false;

    this.events = [];
};

App.prototype.post_input_event = function(name, e) {
    if (name.indexOf('mouse') == 0) {
        this.events.push([name, [e.clientX, e.clientY]]);
    } else {
        this.events.push([name, e.keyCode]);
    }
};

App.prototype.get_filename = function(basename) {
    return this.resman._path(basename);
};

App.prototype.run = function() {
    var self = this;

    this._interval_id = setInterval(function() {
        if (self.quit) {
            clearInterval(self._interval_id);
            return;
        }

        // self._clock.tick(self.fps)

        self.step();
        self.render();
    }, 1000.0 / this.fps);
};

App.prototype.step = function() {
    var fading_out = (this.old_scene !== null && this.scene_transition < 0.5);
    if (!fading_out) {
        var events = this.events;
        this.events = [];

        for (var i=0; i<events.length; i++) {
            this.scene.process_input(events[i]);
        }

        var app = this;
        var next_scene = this.accumulator.update(function() {
            return app.scene.process();
        });
        if (next_scene === 'GoodBye') {
            this.quit = true;
        } else if (next_scene) {
            // scene wants to change!
            this.old_scene = this.scene;
            this.scene_transition = 0.0;
            // XXX: Tell the renderer to snapshot old_scene for transition
            this.scene = this.scenes[next_scene];
            this.scene.resume();
        }
    }
};

App.prototype.render = function() {
    this.renderer.begin();

    if (this.scene_transition >= 0.95) {
        // Scene transition is done
        this.old_scene = null;
    } else {
        // Push forward the transition
        this.scene_transition += 0.05;
    }

    if (this.old_scene !== null) {
        // Fading in of new scene part
        var brightness = this.scene_transition;
        // XXX: Tell renderer to fade between old_scene and this.scene
        this.renderer.global_tint = [brightness, brightness, brightness];
    }

    this.scene.draw();

    this.renderer.global_tint = [1.0, 1.0, 1.0];
    if (this.debug) {
        this.screen.draw_debug();
    }

    this.renderer.finish();
};


function main(use_webgl) {
    var TITLE = 'One Whale Trip';
    //var SCREEN_WIDTH = 800;
    //var SCREEN_HEIGHT = 480;
    var SCREEN_WIDTH = window.innerWidth;
    var SCREEN_HEIGHT = window.innerHeight;

    var SCENES = {
        'Start': Start,
        'Intro': Intro,
        'Game': Game,
        'NextLevelGroup_1_3': NextLevelGroup_1_3,
        'NextLevelGroup_2_3': NextLevelGroup_2_3,
        'NextLevelGroup_3_1': NextLevelGroup_3_1,
        'NextLevelGroup_4_1': NextLevelGroup_4_1,
        'NextLevelGroup_5_1': NextLevelGroup_5_1,
        'LostLife': LostLife,
        'GameOver': GameOver,
        'Victory': Victory,
        'Outro': Outro,
    };

    var app = new App(TITLE, SCREEN_WIDTH, SCREEN_HEIGHT, false, SCENES, 'Start', true, use_webgl);
    app.run();
}

