angular.module('MotainAppModel', ['MotainAppConfig', 'angular-momentjs', 'pascalprecht.translate']);
angular.module('MotainAppDirective', ['MotainAppConfig', 'pascalprecht.translate']);

angular.module('MotainApp', ['MotainAppConfig', 'MotainAppModel', 'MotainAppDirective', 'ngRoute', 'pascalprecht.translate', 'ngCookies', 'mgcrea.ngStrap', 'angular-momentjs', 'ngTouch', 'angular-inview']).
    config(function ($routeProvider, $locationProvider, $compileProvider, $httpProvider) {

    /* AngularJS related security issue, see http://stackoverflow.com/questions/15637133/unsafe-link-in-angular/15637622#15637622 */
    $compileProvider.aHrefSanitizationWhitelist(/^\s*(https?|ftp|mailto|app):/);
    $compileProvider.imgSrcSanitizationWhitelist(/^\s*(https?|app):/);

    $locationProvider
        .html5Mode(false)
        .hashPrefix('!');

    $routeProvider
        .when('/', {
            redirectTo: function () {
                return '/live';
            }
        })
        .when('/section/:competitionId/:seasonId', {
            redirectTo: function (routeParams, path, search) {
                return path + '/schedule';
            }
        })
        .when('/section/:competitionId/:seasonId/:page', {
            controller: 'SectionController',
            templateUrl: 'view/section.html'
        })
        .when('/section/:competitionId/:seasonId/:page/:subpage', {
            controller: 'SectionController',
            templateUrl: 'view/section.html'
        })

        .when('/team/:competitionId/:seasonId/:teamId', {
            redirectTo: function (routeParams, path, search) {
                return path + '/matches';
            }
        })
        .when('/team/:competitionId/:seasonId/:teamId/:page', {
            controller: 'TeamController',
            templateUrl: 'view/team.html'
        })

        .when('/match/:competitionId/:seasonId/:matchId', {
            redirectTo: function (routeParams, path, search) {
                return path + '/overview';
            }
        })
        .when('/match/:competitionId/:seasonId/:matchId/:page', {
            controller: 'MatchController',
            templateUrl: 'view/match.html',
            title: 'Match'
        })

        .when('/live', {
            redirectTo: function (routeParams, path, search) {
                return path + '/today';
            }
        })
        .when('/live/:date', {
            controller: 'LiveController',
            templateUrl: 'view/live.html'
        })
        .when('/about', {
            controller: 'AboutController',
            templateUrl: 'view/about.html'
        })
        .otherwise({
            redirectTo: '/'
        });

    // Keep track of the number of requests still running
    var interceptor = ['$rootScope','$q', 'NetworkRequestsCounter', function(scope, $q, NetworkRequestsCounter) {
        return {
            'request': function (config) {
                NetworkRequestsCounter.plus();
                return config || $q.when(config);
            },
            'requestError': function(rejection) {
                NetworkRequestsCounter.minus();
                return $q.reject(rejection);
            },
            'response': function(response) {
                NetworkRequestsCounter.minus();
                return response || $q.when(response);
            },
            'responseError': function(rejection) {
                NetworkRequestsCounter.minus();
                return $q.reject(rejection);
            }
        }
    }];
    $httpProvider.interceptors.push(interceptor);


});

angular.module('MotainApp').run(function ($rootScope, UserSession, $moment, $route, EventEnum, $cookies) {

    // function setAppLang(lang) {
    //     $translateProvider.preferredLanguage(lang);
    //     $moment.lang(lang);
    // }

    $rootScope.$on(EventEnum.USER_AUTHENTICATION_STATE_CHANGED, function () {
    	$translateProvider.preferredLanguage('en');
        $moment.lang('en');

    });

});
(function() {

    var api = 'http://api.users.thefootballapp.com.dev';

    var config = angular.module('MotainAppConfig', []);

    config.constant('_', _);

    config.constant('CONFIG', {
        default_lang: 'en',
        oauth: {
            url: api + '/auth/oauth/v2/auth',
            clientId: '94b900318b8bf9aec63300bf273a017c52eb899e9b9f3_a45la0wzzh4c8w880ck48448gwcwcokok8co8sgo0c8s0k0w0',
            redirectUri: 'http://localhost:9001/auth.html',
            logoutFix: 'http://api.users.thefootballapp.com.dev/auth/app_dev.php/oauth/v2/logout?callback=JSON_CALLBACK'
        },
        api: {
            user_root: 'http://api.users.thefootballapp.com.dev/app_dev.php',
            teams_overview: 'http://feedmonster.iliga.de/feeds/il/:lang/competitions/:competitionId/:seasonId/teamsOverview.json',
            teams_standings: 'http://feedmonster.iliga.de/feeds/il/:lang/competitions/:competitionId/:seasonId/standings.json',
            live_feed: 'http://feedmonster.iliga.de/live/',
            live_feed_default_params: {
                f: 'il',
                gz: 'false',
                l: 'en',
                o: 'json',
                t: 'scores',
                tz: 'UTC',
                c: '12'
            },
            team_score: 'http://feedmonster.iliga.de/feeds/il/:lang/competitions/:competitionId/:seasonId/seasonscores/:teamId.json', // @deprecated
            team_feed: 'http://vintagemonster.iliga.de/api/teams/:teamId.json',
            team_stats: 'http://vintagemonster.iliga.de/api/season-stats/teams/:teamId/:seasonId.json?games_n=100',
            match_feed: 'http://feedmonster.iliga.de/feeds/il/:lang/competitions/:competitionId/:seasonId/matches/:matchId.json',
            matchdays_feed: 'http://feedmonster.iliga.de/feeds/il/:lang/competitions/:competitionId/:seasonId/matchdaysOverview.json',
            matchday_feed: 'http://feedmonster.iliga.de/feeds/il/:lang/competitions/:competitionId/:seasonId/matchdays/:matchdayId.json',
            season_stats: 'http://feedmonster.iliga.de/feeds/il/:lang/competitions/:competitionId/:seasonId/:statType.json',
            quotes: 'http://vintagemonster.iliga.de/api/feeds/matches/:matchId/bettingQuotes.json',
            ticker_feed: 'http://feedmonster.iliga.de/ticker/il/index.php?lang=:lang&do=match&competitionId=:competitionId&seasonId=:seasonId&matchdayId=:matchdayId&matchId=:matchId&format=json&gzip=0',
            knockout: 'http://vintagemonster.iliga.de/api/roster/:competitionId/:seasonId.json'
        },
        resources: {
            teams_logos: 'http://images.iliga.de/icons/164/:optaId.png',
            teams_logos_internal: 'http://images.iliga.de/icons/internal/164/:id.png',
            sections_flags: 'http://images.iliga.de/icons/leagueSections/:section.png',
            config_file: 'http://feedmonster.iliga.de/feeds/il/:lang/config.json'
        }
    });
})();

angular.module('MotainApp').controller('AboutController', function ($rootScope, $scope, $route, $routeParams) {

    $scope.$on("$routeChangeSuccess", function (event, next, current) {
        $rootScope.pageTitle = 'About';
        $rootScope.pageSecondTitle = undefined;
    });

});
angular.module('MotainApp').controller('LiveController', function ($rootScope, $scope, $route, $routeParams, LiveMapper, UserSession, ApplicationConfigFileLoader, $moment, CONFIG, EventEnum) {
    $scope.isAuthenticated = false;

    $scope.isLoading = true;
    $scope.LEAGUES_FLAGS_URL = CONFIG.resources.sections_flags;
    $scope.TEAMS_LOGOS_URL = CONFIG.resources.teams_logos;

    $scope.day = $routeParams.date;
    $scope.yesterday = $moment().subtract('days', 1).format("YYYY-MM-DD");
    $scope.tomorrow = $moment().add('days', 1).format("YYYY-MM-DD");
    $scope.today = $moment().format("YYYY-MM-DD");

    $scope.fetchLiveFeed = function (configFile) {
        $scope.configFile = configFile || $scope.configFile;
        $scope.isLoading = true;
    };

    pageIndexes = {
        yesterday: 0,
        today: 1,
        tomorrow: 2
    };

    $rootScope.activeTabIndex = pageIndexes[$scope.day];

    ApplicationConfigFileLoader.getConfigFile().then($scope.fetchLiveFeed);
    $scope.$on(EventEnum.USER_AUTHENTICATION_STATE_CHANGED, $scope.fetchLiveFeed);

    $scope.$on("$routeChangeSuccess", function (event, next, current) {
        $rootScope.pageTitle = 'Live';
        $rootScope.pageSecondTitle = undefined;
    });

});

angular.module('MotainApp').controller('MainController', function($scope) {

});

angular.module('MotainApp').controller('MatchController', function ($rootScope, $scope, $route, $routeParams, $interval, MatchMapper, QuotesMapper, TickerMapper, ApplicationConfigFileLoader, CONFIG, UserSession, $alert, $log, $sce, EventEnum) {

    $scope.isLoading = true;
    $scope.match = undefined;

    $scope.TEAMS_LOGOS_URL = CONFIG.resources.teams_logos;

    $scope.page = $routeParams.page && $routeParams.page || 'overview';

    $scope.now = new Date();

    $rootScope.activeTabIndex = 1;

    function loadTicker() {
        if ($scope.hasLiveTicker) {
            $scope.tickerErrorsCounter = 0;
            $scope.tickerRefreshInterval = 10 * 1000; // every 10 seconds
            $scope.fetchTicker = function () {
                if ($scope.tickerErrorsCounter > 3) {
                    $interval.cancel($scope.tickerPromise);
                    return;
                }
                TickerMapper.find($routeParams).then(function (data) {
                    $scope.ticker = data;
                }, function () {
                    $scope.tickerErrorsCounter += 1;
                    $log.error('Could not retrieve ticker for match ' + $routeParams.matchId + " (" + $scope.tickerErrorsCounter + ")");
                });
            };
            $scope.fetchTicker();
            $scope.tickerPromise = $interval($scope.fetchTicker, $scope.tickerRefreshInterval);
            $scope.$on('$destroy', function () {
                $interval.cancel($scope.tickerPromise);
            });
        }
    }

    function reloadCompetitionProperties() {
        $scope.hasLiveTicker = $scope.configFile.sortedCompetitions[$scope.match.competitionId].hasLiveTicker;
        $scope.hasFacts = $scope.configFile.sortedCompetitions[$scope.match.competitionId].hasFacts;
        $scope.hasLiveStatistics = $scope.configFile.sortedCompetitions[$scope.match.competitionId].hasLiveStatistics;
        $scope.hasSquads = $scope.configFile.sortedCompetitions[$scope.match.competitionId].hasSquads;
        $rootScope.pageSecondTitle = $scope.configFile.sortedCompetitions[$scope.match.competitionId].name;
    }

    function getConfigFile() {
        ApplicationConfigFileLoader.getConfigFile().then(function (data) {
            $scope.configFile = data;
            reloadCompetitionProperties();
            loadTicker()
        }, function () {
            $log.error('ConfigFile not found');
        });
    }

    function sortPlayerFormationData(formation_data) {
        var formation = [];
        
        for(player in formation_data) {
            formation.push(formation_data[player]);
        }

        formation.sort(function (a, b) {
            if (a.positionTactical > b.positionTactical) {
                return 1;  
            }
              
            if (a.positionTactical < b.positionTactical) {
                return -1;  
            }

            return 0;
        });

        return formation;
    }


    function buildLineUp(team, formation_layout) {
        var line_up, formation_index, formation_layout_length, row, line_up_index;
        
        line_up = [[team.shift()]];
        formation_index = 0;
        formation_layout_length = formation_layout.length;
        row = [];
        
        for(formation_index; formation_index < formation_layout_length; formation_index = formation_index + 1) {

            for(line_up_index = 0; line_up_index < formation_layout.charAt(formation_index); line_up_index = line_up_index + 1) {
                row.push(team.shift());
            }

            line_up.push(row);
            row = [];
        }

        return line_up;
    }

    /*
    * Formats a teams formation layout from for example 442 => 4-4-2
    */
    function formatTeamFormation(formation_layout) {
        var i, formation, formation_layout_length;
        
        formation = '';
        formation_layout_length = formation_layout.length;
        i = 0;
        for(i; i < formation_layout_length; i = i + 1) {
            formation += formation_layout.charAt(i) + '-';
        }
        
        return formation.substr(0, formation.length-1); ;
    }

    MatchMapper.find(
            angular.extend($routeParams, {lang: UserSession.getCurrentLang()})
        ).then(function (data) {
            $scope.match = data;
            $scope.gameTime = new Date($scope.match.kickoff);
            $scope.isLoading = false;

            $rootScope.pageTitle = 'Match';
            $rootScope.pageSecondTitle = undefined;

            var home_team_formation_layout = data.homeTeam.formationLayout.toString();
            var guest_team_formation_layout = data.guestTeam.formationLayout.toString();

            // Sort the player formations after tactical position
            var home_team_formation = sortPlayerFormationData(data.homeTeam.formation);
            var guest_team_formation = sortPlayerFormationData(data.guestTeam.formation);

            // Initial the pitch object
            $scope.pitch = {
                homeTeam: {
                    name: data.homeTeam.name,
                    lineUp: [],
                },
                guestTeam: {
                    name: data.guestTeam.name,
                    lineUp: []
                }
            };

            // Builds the pitch line ups
            $scope.pitch.homeTeam.lineUp = buildLineUp(home_team_formation, home_team_formation_layout);
            $scope.pitch.guestTeam.lineUp = buildLineUp(guest_team_formation, guest_team_formation_layout);

            // Reverses the guest team formation layout so it is printed in reveser with the goalkeeper last.
            $scope.pitch.guestTeam.lineUp.reverse();
            
            // Format the formation layouts to strings and correct format.
            $scope.pitch.homeTeam.formationLayout = formatTeamFormation(home_team_formation_layout);
            $scope.pitch.guestTeam.formationLayout = formatTeamFormation(guest_team_formation_layout);
            
            $scope.pitch.homeTeam.formationRows = home_team_formation_layout.length;
            $scope.pitch.guestTeam.formationRows = guest_team_formation_layout.length;

            $scope.showPreMatchTimer = (['PreMatch', 'Postponed'].indexOf($scope.match.period) != -1);
            $scope.showInMatchTimer = (['FirstHalf', 'SecondHalf', 'HalfTime',
                'ExtraFirstHal', 'ExtraSecondHalf', 'ShootOut'].indexOf($scope.match.period) != -1);
            $scope.gameIsFinished = (['FullTime', 'Abandoned'].indexOf($scope.match.period) != -1);

            getConfigFile();
            $scope.$on(EventEnum.USER_AUTHENTICATION_STATE_CHANGED, getConfigFile());

            QuotesMapper.find($routeParams.matchId).then(function (data) {

                // Helper function to calculate the percentage of votes and round down the decimals properly
                function calculatePercentage(votes, total_votes, decimals) {
                    var value = ((votes / total_votes) * 100)
                    return Number(Math.round(value+'e'+decimals)+'e-'+decimals);
                }

                // Stone cold counting and coding assuming the fact that the Tipster quotes is the first entry in the data array
                var quotes = data[0];
                var total_votes = (quotes.home + quotes.draw + quotes.guest);
                var quotes_data = {
                    provider: quotes.provider,
                    home: quotes.home,
                    draw: quotes.draw,
                    guest: quotes.guest,
                    totalVotes: total_votes,
                    homePct: calculatePercentage(quotes.home, total_votes, 1),
                    drawPct: calculatePercentage(quotes.draw, total_votes, 1),
                    guestPct: calculatePercentage(quotes.guest, total_votes, 1)
                };

                $scope.quotes = quotes_data;
            }, function () {
                $log.error('Quotes not found for match ' + $routeParams.matchId);
            });
        }, function () {
            $log.error('Match ' + $routeParams.matchId + ' not found');
            $alert({
                title: $sce.trustAsHtml('This match was not found'),
                placement: 'top',
                type: 'info',
                keyboard: false
            });
        });
});


angular.module('MotainApp').controller('SectionControllerStandings', function ($scope, SectionMapper, UserSession, $location) {
    
    // key is also copied to 'value.key' so that it can be easily retrieved later
    addValueToArray = function (ar, key, value) {
        value.key = key;
        ar[key] = value;
    };

    // @TODO check booleans in config file: hasStandings, hasHomeAwayStandings
    $scope.standingsTables = { };
    addValueToArray($scope.standingsTables, 'general', { label: 'Standings', attrName: 'index' });
    addValueToArray($scope.standingsTables, 'home', { label: 'Home', attrName: 'indexHome' });
    addValueToArray($scope.standingsTables, 'away', { label: 'Away', attrName: 'indexAway' });
    $scope.selectedStandingsTable = $scope.subpage
        && $scope.standingsTables[$scope.subpage]
        || $scope.standingsTables['general'];

    function loadStandings() {
        $scope.isLoadingStandings = true;
        SectionMapper.findStandings($scope.competitionId, $scope.seasonId, UserSession.getCurrentLang()).then(function (groups) {
            $scope.groups = groups;
            $scope.isLoadingTable = false;
        });
    }

    loadStandings();

    $scope.onRankingClick = function (ranking) {
        $location.path('/team/' + $scope.competitionId + '/' + $scope.seasonId + '/' + ranking.team.id);
    };

    $scope.onStandingsTablesChange = function (obj) {
        $location.path('/section/'
            + $scope.competitionId + '/'
            + $scope.seasonId + '/'
            + $scope.page + '/'
            + obj.key
        );
    };
});

angular.module('MotainApp').controller('SectionControllerTeams', function ($scope, SectionMapper, UserSession) {

    function loadTeams() {
        $scope.isLoadingTeams = true;
        SectionMapper.findTeams($scope.competitionId, $scope.seasonId, UserSession.getCurrentLang()).then(function (teams) {
            $scope.teams = teams;
            $scope.isLoadingTeams = false;
        });
    }

    loadTeams();
});

angular.module('MotainApp').controller('SectionControllerMatchday', function ($scope, $routeParams, $location, $route, $anchorScroll, $modal, $moment, SectionMapper, UserSession, $window) {

    // Pre-fetch an external template populated with a custom scope
    var matchDayModal = $modal({
        scope: $scope,
        template: 'view/matchday/matchDayModal.tpl.html',
        contentTemplate: 'view/matchday/matchDayModal.html',
        show: false
    });

    var currentMatchDayDate = '';
    var lastRoute = $route.current;

    // Prevents the ngView router routine to run the controller when using $location and $anchorScroll
    // $scope.$on('$locationChangeSuccess', function(event) {
    //     $route.current = lastRoute;
    // });
    $scope.selectedMatchday;
    
    function loadMatchday(competitionId, seasonId, matchdayId) {
        $scope.isLoadingMatchday = true;

        SectionMapper.findMatchdayById(competitionId, seasonId, matchdayId, UserSession.getCurrentLang()).then(function (matchday) {
            $scope.matchday = matchday;
            $scope.isLoadingMatchday = false;
        });
    }

    function loadMatchdays() {
        $scope.isLoadingMatchdays = true;

        SectionMapper.findMatchdays($scope.competitionId, $scope.seasonId, UserSession.getCurrentLang()).then(function (matchdays) {
            $scope.matchdays = matchdays;

            _.forEach($scope.matchdays, function (m) {
                if (m.isCurrentMatchday) {
                    $scope.selectedMatchday = m;
                }
            });
            if ($scope.selectedMatchday == undefined) {
                $scope.selectedMatchday = $scope.matchdays[0];
            }

            $scope.isLoadingMatchdays = false;
            loadMatchday($scope.competitionId, $scope.seasonId, $scope.matchdays[0].id);
        });
    }

    loadMatchdays();

    $scope.onMatchClick = function (match) {
        $location.path('/match/' + $scope.competitionId + '/' + $scope.seasonId + '/' + match.id);
    };

    $scope.openMatchModal = function() {
        // use $promise property to ensure the template has been loaded
        matchDayModal.$promise.then(function() {
            $scope.matchDayContainerHeight=$window.innerHeight -30*2 -40; // .modal-dialog->top-margin x2 - .match-day-modal-header->height
            matchDayModal.show();
        });
    }

    $scope.onSelectMatchDay = function(matchday) {
        loadMatchday($routeParams.competitionId, $routeParams.seasonId, matchday.id);
        $location.hash('');
        matchDayModal.hide();
    }
    
    $scope.isFutureMatchDay = function(currentMatchDay) {

        if(currentMatchDayDate !== '') {
            return $moment(currentMatchDay.kickoffLast) > currentMatchDayDate;
        };

        if(currentMatchDay.isCurrentMatchday) {
            currentMatchDayDate = $moment(currentMatchDay.kickoffLast);
        };

        return false;
    }

});


angular.module('MotainApp').controller('SectionControllerStats', function ($scope, SectionMapper, UserSession, $location) {

    // key is also copied to 'value.key' so that it can be easily retrieved later
    addValueToArray = function (ar, key, value) {
        value.key = key;
        ar[key] = value;
    };

    $scope.stats = { };
    addValueToArray($scope.stats, 'scorer', { label: 'Goals', method: SectionMapper.findStatsScorers, statName: 'seasonGoals' });
    addValueToArray($scope.stats, 'assist', { label: 'Assists', method: SectionMapper.findStatsAssists, statName: 'seasonAssists' });
    addValueToArray($scope.stats, 'goalsassist', { label: 'Goals+Assists', method: SectionMapper.findStatsGoalsAssits, statName: 'seasonGoalsAndAssists' });
    addValueToArray($scope.stats, 'yellowcards', { label: 'Yellow Cards', method: SectionMapper.findStatsYellowCards, statName: 'seasonYellowCards' });
    addValueToArray($scope.stats, 'redcards', { label: 'Red Cards', method: SectionMapper.findStatsRedCards, statName: 'seasonRedCards' });
    $scope.selectedStat = $scope.subpage
        && $scope.stats[$scope.subpage]
        || $scope.stats['scorer'];

    function loadStats() {
        $scope.isLoadingStats = true;
        $scope.selectedStat.method($scope.competitionId, $scope.seasonId, UserSession.getCurrentLang()).then(function (statsTable) {
            $scope.statsTable = statsTable;
            $scope.isLoadingStats = false;
        });
    }

    loadStats();

    $scope.onStatsChange = function (obj) {
        loadStats();
        $scope.page='stats'
        $location.path('/section/'
            + $scope.competitionId + '/'
            + $scope.seasonId + '/'
            + $scope.page + '/'
            + obj.key
        );
    };

});

angular.module('MotainApp').controller('SectionControllerKnockout', function ($scope, SectionMapper, UserSession, $location) {

    // key is also copied to 'value.key' so that it can be easily retrieved later
    addValueToArray = function (ar, key, value) {
        value.key = key;
        ar[key] = value;
    };

    function loadKnockouts() {
        $scope.isLoadingStats = true;
        SectionMapper.getKnockout($scope.competitionId, $scope.seasonId).then(function (knockouts) {
            formatKnockouts(knockouts);
            $scope.isLoadingStats = false;
        });
    }

    loadKnockouts();

    function formatKnockouts(knockouts){
        $scope.knockouts = {}
        $scope.knockouts.final = knockouts.match;
        $scope.knockouts.semifinals = [];
        $scope.knockouts.quarters = [];
        $scope.knockouts.octaves = [];
        knockouts.match.children.forEach(function(semifinal) {
            $scope.knockouts.semifinals.push(semifinal);
            if (typeof semifinal.children != 'undefined') {
                semifinal.children.forEach(function(quarters) {
                    $scope.knockouts.quarters.push(quarters);
                    if (typeof quarters.children != 'undefined') {
                        quarters.children.forEach(function(octaves) {
                            $scope.knockouts.octaves.push(octaves);
                        })
                    }
                })
            };
           
        })
    }

});

angular.module('MotainApp').controller('SectionController', function ($rootScope, CONFIG, $scope, $routeParams, $route, $anchorScroll, ApplicationConfigFileLoader, SectionMapper, UserSession, EventEnum, NetworkRequestsCounter) {

    $anchorScroll();

    $scope.teamsLogosUrl = CONFIG.resources.teams_logos;

    $scope.competitionId = $routeParams.competitionId;
    $scope.seasonId = $routeParams.seasonId;
    $scope.page = $routeParams.page;
    $scope.subpage = $routeParams.subpage;

    $scope.activeTab = $routeParams.page && $routeParams.page || 'teams';
    pageIndexes = {
        teams: 4,
        stats: 3,
        knockout: 2,
        table: 1,
        schedule: 0
    };
    pageTitles = {
        teams: 'Teams List',
        stats: 'Statistics',
        knockout: 'Knockout',
        schedule: 'Matchday',
        table: 'Table'
    };
    
    $rootScope.activeTabIndex = pageIndexes[$scope.activeTab];
    $scope.$watch('activeTab', function (newValue, oldValue) {
        $rootScope.pageTitle = pageTitles[newValue];
    });

    // @TODO handle rejected promise
    $scope.fetchLiveFeed = function (configFile) {
        $scope.configFile = configFile || $scope.configFile;
        $scope.section = $scope.configFile.sortedCompetitions[$routeParams.competitionId];
        $route.current.title = $scope.section.competitionName;
        $rootScope.pageSecondTitle = $scope.section.competitionName;
    };

    $scope.$on(EventEnum.USER_LANG_CHANGED, function () {
        // TODO reload content
    });

    ApplicationConfigFileLoader.getConfigFile().then($scope.fetchLiveFeed);

});

angular.module('MotainApp').controller('TeamController', function ($rootScope, $scope, $http, $routeParams, $moment, ApplicationConfigFileLoader, TeamMapper, TeamStatsMapper, UserSession, EventEnum, $location) {

    $scope.$moment = $moment;

    $scope.activeTab = $routeParams.page && $routeParams.page || 'matches';

    $scope.competitions = {};
    $scope.selectedCompetition;


    $scope.competitionId = $routeParams.competitionId;
    $scope.seasonId = $routeParams.seasonId;
    $scope.teamId = $routeParams.teamId;
    $scope.page = $routeParams.page;

    $rootScope.pageTitle = 'Team';
    $rootScope.pageSecondTitle = undefined;

    $rootScope.activeTabIndex = 0;

    $scope.fetchLiveFeed = function (configFile) {
        $scope.configFile = configFile || $scope.configFile;
        $scope.isLoading = true;

        function loadData() {
            $scope.isLoadingTeam = true;
            TeamMapper.findOneById($routeParams.teamId).then(function (team) {
                $scope.team = team;
                $scope.players = team.players;
                $scope.competitions = (function (competitions) {
                    var c = {};
                    for (var i = 0; i < competitions.length; i++) {
                        var id = competitions[i].competitionId;
                        c[id] = $scope.configFile.sortedCompetitions[id];
                    }
                    return c;
                }(team.competitions));
                $scope.selectedCompetition = $scope.competitions[$routeParams.competitionId];
                $scope.isLoadingTeam = false;
            }, function () {
                $scope.isLoadingTeam = false;
            });

            $scope.isLoadingStats = true;
            TeamStatsMapper.findStatsForSeason($routeParams.teamId, $routeParams.seasonId).then(function (teamSeason) {
                $scope.stats = teamSeason.stats;
                $scope.matches = teamSeason.games;
                $scope.isLoadingStats = false;
            }, function () {
                $scope.isLoadingStats = false;
            });
        }

        loadData();

        $scope.$on(EventEnum.USER_LANG_CHANGED, function () {
            loadData();
        });

        $scope.onCompetitionChange = function (selectedCompetition) {
            $location.path('/team/' + selectedCompetition.competitionId
                + '/' + selectedCompetition.seasonId
                + '/' + $scope.team.id
                + '/' + $scope.activeTab);
        };

        $scope.onMatchClick = function (match) {
            $location.path('/match/' + $routeParams.competitionId + '/' + $routeParams.seasonId + '/' + match.id);
        };
    };

    ApplicationConfigFileLoader.getConfigFile().then($scope.fetchLiveFeed);
});

angular.module('MotainApp').controller('MenuCompetitionListController', function ($scope, $log, ApplicationConfigFileLoader, _, SectionMapper, UserSession) {

    $scope._ = _;

    ApplicationConfigFileLoader.getConfigFile().then(function (data) {
        $scope.configFile = data;

        $scope.wc = $scope.configFile.sortedCompetitions[$scope.competitionId];

        function loadTeams(configFile) {
            $scope.isLoadingTeams = true;
            SectionMapper.findTeams($scope.competitionId, $scope.seasonId, UserSession.getCurrentLang()).then(function (teams) {
                $scope.teams = teams;
                $scope.isLoadingTeams = false;
            });
        }

        loadTeams($scope.configFile);

    }, function () {

    });

});

angular.module('MotainApp').controller('MenuController', function($scope, CONFIG, $translate) {
    $scope.onLinkClick = function() {
        return false;
    };

    $scope.leaguesFlagsUrl = CONFIG.resources.sections_flags;
    $scope.teamsLogosUrl = CONFIG.resources.teams_logos;
    $scope.competitionId = 12;
    $scope.seasonId = 459;
});

angular.module('MotainApp').controller('NavigationController', function($scope, UserSession, UserMapper, $http, CONFIG, $alert, $log, $sce, EventEnum) {
    $scope.isAuthenticated = false;
    $scope.user = null;

    $scope.$on(EventEnum.USER_AUTHENTICATION_STATE_CHANGED, function() {
        $scope.isAuthenticated = UserSession.isAuthenticated;

        if (UserSession.isAuthenticated) {
            $scope.user = UserSession.getUser();

            $alert({
                title: 'Logged in!',
                content: $sce.trustAsHtml('You have been logged in. Welcome ' + $scope.user.username + '!'),
                placement: 'top-right',
                type: 'info',
                show: true,
                duration: 3
            });

        } else {
            $scope.user = null;
        }
    });

    $scope.onButtonLogoutClick = function() {
        $scope.ajaxLogout = true;
        UserMapper.deleteToken().then(function() {

            UserSession.clean();

            // call directly oauth login app, and logout from session.
            try {
                $http.jsonp(CONFIG.oauth.logoutFix).success(function() {});
            } catch (e) {
               $log.error('Logout hack failed');
            }

            $scope.ajaxLogout = false;

            $alert({
                title: 'Logged out!',
                content: $sce.trustAsHtml('You have been logged out. Come back soon!'),
                placement: 'top-right',
                type: 'info',
                show: true,
                duration: 3
            });

        }, function() {
            // @todo handle error
            $scope.ajaxLogout = false;
        });
    };
});

angular.module('MotainApp').controller('NetworkMonitorController', ['$scope', 'NetworkRequestsCounter', 'EventEnum', function ($scope, NetworkRequestsCounter, EventEnum) {

    $scope.isBusy = false;

    $scope.$on(EventEnum.NETWORK_REQUEST_STARTED, function () {
        $scope.isBusy = !NetworkRequestsCounter.areDone();
    });

    $scope.$on(EventEnum.NETWORK_REQUEST_FINISHED, function () {
        $scope.isBusy = !NetworkRequestsCounter.areDone();
    });

}]);

angular.module('MotainApp').controller('LoginController', function($scope, CONFIG, $sce) {
    $scope.getIframeSrc = function() {
        var oauth = CONFIG.oauth;
        var url = oauth.url + '?response_type=token&client_id=' + oauth.clientId + '&redirect_uri='+oauth.redirectUri;
        return $sce.trustAsResourceUrl(url);
    };
});
angular.module('MotainAppDirective')
.directive('comingSoon', function ($rootScope) {
    return {
        restrict: 'EA',
        templateUrl: '/scripts/directives/views/comingSoon.html'
    };
});

angular.module('MotainAppDirective').directive('loading', function () {
    return {
        restrict: 'EA',
        templateUrl: '/scripts/directives/views/loading.html'
    };
});

angular.module('MotainAppDirective').directive('networkIndicator', function () {
    return {
        restrict: 'EA',
        templateUrl: '/scripts/directives/views/networkIndicator.html',
        link: function (scope, element, attrs) {
            scope.isBusy = attrs.isBusy;
        }
    };
});

angular.module('MotainAppDirective').directive("countdown", function($interval, $filter) {
  return {
    restrict: 'E',
    scope: {
     	},
    templateUrl: 'scripts/directives/views/oneGameCountdown.html', 
      
    link: function(scope, element, attrs) {

		scope.startTimer = function  () {
			var duration = scope.inputSeconds;
			scope.countDown = scope.convertToFormatedTime(scope.inputSeconds);
			var inc = scope.inputInterval;
			$interval.cancel(intervalFlag);
			intervalFlag = scope.setInterval(duration, inc);
		}

		scope.setInterval = function(duration, increment) {

			var interval = $interval(function() {
					duration = duration - increment;
					scope.countDown = scope.convertToFormatedTime(duration);
		        }, increment*1000);
			return interval;
		}

		scope.stopTimer = function() {
			$interval.cancel(intervalFlag);
		}

		scope.resetTimer = function() {
			scope.countDown = scope.convertToFormatedTime(scope.inputSeconds);
			scope.stopTimer();
		};

		scope.convertToFormatedTime = function (seconds) {
			var duration = moment.duration(seconds*1000);
			duration = Math.floor(duration.asHours()) + moment(duration.asMilliseconds()).format(":mm:ss");
			var durationArray = scope.splitFormatedTime(duration);
			return durationArray;
		}

		scope.splitFormatedTime = function(formatedTime) {
			var durationArray = formatedTime.split(":");
			durationArray[0] = scope.pad2(durationArray[0]);
			return durationArray;
		};

		scope.pad2 = function(number){
     		return (number < 10 ? '0' : '') + number
		}

		scope.$watch('countDown', function(newValue, oldValue) { 

			var countIsZero = true;
			angular.forEach(newValue, function(value) {
				if (value != '00' && value != '0'){
					countIsZero = false;
				}
			});
			if (countIsZero) {
				scope.stopTimer();
			}
		});

		scope.convertTimeToSecondsRemaining = function(kickoff) {
			var time2 = new Date();
			var time1 = new Date(kickoff);
			var dif = (time1.getTime() - time2.getTime()) / 1000;
			return dif;
		}

		scope.inputInterval=1;
      	scope.inputSeconds = scope.convertTimeToSecondsRemaining(attrs.kickoff);
      	var intervalFlag;
      	scope.startTimer(); 
    }
  };
});

angular.module('MotainAppDirective').directive('oneGameHighlights', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameHighlights.html',
        link: function(scope, element) {
            element.addClass('one-game-highlights');
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameHighlightsGoal', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameHighlightsGoal.html',
        link: function(scope, element, attrs) {
            element.addClass('one-game-highlights-goal');
            var home = true;
            if (attrs.guestTeamId == scope.event.player.internalTeamId) {
                home = false;
            }

            if (scope.event.type == 'Own') {
                home = !home;
            }

            if (home) {
                element.addClass('home-team-highlights');
            } else {
                element.addClass('guest-team-highlights');
            }
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameHighlightsRed', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameHighlightsRed.html',
        link: function(scope, element, attrs) {
            element.addClass('one-game-highlights-red');
            if (attrs.homeTeamId == scope.event.player.internalTeamId) {
                element.addClass('home-team-highlights');
                return
            }

            if (attrs.guestTeamId == scope.event.player.internalTeamId) {
                element.addClass('guest-team-highlights');
                return
            }
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameHighlightsSubstitution', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameHighlightsSubstitution.html',
        link: function(scope, element, attrs) {
            element.addClass('one-game-highlights-substitution');
            if (attrs.homeTeamId == scope.event.playerIn.internalTeamId) {
                element.addClass('home-team-highlights');
                return
            }

            if (attrs.guestTeamId == scope.event.playerIn.internalTeamId) {
                element.addClass('guest-team-highlights');
                return
            }
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameHighlightsYellow', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameHighlightsYellow.html',
        link: function(scope, element, attrs) {
            element.addClass('one-game-highlights-yellow');
            if (attrs.homeTeamId == scope.event.player.internalTeamId) {
                element.addClass('home-team-highlights');
                return
            }

            if (attrs.guestTeamId == scope.event.player.internalTeamId) {
                element.addClass('guest-team-highlights');
                return
            }
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameInfo', function ($translate) {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameInfo.html',
        link: function(scope, element) {
            element.addClass('one-game-info');
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameLineup', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameLineup.html',
        link: function(scope, element) {
            element.addClass('one-game-lineup');
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameMedia', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameMedia.html',
        link: function(scope, element) {
            element.addClass('one-game-media');
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameQuotes', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameQuotes.html',
        link: function(scope, element) {
            element.addClass('one-game-quotes');
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameScore', function () {
    var layouts = {
        extended: 'scripts/directives/views/oneGameScoreExtended.html',
        default: 'scripts/directives/views/oneGameScore.html'
    };

    return {
        restrict: 'EA',
        templateUrl: function (element, attrs) {
            return (layouts[attrs.layout] != undefined ? layouts[attrs.layout] : layouts['default']);
        },
        link: function (scope, element, attrs) {
            element.addClass('one-game-score');
            element.addClass('one-game-score-' + attrs.layout);
            scope.showHalfTimeScore = !!attrs.showHalfTimeScore;
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameStats', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameStats.html',
        link: function (scope, element) {
            element.addClass('one-game-stats');
            scope.elementsShown = 1;
            //loads the next element into the view
            scope.loadMoreItems = function(next, inview) {
                if(next > scope.elementsShown && inview){
                    scope.elementsShown = next;
                }
            }
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameStatsBars', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameStatsBars.html',
        scope: {
            title: '@',
            homeTeam: '@',
            guestTeam: '@',
            percentage: '@'
        },
        link: function (scope, element) {
            element.addClass('one-game-stats-bars');
            scope.percentage = scope.percentage == "true" && true || false;
            if (scope.percentage) {
                scope.homeTeamPercentage = scope.homeTeam;
                scope.guestTeamPercentage = scope.guestTeam;
            } else {
                var total = parseInt(scope.homeTeam, 10) + parseInt(scope.guestTeam, 10);
                scope.homeTeamPercentage = parseInt(scope.homeTeam, 10) * 100 / total;
                scope.guestTeamPercentage = parseInt(scope.guestTeam, 10) * 100 / total;
            }
            scope.homeTeam = scope.homeTeam;
            scope.guestTeam = scope.guestTeam;
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameStatus', function ($translate, $filter) {
    return {
        restrict: 'EA',
        template: '<span>{{gameStatus[period] | translate}}</span>',
        scope:{
        	period : '='
        },
        link: function(scope, element) {
            scope.gameStatus = {
            	'PreMatch' : 'PRE_MATCH',
            	'FirstHalf' : 'FIRST_HALF',
            	'HalfTime' : 'HALF_TIME',
            	'SecondHalf' : 'SECOND_HALF',
            	'ExtraFirstHalf' : 'EXTRA_FIRST_HALF',
            	'ExtraSecondHalf' : 'EXTRA_SECOND_HALF',
            	'ShootOut' : 'SHOOT_OUT',
            	'FullTime' : 'FULL_TIME',
            	'Postponed' : 'POSTPONED',
            	'Abandoned' : 'ABANDONED'
            }
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameTest', function () {
    return {
        restrict: 'EA',
        //templateUrl: 'scripts/directives/views/oneGameTest.html',
        template: '<h1>TEST</h1>',
        link: function(scope, element, attrs) {
            element.addClass('one-game-test');
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameTicker', function () {
    return {
        restrict: 'EA',
        scope: {
            ticker: '=',
            tickerErrorsCounter: '='
        },
        templateUrl: 'scripts/directives/views/oneGameTicker.html',
        link: function(scope, element) {
            element.addClass('one-game-ticker');
            //offset determines with how many elements above the last one the limit increases. This is used for start loading more elements before the end of the page is reached.
            scope.offset = 5;
            scope.limit = 20;
            scope.loadMoreItems = function(index, inview) {
                //if current position is visible and it is greater than the offset position, then increase the limit of elements shown.
                if((index > scope.limit-scope.offset) && inview){
                    scope.limit = scope.limit + 20;
                }
            }
        }

    };
});

angular.module('MotainAppDirective').directive('oneGameTickerMinute', function () {
    return {
        restrict: 'EA',
        scope: {
            minute: '='
        },
        templateUrl: 'scripts/directives/views/oneGameTickerMinute.html',
        link: function(scope, element) {
            var icon_class;

            switch(scope.minute.type) {
                case 'other':
                    icon_class = 'ticker-icon-other'
                    break;
                case 'startEnd':
                    icon_class = 'ticker-icon-start-end'
                    break;
                case 'goal':
                    icon_class = 'ticker-icon-goal'
                    break;
                case 'penalty':
                    icon_class = 'ticker-icon-penalty'
                    break;
                case 'corner':
                    icon_class = 'ticker-icon-corner'
                    break;
                case 'yellowCard':
                    icon_class = 'ticker-icon-yellow-card'
                    break;
                case 'redCard':
                    icon_class = 'ticker-icon-red-card'
                    break;
                case 'substitution':
                    icon_class = 'ticker-icon-substitution'
                    break;
                case 'offside':
                    icon_class = 'ticker-icon-offside'
                    break;
                case 'post':
                    icon_class = 'ticker-icon-post'
                    break;
                case 'yellowRedCard':
                    icon_class = 'ticker-icon-yellow-red-card'
                    break;
                default:
                    icon_class = 'ticker-icon-other'
                    break;
            }
            scope.iconClass = icon_class;
            element.addClass('one-game-ticker-minute');
        }
    };
});

angular.module('MotainAppDirective').directive('oneGameTimer', function () {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneGameTimer.html',
        link: function(scope, element, attrs) {
            element.addClass('one-game-timer');
        }
    };
});

angular.module('MotainAppDirective').directive('oneKnockoutCard', function ($location) {
    return {
        restrict: 'EA',
        scope: {
            game: '=',
            isFinal: '='
        },
        templateUrl: 'scripts/directives/views/oneKnockoutCard.html',
        link: function(scope, element) {
            if (typeof scope.isFinal == 'undefined') {
                scope.isFinal==false;
            };
            scope.goMatch = function (match) {
                console.log(match);
                $location.path('/match/' + match.competitionId + '/' + match.seasonId + '/' + match.id);
            };
            //offset determines with how many elements above the last one the limit increases. This is used for start loading more elements before the end of the page is reached.
        }

    };
});

angular.module('MotainAppDirective').directive('oneLiveScores', function (LiveMapper, $location, ApplicationConfigFileLoader, CONFIG) {

    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/oneLiveScores.html',
        scope: {
            date: '='
        },
        link: function (scope, element) {
            element.addClass('one-live-scores');
        },
        controller: function ($scope) {
            $scope.onClick = function (match) {
                $location.path('/match/' + match.competitionId + '/' + match.seasonId + '/' + match.id);
            };
            $scope.fetchLiveFeed = function (configFile) {
                $scope.configFile = configFile || $scope.configFile;
                $scope.isLoading = true;
            };
            $scope.getFlagUrl = function(groupId) {
                 var imgSrc = $scope.LEAGUES_FLAGS_URL.replace(":section", $scope.configFile.sortedCompetitions[groupId].section);
                 return imgSrc;
            };

            $scope.LEAGUES_FLAGS_URL = CONFIG.resources.sections_flags;
            $scope.isLoading = true;
            var findParams = angular.extend({}, {d: $scope.date});

            //Possible future improvement: Call getConfigFile() only once.
            ApplicationConfigFileLoader.getConfigFile().then($scope.fetchLiveFeed);

            LiveMapper.find(findParams).then(function (data) {
                $scope.feed = data;
                $scope.isLoading = false;
            });
        }
    };
});

angular.module('MotainAppDirective').directive('onePageTitle', function ($rootScope) {
    return {
        restrict: 'EA',
        templateUrl: 'scripts/directives/views/onePageTitle.html',
        link: function (scope, element) {
            element.addClass('one-page-title');
        },
        controller: function () {
            $rootScope.pageTitle = $rootScope.pageTitle || 'Onefootball';
        }
    };
});

angular.module('MotainAppDirective').directive('playerStat', function () {
    return {
        restrict: 'EA',
        scope: {
            statName: '=',
            player: '='
        },
        templateUrl: 'scripts/directives/views/playerStat.html',
        link: function(scope, element) {
            var icon_class = '';
            
            switch(scope.statName) {
                case 'seasonGoals':
                    icon_class = 'player-season-goals'
                    break;
                case 'seasonYellowCards':
                    icon_class = 'player-yellow-cards'
                    break;
                case 'seasonRedCards':
                    icon_class = 'player-red-cards'
                    break;
                case 'seasonAssists':
                    icon_class = 'player-assists'
                    break;
                case 'seasonGoalsAndAssists':
                    icon_class = 'player-goals-assists'
                    break;
                default:
                    icon_class = 'player-stat-icon'
                    break;
            }
            scope.iconClass = icon_class;
        }
    };
});

angular.module('MotainAppDirective')
.directive('swiperColumns', function ($anchorScroll, $location) {

    return {
        require:['swiperColumns', 'swiperTabs'],
        restrict: 'A',
        transclude: false,
        link: function($scope, element, attrs, controllers) {

            var columnsCtrl, tabsCtrl, tabs, start_tab;

            // Assigns the required controllers, using the convention that the first controller in controllers is the directives own controller
            columnsCtrl = controllers[0];
            tabsCtrl = controllers[1];

            // Wraps the different tabs in jQlite to be able to do some manipulation and set the initial current tab
            tabs = angular.element(tabsCtrl.tabs.slides);
            start_tab = angular.element(tabs[columnsCtrl.columns.activeIndex]);
            tabs.removeClass('active');
            start_tab.addClass('active');
 
            columnsCtrl.columns.addCallback('SlideNext', function() {
                tabsCtrl.tabs.swipeNext();
                tabs.removeClass('active');
                angular.element(tabs[columnsCtrl.columns.activeIndex]).addClass('active');
                tabsCtrl.tabs.swipeTo(columnsCtrl.columns.activeIndex);
                $scope.$apply(function(){
                    $scope.activeTabIndex = columnsCtrl.columns.activeIndex;

                });
            });
 
            columnsCtrl.columns.addCallback('SlidePrev', function() {
                tabsCtrl.tabs.swipePrev();
                tabs.removeClass('active');
                angular.element(tabs[columnsCtrl.columns.activeIndex]).addClass('active');
                tabsCtrl.tabs.swipeTo(columnsCtrl.columns.activeIndex);
                $scope.$apply(function(){
                    $scope.activeTabIndex = columnsCtrl.columns.activeIndex;
                });
            });
        },
        controller: function($scope, $rootScope) {
            var columnSwiper, columns_config;

            // Set to true to make it possible to test swiping in a desktop browser
            simulate_touch = false;

            columns_config = {
                mode: 'horizontal',
                simulateTouch: simulate_touch,
                slidesPerView: 'auto',
                initialSlide: $rootScope.activeTabIndex
            }
         
            if(!window.matchMedia("(min-width: 768px)").matches) {
                columns_config.slidesPerView = 1;
                columns_config.initialSlide = $rootScope.activeTabIndex;
            }

            this.columns = new Swiper('#swiper-columns', columns_config);
        }
    };
});
angular.module('MotainAppDirective')
.directive('swiperTabs', function($rootScope) {
    
    return {
        require:['swiperTabs', 'swiperColumns'],
        restrict: 'A',
        transclude: false,
        link: function($scope, element, attrs, controllers) {

            var tabsCtrl, columnsCtrl, tabs;

            // Assigns the required controllers, using the convention that the first controller in controllers is the directives own controller
            tabsCtrl = controllers[0];
            columnsCtrl = controllers[1];
            tabs = angular.element(tabsCtrl.tabs.slides);
            
            tabs.bind('click', function(e) {
                tabs.removeClass('active');
                angular.element(this).addClass('active');
                var index = this.index();
                $scope.$apply(function(){
                    $scope.activeTabIndex = index;
                });
                tabsCtrl.tabs.swipeTo(this.index());
                columnsCtrl.columns.swipeTo(this.index());
            });
        },
        controller: function($scope, $rootScope) {
            var tabSwiper, nr_of_tabs, simulate_touch, tabs_config;

            // Set to true to make it possible to test swiping in a desktop browser
            simulate_touch = false;

            // Get the number of tabs to be used in the tabs config so that the tabs will scale properly for screen sizes.
            nr_of_tabs = document.getElementById('swiper-tabs').getElementsByTagName('a').length;

            tabs_config = {
                mode: 'horizontal',
                simulateTouch: simulate_touch,
                slidesPerView: nr_of_tabs,
                centeredSlides: false,
                calculateHeight: true,
                centeredSlides: true,
                initialSlide: $rootScope.activeTabIndex
            }

            if(!window.matchMedia("(min-width: 768px)").matches) {
                tabs_config.slidesPerView = 3;
                tabs_config.centeredSlides = true;
                tabs_config.initialSlide = $rootScope.activeTabIndex;
            }

            this.tabs = new Swiper('#swiper-tabs', tabs_config);
        }
    };
});
angular.module('MotainApp').constant('EventEnum', {
    USER_LANG_CHANGED: 'userLangChanged',
    USER_AUTHENTICATION_STATE_CHANGED: 'auth:authentication',
    NETWORK_REQUEST_STARTED: 'net:request:start',
    NETWORK_REQUEST_FINISHED: 'net:request:end'
});

// Adds LiveReload script pointing at the client's hostname.
// This is helpful for mobile web development where your desktop might point at localhost while
// your devices point to a local IP address.
document.write('<script src="http://'
 + window.location.hostname
 + ':35729/livereload.js?snipver=1" type="text/javascript"><\/script>')

angular.module('MotainApp').service('LocaleService', function(CONFIG, $window) {

    function LocaleService() {
    }

    LocaleService.prototype.getDefaultLang = function () {
        return CONFIG.default_lang;
    };

    LocaleService.prototype.getBrowserLocale = function () {
        return $window.navigator.language;
    };

    return new LocaleService();
});

// https://motainteam.atlassian.net/wiki/display/MIDWARE/C.+Scores+API+v2
// https://motainteam.atlassian.net/wiki/display/MIDWARE/1%29+Scores+API+v2+Entity+Types

// Note: the API 1.0 sometimes uses different names for the same attributes inside of an object,
// instead of trying to guess how to unserialize it, we first copy every property, then we run some tests on
// the properties we know we want normalized

angular.module('MotainAppModel').factory('Team', function (CONFIG) {
    function Team(data) {
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

        // we should always use id & optaId, not id & internalId
        //      |    id    |  internalId  |  optaId  |
        // old  |    x     |  y           |  x       |
        // new  |    x     |  x           |  z       |
        if (this.optaId == undefined) {
            this.optaId = this.id;
            this.id = this.idInternal || this.idInt; // 'idInt' for live feed
        }
        this.logoUrl = CONFIG.resources.teams_logos_internal.replace(':id', this.id);
    }

    return Team;
});

angular.module('MotainAppModel').factory('TeamSeason', function (Match) {
    function TeamSeason(data) {
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

        this.games = (function (games) {
            var m = [];
            for (var i = 0; games && i < games.length; i++) {
                m.push(new Match(games[i]));
            }
            return m;
        }(this.games));
    }

    return TeamSeason;
});

angular.module('MotainAppModel').factory('Matchday', function (Kickoff) {
    /* matchday.type: https://motainteam.atlassian.net/wiki/display/MIDWARE/B.+iLiga+API#B.iLigaAPI-2.2.MatchdaysOverviewFeed
    "0" = cpmpetitions without groups
    "1" = qualifying
    "2" = group phase
    "3" = rounds of 16
    "4" = quarter final
    "5" = semi final
    "6" = match for 3rd place
    "7" = final
    "8" = round before rounds of 16
    "10" = Intermediate Round
    "11" = round of 32
    "12" = round of 64
    "13" = play off games */
    function Matchday(data) {
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

        // In API v1, kickoffs can be an array of times or an array of objects containing groups data depending on the
        // feed they come from. We'll normalize that a bit by renaming the first one kickoffsTimes.
        // See also API v2 https://motainteam.atlassian.net/wiki/display/MIDWARE/1%29+Scores+API+v2+Entity+Types#id-1%29ScoresAPIv2EntityTypes-Matchday
        if (this.kickoffs && this.kickoffs.length) {
            if (typeof this.kickoffs[0] == 'string') {
                this.kickoffsTimes = this.kickoffs;
                this.kickoffs = undefined;
            } else {
                this.kickoffs = (function (kickoffs) {
                    var k = [];
                    for (var i = 0; kickoffs && i < kickoffs.length; i++) {
                        k.push(new Kickoff(kickoffs[i]));
                    }
                    return k;
                }(this.kickoffs));
            }
        }

    }

    return Matchday;
});


angular.module('MotainAppModel').factory('Match', function (Team) {
    function Match(data) {

        // Note: the API 1.0 sometimes uses different names for the same attributes inside of an object,
        // instead of trying to guess how to unserialize it, we first copy every property, then we run some tests on
        // the properties we know we want normalized
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

        this.gameStatus = function(period) {
            if (['PreMatch', 'Postponed'].indexOf(period) != -1){
                return "PreMatch";
            }
            if (['FirstHalf', 'SecondHalf', 'HalfTime',
                'ExtraFirstHal', 'ExtraSecondHalf', 'ShootOut'].indexOf(period) != -1){
                return "InMatch";
            }
            if (['FullTime', 'Abandoned'].indexOf(period) != -1){
                return "PostMatch";
            }
            return "PreMatch"
        }(this.status || this.period);

        this.id = this.id || this.matchId;

        if (['PreMatch', 'Postponed'].indexOf(this.gameStatus) != -1) {
            this.homeTeam = new Team(angular.extend(this.teamhome, {
                firstHalfScore: '-',
                secondHalfScore: '-',
                finalScore: '-'
            }));
            this.guestTeam = new Team(angular.extend(this.teamaway, {
                firstHalfScore: '-',
                secondHalfScore: '-',
                finalScore: '-'
            }));
        } else {
            this.homeTeam = new Team(angular.extend(this.teamhome, {
                firstHalfScore: this.scorehomefirsthalf,
                secondHalfScore: this.scorehomesecondhalf,
                finalScore: this.scorehome
            }));
            this.guestTeam = new Team(angular.extend(this.teamaway, {
                firstHalfScore: this.scoreawayfirsthalf,
                secondHalfScore: this.scoreawaysecondhalf,
                finalScore: this.scoreaway
            }));
        }

        this.kickoffAt = this.kickoff;
        //this.matchday = Matchday;
        //this.referee = Referee;
        //this.season = BaseSeason;
        //this.stadium = Stadium;
        //this.attendance = integer;

        var substitutions = []
            .concat(this.guestTeam.substitutions)
            .concat(this.homeTeam.substitutions);
        angular.forEach(substitutions, function(o) {
            o && (o['type'] = 'Substitution');
        });
        this.events = []
            .concat(substitutions)
            .concat(this.cards)
            .concat(this.goals);
    };
    return Match;
});

angular.module('MotainAppModel').factory('Group', function (Match) {
    function Group(data) {
        // Note: the API 1.0 sometimes uses different names for the same attributes inside of an object,
        // instead of trying to guess how to unserialize it, we first copy every property, then we run some tests on
        // the properties we know we want normalized
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

        this.matches = (function (matches) {
            var m = [];
            for (var i = 0; i < matches.length; i++) {
                m.push(new Match(matches[i]));
            }
            return m;
        }(this.matches));
        this.id = this.groupId;
        this.name = this.groupName;
    }

    return Group;
});

angular.module('MotainAppModel').factory('Kickoff', function (Group) {
    function Kickoff(data) {
        // Note: the API 1.0 sometimes uses different names for the same attributes inside of an object,
        // instead of trying to guess how to unserialize it, we first copy every property, then we run some tests on
        // the properties we know we want normalized
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

        this.kickoffAt = this.kickoff;
        this.groups = (function (groups) {
            var g = [];
            for (var i = 0; i < groups.length; i++) {
                g.push(new Group(groups[i]));
            }
            return g;
        })(this.groups);
    }

    return Kickoff;
});


angular.module('MotainAppModel').factory('Ticker', function () {
    function Ticker(data) {
        this.events = data.comments;
        this.homeTeamId = data.meta.teamHomeIds[0];
        this.guestTeamId = data.meta.teamAwayIds[0];
    }

    return Ticker;
});

angular.module('MotainAppModel').factory('Quote', function () {
    function Quote(data) {
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }
    };

    return Quote;
});

angular.module('MotainAppModel').factory('User', function() {
    function User(data) {
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }
    }

    return User;
});

angular.module('MotainAppModel').factory('Live', function (Kickoff) {
    function Live(data) {
        // Note: the API 1.0 sometimes uses different names for the same attributes inside of an object,
        // instead of trying to guess how to unserialize it, we first copy every property, then we run some tests on
        // the properties we know we want normalized
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }
        this.kickoffs = (function (kickoffs) {
            var k = [];
            for (var i = 0; i < kickoffs.length; i++) {
                k.push(new Kickoff(kickoffs[i]));
            }
            return k;
        }(this.kickoffs));
    };

    return Live;
});

angular.module('MotainAppModel').factory('Competition', function () {
    function Competition(data) {
        // Note: the API 1.0 sometimes uses different names for the same attributes inside of an object,
        // instead of trying to guess how to unserialize it, we first copy every property, then we run some tests on
        // the properties we know we want normalized
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

        this.id = this.id || data.competitionId;
        this.name = this.name || data.competitionName;
    }

    return Competition;
});

angular.module('MotainAppModel').factory('Knockout', function () {
    function Knockout(data) {
        // Note: the API 1.0 sometimes uses different names for the same attributes inside of an object,
        // instead of trying to guess how to unserialize it, we first copy every property, then we run some tests on
        // the properties we know we want normalized
        for (var prop in data) {
            if (data.hasOwnProperty(prop)) {
                this[prop] = data[prop];
            }
        }

    }

    return Knockout;
});

angular.module('MotainAppModel').service('ApplicationConfigFileMapper', function (Competition, $http, $q, CONFIG) {
    function ApplicationConfigFileMapper() {
    }

    ApplicationConfigFileMapper.prototype.find = function (lang) {

        var d = $q.defer();
        var url = CONFIG.resources.config_file.replace(/:lang/, lang);

        successFn = function (response) {
            var data = response;

            data.sortedCompetitions = {};
            for (var i = 0, l = data.competitions.length; i < l; i++) {
                var competition = new Competition(data.competitions[i]);
                delete data.competitions[i];
                data.sortedCompetitions[competition.id] = competition;
            }

            d.resolve(data);
        };

        $http.get(url)
            .success(successFn)
            .error(function () {
                d.reject(arguments);
            });

        return d.promise;
    };

    return new ApplicationConfigFileMapper();
});

angular.module('MotainAppModel').factory('ApplicationConfigFileLoader', function ($q, $rootScope, UserSession, ApplicationConfigFileMapper, CONFIG) {

    var self = {};

    self.configFiles = [];
    self.user = undefined;

    self.getConfigFile = function () {
        self.isAuthenticated = UserSession.isAuthenticated;
        if (UserSession.isAuthenticated) {
            self.user = UserSession.getUser();
        } else {
            self.user = null;
        }
        var locale = UserSession.getCurrentLang();
        if (!self.configFiles[locale]) {
            var d = $q.defer();
            ApplicationConfigFileMapper.find(locale).then(
                function (data) {
                    // the locale is supported, we found the matching config file
                    d.resolve(data);
                }, function () {
                    // the locale is not supported, we didn't find the matching config file so we ask for the default one
                    ApplicationConfigFileMapper.find(CONFIG.default_lang).then(
                        function (data) {
                            d.resolve(data);
                        }, function () {
                            // TODO handle this case (network error?)
                        })
                });
            self.configFiles[locale] = d.promise;
        }
        return self.configFiles[locale];
    };

    return self;
});

angular.module('MotainAppModel').service('KnockoutMapper', function ($http, $q, Team, CONFIG) {

    function KnockoutMapper() {
    }

    KnockoutMapper.prototype.get = function (competitionId,  teamId) {

        var url = CONFIG.api.knockout;
        url = url.replace(':competitionId', competitionId);
        url = url.replace(':teamId', teamId);

        var d = $q.defer();
        $http.get(url).success(function (response) {
            return d.resolve(response.data);
        }).error(function () {
            return d.reject(arguments);
        });

        return d.promise;
    };

    return new KnockoutMapper();
});

angular.module('MotainAppModel').service('LiveMapper', function ($http, $q, Live, CONFIG) {
    function LiveMapper() {
    }

    LiveMapper.prototype.find = function (params) {
        var p = angular.copy(CONFIG.api.live_feed_default_params);
        angular.extend(p, params);

        var d = $q.defer();
        $http.get(CONFIG.api.live_feed, {params: p}).success(function (response) {
            var data = response;
            d.resolve(new Live(data));
        }).error(function () {
            d.reject(arguments);
        });

        return d.promise;
    };

    return new LiveMapper();
});

angular.module('MotainAppModel').service('MatchMapper', function ($http, $q, Match, CONFIG) {
    function MatchMapper() {
    }

    MatchMapper.prototype.find = function (params) {
        var url = CONFIG.api.match_feed
            .replace(':lang', params.lang)
            .replace(':competitionId', params.competitionId)
            .replace(':seasonId', params.seasonId)
            .replace(':matchId', params.matchId);
        var d = $q.defer();
        $http.get(url).success(function (response) {
            var data = angular.copy(response.match);
            var m = angular.extend(data, response.meta);
            return d.resolve(new Match(m));
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    MatchMapper.prototype.findScores = function (competitionId, seasonId, teamId, lang, useCache) {
        if (useCache === undefined) {
            useCache = true;
        } else {
            useCache = !!useCache;
        }

        var d = $q.defer();
        var url = CONFIG.api.team_score
            .replace(':lang', lang)
            .replace(':competitionId', competitionId)
            .replace(':seasonId', seasonId)
            .replace(':teamId', teamId);
        $http.get(url, {cache: useCache}).success(function (response) {
            var matches = [];
            response.seasonScores = _.forEach(response.seasonScores, function (match) {
                matches.push(new Match(match));
            });
            return d.resolve(matches);
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    return new MatchMapper();
});

angular.module('MotainAppModel').factory('NetworkRequestsCounter', function ($rootScope, EventEnum) {

    var self = {};

    self.count = 0;

    self.add = function (promise) {
        self.plus();
        promise.then(function() {
            self.minus();
        });
    };

    self.areDone = function() {
        return (self.count == 0);
    };

    self.plus = function() {
        self.count += 1;
        $rootScope.$broadcast(EventEnum.NETWORK_REQUEST_STARTED, self.count);
    };

    self.minus = function() {
        self.count -= 1;
        $rootScope.$broadcast(EventEnum.NETWORK_REQUEST_FINISHED, self.count);
    };

    return self;
});

angular.module('MotainAppModel').service('QuotesMapper', function ($http, $q, Quote, CONFIG) {
    function QuotesMapper() {
    }

    QuotesMapper.prototype.find = function (matchId) {
        var url = CONFIG.api.quotes;
        url = url.replace(':matchId', matchId);

        var d = $q.defer();
        $http.get(url).success(function (response) {
            var data = angular.copy(response.quotes);
            var quotes = [];
            for (var i = 0; i < data.length; i++) {
                var q = angular.extend(data[i], response.meta);
                quotes.push(new Quote(q));
            }
            d.resolve(quotes);
        }).error(function () {
                d.reject(arguments);
            });

        return d.promise;
    };

    return new QuotesMapper();
});

angular.module('MotainAppModel').service('SectionMapper', function ($http, $q, Team, Matchday, CONFIG) {

    function SectionMapper() {
    }

    function _cache(useCache) {
        if (useCache === undefined) {
            useCache = true;
        } else {
            useCache = !!useCache;
        }
        return useCache;
    }

    SectionMapper.prototype.getKnockout = function (competitionId,  seasonId) {

        var url = CONFIG.api.knockout;
        url = url.replace(':competitionId', competitionId);
        url = url.replace(':seasonId', seasonId);

        var d = $q.defer();
        $http.get(url).success(function (response) {
            return d.resolve(response.data);
        }).error(function () {
            return d.reject(arguments);
        });

        return d.promise;
    };

    SectionMapper.prototype.findTeams = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();

        var url = CONFIG.api.teams_overview.replace(':lang', lang).replace(':competitionId', competitionId).replace(':seasonId', seasonId);
        $http.get(url, {cache: _cache(useCache)}).success(function (response) {

            var teams = [];
            response.seasonScores = _.forEach(response.teams, function (team) {
                teams.push(new Team(team));
            });

            return d.resolve(teams);
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    SectionMapper.prototype.findStandings = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();

        var url = CONFIG.api.teams_standings.replace(':lang', lang).replace(':competitionId', competitionId).replace(':seasonId', seasonId);
        $http.get(url, {cache: _cache(useCache)}).success(function (response) {

            var groups = [];
            _.forEach(response.groups, function (group) {
                var rankings = [];
                _.forEach(group.ranking, function (ranking) {
                    ranking.team = new Team(ranking.team);
                    rankings.push(ranking);
                });
                groups.push({
                    groupName: group.groupName,
                    rankings: rankings
                });
            });

            return d.resolve(groups);
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    SectionMapper.prototype.findMatchdays = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();

        var url = CONFIG.api.matchdays_feed
            .replace(':lang', lang)
            .replace(':competitionId', competitionId)
            .replace(':seasonId', seasonId);

        $http.get(url, {cache: _cache(useCache)}).success(function (response) {
            var matchdays = [];
            _.forEach(response.matchdays, function (matchday) {
                var m = angular.extend(matchday, response.meta);
                matchdays.push(new Matchday(m));
            });
            return d.resolve(matchdays);
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    SectionMapper.prototype.findMatchdayById = function (competitionId, seasonId, matchdayId, lang, useCache) {
        var d = $q.defer();

        var url = CONFIG.api.matchday_feed
            .replace(':lang', lang)
            .replace(':competitionId', competitionId)
            .replace(':seasonId', seasonId)
            .replace(':matchdayId', matchdayId);

        $http.get(url, {cache: _cache(useCache)}).success(function (response) {
            var matchday = angular.extend({}, response.meta);
            matchday.kickoffs = response.kickoffs;
            return d.resolve(new Matchday(matchday));
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    // common method to get the statistics feed for a season
    // https://motainteam.atlassian.net/wiki/display/MIDWARE/B.+iLiga+API#B.iLigaAPI-2.10StatisticFeeds
    findStat = function (competitionId, seasonId, statType, lang, useCache) {
        var d = $q.defer();

        var url = CONFIG.api.season_stats
            .replace(':lang', lang)
            .replace(':competitionId', competitionId)
            .replace(':seasonId', seasonId)
            .replace(':statType', statType);

        $http.get(url, {cache: _cache(useCache)}).success(function (response) {
            return d.resolve(response);
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    SectionMapper.prototype.findStatsScorers = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();
        findStat(competitionId, seasonId, 'scorer', lang, useCache).then(function (stat) {
            return d.resolve(stat.scorer);
        });
        return d.promise;
    };

    SectionMapper.prototype.findStatsAssists = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();
        findStat(competitionId, seasonId, 'assist', lang, useCache).then(function (stat) {
            return d.resolve(stat.assist);
        });
        return d.promise;
    };

    SectionMapper.prototype.findStatsGoalsAssits = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();
        findStat(competitionId, seasonId, 'goalsassist', lang, useCache).then(function (stat) {
            return d.resolve(stat.goalsassist);
        });
        return d.promise;
    };

    SectionMapper.prototype.findStatsYellowCards = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();
        findStat(competitionId, seasonId, 'yellowcards', lang, useCache).then(function (stat) {
            return d.resolve(stat.yellowcards);
        });
        return d.promise;
    };

    SectionMapper.prototype.findStatsRedCards = function (competitionId, seasonId, lang, useCache) {
        var d = $q.defer();
        findStat(competitionId, seasonId, 'redcards', lang, useCache).then(function (stat) {
            return d.resolve(stat.redcards);
        });
        return d.promise;
    };

    return new SectionMapper();
});

angular.module('MotainAppModel').service('SwipeAndSlide', function () {

    var tabs, columns, tabs_config, columns_config, nr_of_tabs, simulate_touch;

    // Set to true to make it possible to test swiping in a desktop browser
    simulate_touch = false;

    // Utility function for setting the active tab on swipe and tap using the css class active
    function setActiveTab(swiper_tabs, index) {
        angular.element(swiper_tabs.slides).removeClass('active');
        angular.element(swiper_tabs.slides[index]).addClass('active');
    }

    // Get the number of tabs to be used in the tabs config so that the tabs will scale properly for screen sizes.
    nr_of_tabs = document.getElementById('swiper-tabs').getElementsByTagName('a').length;


    // Creating the Swiper configuration here to make it easier to handle runtime configuraion changes.
    // Swiper configuration for tabs
    // Adding the onSlideClick here instead of using the new API callback since it does not work for onSlideClick
    tabs_config = {
        mode: 'horizontal',
        simulateTouch: simulate_touch,
        slidesPerView: nr_of_tabs,
        centeredSlides: false,
        calculateHeight: true,
        centeredSlides: false,
        initialSlide: 0,
        onSlideClick: function(swiperTabs) {
            setActiveTab(swiperTabs, swiperTabs.clickedSlideIndex)
            swiperTabs.swipeTo(swiperTabs.clickedSlideIndex);
            swiperTabs.columns.swipeTo(swiperTabs.clickedSlideIndex);
        }
    }

    // Swiper configuration for columns.
    columns_config = {
        mode: 'horizontal',
        simulateTouch: simulate_touch,
        slidesPerView: 'auto',
        initialSlide: 0
    }

    if(!window.matchMedia("(min-width: 568px)").matches) {
        tabs_config.slidesPerView = 3;
        tabs_config.centeredSlides = true;
        tabs_config.initialSlide = 1;
        columns_config.slidesPerView = 1;
        columns_config.initialSlide = 1;
    }

    // Create the tabs and columns objects
    columns = new Swiper('#swiper-columns', columns_config);
    tabs = new Swiper('#swiper-tabs', tabs_config);

    // Piggybacking the tabs and columns to each other instead of passing them around.
    tabs.columns = columns;
    columns.tabs = tabs;

    // Columns callback for setting the correct active tab
    columns.addCallback('SlideNext', function(swiperColumns) {
        setActiveTab(swiperColumns.tabs, swiperColumns.activeIndex);
        swiperColumns.tabs.swipeTo(swiperColumns.activeIndex);
    });

    columns.addCallback('SlidePrev', function(swiperColumns) {
        setActiveTab(swiperColumns.tabs, swiperColumns.activeIndex);
        swiperColumns.tabs.swipeTo(swiperColumns.activeIndex);
    });

    // Sets the initial tab as active using the css class active
    setActiveTab(tabs, tabs_config.initialSlide);

    return {tabs: tabs, columns: columns};
});
angular.module('MotainAppModel').service('TeamMapper', function ($http, $q, Team, CONFIG) {

    function TeamMapper() {
    }

    TeamMapper.prototype.findOneById = function (id, useCache) {
        if (useCache === undefined) {
            useCache = true;
        } else {
            useCache = !!useCache;
        }

        var d = $q.defer();

        var url = CONFIG.api.team_feed.replace(':teamId', id);
        $http.get(url, {cache: useCache}).success(function (response) {
            var team = new Team(response.data.team);
            return d.resolve(team);
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    return new TeamMapper();
});

angular.module('MotainAppModel').service('TeamStatsMapper', function ($http, $q, TeamSeason, CONFIG) {

    function TeamStatsMapper() {
    }

    TeamStatsMapper.prototype.findStatsForSeason = function (teamId, seasonId, useCache) {
        if (useCache === undefined) {
            useCache = true;
        } else {
            useCache = !!useCache;
        }

        var d = $q.defer();

        var url = CONFIG.api.team_stats.replace(':teamId', teamId).replace(':seasonId', seasonId);
        $http.get(url, {cache: useCache}).success(function (response) {
            var teamSeason = new TeamSeason(response.data);
            return d.resolve(teamSeason);
        }).error(function () {
                return d.reject(arguments);
            });

        return d.promise;
    };

    return new TeamStatsMapper();
});

angular.module('MotainAppModel').service('TickerMapper', function ($http, $q, Ticker, CONFIG) {
    function TickerMapper() {
    }

    TickerMapper.prototype.find = function (params) {
        var url = CONFIG.api.ticker_feed;
        url = url.replace(':lang', 'en');
        url = url.replace(':competitionId', params.competitionId);
        url = url.replace(':seasonId', params.seasonId);
        url = url.replace(':matchId', params.matchId);
        //url = url.replace(':matchdayId', params.matchdayId);
        url = url.replace(':matchdayId', 0);

        var d = $q.defer();
        $http.get(url).success(function (response) {
            if (!response.comments || response.comments.length == 0) {
                d.reject();
                return;
            }
            var data = angular.copy(response);
            d.resolve(new Ticker(data));
        }).error(function () {
                d.reject(arguments);
            });

        return d.promise;
    };

    return new TickerMapper();
});

angular.module('MotainAppModel').service('UserMapper', function(User, $http, $q, CONFIG, UserSession) {

    var meUser = null;

    function UserMapper() {}

    UserMapper.prototype.getMe = function(useCache) {
        if (useCache === undefined) {
            useCache = true;
        } else {
            useCache = !!useCache;
        }

        var d = $q.defer();

        $http.get(CONFIG.api.user_root + '/v1/users/me', {
            params: { access_token: UserSession.accessToken },
            cache: useCache
        }).success(function(response) {
            if (angular.isObject(meUser) && meUser.id === response.data.users.id) {
                for (var prop in response.data.users) {
                    if (response.data.hasOwnProperty(prop)) {
                        meUser[prop] = data[prop];
                    }
                }
            } else {
                meUser = new User(response.data.users);
            }

            return d.resolve(meUser);
        }).error(function() {
            return d.reject(arguments);
        });

        return d.promise;
    };


    UserMapper.prototype.deleteToken = function() {
        var d = $q.defer();

        $http['delete'](CONFIG.api.user_root + '/v1/users/token', {params: { access_token: UserSession.accessToken }})
        .success(function(response) {
            return d.resolve(response);
        }).error(function() {
            return d.reject(arguments);
        });

        return d.promise;
    };
    return new UserMapper();
});

angular.module('MotainApp').config(function ($translateProvider) {

    function transformTranslationCode(lang) {

        if(lang != 'pt-PT' && lang != 'pt-BR'){
            var aux = lang.split('-');
            lang = aux[0];
        }
        return lang;
    }

    $translateProvider.useStaticFilesLoader({
      prefix: 'i18n/',
      suffix: '/strings.json'
    }); 

    var userLang = (navigator.language || navigator.userLanguage);
    userLang = transformTranslationCode(userLang);
    $translateProvider.preferredLanguage(userLang);

});

/**
 * Things to remember:
 *   - there might be user set, but it doesn't mean that he is authenticated
 *   - isAuthentiacted expects user to be injected before invoking.
 *   - basic usage is to invoke functions in this order:
 *     - setAuthInfo(...) - data used in UserMapper when requesting restricted api endpoints
 *     - setUser(...) - user object can be fetched using methods from UserMapper
 *     - isAuthenticated(bool) - set information if current user is authenticated
 *
 */
angular.module('MotainApp').service('UserSession', function($rootScope, $log, EventEnum, LocaleService) {
    /**
     * @todo:
     *  - handle expiration time of token - do automatic call for new
     *  - handle remember me, if it is possible
     */

    function UserSession() {
        /**
         * @type {Boolean}
         */
        this.isAuthenticated = null;

        /**
         * @type {String}
         */
        this.accessToken = null;

        /**
         * @type {String}
         */
        this.refreshToken = null;

        /**
         * @type {Date}
         */
        this.expiresAt = null;

        /**
         * @type {(User|null)}
         */
        this.user = null;

        /**
         * @type {String}
         */
        this.defaultLocale = LocaleService.getDefaultLang();

        /**
         * @type {String}
         */
        this.currentLang = LocaleService.getBrowserLocale().split('-')[0];
    }

    /**
     * @param {String} accessToken
     * @param {String} refreshToken
     * @param {(Number|String)} accessExpiresIn - value in seconds, when access token expires.
     */
    UserSession.prototype.setAuthInfo = function(accessToken, refreshToken, accessExpiresIn) {
        if (angular.isString(accessExpiresIn)) {
            accessExpiresIn = parseInt(accessExpiresIn, 10);
        }
        this.accessToken = accessToken;
        this.refreshToken = refreshToken;
        this.expiresAt = new Date(Date.now() + accessExpiresIn * 1000);
    };

    /**
     * @param {User}
     */
    UserSession.prototype.setUser = function(user) {
        $log.debug('user set: ', user);
        this.user = user;
        this.currentLang = this.user.language || this.defaultLocale;
    };

    UserSession.prototype.getCurrentLang = function() {
        return this.currentLang;
    };

    UserSession.prototype.setCurrentLang = function(lang) {
        if (this.currentLang !== lang) {
            this.currentLang = lang;

            $rootScope('user:langChanged');
        }
    };

    /**
     * @param {Boolean} isAuthenticated
     * @fires UserSession#auth:authentication
     */
    UserSession.prototype.setIsAuthenticated = function(isAuthenticated) {
        this.isAuthenticated = !!isAuthenticated;
        $rootScope.$broadcast(EventEnum.USER_AUTHENTICATION_STATE_CHANGED, this.isAuthenticated);
    };

    UserSession.prototype.getUser = function() {
        if (!angular.isObject(this.user)) {
            throw new Error("this.user is not an object. Did You invoke this.setUser before?");
        }
        return this.user;
    };

    UserSession.prototype.clean = function() {
        this.accessToken = null;
        this.refreshToken = null;
        this.expiresAt = null;
        this.user = null;

        this.setIsAuthenticated(false);
    };

    return new UserSession();
});

//# sourceMappingURL=all.js.map