function Receipts() {
	this.storage_id = 'receipts';
	this.init();
}

Receipts.prototype = {
	templates: {
		unknown: 'We could not verify your purchase receipt yet.<br/>Please turn on your internet connection to prevent this message from showning.',

		outofdate: 'We could not verify your purchase receipt for a while now.<br/>Please go online to prevent this message from showning.',

		needsInstall: 'This app needs to be installed. Please visit the <a href="https://marketplace.firefox.com/app/orbland" target="_blank">Firefox Marketplace</a> to get it.',
		
		invalid: 'Sorry, we could not verify your purchase receipt. If you bought this app at the Firefox Marketplace, please try reinstalling it, or <a href="mailto:hansen@prokapi.com">contact our support</a>.',

		mozAppsNotSupported: 'Sorry, your browser or device does not support the Marketplace Apps system.',
  },

	message_box: '<form id="message" role="dialog" data-type="confirm"><section ><h1></h1><p id="message_text"></p></section><menu><button style="width:100%" class="recommend" id="message_ok">OK</button></menu></form>',


	init: function() {  
		this.storage = new Storage(this.storage_id);
		
		this.initial_run = this.storage.get('initial_run');
		if (!this.initial_run) {
			this.setPersist('initial_run', Date.now());
		}

		//settings
		this.loadCheckSettings();

		this.verifier_settings = {
			//productURL: "https://marketplace.firefox.com/apps/flashlight-plus",
			installs_allowed_from: "*",//https://marketplace.firefox.com",
			typsAllowed: ['purchase-receipt', 'developer-receipt', 'reviewer-receipt', 'test-receipt']
		};


		//initialization
		this.verifier = new mozmarket.receipts.Verifier(this.verifier_settings);
		this.message = this.templates.unknown;
		this.status = 'unknown';
		this.last_checked = Date.now() - this.check_interval;
		this.last_valid = Date.now() - this.check_interval;
		this.last_reminded = Date.now();
		
		//try to get from storage
		var message = this.storage.get('message');
		if (message != null) {
			this.message = message;
		}
		var status = this.storage.get('status');
		if (status != null) {
			this.status = status;
		}
		var last_checked = this.storage.get('last_checked');
		if (last_checked != null) {
			this.last_checked = last_checked;
		} 
		var last_valid = this.storage.get('last_valid');
		if (last_valid != null) {
			this.last_valid = last_valid;
		} 
		var last_reminded = this.storage.get('last_reminded');
		if (last_reminded != null) {
			this.last_reminded = last_reminded;
		} 
	
	},
	
	loadCheckSettings: function() {
		var one_minute = 1000*60;
		var one_hour = one_minute*60;
		var one_day = one_hour*24;
		var one_week = one_day*7;
		var one_month = one_day*30;

		/*
		this.check_interval - periodically check in state 'valid', other states always check
		this.silent_grace_period - try to check silently when in state 'outofdate'
		this.reminding_grace_period - try to check and remind in states 'unknown' and 'outofdate' (after silent period)
		this.remind_interval - message interval during reminding_grace_period
		*/

		if (Date.now() - this.initial_run < one_hour) {
			this.check_interval = 1*one_minute; //once per 10 minutes
			this.silent_grace_period = one_hour; //=> no message in first hour if valid once
			this.reminding_grace_period = one_hour;
			this.remind_interval = 20*one_minute; 
			
			this.check = true;
			this.remind = true;
			this.block_invalid = true;
			this.block_unknown = false;
		
		} else if (Date.now() - this.initial_run < one_day) {
			this.check_interval = one_hour; //once per hour
			this.silent_grace_period = one_day; //=> no message on first day if valid once
			this.reminding_grace_period = one_day;
			this.remind_interval = one_hour; 
			
			this.check = true;
			this.remind = true;
			this.block_invalid = true;
			this.block_unknown = false;
		
		} else if (Date.now() - this.initial_run < one_month) {
			this.check_interval = one_hour; //once per hour
			this.silent_grace_period = one_month; //=> no message in first month if valid once
			this.reminding_grace_period = one_month; 
			this.remind_interval = one_hour;
			
			this.check = true;
			this.remind = true;
			this.block_invalid = true;
			this.block_unknown = false;
		
		} else {
			this.check_interval = one_hour; //once per hour
			this.silent_grace_period = one_month; //=> no message if one successfull validation per month (after valid once)
			this.reminding_grace_period = one_month; 
			this.remind_interval = one_hour;
			
			this.check = true;
			this.remind = true;
			this.block_invalid = true;
			this.block_unknown = false;
		}
	},

	
	update: function() {
		this.verifier.verify(this.updateCallback.bind(this));
		this.setPersist('last_checked', Date.now());
	},

	
	updateCallback: function(verifier) {
		//debug
		console.log(verifier.state);
		//alert(': ' + verifier.state);

		var template = "";
		var status = this.status;
	
		if (verifier.state instanceof verifier.states.VerificationIncomplete) {
			//do nothing

		} else if (verifier.state instanceof verifier.states.OK) {
			//everything's fine (receipt gets cached after refund window, check for stale cache if neccessary)
			status = 'valid';

		} else if (verifier.state instanceof verifier.states.NetworkError) {
			if (status === 'valid' || status === 'outofdate') {
				//valid status is outdated only
				status = 'outofdate';
				template = this.templates.outofdate;
			} else if (status === 'unknown') {
				status = 'unknown';
				template = this.templates.unknown;
			}

		} else if (verifier.state instanceof verifier.states.NeedsInstall) {
			template = this.templates.needsInstall;
			status = 'invalid';

		} else if (verifier.state instanceof verifier.states.NoValidReceipts) {
			//template = this.handleReceiptError(verifier, verifier.receiptErrors);
			template = this.templates.invalid;
			status = 'invalid';

		} else if (verifier.state instanceof verifier.states.MozAppsNotSupported) {
			template = this.templates.mozAppsNotSupported;
			status = 'invalid';

		} else if (verifier.state instanceof verifier.states.InternalError) {
			//do nothing

		} else {
			//do nothing
		
		}
	
		this.setPersist('status', status);	
		if (template != "") {
			this.setPersist('message', template);
		}
		if (status === 'valid') {
			this.setPersist('last_valid', Date.now());
		}

	},


	validate: function() {
		var now = Date.now();
		var valid = true;
		
		this.loadCheckSettings();

		console.log("*RECEIPT status " + this.status + ", next check in " + (this.last_checked - now + this.check_interval)/1000 + "s");
	
		if (this.status === 'invalid') {
			this.update();
			if (this.block_invalid) {
				//show message and block
				console.log('*RECEIPT ... blocking');
				this.showMessage();
				valid = false;
			}
			
		} else if (this.status === 'unknown') {
			this.update();
			if (now > this.last_valid + this.reminding_grace_period) {
				//reminding grace period expired
				console.log('*RECEIPT ... beyond grace period');
				if (this.block_unknown) {
					//show message and block
					console.log('*RECEIPT ... blocking');
					this.showMessage();
					valid = false;
				} else {
					//show message only
					this.showMessage();
				}

			} else {
				//within reminding grace period
				console.log('*RECEIPT ... within reminding grace period');
				if (this.remind) {
					console.log("*RECEIPT ... next message in " + (this.last_reminded - now + this.remind_interval)/1000 + "s");
				}
				if (this.remind && this.last_reminded + this.remind_interval <= now) {
					this.showMessage();
				}
			}
			
		} else if (this.status === 'outofdate') {
			this.update();
			if (this.last_valid + this.reminding_grace_period <= now) {
				//silent & reminding grace period expired
				console.log('*RECEIPT ... beyond grace period');
				if (this.block_unknown) {
					//show message and block
					console.log('*RECEIPT ... blocking');
					this.showMessage();
					valid = false;
				} else {
					//show message only
					this.showMessage();
				}

			} else if (this.last_valid + this.silent_grace_period <= now) {
				//within reminding grace period
				console.log('*RECEIPT ... within reminding grace period');
				if (this.remind) {
					console.log("*RECEIPT ... next message in " + (this.last_reminded - now + this.remind_interval)/1000 + "s");
				}
				if (this.remind && this.last_reminded + this.remind_interval <= now) {
					this.showMessage();
				}
			} else {
				//within silent grace period
				console.log('*RECEIPT ... whithin silent grace period');
				//do nothing
			}

		} else if (this.status === 'valid') {
			if (this.check && this.last_checked + this.check_interval <= now) {
				console.log('*RECEIPT updating');
				this.update();
			}
		}

		return valid;
	},

	
	showMessage: function() {
		this.setPersist('last_reminded', Date.now());

		//check as fast as possible after message show
		this.setPersist('last_checked', (Date.now() - this.check_interval));

		//TODO: limits functionality to obviously invalid receipts
		if (this.status === 'invalid') {
		Zepto('#screen').append(this.message_box);
			Zepto('#message_text').html(this.message);
			Zepto('#message_ok').on('click', function(event) {
				event.preventDefault();
				Zepto('#message_ok').off();
				Zepto('#message').remove();
			});
		}
	},


	//**********************************

	handleReceiptError: function (verifier, error) {
    var template;
    this.error = error;
    
		if (error instanceof verifier.errors.Refunded) {
      template = this.templates.refunded;

    } else if (error instanceof verifier.errors.InvalidReceiptIssuer) {
      template = this.templates.invalidReceiptIssuer;

    } else if (error instanceof verifier.errors.InvalidFromStore) {
      template = this.templates.invalidFromStore;

    } else if (error instanceof verifier.errors.ReceiptFormatError) {
      template = this.templates.receiptFormatError;

    } else {
      template = this.templates.genericError;
    }
    
		return template;
  },


	setPersist: function(name, value) {
		this[name] = value;
		this.storage.set(name, value);
	}


}
/*
	*** verifier.states ***
	VerificationIncomplete: 
	this is the state until the verification actually completes.

	OK: 
	everything went okay!	
	-OKCache: 
	subclass of OK; everything went okay, and we used some cached results to verify this.
	-OKStaleCache: 
	subclass of OK; everything didn't really go okay, there was some network error, but we had previously cached results of a past verification. These cached items were too old, but were an acceptable fallback. Or not acceptable, you can check for this state. The network errors will still be present in verifier.receiptErrors.

	NetworkError: 
	some network error occurred that kept validation from completing. That is, a receipt seemed okay but we weren't able to contact the server to verify if. This will happen when the user agent is offline.
	-ServerError: 
	subclass of NetworkError; the server did something wrong. This might be an invalid response from the server, or a wifi login in the way, or the server is down, etc. Like a network error, it's not the user's fault!

	NeedsInstall: 
	an error that indicates the application needs to be installed.
	-NoReceipts: 
	a subclass of NeedsInstall; the application is installed, but has no receipts. This would probably be the result of a self-install or free install.
	-NotInstalled: 
	a subclass of NeedsInstall; the application is simply not installed.

	NoValidReceipts: 
	the application is installed, and has receipts, but none of the receipts are valid. The receipts may be syntactically invalid, may be from a store that is not allowed, may be rejected by the store as invalid, or may be refunded. Look to verifier.receiptErrors for details.

	MozAppsNotSupported: 
	the navigator.mozApps API is not supported by this client. The app can't be "installed" on this browser.
	
	InternalError: 
	something went wrong with the verifier itself or the navigator.mozApps API. This of course shouldn't happen; please report any such errors.
	-MozAppsError: 
	subclass of InternalError; this is generally an error with the navigator.mozApps.getSelf() call.
	-VerifierError: 
	subclass of InternalError; an exception somewhere in the verifier code.


	** verifier.errors **
	InvalidFromStore: 
	the store responded that the receipt is invalid. This may mean the store has no record of the receipt, doesn't recognize the signature, or some other state.

	ReceiptExpired: 
	the store responded that the receipt has expired. This is generally a recoverable error, in that the receipt can be refreshed once expired. This refreshing has not yet been implemented.

	Refunded: 
	the store reports that the payment was refunded.

	InvalidReceiptIssuer: 
	the receipt was issued by a store not listed in your installs_allowed_from list.

	ConnectionError: 
	subclass of verifier.states.NetworkError; happens when the connection to the server fails.

	RequestTimeout: 
	a subclass of verifier.states.NetworkError; the request timed out. You can set verifier.requestTimeout to a millisecond value to control this.

	ServerStatusError: 
	a subclass of verifier.states.ServerError; the server responded with a non-200 response.

	InvalidServerResponse: 
	a subclass of verifier.states.ServerError; the server responded with a non-JSON response, or a JSON response that didn't contain a valid status.

	ReceiptFormatError: 
	the receipt itself is invalid. It might be badly formatted, or is missing required properties.

	ReceiptParseError: 
	subclass of ReceiptFormatError; the receipt is not valid JWT. This should only happen if the store that issued the receipt is simply broken, or the receipt was corrupted.

*/

