/*
 * OTPWCalc - One time password challenge response calculator client
 * Copyright © 2013 Michael Schloh von Bennewitz <michael@schloh.com>
 * 
 * OTPWCalc is free software: you can redistribute it and/or modify
 * it under the terms of the European Union Public Licence, either
 * version 1.1 of the license, or (at your option) any later version.
 * 
 * OTPWCalc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the European Union Public License for more details.
 * 
 * You should have received a copy of the European Union Public
 * Licence along with OTPWCalc. If not, please refer to
 * <http://joinup.ec.europa.eu/software/page/eupl/>.
 * 
 * This file is part of project OTWPCalc, a one time password challenge
 * response calculator client and is found at http://otpwcalc.europalab.com/
 * 
 * otpalg.js: ECMA JavaScript implementation
 */

// <![CDATA[
function genotpmd4(secret, seed, n) {
    var t = seed.toString().toLowerCase() + secret;
    t = mdxfold(coremd4(str2binl(t), t.length * 8));
    for (var i = n; i > 0; --i) { t = mdxfold(coremd4(t, 64)); }
    return t;
}

function genotpmd5(secret, seed, n) {
    var t = seed.toString().toLowerCase() + secret;
    t = mdxfold(coremd5(str2binl(t), t.length * 8));
    for (var i = n; i > 0; --i) { t = mdxfold(coremd5(t, 64)); }
    return t;
}

function genotpsha1(secret, seed, n) {
    var t = seed.toString().toLowerCase() + secret;
    t = sha1fold(coresha1(str2binb(t), t.length * 8));
    for (var i = n; i > 0; --i) { t = sha1fold(coresha1(t, 64)); }
    t = invertendian(t, true);
    return t;
}

function genotprmd160(secret, seed, n) {
    var t = seed.toString().toLowerCase() + secret;
    t = rmd160fold(corermd160(str2binl(t), t.length * 8));
    for (var i = n; i > 0; --i) { t = rmd160fold(corermd160(t, 64)); }
    return t;
}

function genotpmultmd4(secret, seed, n, m) {
    var res = Array(); var lim = n - m + 1;
    var t = seed.toString().toLowerCase() + secret;
    t = mdxfold(coremd4(str2binl(t), t.length * 8));
    if (lim == 0) res[0] = t;
    for (var i = 1; i <= n; ++i) {
	t = mdxfold(coremd4(t, 64));
	if (i >= lim) res[i-lim] = t;
    }
    return res;
}

function genotpmultmd5(secret, seed, n, m) {
    var res = Array(); var lim = n - m + 1;
    var t = seed.toString().toLowerCase() + secret;
    t = mdxfold(coremd5(str2binl(t), t.length * 8));
    if (lim == 0) res[0] = t;
    for (var i = 1; i <= n; ++i) {
	t = mdxfold(coremd5(t, 64));
	if (i >= lim) res[i-lim] = t;
    }
    return res;
}

function genotpmultsha1(secret, seed, n, m) {
    var res = Array(); var lim = n - m + 1;
    var t = seed.toString().toLowerCase() + secret;
    t = sha1fold(coresha1(str2binb(t), t.length * 8));
    if (lim == 0) res[0] = invertendian(t, false);
    for (var i = 1; i <= n; ++i) {
	t = sha1fold(coresha1(t, 64));
	if (i >= lim) res[i-lim] = invertendian(t, false);
    }
    return res;
}

function genotpmultrmd160(secret, seed, n, m) {
    var res = Array(); var lim = n - m + 1;
    var t = seed.toString().toLowerCase() + secret;
    t = rmd160fold(corermd160(str2binl(t), t.length * 8));
    if (lim == 0) res[0] = t;
    for (var i = 1; i <= n; ++i) {
	t = rmd160fold(corermd160(t, 64));
	if (i >= lim) res[i-lim] = t;
    }
    return res;
}

function mdxfold(h) { return Array(h[0] ^ h[2], h[1] ^ h[3]); }

function sha1fold(h) {
    h = invertendian(h, true);
    return Array(h[0] ^ h[2] ^ h[4], h[1] ^ h[3]);
}

function rmd160fold(h) { return Array(h[0] ^ h[2] ^ h[4], h[1] ^ h[3]); }

function invertendian(a, inpl) {
    var t = inpl ? a : Array(a.length);
    for (var i = 0; i < a.length; ++i) {
	var t1 = (a[i] & 0xff) << 24;
	var t2 = ((a[i] >> 8) & 0xff) << 16;
	var t3 = ((a[i] >> 16) & 0xff) << 8;
	var t4 = (a[i] >> 24) & 0xff;
	t[i] = t1 | t2 | t3 | t4;
    }
    return t;
}

function arrtoboth(a) { return arrtosix(a) + " (" + arrtohex(a) + ")"; }

function arrtohex(a) {
    var s = "";
    for (var i = 0; i < 2; ++i) {
	for (var j = 0; j < 4; ++j) {
	    var t = (a[i] >> (8*j)) & 0xff;
	    t = t.toString(16).toUpperCase();
	    s += (t.length == 1) ? ('0' + t) : t; // 1 octet = 2 hex digits
	    if (j % 2 == 1) s += ' ';
	}
    }
    return s.substr(0, s.length-1); // drop the last space
}

function arrtosix(h) {
    var s = "";
    var parity = 0;
    for (var i = 0; i < 2; ++i) {
	for (var j = 0; j < 32; j += 2) {
	    parity += (h[i] >> j) & 0x3;
	}
    }
    var ind;
    ind = (h[0] & 0xff) << 3;
    ind |= (h[0] >> 13) & 0x7;
    s += words[ind] + " ";
    ind = ((h[0] >> 8) & 0x1f) << 6;
    ind |= (h[0] >> 18) & 0x3f;
    s += words[ind] + " ";
    ind = ((h[0] >> 16) & 0x3) << 9;
    ind |= ((h[0] >> 24) & 0xff) << 1;
    ind |= (h[1] >> 7) & 0x1;
    s += words[ind] + " ";
    ind = (h[1] & 0x7f) << 4;
    ind |= (h[1] >> 12) & 0xf;
    s += words[ind] + " ";
    ind = ((h[1] >> 8) & 0xf) << 7;
    ind |= (h[1] >> 17) & 0x7f;
    s += words[ind] + " ";
    ind = ((h[1] >> 16) & 0x1) << 10;
    ind |= ((h[1] >> 24) & 0xff) << 2;
    ind |= (parity & 0x03);
    s += words[ind];
    return s;
}
// ]]>
