// Built with IMPACT - impactjs.org

Number.prototype.map = function (istart, istop, ostart, ostop) {
    return ostart + (ostop - ostart) * ((this - istart) / (istop - istart));
};
Number.prototype.limit = function (min, max) {
    return Math.min(max, Math.max(min, this));
};
Number.prototype.round = function (precision) {
    precision = Math.pow(10, precision || 0);
    return Math.round(this * precision) / precision;
};
Number.prototype.floor = function () {
    return Math.floor(this);
};
Number.prototype.ceil = function () {
    return Math.ceil(this);
};
Number.prototype.toInt = function () {
    return (this | 0);
};
Array.prototype.erase = function (item) {
    for (var i = this.length; i--; i) {
        if (this[i] === item) this.splice(i, 1);
    }
    return this;
};
Array.prototype.random = function () {
    return this[(Math.random() * this.length).floor()];
};
Function.prototype.bind = function (bind) {
    var self = this;
    return function () {
        var args = Array.prototype.slice.call(arguments);
        return self.apply(bind || null, args);
    };
};
(function (window) {
    window.ig = {
        game: null,
        version: '1.16',
        global: window,
        modules: {},
        resources: [],
        ready: false,
        baked: false,
        nocache: '',
        ua: {},
        lib: 'lib/',
        _current: null,
        _loadQueue: [],
        _waitForOnload: 0,
        $: function (selector) {
            return selector.charAt(0) == '#' ? document.getElementById(selector.substr(1)) : document.getElementsByTagName(selector);
        },
        $new: function (name) {
            return document.createElement(name);
        },
        copy: function (object) {
            if (!object || typeof (object) != 'object' || object instanceof HTMLElement || object instanceof ig.Class) {
                return object;
            } else if (object instanceof Array) {
                var c = [];
                for (var i = 0, l = object.length; i < l; i++) {
                    c[i] = ig.copy(object[i]);
                }
                return c;
            } else {
                var c = {};
                for (var i in object) {
                    c[i] = ig.copy(object[i]);
                }
                return c;
            }
        },
        merge: function (original, extended) {
            for (var key in extended) {
                var ext = extended[key];
                if (typeof (ext) != 'object' || ext instanceof HTMLElement || ext instanceof ig.Class) {
                    original[key] = ext;
                } else {
                    if (!original[key] || typeof (original[key]) != 'object') {
                        original[key] = {};
                    }
                    ig.merge(original[key], ext);
                }
            }
            return original;
        },
        ksort: function (obj) {
            if (!obj || typeof (obj) != 'object') {
                return [];
            }
            var keys = [],
                values = [];
            for (var i in obj) {
                keys.push(i);
            }
            keys.sort();
            for (var i = 0; i < keys.length; i++) {
                values.push(obj[keys[i]]);
            }
            return values;
        },
        module: function (name) {
            if (ig._current) {
                throw ("Module '" + ig._current.name + "' defines nothing");
            }
            if (ig.modules[name] && ig.modules[name].body) {
                throw ("Module '" + name + "' is already defined");
            }
            ig._current = {
                name: name,
                requires: [],
                loaded: false,
                body: null
            };
            ig.modules[name] = ig._current;
            ig._loadQueue.push(ig._current);
            ig._initDOMReady();
            return ig;
        },
        requires: function () {
            ig._current.requires = Array.prototype.slice.call(arguments);
            return ig;
        },
        defines: function (body) {
            name = ig._current.name;
            ig._current.body = body;
            ig._current = null;
            ig._execModules();
        },
        addResource: function (resource) {
            ig.resources.push(resource);
        },
        setNocache: function (set) {
            ig.nocache = set ? '?' + Date.now() : '';
        },
        _loadScript: function (name, requiredFrom) {
            ig.modules[name] = {
                name: name,
                requires: [],
                loaded: false,
                body: null
            };
            ig._waitForOnload++;
            var path = ig.lib + name.replace(/\./g, '/') + '.js' + ig.nocache;
            var script = ig.$new('script');
            script.type = 'text/javascript';
            script.src = path;
            script.onload = function () {
                ig._waitForOnload--;
                ig._execModules();
            };
            script.onerror = function () {
                throw ('Failed to load module ' + name + ' at ' + path + ' ' + 'required from ' + requiredFrom);
            };
            ig.$('head')[0].appendChild(script);
        },
        _execModules: function () {
            var modulesLoaded = false;
            for (var i = 0; i < ig._loadQueue.length; i++) {
                var m = ig._loadQueue[i];
                var dependenciesLoaded = true;
                for (var j = 0; j < m.requires.length; j++) {
                    var name = m.requires[j];
                    if (!ig.modules[name]) {
                        dependenciesLoaded = false;
                        ig._loadScript(name, m.name);
                    } else if (!ig.modules[name].loaded) {
                        dependenciesLoaded = false;
                    }
                }
                if (dependenciesLoaded && m.body) {
                    ig._loadQueue.splice(i, 1);
                    m.loaded = true;
                    m.body();
                    modulesLoaded = true;
                    i--;
                }
            }
            if (modulesLoaded) {
                ig._execModules();
            } else if (!ig.baked && ig._waitForOnload == 0 && ig._loadQueue.length != 0) {
                var unresolved = [];
                for (var i = 0; i < ig._loadQueue.length; i++) {
                    var unloaded = [];
                    var requires = ig._loadQueue[i].requires;
                    for (var j = 0; j < requires.length; j++) {
                        var m = ig.modules[requires[j]];
                        if (!m || !m.loaded) {
                            unloaded.push(requires[j]);
                        }
                    }
                    unresolved.push(ig._loadQueue[i].name + ' (requires: ' + unloaded.join(', ') + ')');
                }
                throw ('Unresolved (circular?) dependencies. ' + "Most likely there's a name/path mismatch for one of the listed modules:\n" +
                    unresolved.join('\n'));
            }
        },
        _DOMReady: function () {
            if (!ig.modules['dom.ready'].loaded) {
                if (!document.body) {
                    return setTimeout(ig._DOMReady, 13);
                }
                ig.modules['dom.ready'].loaded = true;
                ig._waitForOnload--;
                ig._execModules();
            }
            return 0;
        },
        _boot: function () {
            if (document.location.href.match(/\?nocache/)) {
                ig.setNocache(true);
            }
            ig.ua.pixelRatio = window.devicePixelRatio || 1;
            ig.ua.viewport = {
                width: window.innerWidth,
                height: window.innerHeight
            };
            ig.ua.screen = {
                width: window.screen.availWidth * ig.ua.pixelRatio,
                height: window.screen.availHeight * ig.ua.pixelRatio
            };
            ig.ua.iPhone = /iPhone/i.test(navigator.userAgent);
            ig.ua.iPhone4 = (ig.ua.iPhone && ig.ua.pixelRatio == 2);
            ig.ua.iPad = /iPad/i.test(navigator.userAgent);
            ig.ua.android = /android/i.test(navigator.userAgent);
            ig.ua.firefox = /Firefox/i.test(navigator.userAgent);
            ig.ua.iOS = ig.ua.iPhone || ig.ua.iPad;
            ig.ua.mobile = ig.ua.iOS || ig.ua.android;
        },
        _initDOMReady: function () {
            if (ig.modules['dom.ready']) {
                return;
            }
            ig._boot();
            ig.modules['dom.ready'] = {
                requires: [],
                loaded: false,
                body: null
            };
            ig._waitForOnload++;
            if (document.readyState === 'complete') {
                ig._DOMReady();
            } else {
                document.addEventListener('DOMContentLoaded', ig._DOMReady, false);
                window.addEventListener('load', ig._DOMReady, false);
            }
        }
    };
    var initializing = false,
        fnTest = /xyz/.test(function () {
            xyz;
        }) ? /\bparent\b/ : /.*/;
    window.ig.Class = function () {};
    window.ig.Class.extend = function (prop) {
        var parent = this.prototype;
        initializing = true;
        var prototype = new this();
        initializing = false;
        for (var name in prop) {
            if (typeof (prop[name]) == "function" && typeof (parent[name]) == "function" && fnTest.test(prop[name])) {
                prototype[name] = (function (name, fn) {
                    return function () {
                        var tmp = this.parent;
                        this.parent = parent[name];
                        var ret = fn.apply(this, arguments);
                        this.parent = tmp;
                        return ret;
                    };
                })(name, prop[name])
            } else {
                prototype[name] = prop[name];
            }
        }

        function Class() {
            if (!initializing) {
                if (this.staticInstantiate) {
                    var obj = this.staticInstantiate.apply(this, arguments);
                    if (obj) {
                        return obj;
                    }
                }
                for (var p in this) {
                    this[p] = ig.copy(this[p]);
                }
                if (this.init) {
                    this.init.apply(this, arguments);
                }
            }
            return this;
        }
        Class.prototype = prototype;
        Class.constructor = Class;
        Class.extend = arguments.callee;
        return Class;
    };
})(window);

// lib/impact/image.js
ig.baked = true;
ig.module('impact.image').defines(function () {
    ig.Image = ig.Class.extend({
        data: null,
        width: 0,
        height: 0,
        loaded: false,
        failed: false,
        loadCallback: null,
        path: '',
        staticInstantiate: function (path) {
            return ig.Image.cache[path] || null;
        },
        init: function (path) {
            this.path = path;
            this.load();
        },
        load: function (loadCallback) {
            if (this.loaded) {
                if (loadCallback) {
                    loadCallback(this.path, true);
                }
                return;
            } else if (!this.loaded && ig.ready) {
                this.loadCallback = loadCallback || null;
                this.data = new Image();
                this.data.onload = this.onload.bind(this);
                this.data.onerror = this.onerror.bind(this);
                this.data.src = this.path + ig.nocache;
            } else {
                ig.addResource(this);
            }
            ig.Image.cache[this.path] = this;
        },
        reload: function () {
            this.loaded = false;
            this.data = new Image();
            this.data.onload = this.onload.bind(this);
            this.data.src = this.path + '?' + Date.now();
        },
        onload: function (event) {
            this.width = this.data.width;
            this.height = this.data.height;
            if (ig.system.scale != 1) {
                this.resize(ig.system.scale);
            }
            this.loaded = true;
            if (this.loadCallback) {
                this.loadCallback(this.path, true);
            }
        },
        onerror: function (event) {
            this.failed = true;
            if (this.loadCallback) {
                this.loadCallback(this.path, false);
            }
        },
        resize: function (scale) {
            var widthScaled = this.width * scale;
            var heightScaled = this.height * scale;
            var orig = ig.$new('canvas');
            orig.width = this.width;
            orig.height = this.height;
            var origCtx = orig.getContext('2d');
            origCtx.drawImage(this.data, 0, 0, this.width, this.height, 0, 0, this.width, this.height);
            var origPixels = origCtx.getImageData(0, 0, this.width, this.height);
            var scaled = ig.$new('canvas');
            scaled.width = widthScaled;
            scaled.height = heightScaled;
            var scaledCtx = scaled.getContext('2d');
            var scaledPixels = scaledCtx.getImageData(0, 0, widthScaled, heightScaled);
            for (var y = 0; y < heightScaled; y++) {
                for (var x = 0; x < widthScaled; x++) {
                    var index = ((y / scale).floor() * this.width + (x / scale).floor()) * 4;
                    var indexScaled = (y * widthScaled + x) * 4;
                    scaledPixels.data[indexScaled] = origPixels.data[index];
                    scaledPixels.data[indexScaled + 1] = origPixels.data[index + 1];
                    scaledPixels.data[indexScaled + 2] = origPixels.data[index + 2];
                    scaledPixels.data[indexScaled + 3] = origPixels.data[index + 3];
                }
            }
            scaledCtx.putImageData(scaledPixels, 0, 0);
            this.data = scaled;
        },
        draw: function (targetX, targetY, sourceX, sourceY, width, height) {
            if (!this.loaded) {
                return;
            }
            var scale = ig.system.scale;
            sourceX = sourceX ? sourceX * scale : 0;
            sourceY = sourceY ? sourceY * scale : 0;
            width = (width ? width : this.width) * scale;
            height = (height ? height : this.height) * scale;
            ig.system.context.drawImage(this.data, sourceX, sourceY, width, height, ig.system.getDrawPos(targetX), ig.system.getDrawPos(targetY), width, height);
        },
        drawTile: function (targetX, targetY, tile, tileWidth, tileHeight, flipX, flipY) {
            tileHeight = tileHeight ? tileHeight : tileWidth;
            if (!this.loaded || tileWidth > this.width || tileHeight > this.height) {
                return;
            }
            var scale = ig.system.scale;
            var tileWidthScaled = tileWidth * scale;
            var tileHeightScaled = tileHeight * scale;
            var scaleX = flipX ? -1 : 1;
            var scaleY = flipY ? -1 : 1;
            if (flipX || flipY) {
                ig.system.context.save();
                ig.system.context.scale(scaleX, scaleY);
            }
            ig.system.context.drawImage(this.data, ((tile * tileWidth).floor() % this.width) * scale, ((tile * tileWidth / this.width).floor() * tileHeight) * scale, tileWidthScaled, tileHeightScaled, ig.system.getDrawPos(targetX) * scaleX - (flipX ? tileWidthScaled : 0), ig.system.getDrawPos(targetY) * scaleY - (flipY ? tileHeightScaled : 0), tileWidthScaled, tileHeightScaled);
            if (flipX || flipY) {
                ig.system.context.restore();
            }
        }
    });
    ig.Image.cache = {};
    ig.Image.reloadCache = function () {
        for (path in ig.Image.cache) {
            ig.Image.cache[path].reload();
        }
    };
});

// lib/impact/font.js
ig.baked = true;
ig.module('impact.font').requires('impact.image').defines(function () {
    ig.Font = ig.Image.extend({
        widthMap: [],
        indices: [],
        firstChar: 32,
        height: 0,
        onload: function (ev) {
            this._loadMetrics(this.data);
            this.parent(ev);
        },
        widthForString: function (s) {
            var width = 0;
            for (var i = 0; i < s.length; i++) {
                width += this.widthMap[s.charCodeAt(i) - this.firstChar] + 1;
            }
            return width;
        },
        draw: function (text, x, y, align) {
            if (typeof (text) != 'string') {
                text = text.toString();
            }
            if (align == ig.Font.ALIGN.RIGHT || align == ig.Font.ALIGN.CENTER) {
                var width = 0;
                for (var i = 0; i < text.length; i++) {
                    var c = text.charCodeAt(i);
                    width += this.widthMap[c - this.firstChar] + 1;
                }
                x -= align == ig.Font.ALIGN.CENTER ? width / 2 : width;
            }
            for (var i = 0; i < text.length; i++) {
                var c = text.charCodeAt(i);
                x += this._drawChar(c - this.firstChar, x, y);
            }
        },
        _drawChar: function (c, targetX, targetY) {
            if (!this.loaded || c < 0 || c >= this.indices.length) {
                return 0;
            }
            var scale = ig.system.scale;
            var charX = this.indices[c] * scale;
            var charY = 0;
            var charWidth = this.widthMap[c] * scale;
            var charHeight = (this.height - 2) * scale;
            ig.system.context.drawImage(this.data, charX, charY, charWidth, charHeight, ig.system.getDrawPos(targetX), ig.system.getDrawPos(targetY), charWidth, charHeight);
            return this.widthMap[c] + 1;
        },
        _loadMetrics: function (image) {
            this.height = image.height - 1;
            this.widthMap = [];
            this.indices = [];
            var canvas = ig.$new('canvas');
            canvas.width = image.width;
            canvas.height = 1;
            var ctx = canvas.getContext('2d');
            ctx.drawImage(image, 0, image.height - 1, image.width, 1, 0, 0, image.width, 1);
            var px = ctx.getImageData(0, 0, image.width, 1);
            var currentChar = 0;
            var currentWidth = 0;
            for (var x = 0; x < image.width; x++) {
                var index = x * 4 + 3;
                if (px.data[index] != 0) {
                    currentWidth++;
                } else if (px.data[index] == 0 && currentWidth) {
                    this.widthMap.push(currentWidth);
                    this.indices.push(x - currentWidth);
                    currentChar++;
                    currentWidth = 0;
                }
            }
            this.widthMap.push(currentWidth);
            this.indices.push(x - currentWidth);
        }
    });
    ig.Font.ALIGN = {
        LEFT: 0,
        RIGHT: 1,
        CENTER: 2
    };
});

// lib/impact/sound.js
ig.baked = true;
ig.module('impact.sound').defines(function () {
    ig.SoundManager = ig.Class.extend({
        clips: {},
        volume: 1,
        channels: 4,
        format: 'mp3',
        init: function () {
            this.format = ig.$new('audio').canPlayType('audio/mpeg') ? 'mp3' : 'ogg';
        },
        load: function (path, multiChannel, loadCallback) {
            if (this.clips[path]) {
                if (multiChannel && this.clips[path].length < this.channels) {
                    for (var i = this.clips[path].length; i < this.channels; i++) {
                        this.clips[path].push(this.clips[path][0].cloneNode(true));
                    }
                }
                return this.clips[path][0];
            }
            var realPath = path.match(/^(.*)\.[^\.]+$/)[1] + '.' + this.format + ig.nocache;
            var clip = ig.$new('audio');
            if (loadCallback) {
                clip.addEventListener('canplaythrough', function (ev) {
                    this.removeEventListener('canplaythrough', arguments.callee, false)
                    loadCallback(path, true, ev);
                }, false);
                clip.addEventListener('error', function (ev) {
                    loadCallback(path, true, ev);
                }, false);
            }
            clip.autobuffer = true;
            clip.preload = 'auto';
            clip.src = realPath;
            clip.load();
            this.clips[path] = [clip];
            if (multiChannel) {
                for (var i = 1; i < this.channels; i++) {
                    this.clips[path].push(clip.cloneNode(true));
                }
            }
            return clip;
        },
        get: function (path) {
            var channels = this.clips[path];
            for (var i = 0, clip; clip = channels[i++];) {
                if (clip.paused || clip.ended) {
                    if (clip.ended) {
                        clip.currentTime = 0;
                    }
                    return clip;
                }
            }
            channels[0].pause();
            channels[0].currentTime = 0;
            return channels[0];
        }
    });
    ig.Music = ig.Class.extend({
        tracks: [],
        currentTrack: null,
        currentIndex: 0,
        random: false,
        _volume: 1,
        _loop: false,
        _fadeInterval: 0,
        _fadeTimer: null,
        _endedCallbackBound: null,
        init: function () {
            this._endedCallbackBound = this._endedCallback.bind(this);
            if (Object.defineProperty) {
                Object.defineProperty(this, "volume", {
                    get: this.getVolume.bind(this),
                    set: this.setVolume.bind(this)
                });
                Object.defineProperty(this, "loop", {
                    get: this.getLooping.bind(this),
                    set: this.setLooping.bind(this)
                });
            } else if (this.__defineGetter__) {
                this.__defineGetter__('volume', this.getVolume.bind(this));
                this.__defineSetter__('volume', this.setVolume.bind(this));
                this.__defineGetter__('loop', this.getLooping.bind(this));
                this.__defineSetter__('loop', this.setLooping.bind(this));
            }
        },
        add: function (music) {
            if (!ig.Sound.enabled) {
                return;
            }
            var path = music instanceof ig.Sound ? music.path : music;
            var track = ig.soundManager.load(path, false);
            track.loop = this._loop;
            track.volume = this._volume;
            track.addEventListener('ended', this._endedCallbackBound, false);
            this.tracks.push(track);
            if (!this.currentTrack) {
                this.currentTrack = track;
            }
        },
        next: function () {
            if (!this.tracks.length) {
                return;
            }
            this.stop();
            this.currentIndex = this.random ? (Math.random() * this.tracks.length).floor() : (this.currentIndex + 1) % this.tracks.length;
            this.currentTrack = this.tracks[this.currentIndex];
            this.play();
        },
        pause: function () {
            if (!this.currentTrack) {
                return;
            }
            this.currentTrack.pause();
        },
        stop: function () {
            if (!this.currentTrack) {
                return;
            }
            this.currentTrack.pause();
            this.currentTrack.currentTime = 0;
        },
        play: function () {
            if (!this.currentTrack) {
                return;
            }
            this.currentTrack.play();
        },
        getLooping: function () {
            return this._loop;
        },
        setLooping: function (l) {
            this._loop = l;
            for (var i in this.tracks) {
                this.tracks[i].loop = l;
            }
        },
        getVolume: function () {
            return this._volume;
        },
        setVolume: function (v) {
            this._volume = v.limit(0, 1);
            for (var i in this.tracks) {
                this.tracks[i].volume = this._volume;
            }
        },
        fadeOut: function (time) {
            if (!this.currentTrack) {
                return;
            }
            clearInterval(this._fadeInterval);
            this.fadeTimer = new ig.Timer(time);
            this._fadeInterval = setInterval(this._fadeStep.bind(this), 50);
        },
        _fadeStep: function () {
            var v = this.fadeTimer.delta().map(-this.fadeTimer.target, 0, 1, 0).limit(0, 1) * this._volume;
            if (v <= 0.01) {
                this.stop();
                this.currentTrack.volume = this._volume;
                clearInterval(this._fadeInterval);
            } else {
                this.currentTrack.volume = v;
            }
        },
        _endedCallback: function () {
            if (this._loop) {
                this.play();
            } else {
                this.next();
            }
        }
    });
    ig.Sound = ig.Class.extend({
        path: '',
        volume: 1,
        currentClip: null,
        multiChannel: true,
        init: function (path, multiChannel) {
            this.path = path;
            this.multiChannel = (multiChannel !== false);
            this.load();
        },
        load: function (loadCallback) {
            if (!ig.Sound.enabled) {
                if (loadCallback) {
                    loadCallback(this.path, true);
                }
                return;
            }
            if (ig.ready) {
                ig.soundManager.load(this.path, this.multiChannel, loadCallback);
            } else {
                ig.addResource(this);
            }
        },
        play: function () {
            if (!ig.Sound.enabled) {
                return;
            }
            this.currentClip = ig.soundManager.get(this.path);
            this.currentClip.volume = ig.soundManager.volume * this.volume;
            this.currentClip.play();
        },
        stop: function () {
            if (this.currentClip) {
                this.currentClip.pause();
                this.currentClip.currentTime = 0;
            }
        }
    });
    ig.Sound.enabled = true;
});

// lib/impact/loader.js
ig.baked = true;
ig.module('impact.loader').requires('impact.image', 'impact.font', 'impact.sound').defines(function () {
    ig.Loader = ig.Class.extend({
        resources: [],
        gameClass: null,
        status: 0,
        done: false,
        _unloaded: [],
        _drawStatus: 0,
        _intervalId: 0,
        _loadCallbackBound: null,
        init: function (gameClass, resources) {
            this.gameClass = gameClass;
            this.resources = resources;
            this._loadCallbackBound = this._loadCallback.bind(this);
            for (var i = 0; i < this.resources.length; i++) {
                this._unloaded.push(this.resources[i].path);
            }
        },
        load: function () {
            ig.system.clear('#000');
            if (!this.resources.length) {
                this.end();
                return;
            }
            for (var i = 0; i < this.resources.length; i++) {
                this.loadResource(this.resources[i]);
            }
            this._intervalId = setInterval(this.draw.bind(this), 16);
        },
        loadResource: function (res) {
            res.load(this._loadCallbackBound);
        },
        end: function () {
            if (this.done) {
                return;
            }
            this.done = true;
            clearInterval(this._intervalId);
            ig.system.setGame(this.gameClass);
        },
        draw: function () {
            this._drawStatus += (this.status - this._drawStatus) / 5;
            var s = ig.system.scale;
            var w = ig.system.width * 0.6;
            var h = ig.system.height * 0.1;
            var x = ig.system.width * 0.5 - w / 2;
            var y = ig.system.height * 0.5 - h / 2;
            ig.system.context.fillStyle = '#000';
            ig.system.context.fillRect(0, 0, 480, 320);
            ig.system.context.fillStyle = '#fff';
            ig.system.context.fillRect(x * s, y * s, w * s, h * s);
            ig.system.context.fillStyle = '#000';
            ig.system.context.fillRect(x * s + s, y * s + s, w * s - s - s, h * s - s - s);
            ig.system.context.fillStyle = '#fff';
            ig.system.context.fillRect(x * s, y * s, w * s * this._drawStatus, h * s);
        },
        _loadCallback: function (path, status) {
            if (status) {
                this._unloaded.erase(path);
            } else {
                throw ('Failed to load resource: ' + path);
            }
            this.status = 1 - (this._unloaded.length / this.resources.length);
            if (this._unloaded.length == 0) {
                setTimeout(this.end.bind(this), 250);
            }
        }
    });
});

// lib/impact/timer.js
ig.baked = true;
ig.module('impact.timer').defines(function () {
    ig.Timer = ig.Class.extend({
        target: 0,
        base: 0,
        last: 0,
        init: function (seconds) {
            this.base = ig.Timer.time;
            this.last = ig.Timer.time;
            this.target = seconds || 0;
        },
        set: function (seconds) {
            this.target = seconds || 0;
            this.base = ig.Timer.time;
        },
        reset: function () {
            this.base = ig.Timer.time;
        },
        tick: function () {
            var delta = ig.Timer.time - this.last;
            this.last = ig.Timer.time;
            return delta;
        },
        delta: function () {
            return ig.Timer.time - this.base - this.target;
        }
    });
    ig.Timer._last = 0;
    ig.Timer.time = 0;
    ig.Timer.timeScale = 1;
    ig.Timer.maxStep = 0.05;
    ig.Timer.step = function () {
        var current = Date.now();
        var delta = (current - ig.Timer._last) / 1000;
        ig.Timer.time += Math.min(delta, ig.Timer.maxStep) * ig.Timer.timeScale;
        ig.Timer._last = current;
    };
});

// lib/impact/system.js
ig.baked = true;
ig.module('impact.system').requires('impact.timer', 'impact.image').defines(function () {
    ig.System = ig.Class.extend({
        fps: 30,
        width: 320,
        height: 240,
        realWidth: 320,
        realHeight: 240,
        scale: 1,
        tick: 0,
        intervalId: 0,
        newGameClass: null,
        running: false,
        delegate: null,
        clock: null,
        canvas: null,
        context: null,
        smoothPositioning: true,
        init: function (canvasId, fps, width, height, scale) {
            this.fps = fps;
            this.width = width;
            this.height = height;
            this.scale = scale;
            this.realWidth = width * scale;
            this.realHeight = height * scale;
            this.clock = new ig.Timer();
            this.canvas = ig.$(canvasId);
            this.canvas.width = this.realWidth;
            this.canvas.height = this.realHeight;
            this.context = this.canvas.getContext('2d');
        },
        setGame: function (gameClass) {
            if (this.running) {
                this.newGameClass = gameClass;
            } else {
                this.setGameNow(gameClass);
            }
        },
        setGameNow: function (gameClass) {
            ig.game = new(gameClass)();
            ig.system.setDelegate(ig.game);
        },
        setDelegate: function (object) {
            if (typeof (object.run) == 'function') {
                this.delegate = object;
                this.startRunLoop();
            } else {
                throw ('System.setDelegate: No run() function in object');
            }
        },
        stopRunLoop: function () {
            clearInterval(this.intervalId);
            this.running = false;
        },
        startRunLoop: function () {
            this.stopRunLoop();
            this.intervalId = setInterval(this.run.bind(this), 1000 / this.fps);
            this.running = true;
        },
        clear: function (color) {
            this.context.fillStyle = color;
            this.context.fillRect(0, 0, this.realWidth, this.realHeight);
        },
        run: function () {
            ig.Timer.step();
            this.tick = this.clock.tick();
            this.delegate.run();
            ig.input.clearPressed();
            if (this.newGameClass) {
                this.setGameNow(this.newGameClass);
                this.newGameClass = null;
            }
        },
        getDrawPos: function (p) {
            return this.smoothPositioning ? (p * this.scale).round() : p.round() * this.scale;
        }
    });
});

// lib/impact/input.js
ig.baked = true;
ig.module('impact.input').defines(function () {
    ig.KEY = {
        'MOUSE1': -1,
        'MOUSE2': -3,
        'MWHEEL_UP': -4,
        'MWHEEL_DOWN': -5,
        'BACKSPACE': 8,
        'TAB': 9,
        'ENTER': 13,
        'PAUSE': 19,
        'CAPS': 20,
        'ESC': 27,
        'SPACE': 32,
        'PAGE_UP': 33,
        'PAGE_DOWN': 34,
        'END': 35,
        'HOME': 36,
        'LEFT_ARROW': 37,
        'UP_ARROW': 38,
        'RIGHT_ARROW': 39,
        'DOWN_ARROW': 40,
        'INSERT': 45,
        'DELETE': 46,
        '0': 48,
        '1': 49,
        '2': 50,
        '3': 51,
        '4': 52,
        '5': 53,
        '6': 54,
        '7': 55,
        '8': 56,
        '9': 57,
        'A': 65,
        'B': 66,
        'C': 67,
        'D': 68,
        'E': 69,
        'F': 70,
        'G': 71,
        'H': 72,
        'I': 73,
        'J': 74,
        'K': 75,
        'L': 76,
        'M': 77,
        'N': 78,
        'O': 79,
        'P': 80,
        'Q': 81,
        'R': 82,
        'S': 83,
        'T': 84,
        'U': 85,
        'V': 86,
        'W': 87,
        'X': 88,
        'Y': 89,
        'Z': 90,
        'NUMPAD_0': 96,
        'NUMPAD_1': 97,
        'NUMPAD_2': 98,
        'NUMPAD_3': 99,
        'NUMPAD_4': 100,
        'NUMPAD_5': 101,
        'NUMPAD_6': 102,
        'NUMPAD_7': 103,
        'NUMPAD_8': 104,
        'NUMPAD_9': 105,
        'MULTIPLY': 106,
        'ADD': 107,
        'SUBSTRACT': 109,
        'DECIMAL': 110,
        'DIVIDE': 111,
        'F1': 112,
        'F2': 113,
        'F3': 114,
        'F4': 115,
        'F5': 116,
        'F6': 117,
        'F7': 118,
        'F8': 119,
        'F9': 120,
        'F10': 121,
        'F11': 122,
        'F12': 123,
        'SHIFT': 16,
        'CTRL': 17,
        'ALT': 18,
        'PLUS': 187,
        'COMMA': 188,
        'MINUS': 189,
        'PERIOD': 190
    };
    ig.Input = ig.Class.extend({
        bindings: {},
        actions: {},
        locks: {},
        delayedKeyup: [],
        isUsingMouse: false,
        isUsingKeyboard: false,
        mouse: {
            x: 0,
            y: 0
        },
        initMouse: function () {
            if (this.isUsingMouse) {
                return;
            }
            this.isUsingMouse = true;
            window.addEventListener('mousewheel', this.mousewheel.bind(this), false);
            ig.system.canvas.addEventListener('contextmenu', this.contextmenu.bind(this), false);
            ig.system.canvas.addEventListener('mousedown', this.keydown.bind(this), false);
            ig.system.canvas.addEventListener('mouseup', this.keyup.bind(this), false);
            ig.system.canvas.addEventListener('mousemove', this.mousemove.bind(this), false);
            ig.system.canvas.addEventListener('touchstart', this.keydown.bind(this), false);
            ig.system.canvas.addEventListener('touchend', this.keyup.bind(this), false);
            ig.system.canvas.addEventListener('touchmove', this.mousemove.bind(this), false);
        },
        initKeyboard: function () {
            if (this.isUsingKeyboard) {
                return;
            }
            this.isUsingKeyboard = true;
            window.addEventListener('keydown', this.keydown.bind(this), false);
            window.addEventListener('keyup', this.keyup.bind(this), false);
        },
        mousewheel: function (event) {
            var code = event.wheel > 0 ? ig.KEY.MWHEEL_UP : ig.KEY.MWHEEL_DOWN;
            var action = this.bindings[code];
            if (action) {
                this.actions[action] = true;
                event.stopPropagation();
                this.delayedKeyup.push(action);
            }
        },
        mousemove: function (event) {
            var el = ig.system.canvas;
            var pos = {
                left: 0,
                top: 0
            };
            while (el != null) {
                pos.left += el.offsetLeft;
                pos.top += el.offsetTop;
                el = el.offsetParent;
            }
            var tx = event.pageX;
            var ty = event.pageY;
            if (event.touches) {
                tx = event.touches[0].clientX;
                ty = event.touches[0].clientY;
            }
            this.mouse.x = (tx - pos.left) / ig.system.scale;
            this.mouse.y = (ty - pos.top) / ig.system.scale;
        },
        contextmenu: function (event) {
            if (this.bindings[ig.KEY.MOUSE2]) {
                event.stopPropagation();
                event.preventDefault();
            }
        },
        keydown: function (event) {
            if (event.target.type == 'text') {
                return;
            }
            var code = event.type == 'keydown' ? event.keyCode : (event.button == 2 ? ig.KEY.MOUSE2 : ig.KEY.MOUSE1);
            if (event.type == 'touchstart') {
                this.mousemove(event);
            }
            var action = this.bindings[code];
            if (action) {
                this.actions[action] = true;
                event.stopPropagation();
                event.preventDefault();
            }
        },
        keyup: function (event) {
            if (event.target.type == 'text') {
                return;
            }
            var code = event.type == 'keyup' ? event.keyCode : (event.button == 2 ? ig.KEY.MOUSE2 : ig.KEY.MOUSE1);
            var action = this.bindings[code];
            if (action) {
                this.delayedKeyup.push(action);
                event.stopPropagation();
                event.preventDefault();
            }
        },
        bind: function (key, action) {
            if (key < 0) {
                this.initMouse();
            } else if (key > 0) {
                this.initKeyboard();
            }
            this.bindings[key] = action;
        },
        bindTouch: function (selector, action) {
            var element = ig.$(selector);
            var that = this;
            element.addEventListener('touchstart', function (ev) {
                that.touchStart(ev, action);
            }, false);
            element.addEventListener('touchend', function (ev) {
                that.touchEnd(ev, action);
            }, false);
        },
        unbind: function (key) {
            this.bindings[key] = null;
        },
        unbindAll: function () {
            this.bindings = [];
        },
        state: function (action) {
            return this.actions[action];
        },
        pressed: function (action) {
            if (!this.locks[action] && this.actions[action]) {
                this.locks[action] = true;
                return true;
            } else {
                return false;
            }
        },
        clearPressed: function () {
            for (var i = 0; i < this.delayedKeyup.length; i++) {
                var action = this.delayedKeyup[i];
                this.locks[action] = false;
                this.actions[action] = false;
            }
            this.delayedKeyup = [];
        },
        touchStart: function (event, action) {
            this.actions[action] = true;
            event.stopPropagation();
            event.preventDefault();
            return false;
        },
        touchEnd: function (event, action) {
            this.delayedKeyup.push(action);
            event.stopPropagation();
            event.preventDefault();
            return false;
        }
    });
});

// lib/impact/impact.js
ig.baked = true;
ig.module('impact.impact').requires('dom.ready', 'impact.loader', 'impact.system', 'impact.input', 'impact.sound').defines(function () {
    ig.main = function (canvasId, gameClass, fps, width, height, scale, loaderClass) {
        ig.system = new ig.System(canvasId, fps, width, height, scale || 1);
        ig.input = new ig.Input();
        ig.soundManager = new ig.SoundManager();
        ig.music = new ig.Music();
        ig.ready = true;
        var loader = new(loaderClass || ig.Loader)(gameClass, ig.resources);
        loader.load();
    };
});

// lib/impact/animation.js
ig.baked = true;
ig.module('impact.animation').requires('impact.timer', 'impact.image').defines(function () {
    ig.AnimationSheet = ig.Class.extend({
        width: 8,
        height: 8,
        image: null,
        init: function (path, width, height) {
            this.width = width;
            this.height = height;
            this.image = new ig.Image(path);
        }
    });
    ig.Animation = ig.Class.extend({
        sheet: null,
        timer: null,
        sequence: [],
        flip: {
            x: false,
            y: false
        },
        pivot: {
            x: 0,
            y: 0
        },
        frame: 0,
        tile: 0,
        loopCount: 0,
        alpha: 1,
        angle: 0,
        init: function (sheet, frameTime, sequence, stop) {
            this.sheet = sheet;
            this.pivot = {
                x: sheet.width / 2,
                y: sheet.height / 2
            };
            this.timer = new ig.Timer();
            this.frameTime = frameTime;
            this.sequence = sequence;
            this.stop = !!stop;
        },
        rewind: function () {
            this.timer.reset();
            this.loopCount = 0;
            this.tile = this.sequence[0];
            return this;
        },
        gotoFrame: function (f) {
            this.timer.set(this.frameTime * -f);
            this.update();
        },
        gotoRandomFrame: function () {
            this.gotoFrame((Math.random() * this.sequence.length).floor())
        },
        update: function () {
            var frameTotal = (this.timer.delta() / this.frameTime).floor();
            this.loopCount = (frameTotal / this.sequence.length).floor();
            if (this.stop && this.loopCount > 0) {
                this.frame = this.sequence.length - 1;
            } else {
                this.frame = frameTotal % this.sequence.length;
            }
            this.tile = this.sequence[this.frame];
        },
        draw: function (targetX, targetY) {
            var bbsize = Math.max(this.sheet.width, this.sheet.height);
            if (targetX > ig.system.width || targetY > ig.system.height || targetX + bbsize < 0 || targetY + bbsize < 0) {
                return;
            }
            if (this.alpha != 1) {
                ig.system.context.globalAlpha = this.alpha;
            }
            if (this.angle == 0) {
                this.sheet.image.drawTile(targetX, targetY, this.tile, this.sheet.width, this.sheet.height, this.flip.x, this.flip.y);
            } else {
                ig.system.context.save();
                ig.system.context.translate(ig.system.getDrawPos(targetX + this.pivot.x), ig.system.getDrawPos(targetY + this.pivot.y));
                ig.system.context.rotate(this.angle);
                this.sheet.image.drawTile(-this.pivot.x, -this.pivot.y, this.tile, this.sheet.width, this.sheet.height, this.flip.x, this.flip.y);
                ig.system.context.restore();
            }
            if (this.alpha != 1) {
                ig.system.context.globalAlpha = 1;
            }
        }
    });
});

// lib/impact/entity.js
ig.baked = true;
ig.module('impact.entity').requires('impact.animation', 'impact.impact').defines(function () {
    ig.Entity = ig.Class.extend({
        id: 0,
        settings: {},
        size: {
            x: 16,
            y: 16
        },
        offset: {
            x: 0,
            y: 0
        },
        pos: {
            x: 0,
            y: 0
        },
        last: {
            x: 0,
            y: 0
        },
        vel: {
            x: 0,
            y: 0
        },
        accel: {
            x: 0,
            y: 0
        },
        friction: {
            x: 0,
            y: 0
        },
        maxVel: {
            x: 100,
            y: 100
        },
        zIndex: 0,
        gravityFactor: 1,
        standing: false,
        bounciness: 0,
        minBounceVelocity: 40,
        anims: {},
        animSheet: null,
        currentAnim: null,
        health: 10,
        type: 0,
        checkAgainst: 0,
        collides: 0,
        _killed: false,
        init: function (x, y, settings) {
            this.id = ++ig.Entity._lastId;
            this.pos.x = x;
            this.pos.y = y;
            ig.merge(this, settings);
        },
        addAnim: function (name, frameTime, sequence, stop) {
            if (!this.animSheet) {
                throw ('No animSheet to add the animation ' + name + ' to.');
            }
            var a = new ig.Animation(this.animSheet, frameTime, sequence, stop);
            this.anims[name] = a;
            if (!this.currentAnim) {
                this.currentAnim = a;
            }
            return a;
        },
        update: function () {
            this.last.x = this.pos.x;
            this.last.y = this.pos.y;
            this.vel.y += ig.game.gravity * ig.system.tick * this.gravityFactor;
            this.vel.x = this.getNewVelocity(this.vel.x, this.accel.x, this.friction.x, this.maxVel.x);
            this.vel.y = this.getNewVelocity(this.vel.y, this.accel.y, this.friction.y, this.maxVel.y);
            var mx = this.vel.x * ig.system.tick;
            var my = this.vel.y * ig.system.tick;
            var res = ig.game.collisionMap.trace(this.pos.x, this.pos.y, mx, my, this.size.x, this.size.y);
            this.handleMovementTrace(res);
            if (this.currentAnim) {
                this.currentAnim.update();
            }
        },
        getNewVelocity: function (vel, accel, friction, max) {
            if (accel) {
                return (vel + accel * ig.system.tick).limit(-max, max);
            } else if (friction) {
                var delta = friction * ig.system.tick;
                if (vel - delta > 0) {
                    return vel - delta;
                } else if (vel + delta < 0) {
                    return vel + delta;
                } else {
                    return 0;
                }
            }
            return vel.limit(-max, max);
        },
        handleMovementTrace: function (res) {
            this.standing = false;
            if (res.collision.y) {
                if (this.bounciness > 0 && Math.abs(this.vel.y) > this.minBounceVelocity) {
                    this.vel.y *= -this.bounciness;
                } else {
                    if (this.vel.y > 0) {
                        this.standing = true;
                    }
                    this.vel.y = 0;
                }
            }
            if (res.collision.x) {
                if (this.bounciness > 0 && Math.abs(this.vel.x) > this.minBounceVelocity) {
                    this.vel.x *= -this.bounciness;
                } else {
                    this.vel.x = 0;
                }
            }
            this.pos = res.pos;
        },
        draw: function () {
            if (this.currentAnim) {
                this.currentAnim.draw(this.pos.x.round() - this.offset.x - ig.game.screen.x, this.pos.y.round() - this.offset.y - ig.game.screen.y);
            }
        },
        kill: function () {
            ig.game.removeEntity(this);
        },
        receiveDamage: function (amount, from) {
            this.health -= amount;
            if (this.health <= 0) {
                this.kill();
            }
        },
        touches: function (other) {
            return !(this.pos.x > other.pos.x + other.size.x || this.pos.x + this.size.x < other.pos.x || this.pos.y > other.pos.y + other.size.y || this.pos.y + this.size.y < other.pos.y);
        },
        distanceTo: function (other) {
            var xd = (this.pos.x + this.size.x / 2) - (other.pos.x + other.size.x / 2);
            var yd = (this.pos.y + this.size.y / 2) - (other.pos.y + other.size.y / 2);
            return Math.sqrt(xd * xd + yd * yd);
        },
        angleTo: function (other) {
            return Math.atan2((other.pos.y + other.size.y / 2) - (this.pos.y + this.size.y / 2), (other.pos.x + other.size.x / 2) - (this.pos.x + this.size.x / 2));
        },
        check: function (other) {},
        collideWith: function (other, axis) {}
    });
    ig.Entity._lastId = 0;
    ig.Entity.COLLIDES = {
        NEVER: 0,
        LITE: 1,
        PASSIVE: 2,
        ACTIVE: 4,
        FIXED: 8
    };
    ig.Entity.TYPE = {
        NONE: 0,
        A: 1,
        B: 2,
        BOTH: 3
    };
    ig.Entity.checkPair = function (a, b) {
        if (a.checkAgainst & b.type) {
            a.check(b);
        }
        if (b.checkAgainst & a.type) {
            b.check(a);
        }
        if (a.collides && b.collides && a.collides + b.collides > ig.Entity.COLLIDES.ACTIVE) {
            ig.Entity.solveCollision(a, b);
        }
    };
    ig.Entity.solveCollision = function (a, b) {
        var weak = null;
        if (a.collides == ig.Entity.COLLIDES.LITE || b.collides == ig.Entity.COLLIDES.FIXED) {
            weak = a;
        } else if (b.collides == ig.Entity.COLLIDES.LITE || a.collides == ig.Entity.COLLIDES.FIXED) {
            weak = b;
        }
        if (a.last.x + a.size.x > b.last.x && a.last.x < b.last.x + b.size.x) {
            if (a.last.y < b.last.y) {
                ig.Entity.seperateOnYAxis(a, b, weak);
            } else {
                ig.Entity.seperateOnYAxis(b, a, weak);
            }
            a.collideWith(b, 'y');
            b.collideWith(a, 'y');
        } else if (a.last.y + a.size.y > b.last.y && a.last.y < b.last.y + b.size.y) {
            if (a.last.x < b.last.x) {
                ig.Entity.seperateOnXAxis(a, b, weak);
            } else {
                ig.Entity.seperateOnXAxis(b, a, weak);
            }
            a.collideWith(b, 'x');
            b.collideWith(a, 'x');
        }
    };
    ig.Entity.seperateOnXAxis = function (left, right, weak) {
        var nudge = (left.pos.x + left.size.x - right.pos.x);
        if (weak) {
            var strong = left === weak ? right : left;
            weak.vel.x = -weak.vel.x * weak.bounciness + strong.vel.x;
            var resWeak = ig.game.collisionMap.trace(weak.pos.x, weak.pos.y, weak == left ? -nudge : nudge, 0, weak.size.x, weak.size.y);
            weak.pos.x = resWeak.pos.x;
        } else {
            var v2 = (left.vel.x - right.vel.x) / 2;
            left.vel.x = -v2;
            right.vel.x = v2;
            var resLeft = ig.game.collisionMap.trace(left.pos.x, left.pos.y, -nudge / 2, 0, left.size.x, left.size.y);
            left.pos.x = resLeft.pos.x.floor();
            var resRight = ig.game.collisionMap.trace(right.pos.x, right.pos.y, nudge / 2, 0, right.size.x, right.size.y);
            right.pos.x = resRight.pos.x.ceil();
        }
    };
    ig.Entity.seperateOnYAxis = function (top, bottom, weak) {
        var nudge = (top.pos.y + top.size.y - bottom.pos.y);
        if (weak) {
            var strong = top === weak ? bottom : top;
            weak.vel.y = -weak.vel.y * weak.bounciness + strong.vel.y;
            var nudgeX = 0;
            if (weak == top && Math.abs(weak.vel.y - strong.vel.y) < weak.minBounceVelocity) {
                weak.standing = true;
                nudgeX = strong.vel.x * ig.system.tick;
            }
            var resWeak = ig.game.collisionMap.trace(weak.pos.x, weak.pos.y, nudgeX, weak == top ? -nudge : nudge, weak.size.x, weak.size.y);
            weak.pos.y = resWeak.pos.y;
            weak.pos.x = resWeak.pos.x;
        } else if (ig.game.gravity && (bottom.standing || top.vel.y > 0)) {
            var resTop = ig.game.collisionMap.trace(top.pos.x, top.pos.y, 0, -(top.pos.y + top.size.y - bottom.pos.y), top.size.x, top.size.y);
            top.pos.y = resTop.pos.y;
            if (top.bounciness > 0 && top.vel.y > top.minBounceVelocity) {
                top.vel.y *= -top.bounciness;
            } else {
                top.standing = true;
                top.vel.y = 0;
            }
        } else {
            var v2 = (top.vel.y - bottom.vel.y) / 2;
            top.vel.y = -v2;
            bottom.vel.y = v2;
            var nudgeX = bottom.vel.x * ig.system.tick;
            var resTop = ig.game.collisionMap.trace(top.pos.x, top.pos.y, nudgeX, -nudge / 2, top.size.x, top.size.y);
            top.pos.y = resTop.pos.y;
            var resBottom = ig.game.collisionMap.trace(bottom.pos.x, bottom.pos.y, 0, nudge / 2, bottom.size.x, bottom.size.y);
            bottom.pos.y = resBottom.pos.y;
        }
    };
});

// lib/impact/map.js
ig.baked = true;
ig.module('impact.map').defines(function () {
    ig.Map = ig.Class.extend({
        tilesize: 8,
        width: 1,
        height: 1,
        data: [[]],
        init: function (tilesize, data) {
            this.tilesize = tilesize;
            this.data = data;
            this.height = data.length;
            this.width = data[0].length;
        },
        getTile: function (x, y) {
            var tx = (x / this.tilesize).floor();
            var ty = (y / this.tilesize).floor();
            if ((tx >= 0 && tx < this.width) && (ty >= 0 && ty < this.height)) {
                return this.data[ty][tx];
            } else {
                return 0;
            }
        },
        setTile: function (x, y, tile) {
            var tx = (x / this.tilesize).floor();
            var ty = (y / this.tilesize).floor();
            if ((tx >= 0 && tx < this.width) && (ty >= 0 && ty < this.height)) {
                this.data[ty][tx] = tile;
            }
        }
    });
});

// lib/impact/collision-map.js
ig.baked = true;
ig.module('impact.collision-map').requires('impact.map').defines(function () {
    ig.CollisionMap = ig.Map.extend({
        firstSolidTile: 1,
        lastSolidTile: 255,
        init: function (tilesize, data) {
            this.parent(tilesize, data);
        },
        trace: function (x, y, vx, vy, objectWidth, objectHeight) {
            var res = {
                collision: {
                    x: false,
                    y: false
                },
                pos: {
                    x: x,
                    y: y
                },
                tile: {
                    x: 0,
                    y: 0
                }
            };
            var steps = (Math.max(Math.abs(vx), Math.abs(vy)) / this.tilesize).ceil();
            if (steps > 1) {
                var sx = vx / steps;
                var sy = vy / steps;
                for (var i = 0; i < steps && (sx || sy); i++) {
                    this._traceStep(res, x, y, sx, sy, objectWidth, objectHeight);
                    x = res.pos.x;
                    y = res.pos.y;
                    if (res.collision.x) {
                        sx = 0;
                    }
                    if (res.collision.y) {
                        sy = 0;
                    }
                }
            } else {
                this._traceStep(res, x, y, vx, vy, objectWidth, objectHeight);
            }
            return res;
        },
        _traceStep: function (res, x, y, vx, vy, width, height) {
            res.pos.x += vx;
            res.pos.y += vy;
            if (vx) {
                var pxOffsetX = (vx > 0 ? width : 0);
                var tileOffsetX = (vx < 0 ? this.tilesize : 0);
                var firstTileY = (y / this.tilesize).floor();
                var lastTileY = ((y + height) / this.tilesize).ceil();
                var tileX = ((x + vx + pxOffsetX) / this.tilesize).floor();
                if (lastTileY >= 0 && firstTileY < this.height && tileX >= 0 && tileX < this.width) {
                    for (var tileY = firstTileY; tileY < lastTileY; tileY++) {
                        var t = this.data[tileY] && this.data[tileY][tileX];
                        if (t >= this.firstSolidTile && t <= this.lastSolidTile) {
                            res.collision.x = true;
                            res.tile.x = t;
                            res.pos.x = tileX * this.tilesize - pxOffsetX + tileOffsetX;
                            break;
                        }
                    }
                }
            }
            if (vy) {
                var pxOffsetY = (vy > 0 ? height : 0);
                var tileOffsetY = (vy < 0 ? this.tilesize : 0);
                var firstTileX = (res.pos.x / this.tilesize).floor();
                var lastTileX = ((res.pos.x + width) / this.tilesize).ceil();
                var tileY = ((y + vy + pxOffsetY) / this.tilesize).floor();
                if (lastTileX >= 0 && firstTileX < this.width && tileY >= 0 && tileY < this.height) {
                    for (var tileX = firstTileX; tileX < lastTileX; tileX++) {
                        var t = this.data[tileY] && this.data[tileY][tileX];
                        if (t >= this.firstSolidTile && t <= this.lastSolidTile) {
                            res.collision.y = true;
                            res.tile.y = t;
                            res.pos.y = tileY * this.tilesize - pxOffsetY + tileOffsetY;
                            break;
                        }
                    }
                }
            }
        }
    });
    ig.CollisionMap.staticNoCollision = {
        trace: function (x, y, vx, vy) {
            return {
                collision: {
                    x: false,
                    y: false
                },
                pos: {
                    x: x + vx,
                    y: y + vy
                },
                tile: {
                    x: 0,
                    y: 0
                }
            };
        }
    };
});

// lib/impact/background-map.js
ig.baked = true;
ig.module('impact.background-map').requires('impact.map', 'impact.image').defines(function () {
    ig.BackgroundMap = ig.Map.extend({
        tiles: null,
        scroll: {
            x: 0,
            y: 0
        },
        distance: 1,
        repeat: false,
        tilesetName: '',
        preRender: false,
        preRenderedChunks: null,
        chunkSize: 512,
        debugChunks: false,
        anims: {},
        init: function (tilesize, data, tileset) {
            this.parent(tilesize, data);
            this.setTileset(tileset);
        },
        setTileset: function (tileset) {
            this.tilesetName = tileset instanceof ig.Image ? tileset.path : tileset;
            this.tiles = new ig.Image(this.tilesetName);
            this.preRenderedChunks = null;
        },
        setScreenPos: function (x, y) {
            this.scroll.x = x / this.distance;
            this.scroll.y = y / this.distance;
        },
        preRenderMapToChunks: function () {
            var totalWidth = this.width * this.tilesize * ig.system.scale,
                totalHeight = this.height * this.tilesize * ig.system.scale;
            var chunkCols = (totalWidth / this.chunkSize).ceil(),
                chunkRows = (totalHeight / this.chunkSize).ceil();
            this.preRenderedChunks = [];
            for (var y = 0; y < chunkRows; y++) {
                this.preRenderedChunks[y] = [];
                for (var x = 0; x < chunkCols; x++) {
                    var chunkWidth = (x == chunkCols - 1) ? totalWidth - x * this.chunkSize : this.chunkSize;
                    var chunkHeight = (y == chunkRows - 1) ? totalHeight - y * this.chunkSize : this.chunkSize;
                    this.preRenderedChunks[y][x] = this.preRenderChunk(x, y, chunkWidth, chunkHeight);
                }
            }
        },
        preRenderChunk: function (cx, cy, w, h) {
            var tw = w / this.tilesize / ig.system.scale + 1;
            th = h / this.tilesize / ig.system.scale + 1;
            var nx = (cx * this.chunkSize / ig.system.scale) % this.tilesize,
                ny = (cy * this.chunkSize / ig.system.scale) % this.tilesize;
            var tx = (cx * this.chunkSize / this.tilesize / ig.system.scale).floor(),
                ty = (cy * this.chunkSize / this.tilesize / ig.system.scale).floor();
            var chunk = ig.$new('canvas');
            chunk.width = w;
            chunk.height = h;
            var oldContext = ig.system.context;
            ig.system.context = chunk.getContext("2d");
            for (var x = 0; x < tw; x++) {
                for (var y = 0; y < th; y++) {
                    if (x + tx < this.width && y + ty < this.height) {
                        var tile = this.data[y + ty][x + tx];
                        if (tile) {
                            this.tiles.drawTile(x * this.tilesize - nx, y * this.tilesize - ny, tile - 1, this.tilesize);
                        }
                    }
                }
            }
            ig.system.context = oldContext;
            return chunk;
        },
        draw: function () {
            if (!this.tiles.loaded) {
                return;
            }
            if (this.preRender) {
                this.drawPreRendered();
            } else {
                this.drawTiled();
            }
        },
        drawPreRendered: function () {
            if (!this.preRenderedChunks) {
                this.preRenderMapToChunks();
            }
            var dx = ig.system.getDrawPos(this.scroll.x),
                dy = ig.system.getDrawPos(this.scroll.y);
            if (this.repeat) {
                dx %= this.width * this.tilesize * ig.system.scale;
                dy %= this.height * this.tilesize * ig.system.scale;
            }
            var minChunkX = Math.max((dx / this.chunkSize).floor(), 0),
                minChunkY = Math.max((dy / this.chunkSize).floor(), 0),
                maxChunkX = ((dx + ig.system.realWidth) / this.chunkSize).ceil(),
                maxChunkY = ((dy + ig.system.realHeight) / this.chunkSize).ceil(),
                maxRealChunkX = this.preRenderedChunks[0].length,
                maxRealChunkY = this.preRenderedChunks.length;
            if (!this.repeat) {
                maxChunkX = Math.min(maxChunkX, maxRealChunkX);
                maxChunkY = Math.min(maxChunkY, maxRealChunkY);
            }
            var nudgeY = 0;
            for (var cy = minChunkY; cy < maxChunkY; cy++) {
                var nudgeX = 0;
                for (var cx = minChunkX; cx < maxChunkX; cx++) {
                    var chunk = this.preRenderedChunks[cy % maxRealChunkY][cx % maxRealChunkX];
                    var x = -dx + cx * this.chunkSize - nudgeX;
                    var y = -dy + cy * this.chunkSize - nudgeY;
                    ig.system.context.drawImage(chunk, x, y);
                    if (this.debugChunks) {
                        ig.system.context.strokeStyle = '#f0f';
                        ig.system.context.strokeRect(x, y, this.chunkSize, this.chunkSize);
                    }
                    if (this.repeat && chunk.width < this.chunkSize && x + chunk.width < ig.system.realWidth) {
                        nudgeX = this.chunkSize - chunk.width;
                        maxChunkX++;
                    }
                }
                if (this.repeat && chunk.height < this.chunkSize && y + chunk.height < ig.system.realHeight) {
                    nudgeY = this.chunkSize - chunk.height;
                    maxChunkY++;
                }
            }
        },
        drawTiled: function () {
            var tile = 0,
                anim = null,
                tileOffsetX = (this.scroll.x / this.tilesize).toInt(),
                tileOffsetY = (this.scroll.y / this.tilesize).toInt(),
                pxOffsetX = this.scroll.x % this.tilesize,
                pxOffsetY = this.scroll.y % this.tilesize,
                pxMinX = -pxOffsetX - this.tilesize,
                pxMinY = -pxOffsetY - this.tilesize,
                pxMaxX = ig.system.width + this.tilesize - pxOffsetX,
                pxMaxY = ig.system.height + this.tilesize - pxOffsetY;
            for (var mapY = -1, pxY = pxMinY; pxY < pxMaxY; mapY++, pxY += this.tilesize) {
                var tileY = mapY + tileOffsetY;
                if (tileY >= this.height || tileY < 0) {
                    if (!this.repeat) {
                        continue;
                    }
                    tileY = tileY > 0 ? tileY % this.height : ((tileY + 1) % this.height) + this.height - 1;
                }
                for (var mapX = -1, pxX = pxMinX; pxX < pxMaxX; mapX++, pxX += this.tilesize) {
                    var tileX = mapX + tileOffsetX;
                    if (tileX >= this.width || tileX < 0) {
                        if (!this.repeat) {
                            continue;
                        }
                        tileX = tileX > 0 ? tileX % this.width : ((tileX + 1) % this.width) + this.width - 1;
                    }
                    if ((tile = this.data[tileY][tileX])) {
                        if ((anim = this.anims[tile - 1])) {
                            anim.draw(pxX, pxY);
                        } else {
                            this.tiles.drawTile(pxX, pxY, tile - 1, this.tilesize);
                        }
                    }
                }
            }
        }
    });
});

// lib/impact/game.js
ig.baked = true;
ig.module('impact.game').requires('impact.impact', 'impact.entity', 'impact.collision-map', 'impact.background-map').defines(function () {
    ig.Game = ig.Class.extend({
        clearColor: '#000000',
        gravity: 0,
        screen: {
            x: 0,
            y: 0
        },
        entities: [],
        namedEntities: {},
        collisionMap: ig.CollisionMap.staticNoCollision,
        backgroundMaps: [],
        backgroundAnims: {},
        cellSize: 64,
        _deferredKill: [],
        _levelToLoad: null,
        loadLevel: function (data) {
            this.screen = {
                x: 0,
                y: 0
            };
            this.entities = [];
            this.namedEntities = {};
            for (var i = 0; i < data.entities.length; i++) {
                var ent = data.entities[i];
                this.spawnEntity(ent.type, ent.x, ent.y, ent.settings);
            }
            this.sortEntities();
            this.collisionMap = null;
            this.backgroundMaps = [];
            for (var i = 0; i < data.layer.length; i++) {
                var ld = data.layer[i];
                if (ld.name == 'collision') {
                    this.collisionMap = new ig.CollisionMap(ld.tilesize, ld.data);
                } else {
                    var newMap = new ig.BackgroundMap(ld.tilesize, ld.data, ld.tilesetName);
                    newMap.anims = this.backgroundAnims[ld.tilesetName] || {};
                    newMap.repeat = ld.repeat;
                    newMap.distance = ld.distance;
                    this.backgroundMaps.push(newMap);
                }
            }
        },
        loadLevelDeferred: function (data) {
            this._levelToLoad = data;
        },
        getEntityByName: function (name) {
            return this.namedEntities[name];
        },
        getEntitiesByType: function (type) {
            var entityClass = typeof (type) === 'string' ? ig.global[type] : type;
            var a = [];
            for (var i = 0; i < this.entities.length; i++) {
                var ent = this.entities[i];
                if (ent instanceof entityClass && !ent._killed) {
                    a.push(ent);
                }
            }
            return a;
        },
        spawnEntity: function (type, x, y, settings) {
            var entityClass = typeof (type) === 'string' ? ig.global[type] : type;
            if (!entityClass) {
                throw ("Can't spawn entity of type " + type);
            }
            var ent = new(entityClass)(x, y, settings || {});
            this.entities.push(ent);
            if (ent.name) {
                this.namedEntities[ent.name] = ent;
            }
            return ent;
        },
        sortEntities: function () {
            this.entities.sort(function (a, b) {
                return a.zIndex - b.zIndex;
            });
        },
        removeEntity: function (ent) {
            if (ent.name) {
                delete this.namedEntities[ent.name];
            }
            ent._killed = true;
            ent.checkAgainst = ig.Entity.TYPE.NONE;
            ent.collides = ig.Entity.COLLIDES.NEVER;
            this._deferredKill.push(ent);
        },
        run: function () {
            this.update();
            this.draw();
        },
        update: function () {
            if (this._levelToLoad) {
                this.loadLevel(this._levelToLoad);
                this._levelToLoad = null;
            }
            for (var i = 0; i < this.entities.length; i++) {
                var ent = this.entities[i];
                if (!ent._killed) {
                    ent.update();
                }
            }
            this.checkEntities();
            for (var i = 0; i < this._deferredKill.length; i++) {
                this.entities.erase(this._deferredKill[i]);
            }
            this._deferredKill = [];
            for (var tileset in this.backgroundAnims) {
                var anims = this.backgroundAnims[tileset];
                for (var a in anims) {
                    anims[a].update();
                }
            }
            for (var i = 0; i < this.backgroundMaps.length; i++) {
                this.backgroundMaps[i].setScreenPos(this.screen.x, this.screen.y);
            }
        },
        draw: function () {
            ig.system.clear(this.clearColor);
            for (var i = 0; i < this.backgroundMaps.length; i++) {
                this.backgroundMaps[i].draw();
            }
            for (var i = 0; i < this.entities.length; i++) {
                this.entities[i].draw();
            }
        },
        checkEntities: function () {
            var hash = {};
            for (var e = 0; e < this.entities.length; e++) {
                var entity = this.entities[e];
                if (e.type == ig.Entity.TYPE.NONE && e.checkAgainst == ig.Entity.TYPE.NONE && e.collides == ig.Entity.COLLIDES.NEVER) {
                    continue;
                }
                var checked = {},
                    xmin = (entity.pos.x / this.cellSize).floor(),
                    ymin = (entity.pos.y / this.cellSize).floor(),
                    xmax = ((entity.pos.x + entity.size.x) / this.cellSize).floor() + 1,
                    ymax = ((entity.pos.y + entity.size.y) / this.cellSize).floor() + 1;
                for (var x = xmin; x < xmax; x++) {
                    for (var y = ymin; y < ymax; y++) {
                        if (!hash[x]) {
                            hash[x] = {};
                            hash[x][y] = [entity];
                        } else if (!hash[x][y]) {
                            hash[x][y] = [entity];
                        } else {
                            var cell = hash[x][y];
                            for (var c = 0; c < cell.length; c++) {
                                if (entity.touches(cell[c]) && !checked[cell[c].id]) {
                                    checked[cell[c].id] = true;
                                    ig.Entity.checkPair(entity, cell[c]);
                                }
                            }
                            cell.push(entity);
                        }
                    }
                }
            }
        }
    });
});

// lib/game/entities/finalData.js
ig.baked = true;
ig.module('game.entities.finalData').defines(function () {
    if (ig.ua.iPhone || ig.ua.iPhone4 || ig.ua.android || ig.ua.firefox) {
        FinalData = {
            SCREEN_WIDTH: 320,
            SCREEN_HEIGHT: 480,
            SPLASH_SCREEN_VIEW: 0,
            COVER_SCREEN_VIEW: 60,
            MAIN_MENU_VIEW: 1,
            HELP_VIEW: 3,
            MOREGAME_VIEW: 4,
            HIGH_SCORES_VIEW: 5,
            ABOUT_VIEW: 6,
            INSTRUCTION_VIEW: 7,
            HIDE_SCREEN_VIEW: 10,
            MAIN_GAME_VIEW: 11,
            LEVEL_COMPLETE_VIEW: 12,
            GAME_OVER_VIEW: 13,
            INPUT_NAME_VIEW: 14,
            QUIT_VIEW: 15,
            img_splash: 'media/iPhone/Tweensoft-logo320x480-splash.jpg',
            img_cover: 'media/iPhone/WhacCover.jpg',
            img_backgroud: 'media/iPhone/WhacMenuBack.jpg',
            img_start_game_button: 'media/iPhone/WhacbuttonStargame.png',
            img_help_button: 'media/iPhone/WhacbuttonHelp.png',
            img_highscore_button: 'media/iPhone/WhacbuttonHighscore.png',
            img_about_button: 'media/iPhone/WhacbuttonAbout.png',
            img_moregame_button: 'media/iPhone/WhacbuttonMoreGames.png',
            img_instruction: 'media/iPhone/WhacInstructions.png',
            img_help: 'media/iPhone/WhacHelp.png',
            img_about: 'media/iPhone/WhacAbout.png',
            img_highscore: 'media/iPhone/WhacHighScore.png',
            img_game_background: 'media/iPhone/game-background.jpg',
            img_timelevelscore: 'media/iPhone/timelevelscore.png',
            img_number_font: 'media/iPhone/numbers.png',
            img_black_penguin: 'media/iPhone/blackpenguin.png',
            img_red_penguin: 'media/iPhone/badredpenguin.png',
            img_green_penguin: 'media/iPhone/badgreenpenguin.png',
            img_shark: 'media/iPhone/shark.png',
            img_wave: 'media/iPhone/wave.png',
            img_levelComplete: 'media/iPhone/Levelcompleted.png',
            img_tap_to_continue1: 'media/iPhone/taptocontinue.png',
            img_penguin_craked: 'media/iPhone/penguincraked.png',
            img_fish_craked: 'media/iPhone/fishcraked.png',
            img_igloo_craked: 'media/iPhone/igloocraked.png',
            img_miss: 'media/iPhone/miss.png',
            img_score_font: 'media/iPhone/scorenumbers.png',
            img_fault: 'media/iPhone/fault.png',
            img_igloo: 'media/iPhone/igloo.png',
            img_snowman: 'media/iPhone/snowman.png',
            img_fish: 'media/iPhone/fish.png',
            img_snowbear: 'media/iPhone/snowbear.png',
            img_life: 'media/iPhone/lifes.png',
            img_empty_box: 'media/iPhone/empty.png',
            img_totalscore_text: 'media/iPhone/totalscore.png',
            img_gameover_text: 'media/iPhone/GameOver.png',
            img_playagain_text: 'media/iPhone/playagain.png',
            img_enterhighscore_text: 'media/iPhone/enterhighscore.png',
            img_highscore_font: 'media/iPhone/highscoretext.png',
            img_back_button: 'media/iPhone/back.png',
            img_bad_life: 'media/iPhone/badlife.png',
            img_kill: 'media/iPhone/kill_star.png',
            img_ice_crake: 'media/iPhone/ice_craked.png',
            img_quit_button: 'media/iPhone/quitbutton.png',
            img_quit_message: 'media/iPhone/Quitgametext.png',
            img_levelComplete_font: 'media/iPhone/levelCompleteFont.png',
            SPLASH_TIME: 2,
            MENU_ITEM_POSX: 55,
            TOP_Y: 0,
            EDIT_HEIGHT: 320,
            QUIT_BUTTON_INTENT: 20,
            LETTER_SIZE: 14,
            MENU_ITEM_SIZE: {
                x: 216,
                y: 65
            },
            TIMELEVELSCORE_SIZE: {
                x: 263,
                y: 48
            },
            TIME_BOX_SIZE: {
                x: 52,
                y: 20
            },
            LEVEL_BOX_SIZE: {
                x: 52,
                y: 20
            },
            SCORE_BOX_SIZE: {
                x: 58,
                y: 20
            },
            PENGUIN_SIZE: {
                x: 88,
                y: 70
            },
            WEAV_SIZE: {
                x: 120,
                y: 60
            },
            LEVELCOMPLETE_SIZE: {
                x: 164,
                y: 103
            },
            TAP_TO_CONTINUE1: {
                x: 169,
                y: 52
            },
            HOLE_SIZE: {
                x: 80,
                y: 66
            },
            CRAKED_SIZE: {
                x: 113,
                y: 94
            },
            MISS_SIZE: {
                x: 55,
                y: 36
            },
            LIFE_SIZE: {
                x: 30,
                y: 39
            },
            PLAYAGAINTEXT_SIZE: {
                x: 119,
                y: 35
            },
            ENTERHIGHSCORE_SIZE: {
                x: 161,
                y: 35
            },
            BACK_BUTTON_SIZE: {
                x: 60,
                y: 28
            },
            BAD_LIFE_SIZE: {
                x: 26,
                y: 42
            },
            KILL_SIZE: {
                x: 113,
                y: 94
            },
            QUIT_BUTTON_SIZE: {
                x: 30,
                y: 30
            },
            YES_SIZE: {
                x: 44,
                y: 40
            },
            NO_SIZE: {
                x: 36,
                y: 40
            },
            img_start_posY: 138,
            img_help_posY: 198,
            img_highscore_posY: 198,
            img_about_posY: 260,
            img_moregame_posY: 320,
            TIMELEVELSCORE_POS: {
                x: 30,
                y: 19
            },
            LEVELBOX_POS: {
                x: 127,
                y: 53
            },
            TIMEBOX_POS: {
                x: 40,
                y: 53
            },
            SCOREBOX_POS: {
                x: 220,
                y: 53
            },
            LEVELCOMPLETE_POS: {
                x: 78,
                y: 176
            },
            TAP_TO_CONTINUE_POS: {
                x: 79,
                y: 352
            },
            LIFE_POS: {
                x: 285,
                y: 410
            },
            BAD_LIFE_POS: {
                x: 10,
                y: 410
            },
            TOTALSCORE_TEXT_POS: {
                x: 77,
                y: 120
            },
            TOTALSCORE_POS: {
                x: 130,
                y: 180
            },
            GAMEOVER_TEXT_POS: {
                x: 80,
                y: 228
            },
            PLAYAGAINTEXT_POS: {
                x: 105,
                y: 297
            },
            ENTERHIGHSCORE_POS: {
                x: 80,
                y: 393
            },
            INPUTNAME_TEXT_POS: {
                x: 80,
                y: 120
            },
            BACK_BUTTON_POS: {
                x: 5,
                y: 420
            },
            INPUTNAME_POS: {
                x: 165,
                y: 230
            },
            QUIT_BUTTON_POS: {
                x: 1,
                y: 30
            },
            QUIT_MESSAGE_POS: {
                x: 75,
                y: 180
            },
            YES_INTENT_POS: {
                x: 12,
                y: 80
            },
            NO_INTENT_POS: {
                x: 128,
                y: 80
            },
            LEVELCOMPLETE_LEVEL_POS: {
                x: 100,
                y: 150
            },
            HOLE_POSX: [11, 117, 221, 11, 117, 221, 11, 117, 221],
            HOLE_POSY: [105, 105, 105, 208, 208, 208, 330, 330, 330],
            WAVE_INTENT: {
                x: 14,
                y: 0
            },
            CRAKED_INTENT: {
                x: 11,
                y: 27
            },
            MISS_INTENT: {
                x: 16,
                y: 18
            },
            POPSCORE_INTENT: {
                x: 45,
                y: -2
            },
            HIGHSCORE_INTENT: {
                x: 82,
                y: 106
            },
            HIGHSCORE_NAME_INTENT: 27,
            HIGHSCORE_SCORE_INTENT: 140,
            HIGHSCORE_LINE_SPACE: 24,
            ENTERNAME_TEMP_VALUE: -50,
            INPUTNAME_INTENT: -50,
            INIT_TIME: 60,
            HOLE_COUNT: 9,
            PENGUIN_COLOR_ARRAY: ['black', 'green', 'red'],
            POPUP_TIME: 0.2,
            INITPOPUP_TIME: 0.8,
            INIT_POPDOWN: 0.8,
            INITSTAY_TIME: 3,
            CRAKE_STAY_TIME: 0.5,
            MISS_TIME: 0.6,
            TOTAL_ITEMS_COUNT: [60, 60, 60, 70, 70, 70, 80, 80, 80, 90, 90, 90, 100, 100, 100],
            GOOD_ITEMS_COUNT: [60, 50, 45, 55, 52, 50, 60, 58, 56, 66, 64, 62, 72, 68, 65],
            MAX_LIFE: 3,
            MAX_BAD_LIFE: 4,
            MAX_LEVEL: 15,
            LEVEL_GOOD_ITEMS: [['1tabpenguin'], ['1tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ],
            LEVEL_BAD_ITEMS: [['shark'], ['shark'], ['shark', 'snowman'], ['shark', 'snowman'], ['shark', 'snowman'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ],
            POPUP_SCORE_MOVETIME: 0.2,
            POPUP_MOVEHEIGHT: 20,
            POPUP_STAYTIME: 0.5,
            POPUPSCORE_INTENT: 35,
            MAX_HIGHSCORE_COUNT: 10,
        };
    } else {
        FinalData = {
            SCREEN_WIDTH: 700,
            SCREEN_HEIGHT: 881,
            SPLASH_SCREEN_VIEW: 0,
            COVER_SCREEN_VIEW: 60,
            MAIN_MENU_VIEW: 1,
            HELP_VIEW: 3,
            MOREGAME_VIEW: 4,
            HIGH_SCORES_VIEW: 5,
            ABOUT_VIEW: 6,
            INSTRUCTION_VIEW: 7,
            HIDE_SCREEN_VIEW: 10,
            MAIN_GAME_VIEW: 11,
            LEVEL_COMPLETE_VIEW: 12,
            GAME_OVER_VIEW: 13,
            INPUT_NAME_VIEW: 14,
            QUIT_VIEW: 15,
            img_splash: 'media/iPad/Tweensoft-logo320x480-splash.jpg',
            img_cover: 'media/iPad/WhacCover.jpg',
            img_backgroud: 'media/iPad/WhacMenuBack.jpg',
            img_start_game_button: 'media/iPad/WhacbuttonStargame.png',
            img_help_button: 'media/iPad/WhacbuttonHelp.png',
            img_highscore_button: 'media/iPad/WhacbuttonHighscore.png',
            img_about_button: 'media/iPad/WhacbuttonAbout.png',
            img_moregame_button: 'media/iPad/WhacbuttonMoreGames.png',
            img_instruction: 'media/iPad/WhacInstructions.png',
            img_help: 'media/iPad/WhacHelp.png',
            img_about: 'media/iPad/WhacAbout.png',
            img_highscore: 'media/iPad/WhacHighScore.png',
            img_game_background: 'media/iPad/game-background.jpg',
            img_timelevelscore: 'media/iPad/timelevelscore.png',
            img_number_font: 'media/iPad/numbers.png',
            img_black_penguin: 'media/iPad/blackpenguin.png',
            img_red_penguin: 'media/iPad/badredpenguin.png',
            img_green_penguin: 'media/iPad/badgreenpenguin.png',
            img_shark: 'media/iPad/shark.png',
            img_wave: 'media/iPad/wave.png',
            img_levelComplete: 'media/iPad/Levelcompleted.png',
            img_tap_to_continue1: 'media/iPad/taptocontinue.png',
            img_penguin_craked: 'media/iPad/penguincraked.png',
            img_fish_craked: 'media/iPad/fishcraked.png',
            img_igloo_craked: 'media/iPad/igloocraked.png',
            img_miss: 'media/iPad/miss.png',
            img_score_font: 'media/iPad/scorenumbers.png',
            img_fault: 'media/iPad/fault.png',
            img_igloo: 'media/iPad/igloo.png',
            img_snowman: 'media/iPad/snowman.png',
            img_fish: 'media/iPad/fish.png',
            img_snowbear: 'media/iPad/snowbear.png',
            img_life: 'media/iPad/lifes.png',
            img_empty_box: 'media/iPad/empty.png',
            img_totalscore_text: 'media/iPad/totalscore.png',
            img_gameover_text: 'media/iPad/GameOver.png',
            img_playagain_text: 'media/iPad/playagain.png',
            img_enterhighscore_text: 'media/iPad/enterhighscore.png',
            img_highscore_font: 'media/iPad/highscoretext.png',
            img_back_button: 'media/iPad/back.png',
            img_bad_life: 'media/iPad/badlife.png',
            img_kill: 'media/iPad/kill_star.png',
            img_ice_crake: 'media/iPad/ice_craked.png',
            img_quit_button: 'media/iPad/quitbutton.png',
            img_quit_message: 'media/iPad/Quitgametext.png',
            img_levelComplete_font: 'media/iPad/levelCompleteFont.png',
            SPLASH_TIME: 2,
            MENU_ITEM_POSX: 120,
            TOP_Y: 0,
            EDIT_HEIGHT: 600,
            PHONE_TURN_TIME: 1,
            QUIT_BUTTON_INTENT: 0,
            LETTER_SIZE: 14,
            MENU_ITEM_SIZE: {
                x: 469,
                y: 139
            },
            TIMELEVELSCORE_SIZE: {
                x: 263,
                y: 48
            },
            TIME_BOX_SIZE: {
                x: 52,
                y: 20
            },
            LEVEL_BOX_SIZE: {
                x: 52,
                y: 20
            },
            SCORE_BOX_SIZE: {
                x: 58,
                y: 20
            },
            PENGUIN_SIZE: {
                x: 140,
                y: 125
            },
            WEAV_SIZE: {
                x: 119,
                y: 52
            },
            LEVELCOMPLETE_SIZE: {
                x: 164,
                y: 103
            },
            TAP_TO_CONTINUE1: {
                x: 169,
                y: 52
            },
            HOLE_SIZE: {
                x: 150,
                y: 130
            },
            CRAKED_SIZE: {
                x: 200,
                y: 240
            },
            MISS_SIZE: {
                x: 103,
                y: 58
            },
            LIFE_SIZE: {
                x: 65,
                y: 75
            },
            PLAYAGAINTEXT_SIZE: {
                x: 208,
                y: 72
            },
            ENTERHIGHSCORE_SIZE: {
                x: 300,
                y: 63
            },
            BACK_BUTTON_SIZE: {
                x: 118,
                y: 54
            },
            BAD_LIFE_SIZE: {
                x: 57,
                y: 53
            },
            KILL_SIZE: {
                x: 200,
                y: 240
            },
            QUIT_BUTTON_SIZE: {
                x: 30,
                y: 30
            },
            YES_SIZE: {
                x: 90,
                y: 60
            },
            NO_SIZE: {
                x: 60,
                y: 50
            },
            img_start_posY: 195,
            img_help_posY: 354,
            img_highscore_posY: 354,
            img_about_posY: 511,
            img_moregame_posY: 672,
            TIMELEVELSCORE_POS: {
                x: 102,
                y: 19
            },
            LEVELBOX_POS: {
                x: 335,
                y: 80
            },
            TIMEBOX_POS: {
                x: 125,
                y: 80
            },
            SCOREBOX_POS: {
                x: 523,
                y: 80
            },
            LEVELCOMPLETE_POS: {
                x: 205,
                y: 380
            },
            TAP_TO_CONTINUE_POS: {
                x: 216,
                y: 735
            },
            LIFE_POS: {
                x: 640,
                y: 800
            },
            BAD_LIFE_POS: {
                x: 10,
                y: 820
            },
            TOTALSCORE_TEXT_POS: {
                x: 210,
                y: 220
            },
            TOTALSCORE_POS: {
                x: 308,
                y: 324
            },
            GAMEOVER_TEXT_POS: {
                x: 216,
                y: 414
            },
            PLAYAGAINTEXT_POS: {
                x: 256,
                y: 530
            },
            ENTERHIGHSCORE_POS: {
                x: 210,
                y: 700
            },
            INPUTNAME_TEXT_POS: {
                x: 210,
                y: 180
            },
            BACK_BUTTON_POS: {
                x: 5,
                y: 820
            },
            INPUTNAME_POS: {
                x: 365,
                y: 430
            },
            QUIT_BUTTON_POS: {
                x: 1,
                y: 0
            },
            QUIT_MESSAGE_POS: {
                x: 220,
                y: 330
            },
            YES_INTENT_POS: {
                x: 15,
                y: 150
            },
            NO_INTENT_POS: {
                x: 220,
                y: 150
            },
            LEVELCOMPLETE_LEVEL_POS: {
                x: 425,
                y: 385
            },
            HOLE_POSX: [78, 272, 460, 78, 272, 460, 78, 272, 460],
            HOLE_POSY: [192, 192, 192, 384, 384, 384, 612, 612, 612],
            WAVE_INTENT: {
                x: -18,
                y: 35
            },
            CRAKED_INTENT: {
                x: 11,
                y: 100
            },
            MISS_INTENT: {
                x: 28,
                y: 32
            },
            POPSCORE_INTENT: {
                x: 45,
                y: -2
            },
            HIGHSCORE_INTENT: {
                x: 182,
                y: 236
            },
            HIGHSCORE_NAME_INTENT: 45,
            HIGHSCORE_SCORE_INTENT: 240,
            HIGHSCORE_LINE_SPACE: 50,
            ENTERNAME_TEMP_VALUE: 0,
            INPUTNAME_INTENT: 0,
            INIT_TIME: 60,
            HOLE_COUNT: 9,
            PENGUIN_COLOR_ARRAY: ['black', 'green', 'red'],
            POPUP_TIME: 0.2,
            INITPOPUP_TIME: 0.8,
            INIT_POPDOWN: 0.8,
            INITSTAY_TIME: 3,
            CRAKE_STAY_TIME: 0.5,
            MISS_TIME: 0.6,
            TOTAL_ITEMS_COUNT: [60, 60, 60, 70, 70, 70, 80, 80, 80, 90, 90, 90, 100, 100, 100],
            GOOD_ITEMS_COUNT: [60, 50, 45, 55, 52, 50, 60, 58, 56, 66, 64, 62, 72, 68, 65],
            MAX_LIFE: 3,
            MAX_BAD_LIFE: 4,
            MAX_LEVEL: 15,
            LEVEL_GOOD_ITEMS: [['1tabpenguin'], ['1tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ['1tabpenguin', '2tabpenguin', '1tabigloo', '2tabfish', '3tabpenguin', '2tabigloo', '4tabpenguin', '3tabfish'], ],
            LEVEL_BAD_ITEMS: [['shark'], ['shark'], ['shark', 'snowman'], ['shark', 'snowman'], ['shark', 'snowman'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ['shark', 'snowman', 'snowbear'], ],
            POPUP_SCORE_MOVETIME: 0.2,
            POPUP_MOVEHEIGHT: 20,
            POPUP_STAYTIME: 0.5,
            POPUPSCORE_INTENT: 75,
            MAX_HIGHSCORE_COUNT: 10,
        };
    }
    if (ig.ua.iPhone4 || ig.ua.iPhone) {
        if (ig.ua.viewport.height == 480 || ig.ua.viewport.height == 320) {
            FinalData.TOP_Y = 0;
        } else {
            FinalData.TOP_Y = -30;
        }
    } else if (ig.ua.iPad) {
        if (ig.ua.viewport.height == 1024 || ig.ua.viewport.height == 768) {
            FinalData.TOP_Y = 0;
        } else {
            FinalData.TOP_Y = 0;
        }
    } else {
        FinalData.TOP_Y = 0;
    }
});

// lib/game/entities/menu.js
ig.baked = true;
ig.module('game.entities.menu').requires('impact.entity', 'impact.font').defines(function () {
    EntityMenu = ig.Entity.extend({
        currentView: 0,
        engine: null,
        arrHighScore: [],
        splashTimer: new ig.Timer(),
        highScoreFont: new ig.Font(FinalData.img_highscore_font),
        imgSplash: new ig.Image(FinalData.img_splash),
        imgCover: new ig.Image(FinalData.img_cover),
        imgBackground: new ig.Image(FinalData.img_backgroud),
        imgStartGameButton: new ig.Image(FinalData.img_start_game_button),
        imgHelpButton: new ig.Image(FinalData.img_help_button),
        imgHighScoreButton: new ig.Image(FinalData.img_highscore_button),
        imgAboutButton: new ig.Image(FinalData.img_about_button),
        imgMoreGameButton: new ig.Image(FinalData.img_moregame_button),
        imgInstruction: new ig.Image(FinalData.img_instruction),
        imgHelp: new ig.Image(FinalData.img_help),
        imgAbout: new ig.Image(FinalData.img_about),
        imgHighScore: new ig.Image(FinalData.img_highscore),
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            this.splashTimer.set(FinalData.SPLASH_TIME);
            this.openView(FinalData.SPLASH_SCREEN_VIEW);
        },
        update: function () {
            this.parent();
        },
        draw: function () {
            this.parent();
            this.keyCheck();
            switch (this.currentView) {
            case FinalData.SPLASH_SCREEN_VIEW:
                if (this.splashTimer.delta() > 0) {
                    this.openView(FinalData.COVER_SCREEN_VIEW);
                    this.splashTimer = null;
                }
                this.imgSplash.draw(this.pos.x, this.pos.y);
                break;
            case FinalData.COVER_SCREEN_VIEW:
                this.imgCover.draw(0, this.pos.y);
                break;
            case FinalData.MAIN_MENU_VIEW:
                this.imgBackground.draw(0, this.pos.y);
                this.imgStartGameButton.draw(FinalData.MENU_ITEM_POSX, this.pos.y + FinalData.img_start_posY);
                this.imgHelpButton.draw(FinalData.MENU_ITEM_POSX, this.pos.y + FinalData.img_help_posY);
                this.imgAboutButton.draw(FinalData.MENU_ITEM_POSX, this.pos.y + FinalData.img_about_posY);
                break;
            case FinalData.INSTRUCTION_VIEW:
                this.imgBackground.draw(0, this.pos.y);
                this.imgInstruction.draw(0, this.pos.y);
                break;
            case FinalData.HELP_VIEW:
                this.imgBackground.draw(0, this.pos.y);
                this.imgHelp.draw(0, this.pos.y);
                break;
            case FinalData.HIGH_SCORES_VIEW:
                this.imgBackground.draw(0, this.pos.y);
                this.imgHighScore.draw(0, this.pos.y);
                this.showHighScore();
                break;
            case FinalData.ABOUT_VIEW:
                this.imgBackground.draw(0, this.pos.y);
                this.imgAbout.draw(0, this.pos.y);
                break;
            case FinalData.MOREGAME_VIEW:
                this.imgBackground.draw(0, this.pos.y);
                break;
            }
        },
        getHighScoreData: function () {
            var that = this;
            var req = $.ajax({
                url: ("game.php" + '?nocache=' + Math.random()),
                data: {
                    a: 'l'
                },
                dataType: 'json',
                async: false,
                success: function (scores) {
                    that.arrHighScore = scores;
                }
            });
        },
        openView: function (view) {
            switch (view) {
            case FinalData.HIGH_SCORES_VIEW:
                this.getHighScoreData();
                break;
            case FinalData.MOREGAME_VIEW:
                location.href = 'http://www.celanderus.com/appstore';
                break;
            }
            this.currentView = view;
        },
        keyCheck: function () {
            if (ig.input.pressed('mouse')) {
                var x = ig.input.mouse.x;
                var y = ig.input.mouse.y;
                switch (this.currentView) {
                case FinalData.COVER_SCREEN_VIEW:
                    this.openView(FinalData.MAIN_MENU_VIEW);
                    break;
                case FinalData.MAIN_MENU_VIEW:
                    this.keyCheckInMainMenu(x, y);
                    break;
                case FinalData.HELP_VIEW:
                case FinalData.HIGH_SCORES_VIEW:
                case FinalData.ABOUT_VIEW:
                    this.openView(FinalData.MAIN_MENU_VIEW);
                    break;
                case FinalData.INSTRUCTION_VIEW:
                    var self = this;
                    adManager.showAd(function () {
                       self.engine.startGame();
                    });
                    break;
                }
            }
        },
        keyCheckInMainMenu: function (x, y) {
            if (x >= FinalData.MENU_ITEM_POSX && x <= FinalData.MENU_ITEM_POSX + FinalData.MENU_ITEM_SIZE.x && y >= this.pos.y + FinalData.img_start_posY && y <= this.pos.y + FinalData.img_start_posY + FinalData.MENU_ITEM_SIZE.y) {
                this.openView(FinalData.INSTRUCTION_VIEW);
                return;
            }
            if (x >= FinalData.MENU_ITEM_POSX && x <= FinalData.MENU_ITEM_POSX + FinalData.MENU_ITEM_SIZE.x && y >= this.pos.y + FinalData.img_help_posY && y <= FinalData.img_help_posY + this.pos.y + FinalData.MENU_ITEM_SIZE.y) {
                this.openView(FinalData.HELP_VIEW);
                return;
            }
            if (x >= FinalData.MENU_ITEM_POSX && x <= FinalData.MENU_ITEM_POSX + FinalData.MENU_ITEM_SIZE.x && y >= this.pos.y + FinalData.img_about_posY && y <= this.pos.y + FinalData.img_about_posY + FinalData.MENU_ITEM_SIZE.y) {
                this.openView(FinalData.ABOUT_VIEW);
                return;
            }
        },
        showHighScore: function () {
            for (i = 0; i < 10; i++) {
                this.highScoreFont.draw(i + 1 + '.', FinalData.HIGHSCORE_INTENT.x, this.pos.y + FinalData.HIGHSCORE_LINE_SPACE * i + FinalData.HIGHSCORE_INTENT.y);
                if (this.arrHighScore[i] != undefined) {
                    this.highScoreFont.draw(this.arrHighScore[i].name, FinalData.HIGHSCORE_INTENT.x + FinalData.HIGHSCORE_NAME_INTENT, this.pos.y + FinalData.HIGHSCORE_LINE_SPACE * i + FinalData.HIGHSCORE_INTENT.y);
                    this.highScoreFont.draw(this.arrHighScore[i].score, FinalData.HIGHSCORE_INTENT.x + FinalData.HIGHSCORE_SCORE_INTENT, this.pos.y + FinalData.HIGHSCORE_LINE_SPACE * i + FinalData.HIGHSCORE_INTENT.y);
                } else {
                    this.highScoreFont.draw('..............', FinalData.HIGHSCORE_INTENT.x + FinalData.HIGHSCORE_NAME_INTENT, this.pos.y + FinalData.HIGHSCORE_LINE_SPACE * i + FinalData.HIGHSCORE_INTENT.y);
                    this.highScoreFont.draw('.....', FinalData.HIGHSCORE_INTENT.x + FinalData.HIGHSCORE_SCORE_INTENT, this.pos.y + FinalData.HIGHSCORE_LINE_SPACE * i + FinalData.HIGHSCORE_INTENT.y);
                }
            }
        },
    });
});

// lib/game/entities/customlib/box.js
ig.baked = true;
ig.module('game.entities.customlib.box').requires('game.entities.finalData', 'impact.entity').defines(function () {
    EntityBox = ig.Entity.extend({
        imgBox: null,
        imgText: null,
        font: null,
        number: 0,
        init: function (x, y, settings) {
            this.parent(x, y, settings);
        },
        setSize: function (width, height) {
            this.size.x = width;
            this.size.y = height;
        },
        setBoxImage: function (path) {
            this.imgBox = new ig.Image(path);
        },
        setFont: function (path) {
            if (path != undefined) this.font = new ig.Font(path);
        },
        setNumber: function (number) {
            this.number = number;
        },
        getNumber: function () {
            return this.number;
        },
        draw: function () {
            this.parent();
            if (this.imgBox != undefined) this.imgBox.draw(this.pos.x - this.imgBox.width / 2, this.pos.y - this.imgBox.height / 2);
            if (this.font != undefined) {
                var str = this.number + '';
                var x = Math.floor(this.pos.x + this.size.x / 2 - str.length * FinalData.LETTER_SIZE / 2);
                this.font.draw(this.number, x, this.pos.y);
            }
        }
    });
});

// lib/game/entities/levelBox.js
ig.baked = true;
ig.module('game.entities.levelBox').requires('game.entities.customlib.box').defines(function () {
    EntityLevelBox = EntityBox.extend({
        level: 0,
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            this.setSize(FinalData.LEVEL_BOX_SIZE.x, FinalData.LEVEL_BOX_SIZE.y);
            this.setFont(FinalData.img_number_font);
        },
        setLevel: function (level) {
            this.level = level;
            this.setNumber(level);
        },
        getLevel: function () {
            return this.level;
        },
    });
});

// lib/game/entities/timeBox.js
ig.baked = true;
ig.module('game.entities.timeBox').requires('game.entities.customlib.box').defines(function () {
    EntityTimeBox = EntityBox.extend({
        timer: new ig.Timer(),
        stopFlag: false,
        stopTime: 0,
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            this.setSize(FinalData.TIME_BOX_SIZE.x, FinalData.TIME_BOX_SIZE.y);
            this.setFont(FinalData.img_number_font);
            this.setNumber(FinalData.INIT_TIME);
            this.timer.set(FinalData.INIT_TIME);
        },
        setTime: function (time) {
            this.timer.set(time);
            this.setNumber(time);
        },
        getTime: function () {
            return this.number;
        },
        start: function () {
            this.setNumber(FinalData.INIT_TIME);
            this.timer.set(FinalData.INIT_TIME);
            this.stopFlag = false;
        },
        startTimeBox: function () {
            this.setNumber(this.stopTime);
            this.timer.set(this.stopTime);
            this.stopFlag = false;
        },
        initTimeBox: function () {
            this.setNumber(0);
            this.stopFlag = true;
        },
        stop: function () {
            this.stopFlag = true;
            this.stopTime = Math.abs(this.getNumber());
        },
        draw: function () {
            this.parent();
            if (!this.stopFlag) {
                var time = Math.floor(this.timer.delta());
                this.setNumber(time);
            }
        }
    });
});

// lib/game/entities/scoreBox.js
ig.baked = true;
ig.module('game.entities.scoreBox').requires('game.entities.customlib.box').defines(function () {
    EntityScoreBox = EntityBox.extend({
        score: 0,
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            this.setSize(FinalData.SCORE_BOX_SIZE.x, FinalData.SCORE_BOX_SIZE.y);
            this.setFont(FinalData.img_number_font);
        },
        setScore: function (score) {
            this.setNumber(score);
            this.score = score;
        },
        getScore: function () {
            return this.score;
        },
        add: function (score) {
            this.score += score;
            this.setNumber(this.score);
        }
    });
});

// lib/game/entities/craked.js
ig.baked = true;
ig.module('game.entities.craked').requires('game.entities.finalData', 'impact.entity').defines(function () {
    EntityCraked = ig.Entity.extend({
        name: null,
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            switch (this.name) {
            case '1tabpenguin':
            case '2tabpenguin':
            case '3tabpenguin':
            case '4tabpenguin':
                this.animSheet = new ig.AnimationSheet(FinalData.img_penguin_craked, FinalData.CRAKED_SIZE.x, FinalData.CRAKED_SIZE.y);
                break;
            case '2tabfish':
            case '3tabfish':
                this.animSheet = new ig.AnimationSheet(FinalData.img_fish_craked, FinalData.CRAKED_SIZE.x, FinalData.CRAKED_SIZE.y);
                break;
            case '1tabigloo':
            case '2tabigloo':
                this.animSheet = new ig.AnimationSheet(FinalData.img_igloo_craked, FinalData.CRAKED_SIZE.x, FinalData.CRAKED_SIZE.y);
                break;
            }
            this.addAnim('wave', 0.15, [0, 1, 2, 3, 4, 5, 5]);
        },
        update: function () {
            if (this.currentAnim.frame == 6) {
                this.kill();
            }
            this.parent();
        }
    });
});

// lib/game/entities/miss.js
ig.baked = true;
ig.module('game.entities.miss').requires('game.entities.finalData', 'impact.entity').defines(function () {
    EntityMiss = ig.Entity.extend({
        animSheet: new ig.AnimationSheet(FinalData.img_miss, FinalData.MISS_SIZE.x, FinalData.MISS_SIZE.y),
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            this.addAnim('miss', FinalData.MISS_TIME, [0, 0]);
        },
        update: function () {
            if (this.currentAnim.frame == 1) {
                this.kill();
            }
            this.parent();
        },
    });
});

// lib/game/entities/quitButton.js
ig.baked = true;
ig.module('game.entities.quitButton').requires('impact.entity').defines(function () {
    EntityQuitButton = ig.Entity.extend({
        imgButton: new ig.Image(FinalData.img_quit_button),
        init: function (x, y, settings) {
            this.parent(x, y, settings);
        },
        draw: function () {
            this.parent();
            this.imgButton.draw(this.pos.x, this.pos.y);
        }
    });
});

// lib/game/entities/quitMessage.js
ig.baked = true;
ig.module('game.entities.quitMessage').requires('impact.entity').defines(function () {
    EntityQuitMessage = ig.Entity.extend({
        imgQuitMessage: new ig.Image(FinalData.img_quit_message),
        init: function (x, y, settings) {
            this.parent(x, y, settings);
        },
        draw: function () {
            this.parent();
            this.imgQuitMessage.draw(this.pos.x, this.pos.y);
        }
    });
});

// lib/game/entities/popupScore.js
ig.baked = true;
ig.module('game.entities.popupScore').requires('impact.entity').defines(function () {
    EntityPopupScore = ig.Entity.extend({
        score: null,
        startY: 0,
        state: null,
        moveTimer: null,
        faultFlag: false,
        imgFault: null,
        scoreFont: new ig.Font(FinalData.img_score_font),
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            if (this.faultFlag) {
                this.imgFault = new ig.Image(FinalData.img_fault);
            }
            this.moveTimer = new ig.Timer(FinalData.POPUP_SCORE_MOVETIME);
            this.startY = this.pos.y;
            this.state = 'move';
        },
        setScore: function (score) {
            this.score = score;
        },
        getScore: function () {
            return this.score;
        },
        draw: function () {
            this.parent();
            if (this.score != undefined && this.score != 0) {
                this.scoreFont.draw(this.score, this.pos.x, this.pos.y);
            }
            if (this.faultFlag) {
                this.imgFault.draw(this.pos.x, this.pos.y);
            }
            switch (this.state) {
            case 'move':
                this.startMoving();
                break;
            case 'stop':
                this.stop();
                break;
            }
        },
        stop: function () {
            if (this.stopTimer.delta() > 0) {
                this.kill();
            }
        },
        startMoving: function () {
            if (this.moveTimer.delta() >= 0) {
                this.pos.y = this.startY;
                this.stopTimer = new ig.Timer(FinalData.POPUP_STAYTIME);
                this.state = 'stop';
            } else {
                this.pos.y = this.startY + FinalData.POPUP_MOVEHEIGHT * Math.sin(Math.PI * this.moveTimer.delta() / FinalData.POPUP_SCORE_MOVETIME);
            }
        }
    });
});

// lib/game/entities/wave.js
ig.baked = true;
ig.module('game.entities.wave').requires('game.entities.finalData', 'impact.entity').defines(function () {
    EntityWave = ig.Entity.extend({
        animSheet: new ig.AnimationSheet(FinalData.img_wave, FinalData.WEAV_SIZE.x, FinalData.WEAV_SIZE.y),
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            this.addAnim('wave', 0.1, [0, 1, 2, 3, 3]);
        },
        update: function () {
            if (this.currentAnim.frame == 4) {
                this.kill();
            }
            this.parent();
        },
    });
});

// lib/game/entities/starkill.js
ig.baked = true;
ig.module('game.entities.starkill').requires('impact.entity').defines(function () {
    EntityStarkill = ig.Entity.extend({
        item: null,
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            if (this.item.name == '2tabfish' && this.item.tab != 0) {
                this.animSheet = new ig.AnimationSheet(FinalData.img_ice_crake, FinalData.KILL_SIZE.x, FinalData.KILL_SIZE.y);
            } else if (this.item.tab >= 2) {
                this.animSheet = new ig.AnimationSheet(FinalData.img_ice_crake, FinalData.KILL_SIZE.x, FinalData.KILL_SIZE.y);
            } else {
                this.animSheet = new ig.AnimationSheet(FinalData.img_kill, FinalData.KILL_SIZE.x, FinalData.KILL_SIZE.y);
            }
            this.addAnim('kill', 0.1, [0, 1, 2, 3, 4, 4]);
        },
        update: function () {
            this.parent();
            if (this.currentAnim.frame == 5) {
                if (this.item.tab > 0) this.item.crakeFlag = false;
                this.kill();
            }
        }
    });
});

// lib/game/entities/item.js
ig.baked = true;
ig.module('game.entities.item').requires('game.entities.finalData', 'game.entities.popupScore', 'game.entities.miss', 'game.entities.wave', 'game.entities.starkill', 'impact.entity').defines(function () {
    EntityItem = ig.Entity.extend({
        engine: null,
        hole_pos: null,
        name: null,
        originTab: 0,
        tab: null,
        stayTimer: null,
        crakeFlag: false,
        missFlag: true,
        kind: null,
        stayTimeValue: 0,
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            switch (this.name) {
            case '1tabpenguin':
                this.animSheet = new ig.AnimationSheet(FinalData.img_black_penguin, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 1;
                this.kind = 'good';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('pushdown', 0.3, [0, 0, 1, 1]);
                break;
            case '2tabpenguin':
                this.animSheet = new ig.AnimationSheet(FinalData.img_black_penguin, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 2;
                this.kind = 'good';
                this.addAnim('tab1', 0.1, [1]);
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('pushdown', 0.1, [1, 1, 1, 1]);
                break;
            case '3tabpenguin':
                this.animSheet = new ig.AnimationSheet(FinalData.img_black_penguin, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 3;
                this.kind = 'good';
                this.addAnim('stay', 0.1, [2]);
                this.addAnim('tab2', 0.1, [0]);
                this.addAnim('tab1', 0.1, [1]);
                this.addAnim('pushdown', 0.1, [1, 1, 1, 1]);
                break;
            case '4tabpenguin':
                this.animSheet = new ig.AnimationSheet(FinalData.img_black_penguin, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 4;
                this.kind = 'good';
                this.addAnim('stay', 0.1, [2]);
                this.addAnim('tab3', 0.1, [3]);
                this.addAnim('tab2', 0.1, [0]);
                this.addAnim('tab1', 0.1, [1]);
                this.addAnim('pushdown', 0.1, [1, 1, 1, 1]);
                break;
            case '2tabfish':
                this.animSheet = new ig.AnimationSheet(FinalData.img_fish, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 2;
                this.kind = 'good';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('tab1', 0.1, [2]);
                this.addAnim('pushdown', 0.1, [2, 2, 2, 2]);
                break;
            case '3tabfish':
                this.animSheet = new ig.AnimationSheet(FinalData.img_fish, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 3;
                this.kind = 'good';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('tab2', 0.1, [1]);
                this.addAnim('tab1', 0.1, [3]);
                this.addAnim('pushdown', 0.1, [3, 3, 3, 3]);
                break;
            case '1tabigloo':
                this.animSheet = new ig.AnimationSheet(FinalData.img_igloo, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 1;
                this.kind = 'good';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('pushdown', 0.1, [0, 0, 1, 1]);
                break;
            case '2tabigloo':
                this.animSheet = new ig.AnimationSheet(FinalData.img_igloo, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.tab = 2;
                this.kind = 'good';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('tab1', 0.1, [1]);
                this.addAnim('pushdown', 0.1, [1, 1, 1, 1]);
                break;
            case 'shark':
                this.animSheet = new ig.AnimationSheet(FinalData.img_shark, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.kind = 'bad';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('pushdown', 0.1, [0, 0, 1, 1]);
                break;
            case 'snowman':
                this.animSheet = new ig.AnimationSheet(FinalData.img_snowman, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.kind = 'bad';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('pushdown', 0.1, [0, 0, 1, 1]);
                break;
            case 'snowbear':
                this.animSheet = new ig.AnimationSheet(FinalData.img_snowbear, FinalData.PENGUIN_SIZE.x, FinalData.PENGUIN_SIZE.y);
                this.kind = 'bad';
                this.addAnim('stay', 0.1, [0]);
                this.addAnim('pushdown', 0.1, [0, 0, 1, 1]);
                break;
            }
            this.originTab = this.tab;
            this.currentAnim = this.anims.stay;
            this.stayTimer = new ig.Timer(FinalData.INITSTAY_TIME);
        },
        stopItem: function () {
            if (this.stayTimer != undefined) {
                this.stayTimeValue = Math.abs(Math.floor(this.stayTimer.delta()));
                this.stayTimer = null;
            }
        },
        startItem: function () {
            this.stayTimer = new ig.Timer(this.stayTimeValue);
        },
        update: function () {
            this.parent();
            this.changeAnim();
        },
        craked: function () {
            this.tab--;
            if (this.kind == 'good') {
                this.crakeFlag = true;
                ig.game.spawnEntity(EntityStarkill, this.pos.x - FinalData.CRAKED_INTENT.x, this.pos.y - FinalData.CRAKED_INTENT.y, {
                    zIndex: 2,
                    item: this
                });
                ig.game.sortEntities();
            }
            switch (this.name) {
            case '2tabpenguin':
                this.currentAnim = this.anims.tab1;
                break;
            case '3tabpenguin':
                if (this.tab == 2) {
                    this.currentAnim = this.anims.tab2;
                } else {
                    this.currentAnim = this.anims.tab1;
                }
                break;
            case '4tabpenguin':
                if (this.tab == 3) {
                    this.currentAnim = this.anims.tab3;
                } else if (this.tab == 2) {
                    this.currentAnim = this.anims.tab2;
                } else {
                    this.currentAnim = this.anims.tab1;
                }
                break;
            case '2tabfish':
                this.currentAnim = this.anims.tab1;
                break;
            case '3tabfish':
                if (this.tab == 2) {
                    this.currentAnim = this.anims.tab2;
                } else {
                    this.currentAnim = this.anims.tab1;
                }
                break;
            case '2tabigloo':
                this.currentAnim = this.anims.tab1;
                break;
            }
            if (this.tab <= 0) {
                if (this.kind == 'good') {
                    var score = 50 * (this.originTab);
                    ig.game.spawnEntity(EntityPopupScore, this.pos.x + FinalData.POPUPSCORE_INTENT, this.pos.y, {
                        score: '+' + score
                    });
                    this.engine.scoreBox.add(score);
                    this.missFlag = false;
                } else {
                    ig.game.spawnEntity(EntityPopupScore, this.pos.x + FinalData.POPUPSCORE_INTENT, this.pos.y, {
                        faultFlag: true
                    });
                    this.engine.bad_life++;
                }
                this.currentAnim = this.anims.pushdown;
                this.stayTimer = null;
            }
        },
        changeAnim: function () {
            if (this.stayTimer != undefined && this.stayTimer.delta() > 0) {
                this.currentAnim = this.anims.pushdown;
                this.stayTimer = null;
            }
            if (this.currentAnim == this.anims.pushdown && this.currentAnim.frame == 3) {
                this.engine.freeHole(this.hole_pos);
                ig.game.spawnEntity(EntityWave, this.pos.x - FinalData.WAVE_INTENT.x, this.pos.y + FinalData.WAVE_INTENT.y);
                this.kill();
            }
        },
        kill: function () {
            if (this.missFlag && this.kind == 'good') {
                ig.game.spawnEntity(EntityMiss, this.pos.x + FinalData.MISS_INTENT.x, this.pos.y + FinalData.MISS_INTENT.y);
                this.engine.life--;
            }
            this.parent();
        }
    });
});

// lib/game/entities/engine.js
ig.baked = true;
ig.module('game.entities.engine').requires('impact.entity', 'impact.font', 'game.entities.menu', 'game.entities.levelBox', 'game.entities.timeBox', 'game.entities.scoreBox', 'game.entities.craked', 'game.entities.miss', 'game.entities.quitButton', 'game.entities.quitMessage', 'game.entities.item').defines(function () {
    EntityEngine = ig.Entity.extend({
        menu: null,
        startTime: new ig.Timer(),
        popupTimer: new ig.Timer(),
        badItemCreateTimer: new ig.Timer(),
        currentView: null,
        holes: [],
        popupFlags: [],
        levelBox: null,
        timeBox: null,
        scoreBox: null,
        totalScoreBox: null,
        quitButton: null,
        quitMessage: null,
        life: FinalData.MAX_LIFE,
        bad_life: 0,
        popupInterval: 0,
        stayInterval: 0,
        popdownInterval: 0,
        total_item_count: 0,
        good_item_count: 0,
        bad_item_count: 0,
        popFlag: false,
        badItemFlag: false,
        highScoreFont: new ig.Font(FinalData.img_highscore_font),
        levelCompleteFont: new ig.Font(FinalData.img_levelComplete_font),
        imgBackground: new ig.Image(FinalData.img_game_background),
        imgTimeLevelScore: new ig.Image(FinalData.img_timelevelscore),
        imgLevelComplete: new ig.Image(FinalData.img_levelComplete),
        imgTapToContinue: new ig.Image(FinalData.img_tap_to_continue1),
        imgLife: new ig.Image(FinalData.img_life),
        imgBadLife: new ig.Image(FinalData.img_bad_life),
        imgEmptyBox: new ig.Image(FinalData.img_empty_box),
        imgTotalScoreText: new ig.Image(FinalData.img_totalscore_text),
        imgGameOverText: new ig.Image(FinalData.img_gameover_text),
        imgPlayAgainText: new ig.Image(FinalData.img_playagain_text),
        imgEnterHighScoreText: new ig.Image(FinalData.img_enterhighscore_text),
        imgBackButton: new ig.Image(FinalData.img_back_button),
        init: function (x, y, settings) {
            this.parent(x, y, settings);
            this.startMenu();
        },
        update: function () {
            this.parent();
            switch (this.currentView) {
            case FinalData.MAIN_GAME_VIEW:
                this.mouseCheckInMainGame();
                break;
            case FinalData.QUIT_VIEW:
                this.mouseCheckInQuitView();
                break;
            case FinalData.LEVEL_COMPLETE_VIEW:
                this.mouseCheckInLevelComplete();
                break;
            case FinalData.GAME_OVER_VIEW:
                this.mouseCheckInGameOver();
                break;
            case FinalData.INPUT_NAME_VIEW:
                this.MouseCheckInInputNameView();
            }
        },
        draw: function () {
            this.parent();
            switch (this.currentView) {
            case FinalData.QUIT_VIEW:
                this.imgBackground.draw(this.pos.x, this.pos.y);
                this.drawGameInfo();
                this.imgTimeLevelScore.draw(this.pos.x + FinalData.TIMELEVELSCORE_POS.x, this.pos.y + FinalData.TIMELEVELSCORE_POS.y);
                break;
            case FinalData.MAIN_GAME_VIEW:
                this.imgBackground.draw(this.pos.x, this.pos.y);
                this.drawGameInfo();
                this.imgTimeLevelScore.draw(this.pos.x + FinalData.TIMELEVELSCORE_POS.x, this.pos.y + FinalData.TIMELEVELSCORE_POS.y);
                if (this.timeBox.getTime() == 0) {
                    this.clearGame();
                    this.levelComplete();
                }
                this.playPopup();
                break;
            case FinalData.LEVEL_COMPLETE_VIEW:
                this.imgBackground.draw(this.pos.x, this.pos.y);
                this.imgTimeLevelScore.draw(this.pos.x + FinalData.TIMELEVELSCORE_POS.x, this.pos.y + FinalData.TIMELEVELSCORE_POS.y);
                this.imgLevelComplete.draw(this.pos.x + FinalData.LEVELCOMPLETE_POS.x, this.pos.y + FinalData.LEVELCOMPLETE_POS.y);
                if (ig.ua.iPad) {
                    this.levelCompleteFont.draw(this.levelBox.getLevel(), this.pos.x + FinalData.LEVELCOMPLETE_LEVEL_POS.x, this.pos.y + FinalData.LEVELCOMPLETE_LEVEL_POS.y);
                }
                this.imgTapToContinue.draw(this.pos.x + FinalData.TAP_TO_CONTINUE_POS.x, this.pos.y + FinalData.TAP_TO_CONTINUE_POS.y);
                break;
            case FinalData.GAME_OVER_VIEW:
                this.imgBackground.draw(this.pos.x, this.pos.y);
                this.imgTimeLevelScore.draw(this.pos.x + FinalData.TIMELEVELSCORE_POS.x, this.pos.y + FinalData.TIMELEVELSCORE_POS.y);
                this.imgEmptyBox.draw(this.pos.x, this.pos.y);
                this.imgTotalScoreText.draw(this.pos.x + FinalData.TOTALSCORE_TEXT_POS.x, this.pos.y + FinalData.TOTALSCORE_TEXT_POS.y);
                this.imgPlayAgainText.draw(this.pos.x + FinalData.PLAYAGAINTEXT_POS.x, this.pos.y + FinalData.PLAYAGAINTEXT_POS.y);
                break;
            case FinalData.INPUT_NAME_VIEW:
                this.imgBackground.draw(this.pos.x, this.pos.y);
                this.imgEmptyBox.draw(this.pos.x, this.pos.y);
                this.imgEnterHighScoreText.draw(this.pos.x + FinalData.INPUTNAME_TEXT_POS.x, this.pos.y + FinalData.INPUTNAME_TEXT_POS.y);
                this.imgBackButton.draw(this.pos.x + FinalData.BACK_BUTTON_POS.x, this.pos.y + FinalData.BACK_BUTTON_POS.y);
                this.inputName();
                break;
            }
        },
        inputName: function () {
            var str = ig.$('#highscore').value.toUpperCase();
            var length = str.length;
            this.highScoreFont.draw(str, this.pos.x + FinalData.INPUTNAME_POS.x - length * 5, this.pos.y + FinalData.INPUTNAME_POS.y + FinalData.INPUTNAME_INTENT);
        },
        drawGameInfo: function () {
            var x, y;
            if (this.bad_life >= FinalData.MAX_BAD_LIFE) {
                this.openView(FinalData.GAME_OVER_VIEW);
                return;
            }
            if (this.life <= 0) {
                this.openView(FinalData.GAME_OVER_VIEW);
                return;
            }
            for (i = 0; i < this.life; i++) {
                x = FinalData.LIFE_POS.x - FinalData.LIFE_SIZE.x * i;
                y = FinalData.LIFE_POS.y;
                this.imgLife.draw(x, this.pos.y + y);
            }
            for (i = 0; i < this.bad_life; i++) {
                x = FinalData.BAD_LIFE_POS.x + FinalData.BAD_LIFE_SIZE.x * i;
                y = FinalData.BAD_LIFE_POS.y;
                this.imgBadLife.draw(x, this.pos.y + y);
            }
        },
        mouseCheckInQuitView: function () {
            if (ig.input.pressed('mouse')) {
                var x = ig.input.mouse.x;
                var y = ig.input.mouse.y;
                if (x >= FinalData.QUIT_MESSAGE_POS.x + FinalData.YES_INTENT_POS.x && x <= FinalData.QUIT_MESSAGE_POS.x + FinalData.YES_INTENT_POS.x + FinalData.YES_SIZE.x && y >= FinalData.QUIT_MESSAGE_POS.y + FinalData.YES_INTENT_POS.y && y <= FinalData.QUIT_MESSAGE_POS.y + FinalData.YES_INTENT_POS.y + FinalData.YES_SIZE.y) {
                    this.clearGameInfo();
                    this.openView(this.menu.openView(FinalData.MAIN_MENU_VIEW));
                } else if (x >= FinalData.QUIT_MESSAGE_POS.x + FinalData.NO_INTENT_POS.x && x <= FinalData.QUIT_MESSAGE_POS.x + FinalData.NO_INTENT_POS.x + FinalData.NO_SIZE.x && y >= FinalData.QUIT_MESSAGE_POS.y + FinalData.NO_INTENT_POS.y && y <= FinalData.QUIT_MESSAGE_POS.y + FinalData.NO_INTENT_POS.y + FinalData.NO_SIZE.y) {
                    this.timeBox.startTimeBox();
                    this.openView(FinalData.MAIN_GAME_VIEW);
                    this.quitMessage.kill();
                    for (i = 0; i < FinalData.HOLE_COUNT; i++) {
                        if (this.holes[i] != undefined) this.holes[i].startItem();
                    }
                }
            }
        },
        mouseCheckInMainGame: function () {
            if (ig.input.pressed('mouse')) {
                var x = ig.input.mouse.x;
                var y = ig.input.mouse.y;
                if (x >= FinalData.QUIT_BUTTON_POS.x && x <= FinalData.QUIT_BUTTON_POS.x + FinalData.QUIT_BUTTON_SIZE.x && y >= FinalData.QUIT_BUTTON_POS.y && y <= FinalData.QUIT_BUTTON_INTENT + FinalData.QUIT_BUTTON_POS.y + FinalData.QUIT_BUTTON_SIZE.y) {
                    this.quitMessage = ig.game.spawnEntity(EntityQuitMessage, FinalData.QUIT_MESSAGE_POS.x, FinalData.QUIT_MESSAGE_POS.y);
                    this.openView(FinalData.QUIT_VIEW);
                }
                for (i = 0; i < FinalData.HOLE_COUNT; i++) {
                    if (x >= this.pos.x + FinalData.HOLE_POSX[i] && x <= this.pos.x + FinalData.HOLE_POSX[i] + FinalData.HOLE_SIZE.x && y >= this.pos.y + FinalData.HOLE_POSY[i] && y <= this.pos.y + FinalData.HOLE_POSY[i] + FinalData.HOLE_SIZE.y) {
                        if (this.popupFlags[i] && !this.holes[i].crakeFlag) {
                            this.holes[i].craked();
                            crakedFlag = true;
                        }
                        break;
                    }
                }
            }
        },
        mouseCheckInLevelComplete: function () {
            if (ig.input.pressed('mouse')) {
                if (this.levelBox.getLevel() <= FinalData.MAX_LEVEL) {
                    this.startLevel(this.levelBox.getLevel() + 1);
                } else {
                    this.clearGameInfo();
                    this.menu.getHighScoreData();
                    var score = this.scoreBox.getScore();
                    if ((this.menu.arrHighScore[FinalData.MAX_HIGHSCORE_COUNT - 1] == undefined) || (this.menu.arrHighScore[FinalData.MAX_HIGHSCORE_COUNT - 1] != undefined && score > this.menu.arrHighScore[FinalData.MAX_HIGHSCORE_COUNT - 1].score)) {
                        this.openView(FinalData.INPUT_NAME_VIEW);
                    } else {
                        this.openView(this.menu.openView(FinalData.HIGH_SCORES_VIEW));
                    }
                }
            }
        },
        mouseCheckInGameOver: function () {
            if (ig.input.pressed('mouse')) {
                var x = ig.input.mouse.x;
                var y = ig.input.mouse.y;
                if (x >= this.pos.x + FinalData.PLAYAGAINTEXT_POS.x && x <= this.pos.x + FinalData.PLAYAGAINTEXT_POS.x + FinalData.PLAYAGAINTEXT_SIZE.x && y >= this.pos.y + FinalData.PLAYAGAINTEXT_POS.y && y <= this.pos.y + FinalData.PLAYAGAINTEXT_POS.y + FinalData.PLAYAGAINTEXT_SIZE.y) {
                    this.openView(FinalData.MAIN_GAME_VIEW);
                    this.clearGameInfo();
                    var self = this;
                    adManager.showAd(function () {
                       self.startGame();
                    });
                }
                if (x >= this.pos.x + FinalData.ENTERHIGHSCORE_POS.x && x <= this.pos.x + FinalData.ENTERHIGHSCORE_POS.x + FinalData.ENTERHIGHSCORE_SIZE.x && y >= this.pos.y + FinalData.ENTERHIGHSCORE_POS.y && y <= this.pos.y + FinalData.ENTERHIGHSCORE_POS.y + FinalData.ENTERHIGHSCORE_SIZE.y) {
                    this.clearGameInfo();
                    this.menu.getHighScoreData();
                    var score = this.scoreBox.getScore();
                    if ((this.menu.arrHighScore[FinalData.MAX_HIGHSCORE_COUNT - 1] == undefined) || (this.menu.arrHighScore[FinalData.MAX_HIGHSCORE_COUNT - 1] != undefined && score > this.menu.arrHighScore[FinalData.MAX_HIGHSCORE_COUNT - 1].score)) {
                        this.openView(FinalData.INPUT_NAME_VIEW);
                    } else {
                        this.openView(this.menu.openView(FinalData.HIGH_SCORES_VIEW));
                    }
                }
            }
        },
        MouseCheckInInputNameView: function () {
            if (ig.input.pressed('mouse')) {
                var x = ig.input.mouse.x;
                var y = ig.input.mouse.y;
                if (x >= this.pos.x + FinalData.BACK_BUTTON_POS.x && x <= this.pos.x + FinalData.BACK_BUTTON_POS.x + FinalData.BACK_BUTTON_SIZE.x && y >= this.pos.y + FinalData.BACK_BUTTON_POS.y && y <= this.pos.y + FinalData.BACK_BUTTON_POS.y + FinalData.BACK_BUTTON_SIZE.y) {
                    var name = ig.$('#highscore').value.toUpperCase();
                    if (name == '') {
                        return;
                    }
                    var that = this;
                    var req = $.ajax({
                        url: ("game.php" + '?nocache=' + Math.random()),
                        data: {
                            a: 's',
                            n: name,
                            s: this.scoreBox.getScore()
                        },
                        dataType: 'json',
                        async: false,
                        success: function (scores) {
                            that.arrHighScore = scores;
                        }
                    });
                    document.getElementById('highscore').blur();
                    $('#highscore').css('width', 0);
                    $('#highscore').css('height', 0);
                    $('#highscore').css('z-index', 0);
                    $('#canvas').css('z-index', 10);
                    var elem = document.getElementById("myBody");
                    var elemChild = document.getElementById("highscore");
                    elem.removeChild(elemChild);
                    $('#canvas').focus();
                    this.openView(this.menu.openView(FinalData.HIGH_SCORES_VIEW));
                }
            }
        },
        openView: function (view) {
            switch (view) {
            case FinalData.QUIT_VIEW:
                this.timeBox.stop();
                for (i = 0; i < FinalData.HOLE_COUNT; i++) {
                    if (this.holes[i] != undefined) this.holes[i].stopItem();
                }
                break;
            case FinalData.GAME_OVER_VIEW:
                this.clearTargets();
                this.clearGameInfo();
                this.life = 3;
                this.bad_life = 0;
                this.timeBox.stop();
                this.totalScoreBox = ig.game.spawnEntity(EntityScoreBox, this.pos.x + FinalData.TOTALSCORE_POS.x, this.pos.y + FinalData.TOTALSCORE_POS.y);
                this.totalScoreBox.setScore(this.scoreBox.getScore());
                break;
            case FinalData.INPUT_NAME_VIEW:
                this.createEditBox();
                break;
            }
            this.currentView = view;
        },
        createEditBox: function () {
            var elem = document.getElementById("myBody");
            var newChild = document.createElement("input");
            newChild.type = "text";
            newChild.id = "highscore";
            newChild.setAttribute('maxlength', 8);
            elem.appendChild(newChild);
            newChild.focus();
            ig.$('#highscore').value = '';
            $('#highscore').css('position', 'absolute');
            $('#highscore').css('border', '0px solid');
            $('#highscore').css('left', 0);
            $('#highscore').css('top', 0);
            $('#highscore').css('width', FinalData.SCREEN_WIDTH);
            $('#highscore').css('height', FinalData.EDIT_HEIGHT);
            $('#highscore').css('background-color', 'transparent');
            $('#highscore').css('color', 'transparent');
            $('#highscore').css('z-index', 10);
            $('#canvas').css('z-index', 0);
            ig.$("#highscore").focus();
            $('#highscore').bind('blur', function () {
                window.scrollTo(0, 1);
                FinalData.INPUTNAME_INTENT = 0;
            });
            $('#highscore').bind('focus', function () {
                FinalData.INPUTNAME_INTENT = FinalData.ENTERNAME_TEMP_VALUE;
            });
        },
        startMenu: function () {
            this.menu = ig.game.spawnEntity(EntityMenu, this.pos.x, this.pos.y, {
                engine: this
            });
        },
        startGame: function () {
            this.menu.openView(FinalData.HIDE_SCREEN_VIEW);
            this.bad_life = 0;
            this.life = FinalData.MAX_LIFE;
            this.levelBox = ig.game.spawnEntity(EntityLevelBox, this.pos.x + FinalData.LEVELBOX_POS.x, this.pos.y + FinalData.LEVELBOX_POS.y);
            this.timeBox = ig.game.spawnEntity(EntityTimeBox, FinalData.TIMEBOX_POS.x, this.pos.y + FinalData.TIMEBOX_POS.y);
            this.scoreBox = ig.game.spawnEntity(EntityScoreBox, FinalData.SCOREBOX_POS.x, this.pos.y + FinalData.SCOREBOX_POS.y);
            this.quitButton = ig.game.spawnEntity(EntityQuitButton, FinalData.QUIT_BUTTON_POS.x, FinalData.QUIT_BUTTON_POS.y + FinalData.QUIT_BUTTON_INTENT);
            this.startLevel(1);
        },
        startLevel: function (level) {
            if (level > FinalData.MAX_LEVEL) {
                this.menu.openView(FinalData.MAIN_MENU_VIEW);
                return;
            }
            this.levelBox.setLevel(level);
            this.openView(FinalData.MAIN_GAME_VIEW);
            this.timeBox.start();
            this.total_item_count = FinalData.TOTAL_ITEMS_COUNT[level - 1];
            this.good_item_count = FinalData.GOOD_ITEMS_COUNT[level - 1];
            this.bad_item_count = this.total_item_count - this.good_item_count;
            this.popupInterval = FinalData.INIT_TIME / FinalData.TOTAL_ITEMS_COUNT[level - 1];
            this.stayInterval = FinalData.INITSTAY_TIME;
            this.popdownInterval = FinalData.INIT_POPDOWN;
            this.popupTimer.set(this.popupInterval);
            this.badItemCreateTimer.set(FinalData.INIT_TIME / this.bad_item_count);
            for (i = 0; i < FinalData.HOLE_COUNT; i++) {
                this.popupFlags[i] = false;
            }
        },
        levelComplete: function () {
            this.openView(FinalData.LEVEL_COMPLETE_VIEW);
            this.timeBox.initTimeBox();
            this.clearTargets();
        },
        clearTargets: function () {
            for (i = 0; i < FinalData.HOLE_COUNT; i++) {
                if (this.holes[i] != undefined) ig.game.removeEntity(this.holes[i]);
            }
        },
        clearGameInfo: function () {
            if (this.levelBox != undefined) ig.game.removeEntity(this.levelBox);
            if (this.scoreBox != undefined) ig.game.removeEntity(this.scoreBox);
            if (this.timeBox != undefined) ig.game.removeEntity(this.timeBox);
            if (this.totalScoreBox != undefined) ig.game.removeEntity(this.totalScoreBox);
            if (this.quitButton != undefined) ig.game.removeEntity(this.quitButton);
            if (this.quitMessage != undefined) ig.game.removeEntity(this.quitMessage);
            this.clearGame();
        },
        clearGame: function () {
            for (i = 0; i < ig.game.entities.length; i++) {
                if (ig.game.entities[i] instanceof EntityItem || ig.game.entities[i] instanceof EntityPopupScore || ig.game.entities[i] instanceof EntityStarkill || ig.game.entities[i] instanceof EntityMiss || ig.game.entities[i] instanceof EntityWave) {
                    ig.game.removeEntity(ig.game.entities[i]);
                }
            }
        },
        playPopup: function () {
            if (this.popupTimer.delta() > 0 && this.checHoles()) {
                var r = this.getCreateItemPos();
                var name;
                if (this.badItemCreateTimer.delta() > 0) {
                    this.badItemFlag = false;
                    this.badItemCreateTimer.reset();
                }
                if (Math.random() < 0.5 && this.bad_item_count > 0 && !this.badItemFlag) {
                    var bad_itemPos = Math.floor(Math.random() * FinalData.LEVEL_BAD_ITEMS[this.levelBox.getLevel() - 1].length);
                    name = FinalData.LEVEL_BAD_ITEMS[this.levelBox.getLevel() - 1][bad_itemPos];
                    this.bad_item_count--;
                    this.badItemFlag = true;
                } else {
                    var good_itemPos = Math.floor(Math.random() * FinalData.LEVEL_GOOD_ITEMS[this.levelBox.getLevel() - 1].length);
                    name = FinalData.LEVEL_GOOD_ITEMS[this.levelBox.getLevel() - 1][good_itemPos];
                }
                var x = FinalData.HOLE_POSX[r];
                var y = FinalData.HOLE_POSY[r];
                this.popupFlags[r] = true;
                this.holes[r] = ig.game.spawnEntity(EntityItem, x, this.pos.y + y, {
                    name: name,
                    engine: this,
                    hole_pos: r
                });
                this.popupTimer.reset();
            }
        },
        freeHole: function (pos) {
            this.popupFlags[pos] = false;
        },
        getCreateItemPos: function () {
            while (true) {
                var r = Math.floor(Math.random() * FinalData.HOLE_COUNT);
                if (!this.popupFlags[r]) {
                    return r;
                }
            }
        },
        checHoles: function () {
            for (i = 0; i < FinalData.HOLE_COUNT; i++) {
                if (!this.popupFlags[i]) {
                    return true;
                }
            }
            return false;
        }
    });
});

// lib/game/main.js
ig.baked = true;
ig.module('game.main').requires('impact.game', 'game.entities.finalData', 'game.entities.engine', 'impact.font').defines(function () {
    MyGame = ig.Game.extend({
        engine: null,
        bStart: false,
        init: function () {
            if (ig.ua.android) {
                window.scrollTo(0, 8);
            } else {
                setTimeout(function () {
                    window.scrollTo(0, 0);
                }, 1);
            }
            ig.input.bind(ig.KEY.MOUSE1, 'mouse');
            ig.input.initMouse();
        },
        update: function () {
            this.parent();
        },
        draw: function () {
            this.parent();
            if (!this.bStart) {
                this.engine = this.spawnEntity(EntityEngine, 0, FinalData.TOP_Y);
                this.bStart = true;
            }
        }
    });
    ig.main('#canvas', MyGame, 30, FinalData.SCREEN_WIDTH, FinalData.SCREEN_HEIGHT, 1);
});