'use strict';

function ContactItem(contact)
{
	this.contact = contact;
};
ContactItem.prototype =
{
	SerializePim: function (onsuccess)
	{
		var s = "<contact xmlns=\"http://restsync.org/ns/pim/contact/1/\">";

		function wrElement(name, value)
		{
			if (value)
			{
				value = value.toString();
				if (value)
					s += "<" + name + ">" + Utils.XmlEncode(value) + "</" + name + ">";
			}
		}

		function getType(type)
		{
			if (!type) return "other";
			switch (type)
			{
				case "faxHome": case "faxOffice": case "faxOther":
					return "fax";
				case "mobile":
					return "cell";
				case "personal": case "current":
					return "home";
				default:
					return type;
			}
		}

		function wrTypePref(name, obj)
		{
			s += "<" + name + ' pref="' + (obj.pref ? "true" : "false") +
				'" type="' + (obj.type ? obj.type.map(getType) : "other") + '">';
		}

		function wrTypePrefText(name, list)
		{
			if (list)
				list.forEach(function (obj)
				{
					wrTypePref(name, obj);
					s += Utils.XmlEncode(obj.value) + "</" + name + ">";
				});
		}

		var c = this.contact;

		if (c.anniversary) s += '<metadata xmlns="http://restsync.org/ns/pim/metadata/1/">' +
			'<parameter name="VCARD_X-MS-ANNIVERSARY">' + Utils.DateToYYYMMDD(c.anniversary) +
			'</parameter></metadata>';

		wrElement("formattedname", c.name);
		s += "<name>";
		wrElement("family", c.familyName);
		wrElement("given", c.givenName);
		wrElement("other", c.additionalName);
		wrElement("prefix", c.honorificPrefix);
		wrElement("suffix", c.honorificSuffix);
		wrElement("nick", c.nickname);
		wrElement("title", c.jobTitle);
		s += "</name>";

		wrTypePrefText("phone", c.tel);
		wrTypePrefText("email", c.email);

		if (c.adr) c.adr.forEach(function (a)
		{
			wrTypePref("address", a);
			wrElement("pobox", "");
			wrElement("extended", "");
			wrElement("street", a.streetAddress);
			wrElement("city", a.locality);
			wrElement("region", a.region);
			wrElement("code", a.postalCode);
			wrElement("country", a.countryName);
			s += "</address>";
		});

		if (c.bday) wrElement("birthday", c.bday.toISOString());

		wrTypePrefText("url", c.url);

		if (c.org) s += "<org><name>" + Utils.XmlEncode(c.org[0]) + "</name></org>";

		if (c.note) wrElement("note", c.note.join("\r\n"));

		function endElement()
		{
			onsuccess(s + "</contact>");
		};

		if (c.photo && c.photo.length > 0)
		{
			Utils.ToBase64String(c.photo[0],
			function (data)
			{
				wrElement("photo", data);
				endElement();
			});
		}
		else
		{
			endElement();
		}

		//TODO: category,genderIdentity,impp,key,published,sex,updated
	},


	DeserializePim: function (x, c)
	{
		function setType(type)
		{
			switch (type)
			{
				case "fax":
					return "faxOther";
				case "cell":
					return "mobile";
				default:
					return type;
			}
		}

		function rdTypePref(x, list)
		{
			var type = Utils.XmlAttribute(x, "type");
			var field = {
				type: (type ? type.split(',').map(setType) : []),
				pref: Utils.XmlAttribute(x, "pref") == "true"
			};
			list.push(field);
			return field;
		}

		function rdTypePrefText(x, list)
		{
			var field = rdTypePref(x, list);
			field.value = Utils.XmlNodeText(x);
		}

		var c0, c1, i0, i1, x1, x2, value, value2;

		c0 = Utils.ChildNodes(x);
		for (i0 = 0; i0 < c0.length; i0++)
		{
			x1 = c0[i0];
			value = Utils.XmlNodeText(x1);
			switch (Utils.XmlNodeName(x1))
			{
				case "metadata":
					c1 = Utils.ChildNodes(x1);
					for (i1 = 0; i1 < c1.length; i1++)
					{
						x2 = c1[i1];
						if (Utils.XmlNodeName(x2) == "parameter")
						{
							value2 = Utils.XmlNodeText(x2);
							switch (x2.getAttribute("name"))
							{
								case "VCARD_X-MS-ANNIVERSARY":
									var d = Utils.ParseISO8601(value2);
									if (d) c.anniversary = d;
									break;
							}
						}
					}
					break;
				case "formattedname":
					c.name = [value];
					break;
				case "name":
					c1 = Utils.ChildNodes(x1);
					for (i1 = 0; i1 < c1.length; i1++)
					{
						x2 = c1[i1];
						value2 = [Utils.XmlNodeText(x2)];
						switch (Utils.XmlNodeName(x2))
						{
							case "family":
								c.familyName = value2;
								break;
							case "given":
								c.givenName = value2;
								break;
							case "other":
								c.additionalName = value2;
								break;
							case "prefix":
								c.honorificPrefix = value2;
								break;
							case "suffix":
								c.honorificSuffix = value2;
								break;
							case "nick":
								c.nickname = value2;
								break;
							case "title":
								c.jobTitle = value2;
								break;
						}
					}
					break;
				case "phone":
					if (!c.tel) c.tel = [];
					rdTypePrefText(x1, c.tel);
					break;
				case "email":
					if (!c.email) c.email = [];
					rdTypePrefText(x1, c.email);
					break;
				case "address":
					if (!c.adr) c.adr = [];
					var adr = rdTypePref(x1, c.adr);
					c1 = Utils.ChildNodes(x1);
					for (i1 = 0; i1 < c1.length; i1++)
					{
						x2 = c1[i1];
						value2 = Utils.XmlNodeText(x2);
						switch (Utils.XmlNodeName(x2))
						{
							case "street":
								adr.streetAddress = value2;
								break;
							case "city":
								adr.locality = value2;
								break;
							case "region":
								adr.region = value2;
								break;
							case "code":
								adr.postalCode = value2;
								break;
							case "country":
								adr.countryName = value2;
								break;
						}
					}
					break;
				case "birthday":
					var d = Utils.ParseISO8601(value);
					if (d) c.bday = d;
					break;
				case "url":
					if (!c.url) c.url = [];
					rdTypePrefText(x1, c.url);
					break;
				case "org":
					x2 = Utils.ChildNodes(x1)[0];
					if (x2 && Utils.XmlNodeName(x2) == "name")
						c.org = [Utils.XmlNodeText(x2)];
					break;
				case "note":
					c.note = value.split("\r\n");
					break;
				case "photo":
					c.photo = [Utils.FromBase64String(value)];
					break;
			}
		}
	},

	Reset: function(c)
	{
		c.anniversary = null;
		c.name = null;
		c.familyName = null;
		c.givenName = null;
		c.additionalName = null;
		c.honorificPrefix = null;
		c.honorificSuffix = null;
		c.nickname = null;
		c.jobTitle = null;
		c.tel = null;
		c.email = null;
		c.adr = null;
		c.bday = null;
		c.url = null;
		c.org = null;
		c.note = null;
		c.photo = null;
	},

	Serialize: function (modificationNumber, onsuccess)
	{
		var _this = this;
		this.SerializePim(
		function (pim)
		{
			onsuccess('<modification number="' + modificationNumber + '" type="created"><luid>' +
				_this.contact.id + '</luid><pim xmlns="http://restsync.org/ns/pim/pim/1/">' +
				pim + '</pim></modification>');
		});
	},


	EstimateSize: function ()
	{
		var c = this.contact;
		return (c.photo && c.photo.length > 0) ? 40000 : 1000;
	}
};
