// This file expects to be invoked as a Worker (see onmessage below).
importScripts('fileHandler.js');
importScripts('stringView.js');

// Helper functions.
var SendError = function(str) {
    postMessage(new fileHandler.importer.ImportErrorEvent(str));
    // we do not recover from errors
    self.close();
};
var SendWarning = function (str) {
    postMessage(new fileHandler.importer.ImportWarningEvent(str));
};
var SendProgress = function (recordCnt, percentDone) {
    postMessage(new fileHandler.importer.ImportProgressEvent(recordCnt, percentDone));
};
var SendData = function (data) {
    postMessage(new fileHandler.importer.ImportDataEvent(data));
};
var SendSuccess = function (recordCnt) {
    postMessage(new fileHandler.importer.ImportSuccessEvent(recordCnt));
};

// Takes an ArrayBuffer of a zip file in
// returns null on error
// returns an array of DecompressedFile objects on success
var importContacts = function (arrayBuffer) {

    function addAddressElement(contact, type, property, value) {
        if (undefined === contact.adr)
            contact.adr = [];
        var index;
        for (var i = 0; i < contact.adr.length; i++) {
            if (0 <= contact.adr[i].type.indexOf(type)) {
                index = i;
                break;
            }
        }
        if (undefined === index) {
            contact.adr.push({ type: [], pref: false, streetAddress: "", locality: "", region: "", postalCode: "", countryName: "" });
            index = 0;
        }
        contact.adr[index][property] = value;
    }

    function addContactField(contactField, offset, type, value) {

        function addcontactEntries(contactField, Type, Count) {
            for (var i = 0; i < Count; i++) {
                contactField.push({ type: [], value: "", pref: false });
                contactField[contactField.length - 1].type = Type;
            }
            return contactField;
        }

        if (undefined === contactField)
            contactField = [];
        var index;
        var cnt = offset;
        for (var i = 0; i < contactField.length; i++) {
            if ((0 <= contactField[i].type.indexOf("")) || (0 <= contactField[i].type.indexOf(type))) {
                if (cnt-- === 0) {
                    index = i;
                    break;
                }
            }
        }
        if (undefined === index) {
            index = contactField.length + offset;
            contactField = addcontactEntries(contactField, type, 3);
        }
        contactField[index].value = value;
        return contactField;
    }

    /**
    * Translate objects created from PIM contacts files to mozContacts
    *
    * @contact mozConact
    * @key string denominator as named by Microsoft
    * @value string value as read from the file record
    *
    * ContactsAPI:
    * @see https://wiki.mozilla.org/WebAPI/ContactsAPI
    * @spec http://www.w3.org/TR/contacts-manager-api/
    *
    * Microsoft vCard mapping:
    * @see http://msdn.microsoft.com/en-us/library/dd633489(v=exchg.80).aspx
    * @see http://download.microsoft.com/download/5/D/D/5DD33FDF-91F5-496D-9884-0A0B0EE698BB/[MS-OXVCARD].pdf
    */
    function addToContact(contact, key, value) {

        var unmappedKeys = [
            "Ring Tone",            // TODO: Didn't find an equivalent in Mozilla interface
            "Assistant's Name",     // TODO: FN is not implemented with type, so we have no tag to store at
            "Manager's Name",       // TODO: X-MS-ASSISTANT is not implemented in Mozilla interface
            "Government ID Number", // TODO: not implemented in Mozilla interface
            "Account",              // TODO: not implemented in Mozilla interface
            "Customer ID Number",   // TODO: not implemented in Mozilla interface
            "Last Name (Yomi)",     // TODO: FN is not implemented with type, so we have no tag to store at
            "First Name (Yomi)",    // TODO: FN is not implemented with type, so we have no tag to store at
            "Company (Yomi)"        // TODO: FN is not implemented with type, so we have no tag to store at
        ];

        if (0 <= unmappedKeys.indexOf(key)) {
            var name = (contact.name !== undefined) ? contact.name[0] : (contact.familyName !== undefined) ? contact.familyName[0] : (contact.firstName !== undefined) ? contact.firstName[0] : "";
            SendWarning("Could not store " + key + " for: " + name);
            return;
        }

        if (key == "Name") {
            // this is used for sorting, if Display Name also exists, it will override Name
            if (contact.name === undefined)
                contact.name = [value];
        } else if (key == "Title") {
            contact.honorificPrefix = [value];
        } else if (key == "First Name") {
            contact.givenName = [value];
        } else if (key == "Middle Name") {
            contact.additionalName = [value];
        } else if (key == "Last Name") {
            contact.familyName = [value];
        } else if (key === "NickName") {
            contact.nickname = [value];
        } else if (key === "Suffix") {
            contact.honorificSuffix = [value];
        } else if (key === "Display Name") {
            // Maybe it is already set from Display Name, but we will override it
            contact.name = [value];

        } else if (key === "Job Title") {
            contact.jobTitle = [value];
        } else if (key === "Company") {
            if (undefined === contact.org)
                contact.org = ["", "", ""];
            contact.org[0] = value;
        } else if (key === "Department") {
            if (undefined === contact.org)
                contact.org = ["", "", ""];
            contact.org[1] = value;
        } else if (key === "Categories") {
            contact.category = value.split(",");
        } else if (key === "Picture") {
            var u8 = new Uint8Array(value.match(/0x[0-9A-F][0-9A-F]/ig).map(function hex2int(hex) { return parseInt(hex) }));
            contact.photo = [new Blob([u8], { type: "image/jpeg" })];
        } else if (key === "E-mail Address") {
            contact.email = addContactField(contact.email, 0, "internet", value);
        } else if (key === "E-mail 2 Address") {
            contact.email = addContactField(contact.email, 1, "internet", value);
        } else if (key === "E-mail 3 Address") {
            contact.email = addContactField(contact.email, 2, "internet", value);
        } else if (key === "IM") {
            contact.email = addContactField(contact.impp, 0, "im", value);
        } else if (key === "IM2") {
            contact.email = addContactField(contact.impp, 1, "im", value);
        } else if (key === "IM3") {
            contact.email = addContactField(contact.impp, 2, "im", value);
        } else if (key === "Web Page") {
            contact.email = addContactField(contact.url, 0, "home", value);
        } else if (key === "Office Location") {

        } else if (key === "Business Street") {
            addAddressElement(contact, "work", "streetAddress", value);
        } else if (key === "Business City") {
            addAddressElement(contact, "work", "locality", value);
        } else if (key === "Business State") {
            addAddressElement(contact, "work", "region", value);
        } else if (key === "Business Postal Code") {
            addAddressElement(contact, "work", "postalCode", value);
        } else if (key === "Business Country") {
            addAddressElement(contact, "work", "countryName", value);
        } else if (key === "Home Street") {
            addAddressElement(contact, "home", "streetAddress", value);
        } else if (key === "Home City") {
            addAddressElement(contact, "home", "locality", value);
        } else if (key === "Home State") {
            addAddressElement(contact, "home", "region", value);
        } else if (key === "Home Postal Code") {
            addAddressElement(contact, "home", "postalCode", value);
        } else if (key === "Home Country") {
            addAddressElement(contact, "home", "countryName", value);
        } else if (key === "Other Street") {
            addAddressElement(contact, "postal", "streetAddress", value);
        } else if (key === "Other City") {
            addAddressElement(contact, "postal", "locality", value);
        } else if (key === "Other State") {
            addAddressElement(contact, "postal", "region", value);
        } else if (key === "Other Postal Code") {
            addAddressElement(contact, "postal", "postalCode", value);
        } else if (key === "Other Country") {
            addAddressElement(contact, "postal", "countryName", value);

        } else if (key === "Mobile Phone") {
            contact.tel = addContactField(contact.tel, 0, "cell", value);
        } else if (key === "Home Phone") {
            contact.tel = addContactField(contact.tel, 0, "home", value);
        } else if (key === "Home Phone 2") {
            contact.tel = addContactField(contact.tel, 1, "home", value);
        } else if (key === "Business Phone") {
            contact.tel = addContactField(contact.tel, 0, "work", value);
        } else if (key === "Business Phone 2") {
            contact.tel = addContactField(contact.tel, 1, "work", value);
        } else if (key === "Car Phone") {
            contact.tel = addContactField(contact.tel, 0, "car", value);
        } else if (key === "Pager") {
            contact.tel = addContactField(contact.tel, 0, "pager", value);
        } else if (key === "Assistant's Phone") {
            contact.tel = addContactField(contact.tel, 0, "assistent", value);
        } else if (key === "Company Main Phone") {
            contact.tel = addContactField(contact.tel, 0, "company", value);
        } else if (key === "Radio Phone") {
            contact.tel = addContactField(contact.tel, 0, "radio", value);

        } else if (key === "Home Fax") {
            contact.tel = addContactField(contact.tel, 0, "home;fax", value);
        } else if (key === "Business Fax") {
            contact.tel = addContactField(contact.tel, 0, "work;fax", value);

        } else if (key === "Birthday") {
            contact.bday = new Date(value.substr(0, 4), value.substr(0, 2), value.substr(0, 2), 0, 0, 0, 0);
        } else if (key === "Anniversary") {
            contact.bday = new Date(value.substr(0, 4), value.substr(0, 2), value.substr(0, 2), 0, 0, 0, 0);
        } else if (key === "Spouse") {
            // TODO: Didn't find an equivalent in Mozilla interface
        } else if (key === "Children") {
            // TODO: Didn't find an equivalent in Mozilla interface
        } else if (key === "Notes") {
            contact.note = [value];
        } else if (key === "Notes Ink") {
            // encoded duplicate of Notes. Ignored for now.
            var string = value;
        } else {
            var name = (contact.name !== undefined) ? contact.name[0] : (contact.familyName !== undefined) ? contact.familyName[0] : (contact.firstName !== undefined) ? contact.firstName[0] : "";
            SendWarning("Unmatched key for " + name + ": " + key);
        }
    }

    var inStr = arrayBuffer;
    var addedRecords = 0;

    // Looks like StringView implementation of startOffset is buggy, in addition it doesn't handle BOM, so skip BOM manually later!
    var str16 = new StringView(inStr.buffer, "UTF-16");

    // StringView can't handle BOM
    //var BOMCode = StringView.loadUTF16CharCode(str16.rawData, 0);

    // Use byte access to be really sure about byte order
    var byteOrderMark = new Uint8Array(inStr, 2);
    if ((byteOrderMark[0] === parseInt('FE', 16)) && (byteOrderMark[1] === parseInt('FF', 16))) {
        SendError("Can't handle UTF-16 Big Endian");
        return null;
    }
    if ((byteOrderMark[0] !== parseInt('FF', 16)) || (byteOrderMark[1] !== parseInt('FE', 16))) {
        SendError("Can't handle unknown UTF-16 encoding");
        return null;
    }
    //console.log("Little Endian");

    var textLngWord = str16.rawData.length;
    var contactBuffer = "";
    var chunk = 1100;
    var tokenStack = [];
    var linePattern = /(?:([^;]+(?:;[^;]+?)*?)\r\n)+?/mg;

    // Skip BOM: i = 1
    for (var nChrCode, i = 1, nChrIdx = 0; i < textLngWord; i += StringView.getUTF16CharLength(nChrCode), nChrIdx++) {
        nChrCode = StringView.loadUTF16CharCode(str16.rawData, i);
        contactBuffer += String.fromCharCode(nChrCode);

        if ((nChrIdx === chunk) || (i === (textLngWord - 1))) {
            nChrIdx = 0;
            //console.log("contactBuffer: " + contactBuffer);
            var matches;
            while (null !== (matches = linePattern.exec(contactBuffer))) {
                var lineStr = matches[1];
                //console.log("lineStr: " + lineStr);
                //console.log("[2]: " + matches[2]);
                //console.log("lastIndex: %d", linePattern.lastIndex);
                contactBuffer = contactBuffer.substr(linePattern.lastIndex);
                //console.log("Buffer: "+contactBuffer);
                linePattern.lastIndex = 0;
                if (tokenStack.length == 0) {
                    tokenStack = lineStr.split(/;/);
                    //console.log(tokenStack.join("\n"));
                    linePattern = /^("(?:[^"])*?"(?:(;"(?:[^"])*?"))*?)\r\n/mg;
                } else {
                    //console.log("replaced: " + lineStr.replace(/^"\0((?:.|\r|\n)+)"\0/m, '>$1<') + " ");
                    //                    var tokens = lineStr.replace(/^"\0((?:.|\r|\n)+)"\0/m, '$1').split("\"\0;\0\"\0");
                    var tokens = lineStr.replace(/^"((?:.|\r|\n)+)"/m, '$1').split("\";\"");
                    //console.log(tokens.join());
                    if (tokenStack.length != tokens.length) {
                        console.log("TokenCnt: does not match %d (%d) RecordCnt: %d", tokens.length, tokenStack.length, addedRecords);
                        console.log("lineStr: " + lineStr);
                        console.log("contactBuffer: " + contactBuffer);
                    } else {
                        var contact = {}; // new mozContact();
                        tokens.forEach(function (value, index) {
                            //console.log("Index: %d Key: %s Value: %s", index, this[index], value);
                            if (value !== "")
                                addToContact(contact, this[index], value);
                        }, tokenStack);
                        //                            console.log(contact.name[0] );

                        //                            //((contact.familyName instanceof Array) && (contact.familyName.length > 0))
                        //                            var options = ((contact.familyName instanceof Array) && (contact.familyName.length > 0)) ?
                        //                                    {filterBy: [ "familyName" ],
                        //                                           filterOp: "equals",
                        //                                           filterValue: contact.familyName[0],
                        //                                           sortBy: "givenName",
                        //                                           sortOrder: "ascending"
                        //                                } : {filterBy: [ "name" ],
                        //                                           filterOp: "equals",
                        //                                           filterValue: contact.name[0],
                        //                                           sortBy: "givenName",
                        //                                           sortOrder: "ascending"
                        //                            };

                        //                            var request = navigator.mozContacts.find(options);

                        //                            request.onsuccess = function() {
                        //                                if(request.result.length > 0) {
                        //                                    console.log("Found " + request.result.length + " contacts");
                        //                                    console.log(request);
                        //                                    for (var i = 0; i<request.result.length; i++) {
                        //                                        for (var j=0; j<request.result[i].familyName.length; j++) {
                        //                                           console.log("Found contact with familyName[%d][%d]: " + request.result[i].familyName[j], i, j);
                        //                                        }
                        //                                        console.log("Dump request.result[%d]", i);
                        //                                        console.log(request.result[i]);
                        //                                        logContact(request.result[i]);
                        //                                    }
                        //                                    if( request.result.length == 1)
                        //                                        contact.id = request.result[0].id;
                        //                                }
                        //                            };

                        //                            request.onerror = function(e) {
                        //                                console.error('Error while trying to do the matching: '+ e.target.error.name);
                        //                                logObject(e);
                        //                            };
                        SendData( contact);
                        SendProgress(++addedRecords, (i + chunk - contactBuffer.length) / textLngWord);
                    }
                    //console.log("Buffer: "+contactBuffer);
                };
                //console.log("Buffer: "+contactBuffer);
                //console.log(tokenStack.join());
                //console.log(tokenStack.length);
            }
        }
        //console.log("Buffer: "+contactBuffer);
        //console.log("StackLng: "+tokenStack.length);
    }
    if (contactBuffer.length > 0)
        console.log("BufferRest: %d " + contactBuffer, contactBuffer.length);
    SendSuccess(addedRecords);
};

// event.data.file has the ArrayBuffer.
onmessage = function (event) {
    importContacts(event.data.file, true);
    // every worker handles only a single file / request
    self.close();
};
